/*
 * Decompiled with CFR 0.152.
 */
package org.janusgraph.diskstorage.locking.consistentkey;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.janusgraph.core.JanusGraphConfigurationException;
import org.janusgraph.diskstorage.BackendException;
import org.janusgraph.diskstorage.Entry;
import org.janusgraph.diskstorage.PermanentBackendException;
import org.janusgraph.diskstorage.StaticBuffer;
import org.janusgraph.diskstorage.TemporaryBackendException;
import org.janusgraph.diskstorage.configuration.ConfigElement;
import org.janusgraph.diskstorage.configuration.Configuration;
import org.janusgraph.diskstorage.keycolumnvalue.KeyColumnValueStore;
import org.janusgraph.diskstorage.keycolumnvalue.KeySliceQuery;
import org.janusgraph.diskstorage.keycolumnvalue.StoreManager;
import org.janusgraph.diskstorage.keycolumnvalue.StoreTransaction;
import org.janusgraph.diskstorage.locking.AbstractLocker;
import org.janusgraph.diskstorage.locking.LocalLockMediator;
import org.janusgraph.diskstorage.locking.LocalLockMediators;
import org.janusgraph.diskstorage.locking.Locker;
import org.janusgraph.diskstorage.locking.LockerState;
import org.janusgraph.diskstorage.locking.PermanentLockingException;
import org.janusgraph.diskstorage.locking.TemporaryLockingException;
import org.janusgraph.diskstorage.locking.consistentkey.ConsistentKeyLockStatus;
import org.janusgraph.diskstorage.locking.consistentkey.ConsistentKeyLockerSerializer;
import org.janusgraph.diskstorage.locking.consistentkey.ExpiredLockException;
import org.janusgraph.diskstorage.locking.consistentkey.LockCleanerService;
import org.janusgraph.diskstorage.locking.consistentkey.StandardLockCleanerService;
import org.janusgraph.diskstorage.locking.consistentkey.TimestampRid;
import org.janusgraph.diskstorage.util.BufferUtil;
import org.janusgraph.diskstorage.util.KeyColumn;
import org.janusgraph.diskstorage.util.StandardBaseTransactionConfig;
import org.janusgraph.diskstorage.util.StaticArrayBuffer;
import org.janusgraph.diskstorage.util.StaticArrayEntry;
import org.janusgraph.diskstorage.util.time.Timer;
import org.janusgraph.diskstorage.util.time.TimestampProvider;
import org.janusgraph.graphdb.configuration.GraphDatabaseConfiguration;
import org.janusgraph.util.encoding.StringEncoding;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsistentKeyLocker
extends AbstractLocker<ConsistentKeyLockStatus>
implements Locker {
    private final KeyColumnValueStore store;
    private final StoreManager manager;
    private final Duration lockWait;
    private final int lockRetryCount;
    private final LockCleanerService cleanerService;
    private static final StaticBuffer zeroBuf = BufferUtil.getIntBuffer(0);
    public static final StaticBuffer LOCK_COL_START = BufferUtil.zeroBuffer(1);
    public static final StaticBuffer LOCK_COL_END = BufferUtil.oneBuffer(9);
    private static final Logger log = LoggerFactory.getLogger(ConsistentKeyLocker.class);

    private ConsistentKeyLocker(KeyColumnValueStore store, StoreManager manager, StaticBuffer rid, TimestampProvider times, ConsistentKeyLockerSerializer serializer, LocalLockMediator<StoreTransaction> llm, Duration lockWait, int lockRetryCount, Duration lockExpire, LockerState<ConsistentKeyLockStatus> lockState, LockCleanerService cleanerService) {
        super(rid, times, serializer, llm, lockState, lockExpire, log);
        this.store = store;
        this.manager = manager;
        this.lockWait = lockWait;
        this.lockRetryCount = lockRetryCount;
        this.cleanerService = cleanerService;
    }

    @Override
    protected ConsistentKeyLockStatus writeSingleLock(KeyColumn lockID, StoreTransaction txh) throws Throwable {
        StaticBuffer lockKey = this.serializer.toLockKey(lockID.getKey(), lockID.getColumn());
        StaticBuffer oldLockCol = null;
        for (int i = 0; i < this.lockRetryCount; ++i) {
            WriteResult wr = this.tryWriteLockOnce(lockKey, oldLockCol, txh);
            if (wr.isSuccessful() && wr.getDuration().compareTo(this.lockWait) <= 0) {
                Instant writeInstant = wr.getWriteTimestamp();
                Instant expireInstant = writeInstant.plus(this.lockExpire);
                return new ConsistentKeyLockStatus(writeInstant, expireInstant);
            }
            oldLockCol = wr.getLockCol();
            this.handleMutationFailure(lockID, lockKey, wr, txh);
        }
        this.tryDeleteLockOnce(lockKey, oldLockCol, txh);
        throw new TemporaryBackendException("Lock write retry count exceeded");
    }

    /*
     * Enabled aggressive block sorting
     */
    private void handleMutationFailure(KeyColumn lockID, StaticBuffer lockKey, WriteResult wr, StoreTransaction txh) throws Throwable {
        Throwable error = wr.getThrowable();
        if (null == error) {
            log.warn("Lock write succeeded but took too long: duration {} exceeded limit {}", (Object)wr.getDuration(), (Object)this.lockWait);
            return;
        }
        if (error instanceof TemporaryBackendException) {
            log.warn("Temporary exception during lock write", error);
            return;
        }
        log.error("Fatal exception encountered during attempted lock write", error);
        WriteResult dwr = this.tryDeleteLockOnce(lockKey, wr.getLockCol(), txh);
        if (dwr.isSuccessful()) throw error;
        log.warn("Failed to delete lock write: abandoning potentially-unreleased lock on {}", (Object)lockID, (Object)dwr.getThrowable());
        throw error;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private WriteResult tryWriteLockOnce(StaticBuffer key, StaticBuffer del, StoreTransaction txh) {
        BackendException t = null;
        Timer writeTimer = this.times.getTimer().start();
        StaticBuffer newLockCol = this.serializer.toLockCol(writeTimer.getStartTime(), this.rid, this.times);
        Entry newLockEntry = StaticArrayEntry.of(newLockCol, zeroBuf);
        StoreTransaction newTx = null;
        try {
            newTx = this.overrideTimestamp(txh, writeTimer.getStartTime());
            this.store.mutate(key, Collections.singletonList(newLockEntry), null == del ? KeyColumnValueStore.NO_DELETIONS : Collections.singletonList(del), newTx);
            newTx.commit();
            newTx = null;
            this.rollbackIfNotNull(newTx);
        }
        catch (BackendException e) {
            try {
                log.debug("Lock write attempt failed with exception", (Throwable)e);
                t = e;
                this.rollbackIfNotNull(newTx);
            }
            catch (Throwable throwable) {
                this.rollbackIfNotNull(newTx);
                throw throwable;
            }
        }
        writeTimer.stop();
        return new WriteResult(writeTimer.elapsed(), writeTimer.getStartTime(), newLockCol, t);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private WriteResult tryDeleteLockOnce(StaticBuffer key, StaticBuffer col, StoreTransaction txh) {
        BackendException t = null;
        Timer delTimer = this.times.getTimer().start();
        StoreTransaction newTx = null;
        try {
            newTx = this.overrideTimestamp(txh, delTimer.getStartTime());
            this.store.mutate(key, Collections.emptyList(), Collections.singletonList(col), newTx);
            newTx.commit();
            newTx = null;
            this.rollbackIfNotNull(newTx);
        }
        catch (BackendException e) {
            try {
                t = e;
                this.rollbackIfNotNull(newTx);
            }
            catch (Throwable throwable) {
                this.rollbackIfNotNull(newTx);
                throw throwable;
            }
        }
        delTimer.stop();
        return new WriteResult(delTimer.elapsed(), delTimer.getStartTime(), null, t);
    }

    @Override
    protected void checkSingleLock(KeyColumn kc, ConsistentKeyLockStatus ls, StoreTransaction tx) throws BackendException, InterruptedException {
        if (ls.isChecked()) {
            return;
        }
        Instant now = this.times.sleepPast(ls.getWriteTimestamp().plus(this.lockWait));
        KeySliceQuery ksq = new KeySliceQuery(this.serializer.toLockKey(kc.getKey(), kc.getColumn()), LOCK_COL_START, LOCK_COL_END);
        List<Entry> claimEntries = this.getSliceWithRetries(ksq, tx);
        Iterable iterable = Iterables.transform(claimEntries, e -> this.serializer.fromLockColumn(e.getColumnAs(StaticBuffer.STATIC_FACTORY), this.times));
        ArrayList<TimestampRid> unexpiredTRs = new ArrayList<TimestampRid>(Iterables.size((Iterable)iterable));
        Instant cutoffTime = now.minus(this.lockExpire);
        for (TimestampRid tr : iterable) {
            if (tr.getTimestamp().isBefore(cutoffTime)) {
                log.warn("Discarded expired claim on {} with timestamp {}", (Object)kc, (Object)tr.getTimestamp());
                if (null != this.cleanerService) {
                    this.cleanerService.clean(kc, cutoffTime, tx);
                }
                if (!this.rid.equals(tr.getRid()) || !ls.getWriteTimestamp().equals(tr.getTimestamp())) continue;
                throw new ExpiredLockException("Expired lock on " + kc + ": lock timestamp " + tr.getTimestamp() + " " + this.times.getUnit() + " is older than " + ConfigElement.getPath(GraphDatabaseConfiguration.LOCK_EXPIRE, new String[0]) + "=" + this.lockExpire);
            }
            unexpiredTRs.add(tr);
        }
        this.checkSeniority(kc, ls, unexpiredTRs);
        ls.setChecked();
    }

    private List<Entry> getSliceWithRetries(KeySliceQuery ksq, StoreTransaction tx) throws BackendException {
        for (int i = 0; i < this.lockRetryCount; ++i) {
            try {
                return this.store.getSlice(ksq, tx);
            }
            catch (PermanentBackendException e) {
                log.error("Failed to check locks", (Throwable)e);
                throw new PermanentLockingException(e);
            }
            catch (TemporaryBackendException e) {
                log.warn("Temporary storage failure while checking locks", (Throwable)e);
                continue;
            }
        }
        throw new TemporaryBackendException("Maximum retries (" + this.lockRetryCount + ") exceeded while checking locks");
    }

    private void checkSeniority(KeyColumn target, ConsistentKeyLockStatus ls, Iterable<TimestampRid> claimTRs) throws BackendException {
        int trCount = 0;
        for (TimestampRid tr : claimTRs) {
            ++trCount;
            if (!this.rid.equals(tr.getRid())) {
                String msg = "Lock on " + target + " already held by " + tr.getRid() + " (we are " + this.rid + ")";
                log.debug(msg);
                throw new TemporaryLockingException(msg);
            }
            if (tr.getTimestamp().equals(ls.getWriteTimestamp())) {
                log.debug("Checked lock {}", (Object)target);
                return;
            }
            log.warn("Skipping outdated lock on {} with our rid ({}) but mismatched timestamp (actual ts {}, expected ts {})", new Object[]{target, tr.getRid(), tr.getTimestamp(), ls.getWriteTimestamp()});
        }
        if (0 == trCount) {
            throw new TemporaryLockingException("No lock columns found for " + target);
        }
        String msg = "Read " + trCount + " locks with our rid " + this.rid + " but mismatched timestamps; no lock column contained our timestamp (" + ls.getWriteTimestamp() + ")";
        throw new PermanentBackendException(msg);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Loose catch block
     */
    @Override
    protected void deleteSingleLock(KeyColumn kc, ConsistentKeyLockStatus ls, StoreTransaction tx) {
        List<StaticBuffer> deletions = Collections.singletonList(this.serializer.toLockCol(ls.getWriteTimestamp(), this.rid, this.times));
        for (int i = 0; i < this.lockRetryCount; ++i) {
            StoreTransaction newTx = null;
            try {
                newTx = this.overrideTimestamp(tx, this.times.getTime());
                this.store.mutate(this.serializer.toLockKey(kc.getKey(), kc.getColumn()), Collections.emptyList(), deletions, newTx);
                newTx.commit();
                newTx = null;
                this.rollbackIfNotNull(newTx);
                return;
            }
            catch (TemporaryBackendException e) {
                log.warn("Temporary storage exception while deleting lock", (Throwable)e);
                this.rollbackIfNotNull(newTx);
                continue;
            }
            catch (BackendException e2) {
                log.error("Storage exception while deleting lock", (Throwable)e2);
                this.rollbackIfNotNull(newTx);
                return;
                {
                    catch (Throwable throwable) {
                        this.rollbackIfNotNull(newTx);
                        throw throwable;
                    }
                }
            }
        }
    }

    private StoreTransaction overrideTimestamp(StoreTransaction tx, Instant commitTime) throws BackendException {
        StandardBaseTransactionConfig newCfg = new StandardBaseTransactionConfig.Builder(tx.getConfiguration()).commitTime(commitTime).build();
        return this.manager.beginTransaction(newCfg);
    }

    private void rollbackIfNotNull(StoreTransaction tx) {
        if (tx != null) {
            try {
                if (log.isDebugEnabled()) {
                    log.debug("Transaction is still open! Rolling back: " + tx, new Throwable());
                }
                tx.rollback();
            }
            catch (Throwable excp) {
                log.error("Failed to rollback transaction " + tx + ". The transaction may be leaked.", excp);
            }
        }
    }

    private static class WriteResult {
        private final Duration duration;
        private final Instant writeTimestamp;
        private final StaticBuffer lockCol;
        private final Throwable throwable;

        public WriteResult(Duration duration, Instant writeTimestamp, StaticBuffer lockCol, Throwable throwable) {
            this.duration = duration;
            this.writeTimestamp = writeTimestamp;
            this.lockCol = lockCol;
            this.throwable = throwable;
        }

        public Duration getDuration() {
            return this.duration;
        }

        public Instant getWriteTimestamp() {
            return this.writeTimestamp;
        }

        public boolean isSuccessful() {
            return null == this.throwable;
        }

        public StaticBuffer getLockCol() {
            return this.lockCol;
        }

        public Throwable getThrowable() {
            return this.throwable;
        }
    }

    public static class Builder
    extends AbstractLocker.Builder<ConsistentKeyLockStatus, Builder> {
        private final KeyColumnValueStore store;
        private final StoreManager manager;
        private Duration lockWait;
        private int lockRetryCount;
        private CleanerConfig cleanerConfig = CleanerConfig.NONE;
        private LockCleanerService customCleanerService;

        public Builder(KeyColumnValueStore store, StoreManager manager) {
            this.store = store;
            this.manager = manager;
            this.lockWait = GraphDatabaseConfiguration.LOCK_WAIT.getDefaultValue();
            this.lockRetryCount = GraphDatabaseConfiguration.LOCK_RETRY.getDefaultValue();
        }

        public Builder lockWait(Duration d) {
            this.lockWait = d;
            return this.self();
        }

        public Builder lockRetryCount(int count) {
            this.lockRetryCount = count;
            return this.self();
        }

        public Builder standardCleaner() {
            this.cleanerConfig = CleanerConfig.STANDARD;
            this.customCleanerService = null;
            return this.self();
        }

        public Builder customCleaner(LockCleanerService s) {
            this.cleanerConfig = CleanerConfig.CUSTOM;
            this.customCleanerService = s;
            Preconditions.checkNotNull((Object)this.customCleanerService);
            return this.self();
        }

        public Builder fromConfig(Configuration config) {
            this.rid(new StaticArrayBuffer(config.get(GraphDatabaseConfiguration.UNIQUE_INSTANCE_ID, new String[0]).getBytes(StringEncoding.UTF8_CHARSET)));
            String llmPrefix = config.get(GraphDatabaseConfiguration.LOCK_LOCAL_MEDIATOR_GROUP, new String[0]);
            this.times(config.get(GraphDatabaseConfiguration.TIMESTAMP_PROVIDER, new String[0]));
            this.mediator(LocalLockMediators.INSTANCE.get(llmPrefix, this.times));
            this.lockRetryCount(config.get(GraphDatabaseConfiguration.LOCK_RETRY, new String[0]));
            this.lockWait(config.get(GraphDatabaseConfiguration.LOCK_WAIT, new String[0]));
            this.lockExpire(config.get(GraphDatabaseConfiguration.LOCK_EXPIRE, new String[0]));
            if (config.get(GraphDatabaseConfiguration.LOCK_CLEAN_EXPIRED, new String[0]).booleanValue()) {
                this.standardCleaner();
            }
            return this;
        }

        public ConsistentKeyLocker build() {
            LockCleanerService cleaner;
            this.preBuild();
            switch (this.cleanerConfig) {
                case STANDARD: {
                    Preconditions.checkArgument((null == this.customCleanerService ? 1 : 0) != 0);
                    cleaner = new StandardLockCleanerService(this.store, this.serializer, this.times);
                    break;
                }
                case CUSTOM: {
                    Preconditions.checkArgument((null != this.customCleanerService ? 1 : 0) != 0);
                    cleaner = this.customCleanerService;
                    break;
                }
                default: {
                    cleaner = null;
                }
            }
            return new ConsistentKeyLocker(this.store, this.manager, this.rid, this.times, this.serializer, this.llm, this.lockWait, this.lockRetryCount, this.lockExpire, this.lockState, cleaner);
        }

        @Override
        protected Builder self() {
            return this;
        }

        @Override
        protected LocalLockMediator<StoreTransaction> getDefaultMediator() {
            throw new JanusGraphConfigurationException("Local lock mediator prefix must not be empty or null");
        }

        private static enum CleanerConfig {
            NONE,
            STANDARD,
            CUSTOM;

        }
    }
}

