/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.websocket.common;

import java.nio.ByteBuffer;
import java.util.Arrays;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.StringUtil;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.websocket.api.WebSocketPolicy;
import org.eclipse.jetty.websocket.api.extensions.Frame;
import org.eclipse.jetty.websocket.common.CloseInfo;
import org.eclipse.jetty.websocket.common.WebSocketFrame;
import org.eclipse.jetty.websocket.common.frames.BinaryFrame;
import org.eclipse.jetty.websocket.common.frames.CloseFrame;
import org.eclipse.jetty.websocket.common.frames.PingFrame;
import org.eclipse.jetty.websocket.common.frames.TextFrame;
import org.eclipse.jetty.websocket.common.test.IncomingFramesCapture;
import org.eclipse.jetty.websocket.common.test.UnitGenerator;
import org.eclipse.jetty.websocket.common.test.UnitParser;
import org.eclipse.jetty.websocket.common.util.Hex;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Test;

public class GeneratorTest {
    private static final Logger LOG = Log.getLogger(WindowHelper.class);

    private void assertGeneratedBytes(CharSequence expectedBytes, Frame ... frames) {
        ByteBuffer allframes = UnitGenerator.generate(frames);
        String actual = Hex.asHex(allframes);
        MatcherAssert.assertThat((String)"Buffer", (Object)actual, (Matcher)Matchers.is((Object)expectedBytes.toString()));
    }

    private String asMaskedHex(String str, byte[] maskingKey) {
        byte[] utf = StringUtil.getUtf8Bytes((String)str);
        this.mask(utf, maskingKey);
        return Hex.asHex(utf);
    }

    private void mask(byte[] buf, byte[] maskingKey) {
        int size = buf.length;
        for (int i = 0; i < size; ++i) {
            int n = i;
            buf[n] = (byte)(buf[n] ^ maskingKey[i % 4]);
        }
    }

    @Test
    public void testCloseEmpty() {
        this.assertGeneratedBytes("8800", new Frame[]{new CloseFrame()});
    }

    @Test
    public void testCloseCodeNoReason() {
        CloseInfo close = new CloseInfo(1000);
        this.assertGeneratedBytes("880203E8", new Frame[]{close.asFrame()});
    }

    @Test
    public void testCloseCodeOkReason() {
        CloseInfo close = new CloseInfo(1000, "OK");
        this.assertGeneratedBytes("880403E84F4B", new Frame[]{close.asFrame()});
    }

    @Test
    public void testTextHello() {
        TextFrame frame = new TextFrame().setPayload("Hello");
        byte[] utf = StringUtil.getUtf8Bytes((String)"Hello");
        this.assertGeneratedBytes("8105" + Hex.asHex(utf), new Frame[]{frame});
    }

    @Test
    public void testTextMasked() {
        TextFrame frame = new TextFrame().setPayload("Hello");
        byte[] maskingKey = Hex.asByteArray("11223344");
        frame.setMask(maskingKey);
        StringBuilder expected = new StringBuilder();
        expected.append("8185").append("11223344");
        expected.append(this.asMaskedHex("Hello", maskingKey));
        this.assertGeneratedBytes(expected, new Frame[]{frame});
    }

    @Test
    public void testTextMaskedOffsetSourceByteBuffer() {
        ByteBuffer payload = ByteBuffer.allocate(100);
        payload.position(5);
        payload.put(StringUtil.getUtf8Bytes((String)"Hello"));
        payload.flip();
        payload.position(5);
        LOG.debug("Payload = {}", new Object[]{BufferUtil.toDetailString((ByteBuffer)payload)});
        WebSocketFrame frame = new TextFrame().setPayload(payload);
        byte[] maskingKey = Hex.asByteArray("11223344");
        frame.setMask(maskingKey);
        StringBuilder expected = new StringBuilder();
        expected.append("8185").append("11223344");
        expected.append(this.asMaskedHex("Hello", maskingKey));
        this.assertGeneratedBytes(expected, new Frame[]{frame});
    }

    @Test
    public void testManyMasked() {
        int pingCount = 2;
        WebSocketFrame[] frames = new WebSocketFrame[pingCount + 1];
        for (int i = 0; i < pingCount; ++i) {
            frames[i] = new PingFrame().setPayload(String.format("ping-%d", i));
        }
        frames[pingCount] = new CloseInfo(1000).asFrame();
        byte[] maskingKey = Hex.asByteArray("11223344");
        for (WebSocketFrame f : frames) {
            f.setMask(maskingKey);
        }
        StringBuilder expected = new StringBuilder();
        expected.append("8986").append("11223344");
        expected.append(this.asMaskedHex("ping-0", maskingKey));
        expected.append("8986").append("11223344");
        expected.append(this.asMaskedHex("ping-1", maskingKey));
        expected.append("8882").append("11223344");
        byte[] closure = Hex.asByteArray("03E8");
        this.mask(closure, maskingKey);
        expected.append(Hex.asHex(closure));
        this.assertGeneratedBytes(expected, (Frame[])frames);
    }

    @Test
    public void testWindowedGenerate() {
        byte[] payload = new byte[10240];
        Arrays.fill(payload, (byte)68);
        BinaryFrame frame = new BinaryFrame().setPayload(payload);
        int windowSize = 1024;
        WindowHelper helper = new WindowHelper(windowSize);
        ByteBuffer completeBuffer = helper.generateWindowed(new Frame[]{frame});
        int expectedHeaderSize = 4;
        int expectedSize = payload.length + expectedHeaderSize;
        int expectedParts = 1;
        helper.assertTotalParts(expectedParts);
        helper.assertTotalBytes(payload.length + expectedHeaderSize);
        MatcherAssert.assertThat((String)"Generated Buffer", (Object)completeBuffer.remaining(), (Matcher)Matchers.is((Object)expectedSize));
    }

    @Test
    public void testWindowedGenerateWithMasking() {
        byte[] payload = new byte[10240];
        Arrays.fill(payload, (byte)85);
        byte[] mask = new byte[]{42, -16, 15, 0};
        BinaryFrame frame = new BinaryFrame().setPayload(payload);
        frame.setMask(mask);
        int windowSize = 2929;
        WindowHelper helper = new WindowHelper(windowSize);
        ByteBuffer completeBuffer = helper.generateWindowed(new Frame[]{frame});
        int expectedHeaderSize = 8;
        int expectedSize = payload.length + expectedHeaderSize;
        int expectedParts = 1;
        helper.assertTotalParts(expectedParts);
        helper.assertTotalBytes(payload.length + expectedHeaderSize);
        MatcherAssert.assertThat((String)"Generated Buffer", (Object)completeBuffer.remaining(), (Matcher)Matchers.is((Object)expectedSize));
        WebSocketPolicy policy = WebSocketPolicy.newServerPolicy();
        UnitParser parser = new UnitParser(policy);
        IncomingFramesCapture capture = new IncomingFramesCapture();
        parser.setIncomingFramesHandler(capture);
        parser.parse(completeBuffer);
        WebSocketFrame actual = capture.getFrames().poll();
        MatcherAssert.assertThat((String)"Frame.opcode", (Object)actual.getOpCode(), (Matcher)Matchers.is((Object)2));
        MatcherAssert.assertThat((String)"Frame.payloadLength", (Object)actual.getPayloadLength(), (Matcher)Matchers.is((Object)payload.length));
        ByteBuffer actualData = actual.getPayload().slice();
        MatcherAssert.assertThat((String)"Frame.payload.remaining", (Object)actualData.remaining(), (Matcher)Matchers.is((Object)payload.length));
        while (actualData.remaining() > 0) {
            MatcherAssert.assertThat((String)("Actual.payload[" + actualData.position() + "]"), (Object)actualData.get(), (Matcher)Matchers.is((Object)85));
        }
    }

    public static class WindowHelper {
        final int windowSize;
        int totalParts;
        int totalBytes;

        public WindowHelper(int windowSize) {
            this.windowSize = windowSize;
            this.totalParts = 0;
            this.totalBytes = 0;
        }

        public ByteBuffer generateWindowed(Frame ... frames) {
            int completeBufSize = 0;
            for (Frame f : frames) {
                completeBufSize += 28 + f.getPayloadLength();
            }
            ByteBuffer completeBuf = ByteBuffer.allocate(completeBufSize);
            BufferUtil.clearToFill((ByteBuffer)completeBuf);
            UnitGenerator generator = new UnitGenerator();
            for (Frame f : frames) {
                ByteBuffer header = generator.generateHeaderBytes(f);
                this.totalBytes += BufferUtil.put((ByteBuffer)header, (ByteBuffer)completeBuf);
                if (!f.hasPayload()) continue;
                ByteBuffer payload = f.getPayload();
                this.totalBytes += payload.remaining();
                ++this.totalParts;
                completeBuf.put(payload.slice());
            }
            BufferUtil.flipToFlush((ByteBuffer)completeBuf, (int)0);
            return completeBuf;
        }

        public void assertTotalParts(int expectedParts) {
            MatcherAssert.assertThat((String)"Generated Parts", (Object)this.totalParts, (Matcher)Matchers.is((Object)expectedParts));
        }

        public void assertTotalBytes(int expectedBytes) {
            MatcherAssert.assertThat((String)"Generated Bytes", (Object)this.totalBytes, (Matcher)Matchers.is((Object)expectedBytes));
        }
    }
}

