/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ranger.plugin.util;

import com.sun.istack.NotNull;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.ranger.plugin.util.AutoClosableLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RangerCache<K, V> {
    private static final Logger LOG = LoggerFactory.getLogger(RangerCache.class);
    private static final AtomicInteger CACHE_NUMBER = new AtomicInteger(1);
    private static final String CACHE_LOADER_THREAD_PREFIX = "ranger-cache-";
    private static final int DEFAULT_LOADER_THREADS_COUNT = 10;
    private static final RefreshMode DEFAULT_REFRESH_MODE = RefreshMode.ON_ACCESS;
    private static final int DEFAULT_VALUE_VALIDITY_PERIOD_MS = 30000;
    private static final int DEFAULT_VALUE_INIT_TIMEOUT_MS = -1;
    private static final int DEFAULT_VALUE_REFRESH_TIMEOUT_MS = 10;
    private final String name;
    private final Map<K, CachedValue> cache;
    private final int loaderThreadsCount;
    private final RefreshMode refreshMode;
    private final long valueValidityPeriodMs;
    private final long valueInitLoadTimeoutMs;
    private final long valueRefreshLoadTimeoutMs;
    private final ExecutorService loaderThreadPool;
    private ValueLoader<K, V> loader;

    protected RangerCache(String name, ValueLoader<K, V> loader) {
        this(name, loader, 10, DEFAULT_REFRESH_MODE, 30000L, -1L, 10L);
    }

    protected RangerCache(String name, ValueLoader<K, V> loader, int loaderThreadsCount, RefreshMode refreshMode, long valueValidityPeriodMs, long valueInitLoadTimeoutMs, long valueRefreshLoadTimeoutMs) {
        this.name = name;
        this.cache = new ConcurrentHashMap<K, CachedValue>();
        this.loader = loader;
        this.loaderThreadsCount = loaderThreadsCount;
        this.refreshMode = refreshMode;
        this.valueValidityPeriodMs = valueValidityPeriodMs;
        this.valueInitLoadTimeoutMs = valueInitLoadTimeoutMs;
        this.valueRefreshLoadTimeoutMs = valueRefreshLoadTimeoutMs;
        this.loaderThreadPool = this.refreshMode == RefreshMode.ON_SCHEDULE ? Executors.newScheduledThreadPool(loaderThreadsCount, this.createThreadFactory()) : Executors.newFixedThreadPool(loaderThreadsCount, this.createThreadFactory());
        LOG.info("Created RangerCache(name={}): loaderThreadsCount={}, refreshMode={}, valueValidityPeriodMs={}, valueInitLoadTimeoutMs={}, valueRefreshLoadTimeoutMs={}", new Object[]{name, loaderThreadsCount, refreshMode, valueValidityPeriodMs, valueInitLoadTimeoutMs, valueRefreshLoadTimeoutMs});
    }

    public String getName() {
        return this.name;
    }

    public ValueLoader<K, V> getLoader() {
        return this.loader;
    }

    protected void setLoader(ValueLoader<K, V> loader) {
        this.loader = loader;
    }

    public int getLoaderThreadsCount() {
        return this.loaderThreadsCount;
    }

    public RefreshMode getRefreshMode() {
        return this.refreshMode;
    }

    public long getValueValidityPeriodMs() {
        return this.valueValidityPeriodMs;
    }

    public long getValueInitLoadTimeoutMs() {
        return this.valueInitLoadTimeoutMs;
    }

    public long getValueRefreshLoadTimeoutMs() {
        return this.valueRefreshLoadTimeoutMs;
    }

    public V get(K key) {
        return this.get(key, null);
    }

    public Set<K> getKeys() {
        return new HashSet<K>(this.cache.keySet());
    }

    public void addIfAbsent(K key) {
        this.cache.computeIfAbsent(key, f -> new CachedValue(key));
    }

    public V remove(K key) {
        V ret;
        CachedValue value = this.cache.remove(key);
        if (value != null) {
            value.isRemoved = true;
            ret = value.getCurrentValue();
        } else {
            ret = null;
        }
        return ret;
    }

    public boolean isLoaded(K key) {
        CachedValue entry = this.cache.get(key);
        RefreshableValue value = entry != null ? entry.value : null;
        return value != null;
    }

    protected V get(K key, Object context) {
        Object ret;
        long timeoutMs;
        long startTime = System.currentTimeMillis();
        CachedValue value = this.cache.computeIfAbsent(key, f -> new CachedValue(key));
        long l = timeoutMs = value.isInitialized() ? this.valueRefreshLoadTimeoutMs : this.valueInitLoadTimeoutMs;
        if (timeoutMs >= 0L) {
            long timeTaken = System.currentTimeMillis() - startTime;
            if (timeoutMs <= timeTaken) {
                ret = value.getCurrentValue();
                LOG.debug("key={}: cache-lookup={}ms took longer than timeout={}ms. Using current value {}", new Object[]{key, timeTaken, timeoutMs, ret});
            } else {
                ret = value.getValue(timeoutMs - timeTaken);
            }
        } else {
            ret = value.getValue(context);
        }
        return ret;
    }

    private ThreadFactory createThreadFactory() {
        return new ThreadFactory(){
            private final String namePrefix;
            private final AtomicInteger number;
            {
                this.namePrefix = RangerCache.CACHE_LOADER_THREAD_PREFIX + CACHE_NUMBER.getAndIncrement() + "-" + RangerCache.this.name;
                this.number = new AtomicInteger(1);
            }

            @Override
            public Thread newThread(@NotNull Runnable r) {
                Thread t = new Thread(r, this.namePrefix + this.number.getAndIncrement());
                if (!t.isDaemon()) {
                    t.setDaemon(true);
                }
                if (t.getPriority() != 5) {
                    t.setPriority(5);
                }
                return t;
            }
        };
    }

    private class CachedValue {
        private final ReentrantLock lock = new ReentrantLock();
        private final K key;
        private volatile boolean isRemoved;
        private volatile RefreshableValue<V> value;
        private volatile Future<?> refresher;

        private CachedValue(K key) {
            LOG.debug("CachedValue({})", key);
            this.key = key;
        }

        public K getKey() {
            return this.key;
        }

        public V getValue(Object context) {
            this.refreshIfNeeded(context);
            return this.getCurrentValue();
        }

        public V getValue(long timeoutMs, Object context) {
            if (timeoutMs < 0L) {
                this.refreshIfNeeded(context);
            } else {
                this.refreshIfNeeded(timeoutMs, context);
            }
            return this.getCurrentValue();
        }

        public V getCurrentValue() {
            RefreshableValue value = this.value;
            return value != null ? (Object)value.getValue() : null;
        }

        public boolean needsRefresh() {
            return !this.isInitialized() || RangerCache.this.refreshMode == RefreshMode.ON_ACCESS && this.value.needsRefresh();
        }

        public boolean isInitialized() {
            RefreshableValue value = this.value;
            return value != null;
        }

        private void refreshIfNeeded(Object context) {
            if (this.needsRefresh()) {
                try (AutoClosableLock ignored = new AutoClosableLock(this.lock);){
                    if (this.needsRefresh()) {
                        Future<?> future = this.refresher;
                        if (future == null) {
                            LOG.debug("refreshIfNeeded(key={}): using caller thread", this.key);
                            this.refreshValue(context);
                        } else {
                            try {
                                future.get();
                                this.refresher = null;
                            }
                            catch (InterruptedException | ExecutionException excp) {
                                LOG.warn("refreshIfNeeded(key={}) failed", this.key, (Object)excp);
                            }
                        }
                    }
                }
            }
        }

        private void refreshIfNeeded(long timeoutMs, Object context) {
            if (this.needsRefresh()) {
                long startTime = System.currentTimeMillis();
                try (AutoClosableLock.AutoClosableTryLock tryLock = new AutoClosableLock.AutoClosableTryLock(this.lock, timeoutMs, TimeUnit.MILLISECONDS);){
                    if (tryLock.isLocked()) {
                        if (this.needsRefresh()) {
                            Future<Object> future = this.refresher;
                            if (future == null) {
                                future = RangerCache.this.loaderThreadPool.submit(new RefreshWithContext(context));
                                this.refresher = future;
                                LOG.debug("refresher scheduled for key {}", this.key);
                            } else {
                                LOG.debug("refresher already exists for key {}", this.key);
                            }
                            long timeLeftMs = timeoutMs - (System.currentTimeMillis() - startTime);
                            if (timeLeftMs > 0L) {
                                try {
                                    future.get(timeLeftMs, TimeUnit.MILLISECONDS);
                                    this.refresher = null;
                                }
                                catch (InterruptedException | ExecutionException | TimeoutException excp) {
                                    LOG.debug("refreshIfNeeded(key={}, timeoutMs={}) failed", new Object[]{this.key, timeoutMs, excp});
                                }
                            }
                        }
                    } else {
                        LOG.debug("refreshIfNeeded(key={}, timeoutMs={}) couldn't obtain lock", this.key, (Object)timeoutMs);
                    }
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Loose catch block
         */
        private Boolean refreshValue(Object context) {
            RefreshableValue newValue;
            boolean isSuccess;
            long startTime;
            block22: {
                startTime = System.currentTimeMillis();
                isSuccess = false;
                newValue = null;
                try {
                    ValueLoader loader = RangerCache.this.loader;
                    if (loader == null) break block22;
                    newValue = loader.load(this.key, this.value, context);
                    isSuccess = true;
                }
                catch (KeyNotFoundException excp) {
                    LOG.debug("refreshValue(key={}) failed with KeyNotFoundException. Removing it", this.key, (Object)excp);
                    RangerCache.this.remove(this.key);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("refresher {} for key {}, timeTaken={}", new Object[]{isSuccess ? "completed" : "failed", this.key, System.currentTimeMillis() - startTime});
                    }
                    this.setValue(newValue);
                    if (RangerCache.this.refreshMode == RefreshMode.ON_SCHEDULE) {
                        if (!this.isRemoved) {
                            ScheduledExecutorService scheduledExecutor = (ScheduledExecutorService)RangerCache.this.loaderThreadPool;
                            scheduledExecutor.schedule(new RefreshWithContext(context), RangerCache.this.valueValidityPeriodMs, TimeUnit.MILLISECONDS);
                        } else {
                            LOG.debug("key {} was removed. Not scheduling next refresh ", this.key);
                        }
                    }
                }
                catch (Exception excp2) {
                    LOG.warn("refreshValue(key={}) failed", this.key, (Object)excp2);
                    newValue = this.value;
                    {
                        catch (Throwable throwable) {
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("refresher {} for key {}, timeTaken={}", new Object[]{isSuccess ? "completed" : "failed", this.key, System.currentTimeMillis() - startTime});
                            }
                            this.setValue(newValue);
                            if (RangerCache.this.refreshMode == RefreshMode.ON_SCHEDULE) {
                                if (!this.isRemoved) {
                                    ScheduledExecutorService scheduledExecutor = (ScheduledExecutorService)RangerCache.this.loaderThreadPool;
                                    scheduledExecutor.schedule(new RefreshWithContext(context), RangerCache.this.valueValidityPeriodMs, TimeUnit.MILLISECONDS);
                                } else {
                                    LOG.debug("key {} was removed. Not scheduling next refresh ", this.key);
                                }
                            }
                            throw throwable;
                        }
                    }
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("refresher {} for key {}, timeTaken={}", new Object[]{isSuccess ? "completed" : "failed", this.key, System.currentTimeMillis() - startTime});
                    }
                    this.setValue(newValue);
                    if (RangerCache.this.refreshMode == RefreshMode.ON_SCHEDULE) {
                        if (!this.isRemoved) {
                            ScheduledExecutorService scheduledExecutor = (ScheduledExecutorService)RangerCache.this.loaderThreadPool;
                            scheduledExecutor.schedule(new RefreshWithContext(context), RangerCache.this.valueValidityPeriodMs, TimeUnit.MILLISECONDS);
                        } else {
                            LOG.debug("key {} was removed. Not scheduling next refresh ", this.key);
                        }
                    }
                }
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("refresher {} for key {}, timeTaken={}", new Object[]{isSuccess ? "completed" : "failed", this.key, System.currentTimeMillis() - startTime});
            }
            this.setValue(newValue);
            if (RangerCache.this.refreshMode == RefreshMode.ON_SCHEDULE) {
                if (!this.isRemoved) {
                    ScheduledExecutorService scheduledExecutor = (ScheduledExecutorService)RangerCache.this.loaderThreadPool;
                    scheduledExecutor.schedule(new RefreshWithContext(context), RangerCache.this.valueValidityPeriodMs, TimeUnit.MILLISECONDS);
                } else {
                    LOG.debug("key {} was removed. Not scheduling next refresh ", this.key);
                }
            }
            return Boolean.TRUE;
        }

        private void setValue(RefreshableValue<V> value) {
            if (value != null) {
                this.value = value;
                this.value.setNextRefreshTimeMs(System.currentTimeMillis() + RangerCache.this.valueValidityPeriodMs);
            }
        }

        private class RefreshWithContext
        implements Callable<Boolean> {
            private final Object context;

            public RefreshWithContext(Object context) {
                this.context = context;
            }

            @Override
            public Boolean call() {
                return CachedValue.this.refreshValue(this.context);
            }
        }
    }

    public static class KeyNotFoundException
    extends Exception {
        public KeyNotFoundException(String msg) {
            super(msg);
        }
    }

    public static abstract class ValueLoader<K, V> {
        public abstract RefreshableValue<V> load(K var1, RefreshableValue<V> var2, Object var3) throws Exception;
    }

    public static class RefreshableValue<V> {
        private final V value;
        private long nextRefreshTimeMs = -1L;

        public RefreshableValue(V value) {
            this.value = value;
        }

        public V getValue() {
            return this.value;
        }

        public boolean needsRefresh() {
            return this.nextRefreshTimeMs == -1L || System.currentTimeMillis() > this.nextRefreshTimeMs;
        }

        private void setNextRefreshTimeMs(long nextRefreshTimeMs) {
            this.nextRefreshTimeMs = nextRefreshTimeMs;
        }
    }

    public static enum RefreshMode {
        ON_ACCESS,
        ON_SCHEDULE;

    }
}

