/*
 * Decompiled with CFR 0.152.
 */
package org.apache.phoenix.end2end;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apache.phoenix.compile.ExplainPlan;
import org.apache.phoenix.compile.ExplainPlanAttributes;
import org.apache.phoenix.end2end.ParallelStatsDisabledIT;
import org.apache.phoenix.end2end.ParallelStatsDisabledTest;
import org.apache.phoenix.exception.SQLExceptionCode;
import org.apache.phoenix.jdbc.PhoenixPreparedStatement;
import org.apache.phoenix.thirdparty.com.google.common.collect.Lists;
import org.apache.phoenix.util.PropertiesUtil;
import org.apache.phoenix.util.SchemaUtil;
import org.apache.phoenix.util.TestUtil;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@Category(value={ParallelStatsDisabledTest.class})
@RunWith(value=Parameterized.class)
public class SequenceBulkAllocationIT
extends ParallelStatsDisabledIT {
    private static final String SELECT_NEXT_VALUE_SQL = "SELECT NEXT VALUE FOR %s";
    private static final String SELECT_CURRENT_VALUE_SQL = "SELECT CURRENT VALUE FOR %s";
    private static final String CREATE_SEQUENCE_NO_MIN_MAX_TEMPLATE = "CREATE SEQUENCE %s START WITH %s INCREMENT BY %s CACHE %s";
    private static final String CREATE_SEQUENCE_WITH_MIN_MAX_TEMPLATE = "CREATE SEQUENCE %s START WITH %s INCREMENT BY %s MINVALUE %s MAXVALUE %s CACHE %s";
    private static final String CREATE_SEQUENCE_WITH_MIN_MAX_AND_CYCLE_TEMPLATE = "CREATE SEQUENCE %s START WITH %s INCREMENT BY %s MINVALUE %s MAXVALUE %s CYCLE CACHE %s";
    private static final String SCHEMA_NAME = "S";
    private Connection conn;
    private String tenantId;

    public SequenceBulkAllocationIT(String tenantId) {
        this.tenantId = tenantId;
    }

    @Parameterized.Parameters(name="SequenceBulkAllocationIT_tenantId={0}")
    public static synchronized Object[] data() {
        return new Object[]{null, "tenant1"};
    }

    private static String generateTableNameWithSchema() {
        return SchemaUtil.getTableName((String)SCHEMA_NAME, (String)SequenceBulkAllocationIT.generateUniqueName());
    }

    private static String generateSequenceNameWithSchema() {
        return SchemaUtil.getTableName((String)SCHEMA_NAME, (String)SequenceBulkAllocationIT.generateUniqueSequenceName());
    }

    @Before
    public void init() throws Exception {
        this.createConnection();
    }

    @After
    public void tearDown() throws Exception {
        if (this.conn != null) {
            this.conn.close();
        }
    }

    @Test
    public void testSequenceParseNextValuesWithNull() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        try {
            this.conn.createStatement().executeQuery("SELECT NEXT NULL VALUES FOR  " + sequenceName);
            Assert.fail((String)"null is not allowed to be used for <n> in NEXT <n> VALUES FOR <seq>");
        }
        catch (SQLException e) {
            Assert.assertEquals((long)SQLExceptionCode.NUM_SEQ_TO_ALLOCATE_MUST_BE_CONSTANT.getErrorCode(), (long)e.getErrorCode());
            Assert.assertTrue((e.getNextException() == null ? 1 : 0) != 0);
        }
    }

    @Test
    public void testSequenceParseNextValuesWithNonNumber() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        try {
            this.conn.createStatement().executeQuery("SELECT NEXT '89b' VALUES FOR  " + sequenceName);
            Assert.fail((String)"Only integers and longs are allowed to be used for <n> in NEXT <n> VALUES FOR <seq>");
        }
        catch (SQLException e) {
            Assert.assertEquals((long)SQLExceptionCode.NUM_SEQ_TO_ALLOCATE_MUST_BE_CONSTANT.getErrorCode(), (long)e.getErrorCode());
            Assert.assertTrue((e.getNextException() == null ? 1 : 0) != 0);
        }
    }

    @Test
    public void testSequenceParseNextValuesWithNegativeNumber() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        try {
            this.conn.createStatement().executeQuery("SELECT NEXT '-1' VALUES FOR  " + sequenceName);
            Assert.fail((String)"null is not allowed to be used for <n> in NEXT <n> VALUES FOR <seq>");
        }
        catch (SQLException e) {
            Assert.assertEquals((long)SQLExceptionCode.NUM_SEQ_TO_ALLOCATE_MUST_BE_CONSTANT.getErrorCode(), (long)e.getErrorCode());
            Assert.assertTrue((e.getNextException() == null ? 1 : 0) != 0);
        }
    }

    @Test
    public void testParseNextValuesSequenceWithZeroAllocated() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        try {
            this.conn.createStatement().executeQuery("SELECT NEXT 0 VALUES FOR  " + sequenceName);
            Assert.fail((String)"Only integers and longs are allowed to be used for <n> in NEXT <n> VALUES FOR <seq>");
        }
        catch (SQLException e) {
            Assert.assertEquals((long)SQLExceptionCode.NUM_SEQ_TO_ALLOCATE_MUST_BE_CONSTANT.getErrorCode(), (long)e.getErrorCode());
            Assert.assertTrue((e.getNextException() == null ? 1 : 0) != 0);
        }
    }

    @Test
    public void testNextValuesForSequenceWithNoAllocatedValues() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(1).cacheSize(1).numAllocated(100L).build();
        this.createSequenceWithNoMinMax(props);
        int currentValueAfterAllocation = 100;
        this.reserveSlotsInBulkAndAssertValue(sequenceName, 1L, props.numAllocated);
        this.assertExpectedStateInSystemSequence(props, 101L);
        this.assertExpectedNumberOfValuesAllocated(1L, 100L, props.incrementBy, props.numAllocated);
        this.assertExpectedCurrentValueForSequence(sequenceName, 100);
        this.assertExpectedNextValueForSequence(sequenceName, 101);
    }

    @Test
    public void testNextValuesForSequenceUsingBinds() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(1).cacheSize(1).numAllocated(100L).build();
        this.createSequenceWithNoMinMax(props);
        int currentValueAfterAllocation = 100;
        this.reserveSlotsInBulkUsingBindsAndAssertValue(sequenceName, 1, props.numAllocated);
        this.assertExpectedStateInSystemSequence(props, 101L);
        this.assertExpectedNumberOfValuesAllocated(1L, 100L, props.incrementBy, props.numAllocated);
        this.assertExpectedCurrentValueForSequence(sequenceName, 100);
        this.assertExpectedNextValueForSequence(sequenceName, 101);
    }

    @Test
    public void testNextValuesForSequenceWithPreviouslyAllocatedValues() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(1).cacheSize(100).numAllocated(1000L).build();
        this.createSequenceWithNoMinMax(props);
        this.assertExpectedNextValueForSequence(sequenceName, 1);
        this.assertExpectedCurrentValueForSequence(sequenceName, 1);
        this.assertExpectedNextValueForSequence(sequenceName, 2);
        int currentValueAfterAllocation = 1100;
        int nextValueAfterAllocation = currentValueAfterAllocation + props.incrementBy;
        int startValueAfterAllocation = 101;
        this.reserveSlotsInBulkAndAssertValue(sequenceName, startValueAfterAllocation, props.numAllocated);
        this.assertBulkAllocationSucceeded(props, currentValueAfterAllocation, startValueAfterAllocation);
        this.assertExpectedCurrentValueForSequence(sequenceName, currentValueAfterAllocation);
        this.assertExpectedNextValueForSequence(sequenceName, nextValueAfterAllocation);
    }

    @Test
    public void testConnectionCloseReturnsSequenceValuesCorrectly() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(2).startsWith(1).cacheSize(100).numAllocated(100L).build();
        this.createSequenceWithNoMinMax(props);
        this.assertExpectedNextValueForSequence(sequenceName, 1);
        this.assertExpectedCurrentValueForSequence(sequenceName, 1);
        this.assertExpectedNextValueForSequence(sequenceName, 3);
        int currentValueAfterAllocation = 399;
        int nextValueAfterAllocation = currentValueAfterAllocation + props.incrementBy;
        int startValueAfterAllocation = 201;
        this.reserveSlotsInBulkAndAssertValue(sequenceName, startValueAfterAllocation, props.numAllocated);
        this.assertExpectedCurrentValueForSequence(sequenceName, currentValueAfterAllocation);
        this.conn.close();
        this.createConnection();
        this.assertExpectedNextValueForSequence(sequenceName, nextValueAfterAllocation);
        this.assertExpectedCurrentValueForSequence(sequenceName, nextValueAfterAllocation);
    }

    @Test
    public void testNextValuesForSequenceWithUpsert() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        String tableName = SequenceBulkAllocationIT.generateTableNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(1).cacheSize(100).numAllocated(1000L).build();
        this.createSequenceWithNoMinMax(props);
        Connection genericConn = this.createGenericConnection();
        genericConn.createStatement().execute("CREATE TABLE " + tableName + " ( id INTEGER NOT NULL PRIMARY KEY)");
        genericConn.close();
        this.assertExpectedNextValueForSequence(sequenceName, 1);
        this.assertExpectedCurrentValueForSequence(sequenceName, 1);
        this.assertExpectedNextValueForSequence(sequenceName, 2);
        this.assertExpectedStateInSystemSequence(props, 101L);
        this.conn.createStatement().execute("UPSERT INTO " + tableName + " (id) VALUES (NEXT " + props.numAllocated + " VALUES FOR  " + sequenceName + " )");
        this.conn.commit();
        this.assertExpectedStateInSystemSequence(props, 1101L);
        String query = "SELECT id, NEXT VALUE FOR  " + sequenceName + "  FROM " + tableName;
        ResultSet rs = this.conn.prepareStatement(query).executeQuery();
        Assert.assertTrue((boolean)rs.next());
        Assert.assertEquals((long)101L, (long)rs.getInt(1));
        Assert.assertEquals((long)1101L, (long)rs.getInt(2));
        Assert.assertFalse((boolean)rs.next());
    }

    @Test
    public void testNextValuesForSequenceWithIncrementBy() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(3).startsWith(1).cacheSize(100).numAllocated(1000L).build();
        this.createSequenceWithNoMinMax(props);
        this.assertExpectedNextValueForSequence(sequenceName, 1);
        this.assertExpectedCurrentValueForSequence(sequenceName, 1);
        this.assertExpectedNextValueForSequence(sequenceName, 4);
        int currentValueAfterAllocation = 3298;
        int startValueAfterAllocation = 301;
        this.reserveSlotsInBulkAndAssertValue(sequenceName, startValueAfterAllocation, props.numAllocated);
        this.assertBulkAllocationSucceeded(props, currentValueAfterAllocation, startValueAfterAllocation);
        this.assertExpectedCurrentValueForSequence(sequenceName, 3298);
        this.assertExpectedNextValueForSequence(sequenceName, 3301);
    }

    @Test
    public void testNextValuesForSequenceWithNegativeIncrementBy() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(-1).startsWith(2000).cacheSize(100).numAllocated(1000L).build();
        this.createSequenceWithNoMinMax(props);
        this.assertExpectedNextValueForSequence(sequenceName, 2000);
        this.assertExpectedCurrentValueForSequence(sequenceName, 2000);
        this.assertExpectedNextValueForSequence(sequenceName, 1999);
        int currentValueAfterAllocation = 901;
        int startValueAfterAllocation = 1900;
        this.reserveSlotsInBulkAndAssertValue(sequenceName, startValueAfterAllocation, props.numAllocated);
        this.assertBulkAllocationSucceeded(props, currentValueAfterAllocation, startValueAfterAllocation);
        this.assertExpectedCurrentValueForSequence(sequenceName, 901);
        this.assertExpectedNextValueForSequence(sequenceName, 900);
    }

    @Test
    public void testNextValuesForSequenceWithNegativeIncrementByGreaterThanOne() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(-5).startsWith(2000).cacheSize(100).numAllocated(100L).build();
        this.createSequenceWithNoMinMax(props);
        this.assertExpectedNextValueForSequence(sequenceName, 2000);
        this.assertExpectedCurrentValueForSequence(sequenceName, 2000);
        this.assertExpectedNextValueForSequence(sequenceName, 1995);
        int currentValueAfterAllocation = 1005;
        int startValueAfterAllocation = 1500;
        this.reserveSlotsInBulkAndAssertValue(sequenceName, startValueAfterAllocation, props.numAllocated);
        this.assertBulkAllocationSucceeded(props, currentValueAfterAllocation, startValueAfterAllocation);
        this.assertExpectedCurrentValueForSequence(sequenceName, 1005);
        this.assertExpectedNextValueForSequence(sequenceName, 1000);
    }

    @Test
    public void testNextValuesForSequenceExceedsMaxValue() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties sequenceProps = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(100).cacheSize(100).numAllocated(1000L).minValue(100L).maxValue(900L).build();
        this.createSequenceWithMinMax(sequenceProps);
        this.assertExpectedNextValueForSequence(sequenceName, 100);
        this.assertExpectedCurrentValueForSequence(sequenceName, 100);
        this.assertExpectedNextValueForSequence(sequenceName, 101);
        try {
            this.conn.createStatement().executeQuery("SELECT NEXT " + sequenceProps.numAllocated + " VALUES FOR  " + sequenceName + "  LIMIT 1");
            Assert.fail((String)"Invoking SELECT NEXT VALUES should have thrown Reached Max Value Exception");
        }
        catch (SQLException e) {
            Assert.assertEquals((long)SQLExceptionCode.SEQUENCE_VAL_REACHED_MAX_VALUE.getErrorCode(), (long)e.getErrorCode());
            Assert.assertTrue((e.getNextException() == null ? 1 : 0) != 0);
        }
        this.assertExpectedCurrentValueForSequence(sequenceName, 101);
        this.assertExpectedNextValueForSequence(sequenceName, 102);
    }

    @Test
    public void testNextValuesForSequenceExceedsMinValue() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties sequenceProps = new SequenceProperties.Builder().name(sequenceName).incrementBy(-5).startsWith(900).cacheSize(100).numAllocated(160L).minValue(100L).maxValue(900L).build();
        this.createSequenceWithMinMax(sequenceProps);
        this.assertExpectedNextValueForSequence(sequenceName, 900);
        this.assertExpectedCurrentValueForSequence(sequenceName, 900);
        this.assertExpectedNextValueForSequence(sequenceName, 895);
        try {
            this.conn.createStatement().executeQuery("SELECT NEXT " + sequenceProps.numAllocated + " VALUES FOR  " + sequenceName + "  LIMIT 1");
            Assert.fail((String)"Invoking SELECT NEXT VALUES should have thrown Reached Max Value Exception");
        }
        catch (SQLException e) {
            Assert.assertEquals((long)SQLExceptionCode.SEQUENCE_VAL_REACHED_MIN_VALUE.getErrorCode(), (long)e.getErrorCode());
            Assert.assertTrue((e.getNextException() == null ? 1 : 0) != 0);
        }
        this.assertExpectedCurrentValueForSequence(sequenceName, 895);
        this.assertExpectedNextValueForSequence(sequenceName, 890);
    }

    @Test
    public void testNextValuesForSequenceWithMinMaxDefined() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(5).startsWith(100).cacheSize(100).numAllocated(1000L).minValue(100L).maxValue(6000L).build();
        this.createSequenceWithMinMax(props);
        this.assertExpectedNextValueForSequence(sequenceName, 100);
        this.assertExpectedCurrentValueForSequence(sequenceName, 100);
        this.assertExpectedNextValueForSequence(sequenceName, 105);
        int currentValueAfterAllocation = 5595;
        int startValueAfterAllocation = 600;
        this.reserveSlotsInBulkAndAssertValue(sequenceName, startValueAfterAllocation, props.numAllocated);
        this.assertBulkAllocationSucceeded(props, currentValueAfterAllocation, startValueAfterAllocation);
        this.assertExpectedCurrentValueForSequence(sequenceName, 5595);
        this.assertExpectedNextValueForSequence(sequenceName, 5600);
    }

    @Test
    public void testNextValuesForSequenceWithDefaultMax() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(100).cacheSize(100).numAllocated(9223372036854775707L).build();
        this.createSequenceWithMinMax(props);
        long currentValueAfterAllocation = 100L;
        long startValueAfterAllocation = Long.MAX_VALUE;
        this.reserveSlotsInBulkAndAssertValue(sequenceName, currentValueAfterAllocation, props.numAllocated);
        this.assertExpectedStateInSystemSequence(props, startValueAfterAllocation);
        try {
            this.conn.createStatement().executeQuery(String.format(SELECT_NEXT_VALUE_SQL, sequenceName));
        }
        catch (SQLException e) {
            Assert.assertEquals((long)SQLExceptionCode.SEQUENCE_VAL_REACHED_MAX_VALUE.getErrorCode(), (long)e.getErrorCode());
            Assert.assertTrue((e.getNextException() == null ? 1 : 0) != 0);
        }
    }

    @Test
    public void testNextValuesForSequenceOverflowAllocation() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(100).cacheSize(100).numAllocated(Long.MAX_VALUE).build();
        this.createSequenceWithMinMax(props);
        try {
            this.conn.createStatement().executeQuery("SELECT NEXT 9223372036854775807 VALUES FOR  " + sequenceName + " ");
        }
        catch (SQLException e) {
            Assert.assertEquals((long)SQLExceptionCode.SEQUENCE_VAL_REACHED_MAX_VALUE.getErrorCode(), (long)e.getErrorCode());
            Assert.assertTrue((e.getNextException() == null ? 1 : 0) != 0);
        }
    }

    @Test
    public void testNextValuesForSequenceAllocationLessThanCacheSize() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(5).startsWith(100).cacheSize(100).numAllocated(50L).minValue(100L).maxValue(6000L).build();
        this.createSequenceWithMinMax(props);
        this.assertExpectedNextValueForSequence(sequenceName, 100);
        this.assertExpectedCurrentValueForSequence(sequenceName, 100);
        this.assertExpectedNextValueForSequence(sequenceName, 105);
        int currentValueAfterAllocation = 355;
        int startValueAfterAllocation = 110;
        this.reserveSlotsInBulkAndAssertValue(sequenceName, startValueAfterAllocation, props.numAllocated);
        this.assertExpectedStateInSystemSequence(props, 600L);
        this.assertExpectedNumberOfValuesAllocated(startValueAfterAllocation, currentValueAfterAllocation, props.incrementBy, props.numAllocated);
        this.assertExpectedCurrentValueForSequence(sequenceName, 355);
        this.assertExpectedNextValueForSequence(sequenceName, 360);
        this.assertExpectedNextValueForSequence(sequenceName, 365);
        this.assertExpectedNextValueForSequence(sequenceName, 370);
    }

    @Test
    public void testNextValuesForInsufficentCacheValuesAllocationLessThanCacheSize() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(5).startsWith(100).cacheSize(100).numAllocated(50L).minValue(100L).maxValue(6000L).build();
        this.createSequenceWithMinMax(props);
        int currentValueAfter51Allocations = 355;
        for (int i = 100; i <= currentValueAfter51Allocations; i += 5) {
            this.assertExpectedNextValueForSequence(sequenceName, i);
        }
        this.assertExpectedCurrentValueForSequence(sequenceName, currentValueAfter51Allocations);
        int currentValueAfterAllocation = 845;
        int startValueAfterAllocation = 600;
        this.reserveSlotsInBulkAndAssertValue(sequenceName, startValueAfterAllocation, props.numAllocated);
        this.assertBulkAllocationSucceeded(props, currentValueAfterAllocation, startValueAfterAllocation);
        this.assertExpectedCurrentValueForSequence(sequenceName, 845);
        this.assertExpectedNextValueForSequence(sequenceName, 850);
        this.assertExpectedNextValueForSequence(sequenceName, 855);
        this.assertExpectedNextValueForSequence(sequenceName, 860);
    }

    @Test
    public void testNextValuesForSequenceWithCycles() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties sequenceProps = new SequenceProperties.Builder().name(sequenceName).incrementBy(5).startsWith(100).cacheSize(100).numAllocated(1000L).minValue(100L).maxValue(900L).build();
        this.createSequenceWithMinMaxAndCycle(sequenceProps);
        this.assertExpectedNextValueForSequence(sequenceName, 100);
        this.assertExpectedCurrentValueForSequence(sequenceName, 100);
        this.assertExpectedNextValueForSequence(sequenceName, 105);
        try {
            this.conn.createStatement().executeQuery("SELECT NEXT " + sequenceProps.numAllocated + " VALUES FOR  " + sequenceName + "  LIMIT 1");
            Assert.fail((String)"Invoking SELECT NEXT VALUES should have failed as operation is not supported for sequences with Cycles.");
        }
        catch (SQLException e) {
            Assert.assertEquals((long)SQLExceptionCode.NUM_SEQ_TO_ALLOCATE_NOT_SUPPORTED.getErrorCode(), (long)e.getErrorCode());
            Assert.assertTrue((e.getNextException() == null ? 1 : 0) != 0);
        }
        this.assertExpectedCurrentValueForSequence(sequenceName, 105);
        this.assertExpectedNextValueForSequence(sequenceName, 110);
        this.assertExpectedNextValueForSequence(sequenceName, 115);
    }

    @Test
    public void testCurrentValueForAndNextValuesForExpressionsForSameSequence() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(1).cacheSize(100).numAllocated(1000L).build();
        this.createSequenceWithNoMinMax(props);
        this.assertExpectedNextValueForSequence(sequenceName, 1);
        this.assertExpectedCurrentValueForSequence(sequenceName, 1);
        this.assertExpectedNextValueForSequence(sequenceName, 2);
        int currentValueAfterAllocation = 1100;
        int nextValueAfterAllocation = currentValueAfterAllocation + props.incrementBy;
        int startValueAfterAllocation = 101;
        ResultSet rs = this.conn.createStatement().executeQuery("SELECT CURRENT VALUE FOR  " + sequenceName + " , NEXT " + props.numAllocated + " VALUES FOR  " + sequenceName + " ");
        Assert.assertTrue((boolean)rs.next());
        this.assertBulkAllocationSucceeded(props, currentValueAfterAllocation, startValueAfterAllocation);
        int currentValueFor = rs.getInt(1);
        int nextValuesFor = rs.getInt(2);
        Assert.assertEquals((String)"Expected the next value to be first value reserved", (long)startValueAfterAllocation, (long)nextValuesFor);
        Assert.assertEquals((String)"Expected current value to be the same as next value", (long)startValueAfterAllocation, (long)currentValueFor);
        this.assertExpectedCurrentValueForSequence(sequenceName, currentValueAfterAllocation);
        this.assertExpectedNextValueForSequence(sequenceName, nextValueAfterAllocation);
    }

    @Test
    public void testMultipleNextValuesForExpressionsForSameSequence() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(1).cacheSize(100).numAllocated(1000L).build();
        this.createSequenceWithNoMinMax(props);
        this.assertExpectedNextValueForSequence(sequenceName, 1);
        this.assertExpectedCurrentValueForSequence(sequenceName, 1);
        this.assertExpectedNextValueForSequence(sequenceName, 2);
        int currentValueAfterAllocation = 1100;
        int nextValueAfterAllocation = currentValueAfterAllocation + props.incrementBy;
        int startValueAfterAllocation = 101;
        ResultSet rs = this.conn.createStatement().executeQuery("SELECT NEXT 5 VALUES FOR  " + sequenceName + " , NEXT " + props.numAllocated + " VALUES FOR  " + sequenceName + " FROM \"SYSTEM\".\"SEQUENCE\"");
        Assert.assertTrue((boolean)rs.next());
        int firstValue = rs.getInt(1);
        int secondValue = rs.getInt(2);
        Assert.assertEquals((String)"Expected both expressions to return the same value", (long)firstValue, (long)secondValue);
        Assert.assertEquals((String)"Expected the value returned to be the highest allocation", (long)startValueAfterAllocation, (long)firstValue);
        this.assertBulkAllocationSucceeded(props, currentValueAfterAllocation, startValueAfterAllocation);
        this.assertExpectedCurrentValueForSequence(sequenceName, currentValueAfterAllocation);
        this.assertExpectedNextValueForSequence(sequenceName, nextValueAfterAllocation);
    }

    @Test
    public void testMultipleDifferentExpressionsForSameSequence() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(1).cacheSize(100).numAllocated(1000L).build();
        this.createSequenceWithNoMinMax(props);
        this.assertExpectedNextValueForSequence(sequenceName, 1);
        int currentValueAfterAllocation = 1100;
        int nextValueAfterAllocation = currentValueAfterAllocation + props.incrementBy;
        int startValueAfterAllocation = 101;
        ResultSet rs = this.conn.createStatement().executeQuery("SELECT NEXT VALUE FOR  " + sequenceName + " , NEXT " + props.numAllocated + " VALUES FOR  " + sequenceName + " , CURRENT VALUE FOR  " + sequenceName + " , NEXT 999 VALUES FOR  " + sequenceName + "  FROM \"SYSTEM\".\"SEQUENCE\"");
        Assert.assertTrue((boolean)rs.next());
        this.assertBulkAllocationSucceeded(props, currentValueAfterAllocation, startValueAfterAllocation);
        int previousVal = 0;
        for (int i = 1; i <= 4; ++i) {
            int currentVal = rs.getInt(i);
            if (i != 1) {
                Assert.assertEquals((String)"Expected all NEXT VALUE FOR and NEXT <n> VALUES FOR expressions to return the same value", (long)previousVal, (long)currentVal);
            }
            previousVal = currentVal;
        }
        this.assertExpectedCurrentValueForSequence(sequenceName, currentValueAfterAllocation);
        this.assertExpectedNextValueForSequence(sequenceName, nextValueAfterAllocation);
    }

    @Test
    public void testMultipleNextValuesForExpressionsForDifferentSequences() throws Exception {
        int i;
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        String secondSequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        this.conn.createStatement().execute("CREATE SEQUENCE  " + sequenceName + "  START WITH 30 INCREMENT BY 3 CACHE 100");
        this.conn.createStatement().execute("CREATE SEQUENCE " + secondSequenceName + " START WITH 100 INCREMENT BY 5 CACHE 50");
        ResultSet rs = this.conn.createStatement().executeQuery("SELECT NEXT 100 VALUES FOR  " + sequenceName + " , NEXT 1000 VALUES FOR " + secondSequenceName + "");
        Assert.assertTrue((boolean)rs.next());
        Assert.assertEquals((long)30L, (long)rs.getInt(1));
        Assert.assertEquals((long)100L, (long)rs.getInt(2));
        for (i = 330; i < 530; i += 3) {
            this.assertExpectedCurrentValueForSequence(sequenceName, i - 3);
            this.assertExpectedNextValueForSequence(sequenceName, i);
        }
        for (i = 5100; i < 7100; i += 5) {
            this.assertExpectedCurrentValueForSequence(secondSequenceName, i - 5);
            this.assertExpectedNextValueForSequence(secondSequenceName, i);
        }
    }

    @Test
    public void testExplainPlanValidatesSequences() throws Exception {
        int startValue;
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        String tableName = SequenceBulkAllocationIT.generateTableNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(3).startsWith(30).cacheSize(100).numAllocated(1000L).build();
        this.createSequenceWithNoMinMax(props);
        Connection genericConn = this.createGenericConnection();
        genericConn.createStatement().execute("CREATE TABLE " + tableName + " (k BIGINT NOT NULL PRIMARY KEY)");
        genericConn.close();
        int startValueAfterAllocation = 30;
        this.reserveSlotsInBulkAndAssertValue(sequenceName, startValueAfterAllocation, props.numAllocated);
        for (int i = 0; i < 3; ++i) {
            this.conn.createStatement().executeQuery("EXPLAIN SELECT NEXT 1000 VALUES FOR  " + sequenceName + "  FROM " + tableName);
        }
        this.assertExpectedStateInSystemSequence(props, 3030L);
        for (int i = startValue = 3030; i < startValue + 2 * props.cacheSize; i += props.incrementBy) {
            this.assertExpectedCurrentValueForSequence(sequenceName, i - props.incrementBy);
            this.assertExpectedNextValueForSequence(sequenceName, i);
        }
    }

    @Test
    public void testExplainPlanForNextValuesFor() throws Exception {
        String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        String tableName = SequenceBulkAllocationIT.generateTableNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(3).startsWith(30).cacheSize(100).numAllocated(1000L).build();
        this.createSequenceWithNoMinMax(props);
        Connection genericConn = this.createGenericConnection();
        genericConn.createStatement().execute("CREATE TABLE " + tableName + " (k BIGINT NOT NULL PRIMARY KEY)");
        genericConn.close();
        String query = "SELECT NEXT 1000 VALUES FOR  " + sequenceName + "  FROM " + tableName;
        ExplainPlan plan = this.conn.prepareStatement(query).unwrap(PhoenixPreparedStatement.class).optimizeQuery().getExplainPlan();
        ExplainPlanAttributes explainPlanAttributes = plan.getPlanStepsAsAttributes();
        Assert.assertEquals((Object)"PARALLEL 1-WAY", (Object)explainPlanAttributes.getIteratorTypeAndScanSize());
        Assert.assertEquals((Object)"FULL SCAN ", (Object)explainPlanAttributes.getExplainScanType());
        Assert.assertEquals((Object)tableName, (Object)explainPlanAttributes.getTableName());
        Assert.assertEquals((Object)"SERVER FILTER BY FIRST KEY ONLY", (Object)explainPlanAttributes.getServerWhereFilter());
        Assert.assertEquals((long)1L, (long)explainPlanAttributes.getClientSequenceCount().intValue());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testNextValuesForMixedWithNextValueForMultiThreaded() throws Exception {
        final String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(1).cacheSize(100).numAllocated(1000L).build();
        this.createSequenceWithNoMinMax(props);
        this.assertExpectedNextValueForSequence(sequenceName, 1);
        this.assertExpectedCurrentValueForSequence(sequenceName, 1);
        this.assertExpectedNextValueForSequence(sequenceName, 2);
        long startValueAfterAllocation1 = 101L;
        long startValueAfterAllocation2 = 1101L;
        final long numSlotToAllocate = props.numAllocated;
        ExecutorService executorService = Executors.newCachedThreadPool();
        try {
            final CountDownLatch latch1 = new CountDownLatch(1);
            final CountDownLatch latch2 = new CountDownLatch(1);
            Callable<Long> task1 = new Callable<Long>(){

                @Override
                public Long call() throws Exception {
                    ResultSet rs = SequenceBulkAllocationIT.this.conn.createStatement().executeQuery("SELECT NEXT " + numSlotToAllocate + " VALUES FOR  " + sequenceName + " ");
                    latch1.countDown();
                    latch2.await();
                    rs.next();
                    return rs.getLong(1);
                }
            };
            Callable<Long> task2 = new Callable<Long>(){

                @Override
                public Long call() throws Exception {
                    latch1.await();
                    ResultSet rs = SequenceBulkAllocationIT.this.conn.createStatement().executeQuery("SELECT NEXT VALUE FOR  " + sequenceName);
                    rs.next();
                    long retVal = rs.getLong(1);
                    latch2.countDown();
                    return retVal;
                }
            };
            List futures = executorService.invokeAll(Lists.newArrayList((Object[])new Callable[]{task1, task2}), 20L, TimeUnit.SECONDS);
            Assert.assertEquals((long)101L, (long)((Long)futures.get(0).get(10L, TimeUnit.SECONDS)));
            Assert.assertEquals((long)1101L, (long)((Long)futures.get(1).get(10L, TimeUnit.SECONDS)));
        }
        finally {
            executorService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testMultipleNextValuesWithDiffAllocsForMultiThreaded() throws Exception {
        final String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(1).cacheSize(100).numAllocated(1000L).build();
        this.createSequenceWithNoMinMax(props);
        this.assertExpectedNextValueForSequence(sequenceName, 1);
        this.assertExpectedCurrentValueForSequence(sequenceName, 1);
        this.assertExpectedNextValueForSequence(sequenceName, 2);
        long startValueAfterAllocation1 = 101L;
        long startValueAfterAllocation2 = 1101L;
        long numSlotToAllocate1 = 1000L;
        long numSlotToAllocate2 = 100L;
        ExecutorService executorService = Executors.newCachedThreadPool();
        try {
            final CountDownLatch latch1 = new CountDownLatch(1);
            final CountDownLatch latch2 = new CountDownLatch(1);
            Callable<Long> task1 = new Callable<Long>(){

                @Override
                public Long call() throws Exception {
                    ResultSet rs = SequenceBulkAllocationIT.this.conn.createStatement().executeQuery("SELECT NEXT 1000 VALUES FOR  " + sequenceName + " ");
                    rs.next();
                    latch1.countDown();
                    latch2.await();
                    return rs.getLong(1);
                }
            };
            Callable<Long> task2 = new Callable<Long>(){

                @Override
                public Long call() throws Exception {
                    latch1.await();
                    ResultSet rs = SequenceBulkAllocationIT.this.conn.createStatement().executeQuery("SELECT NEXT 100 VALUES FOR  " + sequenceName + " ");
                    rs.next();
                    long retVal = rs.getLong(1);
                    latch2.countDown();
                    return retVal;
                }
            };
            List futures = executorService.invokeAll(Lists.newArrayList((Object[])new Callable[]{task1, task2}), 5L, TimeUnit.SECONDS);
            Long retValue1 = (Long)futures.get(0).get(5L, TimeUnit.SECONDS);
            Assert.assertEquals((long)101L, (long)retValue1);
            Long retValue2 = (Long)futures.get(1).get(5L, TimeUnit.SECONDS);
            Assert.assertEquals((long)1101L, (long)retValue2);
        }
        finally {
            executorService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testMultipleNextValuesWithSameAllocsForMultiThreaded() throws Exception {
        final String sequenceName = SequenceBulkAllocationIT.generateSequenceNameWithSchema();
        SequenceProperties props = new SequenceProperties.Builder().name(sequenceName).incrementBy(1).startsWith(1).cacheSize(100).numAllocated(1000L).build();
        this.createSequenceWithNoMinMax(props);
        long startValueAfterAllocation1 = 1L;
        long startValueAfterAllocation2 = 1001L;
        long numSlotToAllocate1 = 1000L;
        long numSlotToAllocate2 = 1000L;
        ExecutorService executorService = Executors.newCachedThreadPool();
        try {
            final CountDownLatch latch1 = new CountDownLatch(1);
            final CountDownLatch latch2 = new CountDownLatch(1);
            Callable<Long> task1 = new Callable<Long>(){

                @Override
                public Long call() throws Exception {
                    ResultSet rs = SequenceBulkAllocationIT.this.conn.createStatement().executeQuery("SELECT NEXT 1000 VALUES FOR  " + sequenceName + " ");
                    latch1.countDown();
                    latch2.await();
                    rs.next();
                    return rs.getLong(1);
                }
            };
            Callable<Long> task2 = new Callable<Long>(){

                @Override
                public Long call() throws Exception {
                    latch1.await();
                    ResultSet rs = SequenceBulkAllocationIT.this.conn.createStatement().executeQuery("SELECT NEXT 1000 VALUES FOR  " + sequenceName + " ");
                    rs.next();
                    long retVal = rs.getLong(1);
                    latch2.countDown();
                    return retVal;
                }
            };
            List futures = executorService.invokeAll(Lists.newArrayList((Object[])new Callable[]{task1, task2}), 5L, TimeUnit.SECONDS);
            Assert.assertEquals((long)1001L, (long)((Long)futures.get(0).get(5L, TimeUnit.SECONDS)));
            Assert.assertEquals((long)1L, (long)((Long)futures.get(1).get(5L, TimeUnit.SECONDS)));
        }
        finally {
            executorService.shutdown();
        }
    }

    private void assertBulkAllocationSucceeded(SequenceProperties props, int currentValueAfterAllocation, int startValueAfterAllocation) throws SQLException {
        int nextValueAfterAllocation = currentValueAfterAllocation + props.incrementBy;
        this.assertExpectedStateInSystemSequence(props, nextValueAfterAllocation);
        this.assertExpectedNumberOfValuesAllocated(startValueAfterAllocation, currentValueAfterAllocation, props.incrementBy, props.numAllocated);
    }

    private void createSequenceWithNoMinMax(SequenceProperties props) throws SQLException {
        this.conn.createStatement().execute(String.format(CREATE_SEQUENCE_NO_MIN_MAX_TEMPLATE, props.name, props.startsWith, props.incrementBy, props.cacheSize));
    }

    private void createSequenceWithMinMax(SequenceProperties props) throws SQLException {
        this.conn.createStatement().execute(String.format(CREATE_SEQUENCE_WITH_MIN_MAX_TEMPLATE, props.name, props.startsWith, props.incrementBy, props.minValue, props.maxValue, props.cacheSize));
    }

    private void createSequenceWithMinMaxAndCycle(SequenceProperties props) throws SQLException {
        this.conn.createStatement().execute(String.format(CREATE_SEQUENCE_WITH_MIN_MAX_AND_CYCLE_TEMPLATE, props.name, props.startsWith, props.incrementBy, props.minValue, props.maxValue, props.cacheSize));
    }

    private void reserveSlotsInBulkAndAssertValue(String sequenceName, long expectedValue, long numSlotToAllocate) throws SQLException {
        ResultSet rs = this.conn.createStatement().executeQuery("SELECT NEXT " + numSlotToAllocate + " VALUES FOR  " + sequenceName + " ");
        Assert.assertTrue((boolean)rs.next());
        Assert.assertEquals((long)expectedValue, (long)rs.getInt(1));
    }

    private void reserveSlotsInBulkUsingBindsAndAssertValue(String sequenceName, int expectedValue, long numSlotToAllocate) throws SQLException {
        PreparedStatement ps = this.conn.prepareStatement("SELECT NEXT ? VALUES FOR  " + sequenceName + " ");
        ps.getMetaData();
        ps.setLong(1, numSlotToAllocate);
        ResultSet rs = ps.executeQuery();
        Assert.assertTrue((boolean)rs.next());
        int retValue = rs.getInt(1);
        Assert.assertEquals((long)expectedValue, (long)retValue);
    }

    private void assertExpectedCurrentValueForSequence(String sequenceName, int expectedValue) throws SQLException {
        ResultSet rs = this.conn.createStatement().executeQuery(String.format(SELECT_CURRENT_VALUE_SQL, sequenceName));
        Assert.assertTrue((boolean)rs.next());
        Assert.assertEquals((long)expectedValue, (long)rs.getInt(1));
    }

    private void assertExpectedNextValueForSequence(String sequenceName, int expectedValue) throws SQLException {
        ResultSet rs = this.conn.createStatement().executeQuery(String.format(SELECT_NEXT_VALUE_SQL, sequenceName));
        Assert.assertTrue((boolean)rs.next());
        Assert.assertEquals((long)expectedValue, (long)rs.getInt(1));
    }

    private Connection createGenericConnection() throws Exception {
        Properties props = PropertiesUtil.deepCopy((Properties)TestUtil.TEST_PROPERTIES);
        return DriverManager.getConnection(SequenceBulkAllocationIT.getUrl(), props);
    }

    private void createConnection() throws Exception {
        if (this.conn != null) {
            this.conn.close();
        }
        if (this.tenantId != null) {
            Properties props = PropertiesUtil.deepCopy((Properties)TestUtil.TEST_PROPERTIES);
            this.conn = DriverManager.getConnection(SequenceBulkAllocationIT.getUrl() + ';' + "TenantId" + '=' + "tenant1", props);
        } else {
            Properties props = PropertiesUtil.deepCopy((Properties)TestUtil.TEST_PROPERTIES);
            this.conn = DriverManager.getConnection(SequenceBulkAllocationIT.getUrl(), props);
        }
    }

    private void assertExpectedStateInSystemSequence(SequenceProperties props, long currentValue) throws SQLException {
        ResultSet rs = this.conn.createStatement().executeQuery("SELECT start_with, current_value, increment_by, cache_size, min_value, max_value, cycle_flag, sequence_schema, sequence_name FROM \"SYSTEM\".\"SEQUENCE\" where sequence_name='" + props.getNameWithoutSchema() + "'");
        Assert.assertTrue((boolean)rs.next());
        Assert.assertEquals((String)"start_with", (long)props.startsWith, (long)rs.getLong("start_with"));
        Assert.assertEquals((String)"increment_by", (long)props.incrementBy, (long)rs.getLong("increment_by"));
        Assert.assertEquals((String)"cache_size", (long)props.cacheSize, (long)rs.getLong("cache_size"));
        Assert.assertEquals((String)"cycle_flag", (Object)false, (Object)rs.getBoolean("cycle_flag"));
        Assert.assertEquals((String)"sequence_schema", (Object)props.getSchemaName(), (Object)rs.getString("sequence_schema"));
        Assert.assertEquals((String)"sequence_name", (Object)props.getNameWithoutSchema(), (Object)rs.getString("sequence_name"));
        Assert.assertEquals((String)"current_value", (long)currentValue, (long)rs.getLong("current_value"));
        Assert.assertEquals((String)"min_value", (long)props.minValue, (long)rs.getLong("min_value"));
        Assert.assertEquals((String)"max_value", (long)props.maxValue, (long)rs.getLong("max_value"));
        Assert.assertFalse((boolean)rs.next());
    }

    private void assertExpectedNumberOfValuesAllocated(long firstValue, long lastValue, int incrementBy, long numAllocated) {
        int cnt = 0;
        long i = firstValue;
        while (incrementBy > 0 ? i <= lastValue : i >= lastValue) {
            ++cnt;
            i += (long)incrementBy;
        }
        Assert.assertEquals((String)("Incorrect number of values allocated: " + cnt), (long)numAllocated, (long)cnt);
    }

    private static class SequenceProperties {
        private final long numAllocated;
        private final int incrementBy;
        private final int startsWith;
        private final int cacheSize;
        private final long minValue;
        private final long maxValue;
        private final String name;

        public SequenceProperties(Builder builder) {
            this.numAllocated = builder.numAllocated;
            this.incrementBy = builder.incrementBy;
            this.startsWith = builder.startsWith;
            this.cacheSize = builder.cacheSize;
            this.minValue = builder.minValue;
            this.maxValue = builder.maxValue;
            this.name = builder.name;
        }

        private String getSchemaName() {
            return this.name.substring(0, this.name.indexOf("."));
        }

        private String getNameWithoutSchema() {
            return this.name.substring(this.name.indexOf(".") + 1, this.name.length());
        }

        private static class Builder {
            long maxValue = Long.MAX_VALUE;
            long minValue = Long.MIN_VALUE;
            long numAllocated = 100L;
            int incrementBy = 1;
            int startsWith = 1;
            int cacheSize = 100;
            String name = null;

            private Builder() {
            }

            public Builder numAllocated(long numAllocated) {
                this.numAllocated = numAllocated;
                return this;
            }

            public Builder startsWith(int startsWith) {
                this.startsWith = startsWith;
                return this;
            }

            public Builder cacheSize(int cacheSize) {
                this.cacheSize = cacheSize;
                return this;
            }

            public Builder incrementBy(int incrementBy) {
                this.incrementBy = incrementBy;
                return this;
            }

            public Builder minValue(long minValue) {
                this.minValue = minValue;
                return this;
            }

            public Builder maxValue(long maxValue) {
                this.maxValue = maxValue;
                return this;
            }

            public Builder name(String name) {
                this.name = name;
                return this;
            }

            public SequenceProperties build() {
                return new SequenceProperties(this);
            }
        }
    }
}

