/*
 * Decompiled with CFR 0.152.
 */
package org.apache.phoenix.mapreduce.index;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.logging.FileHandler;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.CoprocessorDescriptorBuilder;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.client.TableDescriptorBuilder;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.apache.phoenix.hbase.index.IndexRegionObserver;
import org.apache.phoenix.hbase.index.Indexer;
import org.apache.phoenix.index.GlobalIndexChecker;
import org.apache.phoenix.index.PhoenixIndexBuilder;
import org.apache.phoenix.index.PhoenixIndexCodec;
import org.apache.phoenix.jdbc.PhoenixConnection;
import org.apache.phoenix.mapreduce.index.IndexTool;
import org.apache.phoenix.mapreduce.util.ConnectionUtil;
import org.apache.phoenix.query.ConnectionQueryServices;
import org.apache.phoenix.schema.PName;
import org.apache.phoenix.schema.PTable;
import org.apache.phoenix.schema.PTableType;
import org.apache.phoenix.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.phoenix.thirdparty.com.google.common.base.Strings;
import org.apache.phoenix.thirdparty.org.apache.commons.cli.CommandLine;
import org.apache.phoenix.thirdparty.org.apache.commons.cli.DefaultParser;
import org.apache.phoenix.thirdparty.org.apache.commons.cli.HelpFormatter;
import org.apache.phoenix.thirdparty.org.apache.commons.cli.Option;
import org.apache.phoenix.thirdparty.org.apache.commons.cli.Options;
import org.apache.phoenix.thirdparty.org.apache.commons.cli.ParseException;
import org.apache.phoenix.util.EnvironmentEdgeManager;
import org.apache.phoenix.util.IndexUtil;
import org.apache.phoenix.util.MetaDataUtil;
import org.apache.phoenix.util.SchemaUtil;

public class IndexUpgradeTool
extends Configured
implements Tool {
    private static final Logger LOGGER = Logger.getLogger(IndexUpgradeTool.class.getName());
    private static final String INDEX_REBUILD_OPTION_SHORT_OPT = "rb";
    private static final String INDEX_TOOL_OPTION_SHORT_OPT = "tool";
    private static final Option OPERATION_OPTION = new Option("o", "operation", true, "[Required] Operation to perform (upgrade/rollback)");
    private static final Option TABLE_OPTION = new Option("tb", "table", true, "[Required] Tables list ex. table1,table2");
    private static final Option TABLE_CSV_FILE_OPTION = new Option("f", "file", true, "[Optional] Tables list in a csv file");
    private static final Option DRY_RUN_OPTION = new Option("d", "dry-run", false, "[Optional] If passed this will output steps that will be executed");
    private static final Option HELP_OPTION = new Option("h", "help", false, "Help");
    private static final Option LOG_FILE_OPTION = new Option("lf", "logfile", true, "[Optional] Log file path where the logs are written");
    private static final Option INDEX_REBUILD_OPTION = new Option("rb", "index-rebuild", false, "[Optional] Rebuild the indexes. Set -tool to pass options to IndexTool.");
    private static final Option INDEX_TOOL_OPTION = new Option("tool", "index-tool", true, "[Optional] Options to pass to indexTool when rebuilding indexes. Set -rb to rebuild the index.");
    public static final String UPGRADE_OP = "upgrade";
    public static final String ROLLBACK_OP = "rollback";
    private static final String GLOBAL_INDEX_ID = "#NA#";
    private IndexTool indexingTool;
    private HashMap<String, HashSet<String>> tablesAndIndexes = new HashMap();
    private HashMap<String, HashMap<String, IndexInfo>> rebuildMap = new HashMap();
    private HashMap<String, String> prop = new HashMap();
    private HashMap<String, String> emptyProp = new HashMap();
    private boolean dryRun;
    private boolean upgrade;
    private boolean rebuild;
    private String operation;
    private String inputTables;
    private String logFile;
    private String inputFile;
    private boolean isWaitComplete = false;
    private String indexToolOpts;
    private boolean test = false;
    private boolean failUpgradeTask = false;
    private boolean failDowngradeTask = false;
    private boolean hasFailure = false;

    public void setDryRun(boolean dryRun) {
        this.dryRun = dryRun;
    }

    public void setInputTables(String inputTables) {
        this.inputTables = inputTables;
    }

    public void setLogFile(String logFile) {
        this.logFile = logFile;
    }

    public void setInputFile(String inputFile) {
        this.inputFile = inputFile;
    }

    public void setTest(boolean test) {
        this.test = test;
    }

    public boolean getIsWaitComplete() {
        return this.isWaitComplete;
    }

    public boolean getDryRun() {
        return this.dryRun;
    }

    public String getInputTables() {
        return this.inputTables;
    }

    public String getLogFile() {
        return this.logFile;
    }

    public String getOperation() {
        return this.operation;
    }

    public boolean getIsRebuild() {
        return this.rebuild;
    }

    public String getIndexToolOpts() {
        return this.indexToolOpts;
    }

    @VisibleForTesting
    public void setFailUpgradeTask(boolean failInitialTask) {
        this.failUpgradeTask = failInitialTask;
    }

    public void setFailDowngradeTask(boolean failRollbackTask) {
        this.failDowngradeTask = failRollbackTask;
    }

    public IndexUpgradeTool(String mode, String tables, String inputFile, String outputFile, boolean dryRun, IndexTool indexTool, boolean rebuild) {
        this.operation = mode;
        this.inputTables = tables;
        this.inputFile = inputFile;
        this.logFile = outputFile;
        this.dryRun = dryRun;
        this.indexingTool = indexTool;
        this.rebuild = rebuild;
    }

    public IndexUpgradeTool() {
    }

    public int run(String[] args) throws Exception {
        CommandLine cmdLine = null;
        try {
            cmdLine = this.parseOptions(args);
            LOGGER.info("Index Upgrade tool initiated: " + String.join((CharSequence)",", args));
        }
        catch (IllegalStateException e) {
            this.printHelpAndExit(e.getMessage(), this.getOptions());
        }
        try {
            this.initializeTool(cmdLine);
            this.prepareToolSetup();
            this.executeTool();
        }
        catch (Exception e) {
            e.printStackTrace();
            this.hasFailure = true;
        }
        if (this.hasFailure) {
            return -1;
        }
        return 0;
    }

    @VisibleForTesting
    public CommandLine parseOptions(String[] args) {
        Options options = this.getOptions();
        DefaultParser parser = DefaultParser.builder().setAllowPartialMatching(false).setStripLeadingAndTrailingQuotes(Boolean.valueOf(false)).build();
        CommandLine cmdLine = null;
        try {
            cmdLine = parser.parse(options, args);
        }
        catch (ParseException e) {
            this.printHelpAndExit("severe parsing command line options: " + e.getMessage(), options);
        }
        if (cmdLine.hasOption(HELP_OPTION.getOpt())) {
            this.printHelpAndExit(options, 0);
        }
        if (!cmdLine.hasOption(OPERATION_OPTION.getOpt())) {
            throw new IllegalStateException(OPERATION_OPTION.getLongOpt() + " is a mandatory parameter");
        }
        if (cmdLine.hasOption(DRY_RUN_OPTION.getOpt()) && !cmdLine.hasOption(LOG_FILE_OPTION.getOpt())) {
            throw new IllegalStateException("Log file with " + TABLE_OPTION.getLongOpt() + " is mandatory if " + DRY_RUN_OPTION.getLongOpt() + " is passed");
        }
        if (!cmdLine.hasOption(TABLE_OPTION.getOpt()) && !cmdLine.hasOption(TABLE_CSV_FILE_OPTION.getOpt())) {
            throw new IllegalStateException("Tables list should be passed in either with" + TABLE_OPTION.getLongOpt() + " or " + TABLE_CSV_FILE_OPTION.getLongOpt());
        }
        if (cmdLine.hasOption(TABLE_OPTION.getOpt()) && cmdLine.hasOption(TABLE_CSV_FILE_OPTION.getOpt())) {
            throw new IllegalStateException("Tables list passed in with" + TABLE_OPTION.getLongOpt() + " and " + TABLE_CSV_FILE_OPTION.getLongOpt() + "; specify only one.");
        }
        if (cmdLine.hasOption(INDEX_TOOL_OPTION.getOpt()) && !cmdLine.hasOption(INDEX_REBUILD_OPTION.getOpt())) {
            throw new IllegalStateException("Index tool options should be passed in with " + INDEX_REBUILD_OPTION.getLongOpt());
        }
        return cmdLine;
    }

    private void printHelpAndExit(String severeMessage, Options options) {
        System.err.println(severeMessage);
        this.printHelpAndExit(options, 1);
    }

    private void printHelpAndExit(Options options, int exitCode) {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp("help", options);
        System.exit(exitCode);
    }

    private Options getOptions() {
        Options options = new Options();
        options.addOption(OPERATION_OPTION);
        TABLE_OPTION.setOptionalArg(true);
        options.addOption(TABLE_OPTION);
        TABLE_CSV_FILE_OPTION.setOptionalArg(true);
        options.addOption(TABLE_CSV_FILE_OPTION);
        DRY_RUN_OPTION.setOptionalArg(true);
        options.addOption(DRY_RUN_OPTION);
        LOG_FILE_OPTION.setOptionalArg(true);
        options.addOption(LOG_FILE_OPTION);
        options.addOption(HELP_OPTION);
        INDEX_REBUILD_OPTION.setOptionalArg(true);
        options.addOption(INDEX_REBUILD_OPTION);
        INDEX_TOOL_OPTION.setOptionalArg(true);
        options.addOption(INDEX_TOOL_OPTION);
        return options;
    }

    @VisibleForTesting
    public void initializeTool(CommandLine cmdLine) {
        this.operation = cmdLine.getOptionValue(OPERATION_OPTION.getOpt());
        this.inputTables = cmdLine.getOptionValue(TABLE_OPTION.getOpt());
        this.logFile = cmdLine.getOptionValue(LOG_FILE_OPTION.getOpt());
        this.inputFile = cmdLine.getOptionValue(TABLE_CSV_FILE_OPTION.getOpt());
        this.dryRun = cmdLine.hasOption(DRY_RUN_OPTION.getOpt());
        this.rebuild = cmdLine.hasOption(INDEX_REBUILD_OPTION.getOpt());
        this.indexToolOpts = cmdLine.getOptionValue(INDEX_TOOL_OPTION.getOpt());
    }

    @VisibleForTesting
    public void prepareToolSetup() {
        try {
            if (this.logFile != null) {
                FileHandler fh = new FileHandler(this.logFile);
                fh.setFormatter(new SimpleFormatter());
                LOGGER.addHandler(fh);
            }
            this.prop.put("index.builder", PhoenixIndexBuilder.class.getName());
            this.prop.put("org.apache.hadoop.hbase.index.codec.class", PhoenixIndexCodec.class.getName());
            if (this.inputTables == null) {
                this.inputTables = new String(Files.readAllBytes(Paths.get(this.inputFile, new String[0])), StandardCharsets.UTF_8);
            }
            if (this.inputTables == null) {
                LOGGER.severe("Tables' list is not available; use -tb or -f option");
            }
            LOGGER.info("list of tables passed: " + this.inputTables);
            if (this.operation.equalsIgnoreCase(UPGRADE_OP)) {
                this.upgrade = true;
            } else if (this.operation.equalsIgnoreCase(ROLLBACK_OP)) {
                this.upgrade = false;
            } else {
                throw new IllegalStateException("Invalid option provided for " + OPERATION_OPTION.getOpt() + " expected values: {upgrade, rollback}");
            }
            if (this.dryRun) {
                LOGGER.info("This is the beginning of the tool with dry run.");
            }
        }
        catch (IOException e) {
            LOGGER.severe("Something went wrong " + e);
            System.exit(-1);
        }
    }

    private static void setRpcRetriesAndTimeouts(Configuration conf) {
        long indexRebuildQueryTimeoutMs = conf.getLong("phoenix.index.rebuild.query.timeout", 9002100L);
        long indexRebuildRPCTimeoutMs = conf.getLong("phoenix.index.rebuild.rpc.timeout", 1800000L);
        long indexRebuildClientScannerTimeOutMs = conf.getLong("phoenix.index.rebuild.client.scanner.timeout", 1800000L);
        int indexRebuildRpcRetriesCounter = conf.getInt("phoenix.index.rebuild.rpc.retries.counter", 5);
        conf.setLong("phoenix.query.timeoutMs", indexRebuildQueryTimeoutMs);
        conf.setLong("hbase.rpc.timeout", indexRebuildRPCTimeoutMs);
        conf.setLong("hbase.client.scanner.timeout.period", indexRebuildClientScannerTimeOutMs);
        conf.setInt("hbase.client.retries.number", indexRebuildRpcRetriesCounter);
    }

    @VisibleForTesting
    public static Connection getConnection(Configuration conf) throws SQLException {
        IndexUpgradeTool.setRpcRetriesAndTimeouts(conf);
        return ConnectionUtil.getInputConnection((Configuration)conf);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @VisibleForTesting
    public int executeTool() {
        Configuration conf = HBaseConfiguration.addHbaseResources((Configuration)this.getConf());
        try (Connection conn = IndexUpgradeTool.getConnection(conf);){
            ConnectionQueryServices queryServices = conn.unwrap(PhoenixConnection.class).getQueryServices();
            boolean status = this.extractTablesAndIndexes(conn.unwrap(PhoenixConnection.class));
            if (!status) return -1;
            int n = this.executeTool(conn, queryServices, conf);
            return n;
        }
        catch (SQLException e) {
            LOGGER.severe("Something went wrong in executing tool " + e);
        }
        return -1;
    }

    private int executeTool(Connection conn, ConnectionQueryServices queryServices, Configuration conf) {
        ArrayList<String> immutableList = new ArrayList<String>();
        ArrayList<String> mutableList = new ArrayList<String>();
        for (Map.Entry<String, HashSet<String>> entry : this.tablesAndIndexes.entrySet()) {
            String dataTableFullName = entry.getKey();
            try {
                PTable dataTable = conn.unwrap(PhoenixConnection.class).getTableNoCache(dataTableFullName);
                if (dataTable.isImmutableRows()) {
                    immutableList.add(dataTableFullName);
                    continue;
                }
                mutableList.add(dataTableFullName);
            }
            catch (SQLException e) {
                LOGGER.severe("Something went wrong while getting the PTable " + dataTableFullName + " " + e);
                return -1;
            }
        }
        long startWaitTime = this.executeToolForImmutableTables(queryServices, immutableList);
        this.executeToolForMutableTables(conn, queryServices, conf, mutableList);
        this.enableImmutableTables(queryServices, immutableList, startWaitTime);
        this.rebuildIndexes(conn, conf, immutableList);
        if (this.hasFailure) {
            return -1;
        }
        return 0;
    }

    private long executeToolForImmutableTables(ConnectionQueryServices queryServices, List<String> immutableList) {
        if (immutableList.isEmpty()) {
            return 0L;
        }
        LOGGER.info("Started " + this.operation + " for immutable tables");
        ArrayList<String> failedTables = new ArrayList<String>();
        for (String dataTableFullName : immutableList) {
            try {
                Admin admin = queryServices.getAdmin();
                try {
                    HashSet<String> indexes = this.tablesAndIndexes.get(dataTableFullName);
                    LOGGER.info("Executing " + this.operation + " of " + dataTableFullName + " (immutable)");
                    this.disableTable(admin, dataTableFullName, indexes);
                    this.modifyTable(admin, dataTableFullName, indexes);
                }
                finally {
                    if (admin == null) continue;
                    admin.close();
                }
            }
            catch (Throwable e) {
                LOGGER.severe("Something went wrong while disabling or modifying immutable table " + e);
                this.handleFailure(queryServices, dataTableFullName, immutableList, failedTables);
            }
        }
        immutableList.removeAll(failedTables);
        long startWaitTime = EnvironmentEdgeManager.currentTimeMillis();
        return startWaitTime;
    }

    private void executeToolForMutableTables(Connection conn, ConnectionQueryServices queryServices, Configuration conf, ArrayList<String> mutableTables) {
        if (mutableTables.isEmpty()) {
            return;
        }
        LOGGER.info("Started " + this.operation + " for mutable tables");
        ArrayList<String> failedTables = new ArrayList<String>();
        for (String dataTableFullName : mutableTables) {
            try {
                Admin admin = queryServices.getAdmin();
                try {
                    HashSet<String> indexes = this.tablesAndIndexes.get(dataTableFullName);
                    LOGGER.info("Executing " + this.operation + " of " + dataTableFullName);
                    this.disableTable(admin, dataTableFullName, indexes);
                    this.modifyTable(admin, dataTableFullName, indexes);
                    this.enableTable(admin, dataTableFullName, indexes);
                    LOGGER.info("Completed " + this.operation + " of " + dataTableFullName);
                }
                finally {
                    if (admin == null) continue;
                    admin.close();
                }
            }
            catch (Throwable e) {
                LOGGER.severe("Something went wrong while executing " + this.operation + " steps for " + dataTableFullName + " " + e);
                this.handleFailure(queryServices, dataTableFullName, mutableTables, failedTables);
            }
        }
        mutableTables.removeAll(failedTables);
        this.rebuildIndexes(conn, conf, mutableTables);
    }

    private void handleFailure(ConnectionQueryServices queryServices, String dataTableFullName, List<String> tableList, List<String> failedTables) {
        this.hasFailure = true;
        LOGGER.info("Performing error handling to revert the steps taken during " + this.operation);
        HashSet<String> indexes = this.tablesAndIndexes.get(dataTableFullName);
        try (Admin admin = queryServices.getAdmin();){
            this.upgrade = !this.upgrade;
            this.disableTable(admin, dataTableFullName, indexes);
            this.modifyTable(admin, dataTableFullName, indexes);
            this.enableTable(admin, dataTableFullName, indexes);
            this.upgrade = !this.upgrade;
            this.tablesAndIndexes.remove(dataTableFullName);
            failedTables.add(dataTableFullName);
            LOGGER.severe(dataTableFullName + " has been removed from the list as tool failed to perform " + this.operation);
        }
        catch (Throwable e) {
            LOGGER.severe("Revert of the " + this.operation + " failed in error handling, re-enabling tables and then throwing runtime exception");
            LOGGER.severe("Confirm the state for " + this.getSubListString(tableList, dataTableFullName));
            try (Admin admin2 = queryServices.getAdmin();){
                this.enableTable(admin2, dataTableFullName, indexes);
            }
            catch (Exception ex) {
                throw new RuntimeException("Error re-enabling tables after rollback failure. Original exception that caused the rollback: [" + e.toString() + " ]", ex);
            }
            throw new RuntimeException(e);
        }
    }

    private void enableImmutableTables(ConnectionQueryServices queryServices, ArrayList<String> immutableList, long startWaitTime) {
        if (immutableList.isEmpty()) {
            return;
        }
        while (true) {
            long waitMore;
            if ((waitMore = this.getWaitMoreTime(startWaitTime)) <= 0L) break;
            try {
                Thread.sleep(waitMore);
                this.isWaitComplete = true;
            }
            catch (InterruptedException e) {
                LOGGER.warning("Sleep before starting index rebuild is interrupted. Attempting to sleep again! " + e.getMessage());
            }
        }
        this.isWaitComplete = true;
        for (String dataTableFullName : immutableList) {
            try {
                Admin admin = queryServices.getAdmin();
                try {
                    HashSet<String> indexes = this.tablesAndIndexes.get(dataTableFullName);
                    this.enableTable(admin, dataTableFullName, indexes);
                }
                finally {
                    if (admin == null) continue;
                    admin.close();
                }
            }
            catch (IOException | SQLException e) {
                LOGGER.severe("Something went wrong while enabling immutable table " + e);
                this.tablesAndIndexes.remove(dataTableFullName);
                immutableList.remove(dataTableFullName);
                throw new RuntimeException("Manually enable the following tables " + this.getSubListString(immutableList, dataTableFullName) + " and run the index rebuild ", e);
            }
        }
    }

    private String getSubListString(List<String> tableList, String dataTableFullName) {
        return StringUtils.join((CharSequence)",", tableList.subList(tableList.indexOf(dataTableFullName), tableList.size()));
    }

    private long getWaitMoreTime(long startWaitTime) {
        int waitTime = 11;
        long endWaitTime = EnvironmentEdgeManager.currentTimeMillis();
        if (this.test || this.dryRun) {
            return 0L;
        }
        return (long)(waitTime * 60000) - Math.abs(endWaitTime - startWaitTime);
    }

    private void disableTable(Admin admin, String dataTable, HashSet<String> indexes) throws IOException {
        if (admin.isTableEnabled(TableName.valueOf((String)dataTable))) {
            if (!this.dryRun) {
                admin.disableTable(TableName.valueOf((String)dataTable));
            }
            LOGGER.info("Disabled data table " + dataTable);
        } else {
            LOGGER.info("Data table " + dataTable + " is already disabled");
        }
        for (String indexName : indexes) {
            if (admin.isTableEnabled(TableName.valueOf((String)indexName))) {
                if (!this.dryRun) {
                    admin.disableTable(TableName.valueOf((String)indexName));
                }
                LOGGER.info("Disabled index table " + indexName);
                continue;
            }
            LOGGER.info("Index table " + indexName + " is already disabled");
        }
    }

    private void modifyTable(Admin admin, String dataTableFullName, HashSet<String> indexes) throws IOException {
        if (this.upgrade) {
            this.modifyIndexTable(admin, indexes);
            this.modifyDataTable(admin, dataTableFullName);
            if (this.test && this.failUpgradeTask) {
                throw new RuntimeException("Test requested upgrade failure");
            }
        } else {
            this.modifyDataTable(admin, dataTableFullName);
            this.modifyIndexTable(admin, indexes);
            if (this.test && this.failDowngradeTask) {
                throw new RuntimeException("Test requested downgrade failure");
            }
        }
    }

    private void enableTable(Admin admin, String dataTable, Set<String> indexes) throws IOException {
        if (!admin.isTableEnabled(TableName.valueOf((String)dataTable))) {
            if (!this.dryRun) {
                admin.enableTable(TableName.valueOf((String)dataTable));
            }
            LOGGER.info("Enabled data table " + dataTable);
        } else {
            LOGGER.info("Data table " + dataTable + " is already enabled");
        }
        for (String indexName : indexes) {
            if (!admin.isTableEnabled(TableName.valueOf((String)indexName))) {
                if (!this.dryRun) {
                    admin.enableTable(TableName.valueOf((String)indexName));
                }
                LOGGER.info("Enabled index table " + indexName);
                continue;
            }
            LOGGER.info("Index table " + indexName + " is already enabled");
        }
    }

    private void rebuildIndexes(Connection conn, Configuration conf, ArrayList<String> tableList) {
        if (!this.upgrade || !this.rebuild) {
            return;
        }
        for (String table : tableList) {
            this.rebuildIndexes(conn, conf, table);
        }
    }

    private void rebuildIndexes(Connection conn, Configuration conf, String dataTableFullName) {
        try {
            HashMap<String, IndexInfo> rebuildMap = this.prepareToRebuildIndexes(conn, dataTableFullName);
            if (rebuildMap.isEmpty()) {
                LOGGER.info("No indexes to rebuild for table " + dataTableFullName);
                return;
            }
            if (!this.test) {
                this.indexingTool = new IndexTool();
                this.indexingTool.setConf(conf);
            }
            this.startIndexRebuilds(rebuildMap, this.indexingTool);
        }
        catch (SQLException e) {
            LOGGER.severe("Failed to prepare the map for index rebuilds " + e);
            throw new RuntimeException("Failed to prepare the map for index rebuilds");
        }
    }

    private void modifyDataTable(Admin admin, String tableName) throws IOException {
        TableDescriptorBuilder tableDescBuilder = TableDescriptorBuilder.newBuilder((TableDescriptor)admin.getDescriptor(TableName.valueOf((String)tableName)));
        if (this.upgrade) {
            this.removeCoprocessor(admin, tableName, tableDescBuilder, Indexer.class.getName());
            this.addCoprocessor(admin, tableName, tableDescBuilder, IndexRegionObserver.class.getName());
        } else {
            this.removeCoprocessor(admin, tableName, tableDescBuilder, IndexRegionObserver.class.getName());
            this.addCoprocessor(admin, tableName, tableDescBuilder, Indexer.class.getName());
        }
        if (!this.dryRun) {
            admin.modifyTable(tableDescBuilder.build());
        }
    }

    private void addCoprocessor(Admin admin, String tableName, TableDescriptorBuilder tableDescBuilder, String coprocName) throws IOException {
        this.addCoprocessor(admin, tableName, tableDescBuilder, coprocName, 0x2FFFFFFE, this.prop);
    }

    private void addCoprocessor(Admin admin, String tableName, TableDescriptorBuilder tableDescBuilder, String coprocName, int priority, Map<String, String> propsToAdd) throws IOException {
        if (!admin.getDescriptor(TableName.valueOf((String)tableName)).hasCoprocessor(coprocName)) {
            if (!this.dryRun) {
                CoprocessorDescriptorBuilder coprocBuilder = CoprocessorDescriptorBuilder.newBuilder((String)coprocName);
                coprocBuilder.setPriority(priority).setProperties(propsToAdd);
                tableDescBuilder.setCoprocessor(coprocBuilder.build());
            }
            LOGGER.info("Loaded " + coprocName + " coprocessor on table " + tableName);
        } else {
            LOGGER.info(coprocName + " coprocessor on table " + tableName + "is already loaded");
        }
    }

    private void removeCoprocessor(Admin admin, String tableName, TableDescriptorBuilder tableDescBuilder, String coprocName) throws IOException {
        if (admin.getDescriptor(TableName.valueOf((String)tableName)).hasCoprocessor(coprocName)) {
            if (!this.dryRun) {
                tableDescBuilder.removeCoprocessor(coprocName);
            }
            LOGGER.info("Unloaded " + coprocName + "coprocessor on table " + tableName);
        } else {
            LOGGER.info(coprocName + " coprocessor on table " + tableName + " is already unloaded");
        }
    }

    private void modifyIndexTable(Admin admin, HashSet<String> indexes) throws IOException {
        for (String indexName : indexes) {
            TableDescriptorBuilder indexTableDescBuilder = TableDescriptorBuilder.newBuilder((TableDescriptor)admin.getDescriptor(TableName.valueOf((String)indexName)));
            if (this.upgrade) {
                this.addCoprocessor(admin, indexName, indexTableDescBuilder, GlobalIndexChecker.class.getName(), 0x2FFFFFFD, this.emptyProp);
            } else {
                this.removeCoprocessor(admin, indexName, indexTableDescBuilder, GlobalIndexChecker.class.getName());
            }
            if (this.dryRun) continue;
            admin.modifyTable(indexTableDescBuilder.build());
        }
    }

    private int startIndexRebuilds(HashMap<String, IndexInfo> indexInfos, IndexTool indexingTool) {
        for (Map.Entry<String, IndexInfo> entry : indexInfos.entrySet()) {
            String index = entry.getKey();
            IndexInfo indexInfo = entry.getValue();
            String indexName = SchemaUtil.getTableNameFromFullName((String)index);
            String tenantId = indexInfo.getTenantId();
            String baseTable = indexInfo.getBaseTable();
            String schema = indexInfo.getSchemaName();
            String outFile = "/tmp/index_rebuild_" + schema + "_" + indexName + (String)(GLOBAL_INDEX_ID.equals(tenantId) ? "" : "_" + tenantId) + "_" + UUID.randomUUID().toString();
            CharSequence[] args = this.getIndexToolArgValues(schema, baseTable, indexName, outFile, tenantId);
            try {
                LOGGER.info("Rebuilding index: " + String.join((CharSequence)",", args));
                if (this.dryRun) continue;
                indexingTool.run((String[])args);
            }
            catch (Exception e) {
                LOGGER.severe("Something went wrong while building the index " + index + " " + e);
                return -1;
            }
        }
        return 0;
    }

    public String[] getIndexToolArgValues(String schema, String baseTable, String indexName, String outFile, String tenantId) {
        String[] args = new String[]{"-s", schema, "-dt", baseTable, "-it", indexName, "-direct", "-op", outFile};
        ArrayList<String> list = new ArrayList<String>(Arrays.asList(args));
        if (!GLOBAL_INDEX_ID.equals(tenantId)) {
            list.add("-tenant");
            list.add(tenantId);
        }
        if (!Strings.isNullOrEmpty((String)this.indexToolOpts)) {
            String[] options;
            for (String opt : options = this.indexToolOpts.split("\\s+")) {
                list.add(opt);
            }
        }
        return list.toArray(new String[list.size()]);
    }

    private boolean extractTablesAndIndexes(PhoenixConnection conn) {
        String[] tables = this.inputTables.trim().split(",");
        PTable dataTable = null;
        try {
            for (String tableName : tables) {
                HashSet<String> physicalIndexes = new HashSet<String>();
                dataTable = conn.getTableNoCache(tableName);
                String physicalTableName = dataTable.getPhysicalName().getString();
                if (!dataTable.isTransactional() && dataTable.getType().equals((Object)PTableType.TABLE)) {
                    for (PTable indexTable : dataTable.getIndexes()) {
                        if (!IndexUtil.isGlobalIndex((PTable)indexTable)) continue;
                        String physicalIndexName = indexTable.getPhysicalName().getString();
                        physicalIndexes.add(physicalIndexName);
                    }
                    if (MetaDataUtil.hasViewIndexTable((PhoenixConnection)conn, (PName)dataTable.getPhysicalName())) {
                        String viewIndexPhysicalName = MetaDataUtil.getViewIndexPhysicalName((String)physicalTableName);
                        physicalIndexes.add(viewIndexPhysicalName);
                    }
                    this.tablesAndIndexes.put(physicalTableName, physicalIndexes);
                    continue;
                }
                LOGGER.info("Skipping Table " + tableName + " because it is " + (dataTable.isTransactional() ? "transactional" : "not a data table"));
            }
            return true;
        }
        catch (SQLException e) {
            LOGGER.severe("Failed to find list of indexes " + e);
            if (dataTable == null) {
                LOGGER.severe("Unable to find the provided data table");
            }
            return false;
        }
    }

    private HashMap<String, IndexInfo> prepareToRebuildIndexes(Connection conn, String dataTableFullName) throws SQLException {
        HashMap<String, IndexInfo> indexInfos = new HashMap<String, IndexInfo>();
        HashSet<String> physicalIndexes = this.tablesAndIndexes.get(dataTableFullName);
        String viewIndexPhysicalName = MetaDataUtil.getViewIndexPhysicalName((String)dataTableFullName);
        boolean hasViewIndex = physicalIndexes.contains(viewIndexPhysicalName);
        String schemaName = SchemaUtil.getSchemaNameFromFullName((String)dataTableFullName);
        String tableName = SchemaUtil.getTableNameFromFullName((String)dataTableFullName);
        for (String physicalIndexName : physicalIndexes) {
            if (physicalIndexName.equals(viewIndexPhysicalName)) continue;
            String indexTableName = SchemaUtil.getTableNameFromFullName((String)physicalIndexName);
            String pIndexName = SchemaUtil.getTableName((String)schemaName, (String)indexTableName);
            IndexInfo indexInfo = new IndexInfo(schemaName, tableName, GLOBAL_INDEX_ID, pIndexName);
            indexInfos.put(physicalIndexName, indexInfo);
        }
        if (hasViewIndex) {
            String viewSql = IndexUpgradeTool.getViewSql(tableName, schemaName);
            ResultSet rs = conn.createStatement().executeQuery(viewSql);
            while (rs.next()) {
                String viewFullName = rs.getString(1);
                String viewName = SchemaUtil.getTableNameFromFullName((String)viewFullName);
                String tenantId = rs.getString(2);
                ArrayList<String> viewIndexes = this.findViewIndexes(conn, schemaName, viewName, tenantId);
                for (String viewIndex : viewIndexes) {
                    IndexInfo indexInfo = new IndexInfo(schemaName, viewName, tenantId == null ? GLOBAL_INDEX_ID : tenantId, viewIndex);
                    indexInfos.put(viewIndex, indexInfo);
                }
            }
        }
        return indexInfos;
    }

    @VisibleForTesting
    public static String getViewSql(String tableName, String schemaName) {
        return "SELECT DISTINCT COLUMN_FAMILY, COLUMN_NAME FROM SYSTEM.CHILD_LINK WHERE TABLE_NAME = '" + tableName + "'" + (String)(!Strings.isNullOrEmpty((String)schemaName) ? " AND TABLE_SCHEM = '" + schemaName + "'" : "") + " AND LINK_TYPE = " + PTable.LinkType.CHILD_TABLE.getSerializedValue();
    }

    private ArrayList<String> findViewIndexes(Connection conn, String schemaName, String viewName, String tenantId) throws SQLException {
        String viewIndexesSql = IndexUpgradeTool.getViewIndexesSql(viewName, schemaName, tenantId);
        ArrayList<String> viewIndexes = new ArrayList<String>();
        long stime = EnvironmentEdgeManager.currentTimeMillis();
        ResultSet rs = conn.createStatement().executeQuery(viewIndexesSql);
        long etime = EnvironmentEdgeManager.currentTimeMillis();
        LOGGER.info(String.format("Query %s took %d ms ", viewIndexesSql, etime - stime));
        while (rs.next()) {
            String viewIndexName = rs.getString(1);
            viewIndexes.add(viewIndexName);
        }
        return viewIndexes;
    }

    @VisibleForTesting
    public static String getViewIndexesSql(String viewName, String schemaName, String tenantId) {
        return "SELECT DISTINCT COLUMN_FAMILY FROM SYSTEM.CATALOG WHERE TABLE_NAME = '" + viewName + "'" + (String)(!Strings.isNullOrEmpty((String)schemaName) ? " AND TABLE_SCHEM = '" + schemaName + "'" : "") + " AND LINK_TYPE = " + PTable.LinkType.INDEX_TABLE.getSerializedValue() + (String)(tenantId != null ? " AND TENANT_ID = '" + tenantId + "'" : " AND TENANT_ID IS NULL");
    }

    public static void main(String[] args) throws Exception {
        int result = ToolRunner.run((Tool)new IndexUpgradeTool(), (String[])args);
        System.exit(result);
    }

    private static class IndexInfo {
        private final String schemaName;
        private final String baseTable;
        private final String tenantId;
        private final String indexName;

        public IndexInfo(String schemaName, String baseTable, String tenantId, String indexName) {
            this.schemaName = schemaName;
            this.baseTable = baseTable;
            this.tenantId = tenantId;
            this.indexName = indexName;
        }

        public String getSchemaName() {
            return this.schemaName;
        }

        public String getBaseTable() {
            return this.baseTable;
        }

        public String getTenantId() {
            return this.tenantId;
        }

        public String getIndexName() {
            return this.indexName;
        }
    }
}

