/*
 * Decompiled with CFR 0.152.
 */
package org.apache.phoenix.jdbc;

import java.io.Closeable;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.PathChildrenCache;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheListener;
import org.apache.hadoop.conf.Configuration;
import org.apache.phoenix.jdbc.ClusterRoleRecord;
import org.apache.phoenix.jdbc.PhoenixHAAdmin;
import org.apache.phoenix.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.phoenix.thirdparty.com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HAGroupStoreClient
implements Closeable {
    private static final long HA_GROUP_STORE_CLIENT_INITIALIZATION_TIMEOUT_MS = 30000L;
    private PhoenixHAAdmin phoenixHaAdmin;
    private static final Logger LOGGER = LoggerFactory.getLogger(HAGroupStoreClient.class);
    private final ConcurrentHashMap<ClusterRoleRecord.ClusterRole, ConcurrentHashMap<String, ClusterRoleRecord>> clusterRoleToCRRMap = new ConcurrentHashMap();
    private PathChildrenCache pathChildrenCache;
    private volatile boolean isHealthy;
    private static final Map<String, HAGroupStoreClient> instances = new ConcurrentHashMap<String, HAGroupStoreClient>();

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static HAGroupStoreClient getInstance(Configuration conf) {
        String zkUrl = PhoenixHAAdmin.getLocalZkUrl(conf);
        HAGroupStoreClient result = instances.get(zkUrl);
        if (result != null && result.isHealthy) return result;
        Class<HAGroupStoreClient> clazz = HAGroupStoreClient.class;
        synchronized (HAGroupStoreClient.class) {
            result = instances.get(zkUrl);
            if (result != null && result.isHealthy) return result;
            result = new HAGroupStoreClient(conf, null);
            if (!result.isHealthy) {
                result.close();
                result = null;
            }
            instances.put(zkUrl, result);
            // ** MonitorExit[var3_3] (shouldn't be in output)
            return result;
        }
    }

    @VisibleForTesting
    HAGroupStoreClient(Configuration conf, PathChildrenCacheListener pathChildrenCacheListener) {
        try {
            this.phoenixHaAdmin = new PhoenixHAAdmin(conf);
            PathChildrenCache pathChildrenCache = new PathChildrenCache(this.phoenixHaAdmin.getCurator(), "/", true);
            CountDownLatch latch = new CountDownLatch(1);
            if (pathChildrenCacheListener != null) {
                pathChildrenCache.getListenable().addListener((Object)pathChildrenCacheListener);
            } else {
                pathChildrenCache.getListenable().addListener((client, event) -> {
                    LOGGER.info("HAGroupStoreClient PathChildrenCache Received event for type {}", (Object)event.getType());
                    ChildData childData = event.getData();
                    ClusterRoleRecord eventCRR = this.extractCRROrNull(childData);
                    switch (event.getType()) {
                        case CHILD_ADDED: 
                        case CHILD_UPDATED: {
                            if (eventCRR == null || eventCRR.getHaGroupName() == null) break;
                            this.updateClusterRoleRecordMap(eventCRR);
                            break;
                        }
                        case CHILD_REMOVED: {
                            if (eventCRR == null || eventCRR.getHaGroupName() == null || eventCRR.getHaGroupName().isEmpty() || eventCRR.getRole(this.phoenixHaAdmin.getZkUrl()) == null) break;
                            LOGGER.info("Received CHILD_REMOVED event, Removing CRR {} from existing CRR Map {}", (Object)eventCRR, this.clusterRoleToCRRMap);
                            ClusterRoleRecord.ClusterRole role = eventCRR.getRole(this.phoenixHaAdmin.getZkUrl());
                            this.clusterRoleToCRRMap.putIfAbsent(role, new ConcurrentHashMap());
                            this.clusterRoleToCRRMap.get((Object)role).remove(eventCRR.getHaGroupName());
                            break;
                        }
                        case INITIALIZED: {
                            latch.countDown();
                            break;
                        }
                        case CONNECTION_LOST: 
                        case CONNECTION_SUSPENDED: {
                            this.isHealthy = false;
                            break;
                        }
                        case CONNECTION_RECONNECTED: {
                            this.isHealthy = true;
                            break;
                        }
                        default: {
                            LOGGER.warn("Unexpected event type {}, complete event {}", (Object)event.getType(), (Object)event);
                        }
                    }
                });
            }
            pathChildrenCache.start(PathChildrenCache.StartMode.POST_INITIALIZED_EVENT);
            this.pathChildrenCache = pathChildrenCache;
            this.isHealthy = latch.await(30000L, TimeUnit.MILLISECONDS);
            this.buildClusterRoleToCRRMap();
        }
        catch (Exception e) {
            this.isHealthy = false;
            LOGGER.error("Unexpected error occurred while initializing HAGroupStoreClient, marking cache as unhealthy", (Throwable)e);
        }
    }

    private ClusterRoleRecord extractCRROrNull(ChildData childData) {
        if (childData != null) {
            byte[] data = childData.getData();
            return ClusterRoleRecord.fromJson(data).orElse(null);
        }
        return null;
    }

    private void updateClusterRoleRecordMap(ClusterRoleRecord crr) {
        if (crr != null && crr.getHaGroupName() != null && crr.getRole(this.phoenixHaAdmin.getZkUrl()) != null) {
            ClusterRoleRecord.ClusterRole role = crr.getRole(this.phoenixHaAdmin.getZkUrl());
            LOGGER.info("Updating Existing CRR Map {} with new CRR {}", this.clusterRoleToCRRMap, (Object)crr);
            this.clusterRoleToCRRMap.putIfAbsent(role, new ConcurrentHashMap());
            this.clusterRoleToCRRMap.get((Object)role).put(crr.getHaGroupName(), crr);
            LOGGER.info("Added new CRR {} to CRR Map", (Object)crr);
            for (ClusterRoleRecord.ClusterRole mapRole : this.clusterRoleToCRRMap.keySet()) {
                ConcurrentHashMap<String, ClusterRoleRecord> roleWiseMap;
                if (mapRole == role || !(roleWiseMap = this.clusterRoleToCRRMap.get((Object)mapRole)).containsKey(crr.getHaGroupName())) continue;
                LOGGER.info("Removing any pre-existing mapping with role {} for HAGroupName {}", (Object)mapRole, (Object)crr.getHaGroupName());
                roleWiseMap.remove(crr.getHaGroupName());
            }
            LOGGER.info("Final Updated CRR Map {}", this.clusterRoleToCRRMap);
        }
    }

    private void buildClusterRoleToCRRMap() {
        List clusterRoleRecords = this.pathChildrenCache.getCurrentData().stream().map(this::extractCRROrNull).filter(Objects::nonNull).collect(Collectors.toList());
        for (ClusterRoleRecord crr : clusterRoleRecords) {
            this.updateClusterRoleRecordMap(crr);
        }
    }

    public void rebuild() throws Exception {
        if (!this.isHealthy) {
            throw new IOException("HAGroupStoreClient is not healthy");
        }
        LOGGER.info("Rebuilding HAGroupStoreClient for HA groups");
        this.pathChildrenCache.rebuild();
        this.buildClusterRoleToCRRMap();
        LOGGER.info("Rebuild Complete for HAGroupStoreClient");
    }

    @Override
    public void close() {
        try {
            LOGGER.info("Closing HAGroupStoreClient");
            this.clusterRoleToCRRMap.clear();
            if (this.pathChildrenCache != null) {
                this.pathChildrenCache.close();
            }
            LOGGER.info("Closed HAGroupStoreClient");
        }
        catch (IOException e) {
            LOGGER.error("Exception closing HAGroupStoreClient", (Throwable)e);
        }
    }

    public List<ClusterRoleRecord> getCRRsByClusterRole(ClusterRoleRecord.ClusterRole clusterRole) throws IOException {
        if (!this.isHealthy) {
            throw new IOException("HAGroupStoreClient is not healthy");
        }
        return ImmutableList.copyOf(this.clusterRoleToCRRMap.getOrDefault((Object)clusterRole, new ConcurrentHashMap()).values());
    }
}

