/*
 * Decompiled with CFR 0.152.
 */
package org.apache.phoenix.expression.util.bson;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.phoenix.expression.util.bson.CommonComparisonExpressionUtils;
import org.apache.phoenix.thirdparty.com.google.common.base.Preconditions;
import org.bson.BsonArray;
import org.bson.BsonBoolean;
import org.bson.BsonDocument;
import org.bson.BsonValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DocumentComparisonExpressionUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(DocumentComparisonExpressionUtils.class);
    private static final String EXISTS_OP = "$exists";
    private static final String EQUALS_OP = "$eq";
    private static final String NOT_EQUALS_OP = "$ne";
    private static final String LESS_THAN_OP = "$lt";
    private static final String LESS_THAN_OR_EQUALS_OP = "$lte";
    private static final String GREATER_THAN_OP = "$gt";
    private static final String GREATER_THAN_OR_EQUALS_OP = "$gte";

    private DocumentComparisonExpressionUtils() {
    }

    public static boolean evaluateConditionExpression(BsonDocument document, BsonDocument conditionExpression) {
        if (document == null || conditionExpression == null) {
            LOGGER.warn("Document and/or Condition Expression document are empty. Document: {}, conditionExpression: {}", (Object)document, (Object)conditionExpression);
            return false;
        }
        return DocumentComparisonExpressionUtils.evaluateExpression(document, conditionExpression);
    }

    private static boolean evaluateExpression(BsonDocument document, BsonDocument conditionExpression) {
        String firstFieldKey = conditionExpression.getFirstKey();
        Preconditions.checkArgument((conditionExpression.size() == 1 ? 1 : 0) != 0, (Object)"Expected num of document entries is 1");
        if (!firstFieldKey.startsWith("$")) {
            BsonValue bsonValue = conditionExpression.get((Object)firstFieldKey);
            Preconditions.checkArgument((boolean)(bsonValue instanceof BsonDocument), (Object)"Expected type for Bson value is Document for field based condition operation");
            BsonDocument bsonDocument = (BsonDocument)bsonValue;
            if (bsonDocument.containsKey((Object)EXISTS_OP)) {
                return DocumentComparisonExpressionUtils.isExists(document, conditionExpression);
            }
            if (bsonDocument.containsKey((Object)EQUALS_OP)) {
                return DocumentComparisonExpressionUtils.equals(document, conditionExpression);
            }
            if (bsonDocument.containsKey((Object)NOT_EQUALS_OP)) {
                return DocumentComparisonExpressionUtils.notEquals(document, conditionExpression);
            }
            if (bsonDocument.containsKey((Object)LESS_THAN_OP)) {
                return DocumentComparisonExpressionUtils.lessThan(document, conditionExpression);
            }
            if (bsonDocument.containsKey((Object)LESS_THAN_OR_EQUALS_OP)) {
                return DocumentComparisonExpressionUtils.lessThanOrEquals(document, conditionExpression);
            }
            if (bsonDocument.containsKey((Object)GREATER_THAN_OP)) {
                return DocumentComparisonExpressionUtils.greaterThan(document, conditionExpression);
            }
            if (bsonDocument.containsKey((Object)GREATER_THAN_OR_EQUALS_OP)) {
                return DocumentComparisonExpressionUtils.greaterThanOrEquals(document, conditionExpression);
            }
            throw new IllegalArgumentException("Operator " + firstFieldKey + " is not supported");
        }
        switch (firstFieldKey) {
            case "$or": {
                BsonValue bsonValue = conditionExpression.get((Object)firstFieldKey);
                Preconditions.checkArgument((boolean)(bsonValue instanceof BsonArray), (Object)"Expected type for Bson value is Array for $or operator");
                BsonArray bsonArray = (BsonArray)bsonValue;
                List bsonValues = bsonArray.getValues();
                for (BsonValue value : bsonValues) {
                    if (!DocumentComparisonExpressionUtils.evaluateExpression(document, (BsonDocument)value)) continue;
                    return true;
                }
                return false;
            }
            case "$and": {
                BsonValue bsonValue = conditionExpression.get((Object)firstFieldKey);
                Preconditions.checkArgument((boolean)(bsonValue instanceof BsonArray), (Object)"Expected type for Bson value is Array for $and operator");
                BsonArray bsonArray = (BsonArray)bsonValue;
                List bsonValues = bsonArray.getValues();
                for (BsonValue value : bsonValues) {
                    if (DocumentComparisonExpressionUtils.evaluateExpression(document, (BsonDocument)value)) continue;
                    return false;
                }
                return true;
            }
        }
        throw new IllegalArgumentException(firstFieldKey + " is not a known operator");
    }

    private static boolean isExists(BsonDocument document, BsonDocument conditionExpression) {
        Set entrySet = conditionExpression.entrySet();
        Preconditions.checkArgument((entrySet.size() == 1 ? 1 : 0) != 0, (Object)"Expected entry for the exists operation is 1");
        Iterator iterator = entrySet.iterator();
        if (iterator.hasNext()) {
            Map.Entry bsonValueEntry = (Map.Entry)iterator.next();
            String fieldKey = (String)bsonValueEntry.getKey();
            BsonValue bsonValue = (BsonValue)bsonValueEntry.getValue();
            Preconditions.checkArgument((boolean)(bsonValue instanceof BsonDocument), (Object)"Expected type for Bson value is Document for exists operation");
            BsonDocument bsonDocument = (BsonDocument)bsonValue;
            BsonValue existsValue = bsonDocument.get((Object)EXISTS_OP);
            Preconditions.checkArgument((boolean)(existsValue instanceof BsonBoolean), (Object)"Expected type for $exists value is boolean");
            BsonBoolean existsValBoolean = (BsonBoolean)existsValue;
            if (existsValBoolean.getValue()) {
                return DocumentComparisonExpressionUtils.exists(fieldKey, document);
            }
            return !DocumentComparisonExpressionUtils.exists(fieldKey, document);
        }
        return false;
    }

    private static String getCompareOperator(CommonComparisonExpressionUtils.CompareOp compareOp) {
        if (compareOp == null) {
            return null;
        }
        switch (compareOp) {
            case LESS: {
                return LESS_THAN_OP;
            }
            case LESS_OR_EQUAL: {
                return LESS_THAN_OR_EQUALS_OP;
            }
            case GREATER: {
                return GREATER_THAN_OP;
            }
            case GREATER_OR_EQUAL: {
                return GREATER_THAN_OR_EQUALS_OP;
            }
            case EQUALS: {
                return EQUALS_OP;
            }
            case NOT_EQUALS: {
                return NOT_EQUALS_OP;
            }
        }
        return null;
    }

    private static boolean compare(BsonDocument document, BsonDocument conditionExpression, CommonComparisonExpressionUtils.CompareOp compareOp) {
        Set entrySet = conditionExpression.entrySet();
        Preconditions.checkArgument((entrySet.size() == 1 ? 1 : 0) != 0, (Object)("Expected entry for the " + (Object)((Object)compareOp) + " operation is 1"));
        Iterator iterator = entrySet.iterator();
        if (iterator.hasNext()) {
            Map.Entry bsonValueEntry = (Map.Entry)iterator.next();
            String fieldKey = (String)bsonValueEntry.getKey();
            BsonValue bsonValue = (BsonValue)bsonValueEntry.getValue();
            Preconditions.checkArgument((boolean)(bsonValue instanceof BsonDocument), (Object)("Expected type for Bson value is Document for " + (Object)((Object)compareOp) + " operation"));
            BsonDocument bsonDocument = (BsonDocument)bsonValue;
            BsonValue compareValue = bsonDocument.get((Object)DocumentComparisonExpressionUtils.getCompareOperator(compareOp));
            BsonValue topLevelValue = document.get((Object)fieldKey);
            BsonValue actualValue = topLevelValue != null ? topLevelValue : CommonComparisonExpressionUtils.getFieldFromDocument(fieldKey, document);
            return actualValue != null && CommonComparisonExpressionUtils.compareValues(actualValue, compareValue, compareOp);
        }
        return false;
    }

    private static boolean greaterThanOrEquals(BsonDocument document, BsonDocument conditionExpression) {
        return DocumentComparisonExpressionUtils.compare(document, conditionExpression, CommonComparisonExpressionUtils.CompareOp.GREATER_OR_EQUAL);
    }

    private static boolean greaterThan(BsonDocument document, BsonDocument conditionExpression) {
        return DocumentComparisonExpressionUtils.compare(document, conditionExpression, CommonComparisonExpressionUtils.CompareOp.GREATER);
    }

    private static boolean lessThanOrEquals(BsonDocument document, BsonDocument conditionExpression) {
        return DocumentComparisonExpressionUtils.compare(document, conditionExpression, CommonComparisonExpressionUtils.CompareOp.LESS_OR_EQUAL);
    }

    private static boolean lessThan(BsonDocument document, BsonDocument conditionExpression) {
        return DocumentComparisonExpressionUtils.compare(document, conditionExpression, CommonComparisonExpressionUtils.CompareOp.LESS);
    }

    private static boolean equals(BsonDocument document, BsonDocument conditionExpression) {
        return DocumentComparisonExpressionUtils.compare(document, conditionExpression, CommonComparisonExpressionUtils.CompareOp.EQUALS);
    }

    private static boolean notEquals(BsonDocument document, BsonDocument conditionExpression) {
        return DocumentComparisonExpressionUtils.compare(document, conditionExpression, CommonComparisonExpressionUtils.CompareOp.NOT_EQUALS);
    }

    private static boolean exists(String documentField, BsonDocument bsonDocument) {
        BsonValue topLevelValue = bsonDocument.get((Object)documentField);
        if (topLevelValue != null) {
            return true;
        }
        return CommonComparisonExpressionUtils.getFieldFromDocument(documentField, bsonDocument) != null;
    }
}

