/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.fs;

import jakarta.annotation.Nullable;
import java.time.Duration;
import java.util.OptionalLong;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.hadoop.hdds.annotation.InterfaceAudience;
import org.apache.hadoop.hdds.annotation.InterfaceStability;
import org.apache.hadoop.hdds.fs.SpaceUsageCheckParams;
import org.apache.hadoop.hdds.fs.SpaceUsagePersistence;
import org.apache.hadoop.hdds.fs.SpaceUsageSource;
import org.apache.hadoop.ozone.shaded.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.ratis.util.AutoCloseableLock;
import org.apache.ratis.util.AutoCloseableReadWriteLock;
import org.apache.ratis.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class CachingSpaceUsageSource
implements SpaceUsageSource {
    private static final Logger LOG = LoggerFactory.getLogger(CachingSpaceUsageSource.class);
    private final ScheduledExecutorService executor;
    private final AutoCloseableReadWriteLock lock;
    private long cachedUsedSpace;
    private long cachedAvailable;
    private long cachedCapacity;
    private final Duration refresh;
    private final SpaceUsageSource source;
    private final SpaceUsagePersistence persistence;
    private boolean running;
    private ScheduledFuture<?> updateUsedSpaceFuture;
    private ScheduledFuture<?> updateAvailableFuture;
    private final AtomicBoolean isRefreshRunning;

    public CachingSpaceUsageSource(SpaceUsageCheckParams params) {
        this(params, CachingSpaceUsageSource.createExecutor(params));
    }

    CachingSpaceUsageSource(SpaceUsageCheckParams params, ScheduledExecutorService executor) {
        Preconditions.assertNotNull(params, "params == null");
        this.refresh = params.getRefresh();
        this.source = params.getSource();
        this.lock = new AutoCloseableReadWriteLock(this.source.toString());
        this.persistence = params.getPersistence();
        this.executor = executor;
        this.isRefreshRunning = new AtomicBoolean();
        Preconditions.assertTrue(this.refresh.isZero() == (executor == null), "executor should be provided if and only if refresh is requested");
        this.loadInitialValue();
    }

    @Override
    public long getCapacity() {
        try (AutoCloseableLock ignored = this.lock.readLock(null, null);){
            long l = this.cachedCapacity;
            return l;
        }
    }

    @Override
    public long getAvailable() {
        try (AutoCloseableLock ignored = this.lock.readLock(null, null);){
            long l = this.cachedAvailable;
            return l;
        }
    }

    @Override
    public long getUsedSpace() {
        try (AutoCloseableLock ignored = this.lock.readLock(null, null);){
            long l = this.cachedUsedSpace;
            return l;
        }
    }

    @Override
    public SpaceUsageSource snapshot() {
        try (AutoCloseableLock ignored = this.lock.readLock(null, null);){
            SpaceUsageSource.Fixed fixed = new SpaceUsageSource.Fixed(this.cachedCapacity, this.cachedAvailable, this.cachedUsedSpace);
            return fixed;
        }
    }

    public void incrementUsedSpace(long usedSpace) {
        long change;
        long current;
        if (usedSpace == 0L) {
            return;
        }
        Preconditions.assertTrue(usedSpace > 0L, () -> usedSpace + " < 0");
        try (AutoCloseableLock ignored = this.lock.writeLock(null, null);){
            current = this.cachedAvailable;
            change = Math.min(current, usedSpace);
            this.cachedAvailable -= change;
            this.cachedUsedSpace += change;
        }
        if (change != usedSpace) {
            LOG.warn("Attempted to decrement available space to a negative value. Current: {}, Decrement: {}, Source: {}", new Object[]{current, usedSpace, this.source});
        }
    }

    public void decrementUsedSpace(long reclaimedSpace) {
        long change;
        long current;
        if (reclaimedSpace == 0L) {
            return;
        }
        Preconditions.assertTrue(reclaimedSpace > 0L, () -> reclaimedSpace + " < 0");
        try (AutoCloseableLock ignored = this.lock.writeLock(null, null);){
            current = this.cachedUsedSpace;
            change = Math.min(current, reclaimedSpace);
            this.cachedUsedSpace -= change;
            this.cachedAvailable += change;
        }
        if (change != reclaimedSpace) {
            LOG.warn("Attempted to decrement used space to a negative value. Current: {}, Decrement: {}, Source: {}", new Object[]{current, reclaimedSpace, this.source});
        }
    }

    public void start() {
        if (this.executor != null) {
            long initialDelay;
            long l = initialDelay = this.getUsedSpace() > 0L ? this.refresh.toMillis() : 0L;
            if (!this.running) {
                this.updateUsedSpaceFuture = this.executor.scheduleWithFixedDelay(this::refresh, initialDelay, this.refresh.toMillis(), TimeUnit.MILLISECONDS);
                long availableUpdateDelay = Math.min(this.refresh.toMillis(), Duration.ofMinutes(1L).toMillis());
                this.updateAvailableFuture = this.executor.scheduleWithFixedDelay(this::updateAvailable, availableUpdateDelay, availableUpdateDelay, TimeUnit.MILLISECONDS);
                this.running = true;
            }
        } else {
            this.refresh();
        }
    }

    public void shutdown() {
        this.persistence.save(this);
        if (this.executor != null) {
            if (this.running) {
                if (this.updateUsedSpaceFuture != null) {
                    this.updateUsedSpaceFuture.cancel(true);
                }
                if (this.updateAvailableFuture != null) {
                    this.updateAvailableFuture.cancel(true);
                }
            }
            this.running = false;
            this.executor.shutdown();
        }
    }

    public void refreshNow() {
        this.executor.schedule(this::refresh, 0L, TimeUnit.MILLISECONDS);
    }

    private void loadInitialValue() {
        OptionalLong initialValue = this.persistence.load();
        this.updateCachedValues(initialValue.orElse(0L));
    }

    private void updateAvailable() {
        long capacity = this.source.getCapacity();
        long available = this.source.getAvailable();
        try (AutoCloseableLock ignored = this.lock.writeLock(null, null);){
            this.cachedAvailable = available;
            this.cachedCapacity = capacity;
        }
    }

    private void updateCachedValues(long used) {
        long capacity = this.source.getCapacity();
        long available = this.source.getAvailable();
        try (AutoCloseableLock ignored = this.lock.writeLock(null, null);){
            this.cachedAvailable = available;
            this.cachedCapacity = capacity;
            this.cachedUsedSpace = used;
        }
    }

    private void refresh() {
        if (this.isRefreshRunning.compareAndSet(false, true)) {
            try {
                this.updateCachedValues(this.source.getUsedSpace());
            }
            catch (RuntimeException e) {
                LOG.warn("Error refreshing space usage for {}", (Object)this.source, (Object)e);
            }
            finally {
                this.isRefreshRunning.set(false);
            }
        }
    }

    @Nullable
    private static ScheduledExecutorService createExecutor(SpaceUsageCheckParams params) {
        if (params.getRefresh().isZero()) {
            return null;
        }
        return Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("DiskUsage-" + params.getPath() + "-%n").build());
    }
}

