/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om.lock;

import com.google.common.annotations.VisibleForTesting;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.ozone.lock.LockManager;
import org.apache.hadoop.ozone.om.lock.LockUsageInfo;
import org.apache.hadoop.ozone.om.lock.OMLockMetrics;
import org.apache.hadoop.ozone.om.lock.OzoneManagerLockUtil;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OzoneManagerLock {
    private static final Logger LOG = LoggerFactory.getLogger(OzoneManagerLock.class);
    private static final String READ_LOCK = "read";
    private static final String WRITE_LOCK = "write";
    private final LockManager<String> manager;
    private OMLockMetrics omLockMetrics;
    private final ThreadLocal<Short> lockSet = ThreadLocal.withInitial(() -> (short)0);

    public OzoneManagerLock(ConfigurationSource conf) {
        boolean fair = conf.getBoolean("ozone.om.lock.fair", false);
        this.manager = new LockManager(conf, fair);
        this.omLockMetrics = OMLockMetrics.create();
    }

    @Deprecated
    public boolean acquireLock(Resource resource, String ... resources) {
        String resourceName = this.generateResourceName(resource, resources);
        return this.lock(resource, resourceName, arg_0 -> this.manager.writeLock(arg_0), WRITE_LOCK);
    }

    public boolean acquireReadLock(Resource resource, String ... resources) {
        String resourceName = this.generateResourceName(resource, resources);
        return this.lock(resource, resourceName, arg_0 -> this.manager.readLock(arg_0), READ_LOCK);
    }

    public boolean acquireReadHashedLock(Resource resource, String resourceName) {
        return this.lock(resource, resourceName, arg_0 -> this.manager.readLock(arg_0), READ_LOCK);
    }

    public boolean acquireWriteLock(Resource resource, String ... resources) {
        String resourceName = this.generateResourceName(resource, resources);
        return this.lock(resource, resourceName, arg_0 -> this.manager.writeLock(arg_0), WRITE_LOCK);
    }

    public boolean acquireWriteHashedLock(Resource resource, String resourceName) {
        return this.lock(resource, resourceName, arg_0 -> this.manager.writeLock(arg_0), WRITE_LOCK);
    }

    private boolean lock(Resource resource, String resourceName, Consumer<String> lockFn, String lockType) {
        if (!resource.canLock(this.lockSet.get())) {
            String errorMessage = this.getErrorMessage(resource);
            LOG.error(errorMessage);
            throw new RuntimeException(errorMessage);
        }
        long startWaitingTimeNanos = Time.monotonicNowNanos();
        lockFn.accept(resourceName);
        switch (lockType) {
            case "read": {
                this.updateReadLockMetrics(resource, resourceName, startWaitingTimeNanos);
                break;
            }
            case "write": {
                this.updateWriteLockMetrics(resource, resourceName, startWaitingTimeNanos);
                break;
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Acquired {} {} lock on resource {}", new Object[]{lockType, resource.name, resourceName});
        }
        this.lockSet.set(resource.setLock(this.lockSet.get()));
        return true;
    }

    private void updateReadLockMetrics(Resource resource, String resourceName, long startWaitingTimeNanos) {
        if (this.manager.getReadHoldCount((Object)resourceName) == 1) {
            long readLockWaitingTimeNanos = Time.monotonicNowNanos() - startWaitingTimeNanos;
            this.omLockMetrics.setReadLockWaitingTimeMsStat(TimeUnit.NANOSECONDS.toMillis(readLockWaitingTimeNanos));
            resource.setStartReadHeldTimeNanos(Time.monotonicNowNanos());
        }
    }

    private void updateWriteLockMetrics(Resource resource, String resourceName, long startWaitingTimeNanos) {
        if (this.manager.getWriteHoldCount((Object)resourceName) == 1 && this.manager.isWriteLockedByCurrentThread((Object)resourceName)) {
            long writeLockWaitingTimeNanos = Time.monotonicNowNanos() - startWaitingTimeNanos;
            this.omLockMetrics.setWriteLockWaitingTimeMsStat(TimeUnit.NANOSECONDS.toMillis(writeLockWaitingTimeNanos));
            resource.setStartWriteHeldTimeNanos(Time.monotonicNowNanos());
        }
    }

    public String generateResourceName(Resource resource, String ... resources) {
        if (resources.length == 1 && resource != Resource.BUCKET_LOCK) {
            return OzoneManagerLockUtil.generateResourceLockName(resource, resources[0]);
        }
        if (resources.length == 2 && resource == Resource.BUCKET_LOCK) {
            return OzoneManagerLockUtil.generateBucketLockName(resources[0], resources[1]);
        }
        if (resources.length == 3 && resource == Resource.KEY_PATH_LOCK) {
            return OzoneManagerLockUtil.generateKeyPathLockName(resources[0], resources[1], resources[2]);
        }
        throw new IllegalArgumentException("acquire lock is supported on single resource for all locks except for resource bucket");
    }

    private String getErrorMessage(Resource resource) {
        return "Thread '" + Thread.currentThread().getName() + "' cannot acquire " + resource.name + " lock while holding " + this.getCurrentLocks().toString() + " lock(s).";
    }

    @VisibleForTesting
    List<String> getCurrentLocks() {
        ArrayList<String> currentLocks = new ArrayList<String>();
        short lockSetVal = this.lockSet.get();
        for (Resource value : Resource.values()) {
            if (!value.isLevelLocked(lockSetVal)) continue;
            currentLocks.add(value.getName());
        }
        return currentLocks;
    }

    public boolean acquireMultiUserLock(String firstUser, String secondUser) {
        Resource resource = Resource.USER_LOCK;
        firstUser = this.generateResourceName(resource, firstUser);
        secondUser = this.generateResourceName(resource, secondUser);
        if (!resource.canLock(this.lockSet.get())) {
            String errorMessage = this.getErrorMessage(resource);
            LOG.error(errorMessage);
            throw new RuntimeException(errorMessage);
        }
        int compare = firstUser.compareTo(secondUser);
        if (compare > 0) {
            String temp = secondUser;
            secondUser = firstUser;
            firstUser = temp;
        }
        if (compare == 0) {
            this.manager.writeLock((Object)firstUser);
        } else {
            this.manager.writeLock((Object)firstUser);
            try {
                this.manager.writeLock((Object)secondUser);
            }
            catch (Exception ex) {
                this.manager.writeUnlock((Object)firstUser);
                throw ex;
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Acquired Write {} lock on resource {} and {}", new Object[]{resource.name, firstUser, secondUser});
        }
        this.lockSet.set(resource.setLock(this.lockSet.get()));
        return true;
    }

    public void releaseMultiUserLock(String firstUser, String secondUser) {
        Resource resource = Resource.USER_LOCK;
        firstUser = this.generateResourceName(resource, firstUser);
        int compare = firstUser.compareTo(secondUser = this.generateResourceName(resource, secondUser));
        if (compare > 0) {
            String temp = secondUser;
            secondUser = firstUser;
            firstUser = temp;
        }
        if (compare == 0) {
            this.manager.writeUnlock((Object)firstUser);
        } else {
            this.manager.writeUnlock((Object)firstUser);
            this.manager.writeUnlock((Object)secondUser);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Release Write {} lock on resource {} and {}", new Object[]{resource.name, firstUser, secondUser});
        }
        this.lockSet.set(resource.clearLock(this.lockSet.get()));
    }

    public void releaseWriteLock(Resource resource, String ... resources) {
        String resourceName = this.generateResourceName(resource, resources);
        this.unlock(resource, resourceName, arg_0 -> this.manager.writeUnlock(arg_0), WRITE_LOCK);
    }

    public void releaseWriteHashedLock(Resource resource, String resourceName) {
        this.unlock(resource, resourceName, arg_0 -> this.manager.writeUnlock(arg_0), WRITE_LOCK);
    }

    public void releaseReadLock(Resource resource, String ... resources) {
        String resourceName = this.generateResourceName(resource, resources);
        this.unlock(resource, resourceName, arg_0 -> this.manager.readUnlock(arg_0), READ_LOCK);
    }

    public void releaseReadHashedLock(Resource resource, String resourceName) {
        this.unlock(resource, resourceName, arg_0 -> this.manager.readUnlock(arg_0), READ_LOCK);
    }

    @Deprecated
    public void releaseLock(Resource resource, String ... resources) {
        String resourceName = this.generateResourceName(resource, resources);
        this.unlock(resource, resourceName, arg_0 -> this.manager.writeUnlock(arg_0), WRITE_LOCK);
    }

    private void unlock(Resource resource, String resourceName, Consumer<String> lockFn, String lockType) {
        boolean isWriteLocked = this.manager.isWriteLockedByCurrentThread((Object)resourceName);
        lockFn.accept(resourceName);
        switch (lockType) {
            case "read": {
                this.updateReadUnlockMetrics(resource, resourceName);
                break;
            }
            case "write": {
                this.updateWriteUnlockMetrics(resource, resourceName, isWriteLocked);
                break;
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Release {} {}, lock on resource {}", new Object[]{lockType, resource.name, resourceName});
        }
        this.lockSet.set(resource.clearLock(this.lockSet.get()));
    }

    private void updateReadUnlockMetrics(Resource resource, String resourceName) {
        if (this.manager.getReadHoldCount((Object)resourceName) == 0) {
            long readLockHeldTimeNanos = Time.monotonicNowNanos() - resource.getStartReadHeldTimeNanos();
            this.omLockMetrics.setReadLockHeldTimeMsStat(TimeUnit.NANOSECONDS.toMillis(readLockHeldTimeNanos));
        }
    }

    private void updateWriteUnlockMetrics(Resource resource, String resourceName, boolean isWriteLocked) {
        if (this.manager.getWriteHoldCount((Object)resourceName) == 0 && isWriteLocked) {
            long writeLockHeldTimeNanos = Time.monotonicNowNanos() - resource.getStartWriteHeldTimeNanos();
            this.omLockMetrics.setWriteLockHeldTimeMsStat(TimeUnit.NANOSECONDS.toMillis(writeLockHeldTimeNanos));
        }
    }

    @VisibleForTesting
    public int getReadHoldCount(String resourceName) {
        return this.manager.getReadHoldCount((Object)resourceName);
    }

    @VisibleForTesting
    public String getReadLockWaitingTimeMsStat() {
        return this.omLockMetrics.getReadLockWaitingTimeMsStat();
    }

    @VisibleForTesting
    public long getLongestReadLockWaitingTimeMs() {
        return this.omLockMetrics.getLongestReadLockWaitingTimeMs();
    }

    @VisibleForTesting
    public String getReadLockHeldTimeMsStat() {
        return this.omLockMetrics.getReadLockHeldTimeMsStat();
    }

    @VisibleForTesting
    public long getLongestReadLockHeldTimeMs() {
        return this.omLockMetrics.getLongestReadLockHeldTimeMs();
    }

    @VisibleForTesting
    public int getWriteHoldCount(String resourceName) {
        return this.manager.getWriteHoldCount((Object)resourceName);
    }

    @VisibleForTesting
    public boolean isWriteLockedByCurrentThread(String resourceName) {
        return this.manager.isWriteLockedByCurrentThread((Object)resourceName);
    }

    @VisibleForTesting
    public String getWriteLockWaitingTimeMsStat() {
        return this.omLockMetrics.getWriteLockWaitingTimeMsStat();
    }

    @VisibleForTesting
    public long getLongestWriteLockWaitingTimeMs() {
        return this.omLockMetrics.getLongestWriteLockWaitingTimeMs();
    }

    @VisibleForTesting
    public String getWriteLockHeldTimeMsStat() {
        return this.omLockMetrics.getWriteLockHeldTimeMsStat();
    }

    @VisibleForTesting
    public long getLongestWriteLockHeldTimeMs() {
        return this.omLockMetrics.getLongestWriteLockHeldTimeMs();
    }

    public void cleanup() {
        this.omLockMetrics.unRegister();
    }

    public OMLockMetrics getOMLockMetrics() {
        return this.omLockMetrics;
    }

    public static enum Resource {
        S3_BUCKET_LOCK(0, "S3_BUCKET_LOCK"),
        VOLUME_LOCK(1, "VOLUME_LOCK"),
        BUCKET_LOCK(2, "BUCKET_LOCK"),
        USER_LOCK(3, "USER_LOCK"),
        S3_SECRET_LOCK(4, "S3_SECRET_LOCK"),
        KEY_PATH_LOCK(5, "KEY_PATH_LOCK"),
        PREFIX_LOCK(6, "PREFIX_LOCK");

        private byte lockLevel;
        private short mask;
        private short setMask;
        private String name;
        private final ThreadLocal<LockUsageInfo> readLockTimeStampNanos = ThreadLocal.withInitial(LockUsageInfo::new);
        private final ThreadLocal<LockUsageInfo> writeLockTimeStampNanos = ThreadLocal.withInitial(LockUsageInfo::new);

        void setStartReadHeldTimeNanos(long startReadHeldTimeNanos) {
            this.readLockTimeStampNanos.get().setStartReadHeldTimeNanos(startReadHeldTimeNanos);
        }

        void setStartWriteHeldTimeNanos(long startWriteHeldTimeNanos) {
            this.writeLockTimeStampNanos.get().setStartWriteHeldTimeNanos(startWriteHeldTimeNanos);
        }

        long getStartReadHeldTimeNanos() {
            long startReadHeldTimeNanos = this.readLockTimeStampNanos.get().getStartReadHeldTimeNanos();
            this.readLockTimeStampNanos.remove();
            return startReadHeldTimeNanos;
        }

        long getStartWriteHeldTimeNanos() {
            long startWriteHeldTimeNanos = this.writeLockTimeStampNanos.get().getStartWriteHeldTimeNanos();
            this.writeLockTimeStampNanos.remove();
            return startWriteHeldTimeNanos;
        }

        private Resource(byte pos, String name) {
            this.lockLevel = pos;
            this.mask = (short)(Math.pow(2.0, this.lockLevel + 1) - 1.0);
            this.setMask = (short)Math.pow(2.0, this.lockLevel);
            this.name = name;
        }

        boolean canLock(short lockSetVal) {
            if (((Resource.USER_LOCK.setMask & lockSetVal) == Resource.USER_LOCK.setMask || (Resource.S3_SECRET_LOCK.setMask & lockSetVal) == Resource.S3_SECRET_LOCK.setMask || (Resource.PREFIX_LOCK.setMask & lockSetVal) == Resource.PREFIX_LOCK.setMask) && this.setMask <= lockSetVal) {
                return false;
            }
            return lockSetVal <= this.mask;
        }

        short setLock(short lockSetVal) {
            return (short)(lockSetVal | this.setMask);
        }

        short clearLock(short lockSetVal) {
            return (short)(lockSetVal & ~this.setMask);
        }

        boolean isLevelLocked(short lockSetVal) {
            return (lockSetVal & this.setMask) == this.setMask;
        }

        String getName() {
            return this.name;
        }

        short getMask() {
            return this.mask;
        }
    }
}

