/*
 * Decompiled with CFR 0.152.
 */
package org.apache.omid.tls;

import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.Security;
import java.security.cert.CertSelector;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.X509CertSelector;
import java.util.Arrays;
import java.util.Objects;
import javax.net.ssl.CertPathTrustManagerParameters;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509ExtendedTrustManager;
import javax.net.ssl.X509KeyManager;
import javax.net.ssl.X509TrustManager;
import org.apache.omid.tls.X509Exception;
import org.apache.phoenix.thirdparty.com.google.common.collect.ObjectArrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class X509Util {
    private static final Logger LOG = LoggerFactory.getLogger(X509Util.class);
    private static final char[] EMPTY_CHAR_ARRAY = new char[0];
    public static final int DEFAULT_HANDSHAKE_DETECTION_TIMEOUT_MILLIS = 5000;
    public static final String DEFAULT_PROTOCOL = "TLSv1.2";
    private static final String[] DEFAULT_CIPHERS_JAVA8 = (String[])ObjectArrays.concat((Object[])X509Util.getCBCCiphers(), (Object[])X509Util.getGCMCiphers(), String.class);
    private static final String[] DEFAULT_CIPHERS_JAVA9 = (String[])ObjectArrays.concat((Object[])X509Util.getGCMCiphers(), (Object[])X509Util.getCBCCiphers(), String.class);

    private static String[] getGCMCiphers() {
        return new String[]{"TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256", "TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384", "TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384"};
    }

    private static String[] getCBCCiphers() {
        return new String[]{"TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256", "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA", "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA", "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384", "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384", "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA", "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA"};
    }

    private X509Util() {
    }

    static String[] getDefaultCipherSuites() {
        return X509Util.getDefaultCipherSuitesForJavaVersion(System.getProperty("java.specification.version"));
    }

    static String[] getDefaultCipherSuitesForJavaVersion(String javaVersion) {
        Objects.requireNonNull(javaVersion);
        if (javaVersion.matches("\\d+")) {
            LOG.debug("Using Java9+ optimized cipher suites for Java version {}", (Object)javaVersion);
            return DEFAULT_CIPHERS_JAVA9;
        }
        if (javaVersion.startsWith("1.")) {
            LOG.debug("Using Java8 optimized cipher suites for Java version {}", (Object)javaVersion);
            return DEFAULT_CIPHERS_JAVA8;
        }
        LOG.debug("Could not parse java version {}, using Java8 optimized cipher suites", (Object)javaVersion);
        return DEFAULT_CIPHERS_JAVA8;
    }

    public static SslContext createSslContextForClient(String keyStoreLocation, char[] keyStorePassword, String keyStoreType, String trustStoreLocation, char[] trustStorePassword, String trustStoreType, boolean sslCrlEnabled, boolean sslOcspEnabled, String enabledProtocols, String cipherSuites, String tlsConfigProtocols) throws X509Exception, IOException {
        SslContextBuilder sslContextBuilder = SslContextBuilder.forClient();
        if (keyStoreLocation.isEmpty()) {
            LOG.warn("keyStoreLocation is not specified");
        } else {
            sslContextBuilder.keyManager((KeyManager)X509Util.createKeyManager(keyStoreLocation, keyStorePassword, keyStoreType));
        }
        if (trustStoreLocation.isEmpty()) {
            LOG.warn("trustStoreLocation is not specified");
        } else {
            sslContextBuilder.trustManager((TrustManager)X509Util.createTrustManager(trustStoreLocation, trustStorePassword, trustStoreType, sslCrlEnabled, sslOcspEnabled));
        }
        sslContextBuilder.enableOcsp(sslOcspEnabled);
        sslContextBuilder.protocols(X509Util.getEnabledProtocols(enabledProtocols, tlsConfigProtocols));
        sslContextBuilder.ciphers(Arrays.asList(X509Util.getCipherSuites(cipherSuites)));
        return sslContextBuilder.build();
    }

    public static SslContext createSslContextForServer(String keyStoreLocation, char[] keyStorePassword, String keyStoreType, String trustStoreLocation, char[] trustStorePassword, String trustStoreType, boolean sslCrlEnabled, boolean sslOcspEnabled, String enabledProtocols, String cipherSuites, String tlsConfigProtocols) throws X509Exception, IOException {
        if (keyStoreLocation.isEmpty()) {
            throw new X509Exception.SSLContextException("keyStoreLocation is required for SSL server: ");
        }
        SslContextBuilder sslContextBuilder = SslContextBuilder.forServer((KeyManager)X509Util.createKeyManager(keyStoreLocation, keyStorePassword, keyStoreType));
        if (trustStoreLocation.isEmpty()) {
            LOG.warn("trustStoreLocation is not specified");
        } else {
            sslContextBuilder.trustManager((TrustManager)X509Util.createTrustManager(trustStoreLocation, trustStorePassword, trustStoreType, sslCrlEnabled, sslOcspEnabled));
        }
        sslContextBuilder.enableOcsp(sslOcspEnabled);
        sslContextBuilder.protocols(X509Util.getEnabledProtocols(enabledProtocols, tlsConfigProtocols));
        sslContextBuilder.ciphers(Arrays.asList(X509Util.getCipherSuites(cipherSuites)));
        return sslContextBuilder.build();
    }

    static X509KeyManager createKeyManager(String keyStoreLocation, char[] keyStorePassword, String keyStoreType) throws X509Exception.KeyManagerException {
        if (keyStoreType == null) {
            keyStoreType = "jks";
        }
        if (keyStorePassword == null) {
            keyStorePassword = EMPTY_CHAR_ARRAY;
        }
        try {
            KeyStore ks = KeyStore.getInstance(keyStoreType);
            KeyManager[] keyManagerArray = null;
            try (InputStream inputStream = Files.newInputStream(new File(keyStoreLocation).toPath(), new OpenOption[0]);){
                ks.load(inputStream, keyStorePassword);
            }
            catch (Throwable object) {
                keyManagerArray = object;
                throw object;
            }
            KeyManagerFactory kmf = KeyManagerFactory.getInstance("PKIX");
            kmf.init(ks, keyStorePassword);
            for (KeyManager km : kmf.getKeyManagers()) {
                if (!(km instanceof X509KeyManager)) continue;
                return (X509KeyManager)km;
            }
            throw new X509Exception.KeyManagerException("Couldn't find X509KeyManager");
        }
        catch (IOException | IllegalArgumentException | GeneralSecurityException e) {
            throw new X509Exception.KeyManagerException(e);
        }
    }

    static X509TrustManager createTrustManager(String trustStoreLocation, char[] trustStorePassword, String trustStoreType, boolean crlEnabled, boolean ocspEnabled) throws X509Exception.TrustManagerException {
        if (trustStoreType == null) {
            trustStoreType = "jks";
        }
        if (trustStorePassword == null) {
            trustStorePassword = EMPTY_CHAR_ARRAY;
        }
        try {
            KeyStore ts = KeyStore.getInstance(trustStoreType);
            try (InputStream inputStream = Files.newInputStream(new File(trustStoreLocation).toPath(), new OpenOption[0]);){
                ts.load(inputStream, trustStorePassword);
            }
            PKIXBuilderParameters pbParams = new PKIXBuilderParameters(ts, (CertSelector)new X509CertSelector());
            if (crlEnabled || ocspEnabled) {
                pbParams.setRevocationEnabled(true);
                System.setProperty("com.sun.net.ssl.checkRevocation", "true");
                if (crlEnabled) {
                    System.setProperty("com.sun.security.enableCRLDP", "true");
                }
                if (ocspEnabled) {
                    Security.setProperty("ocsp.enable", "true");
                }
            } else {
                pbParams.setRevocationEnabled(false);
            }
            TrustManagerFactory tmf = TrustManagerFactory.getInstance("PKIX");
            tmf.init(new CertPathTrustManagerParameters(pbParams));
            for (TrustManager tm : tmf.getTrustManagers()) {
                if (!(tm instanceof X509ExtendedTrustManager)) continue;
                return (X509ExtendedTrustManager)tm;
            }
            throw new X509Exception.TrustManagerException("Couldn't find X509TrustManager");
        }
        catch (IOException | IllegalArgumentException | GeneralSecurityException e) {
            throw new X509Exception.TrustManagerException(e);
        }
    }

    private static String[] getEnabledProtocols(String enabledProtocolsInput, String tlsConfigProtocols) {
        if (enabledProtocolsInput == null) {
            return new String[]{tlsConfigProtocols};
        }
        return enabledProtocolsInput.split(",");
    }

    private static String[] getCipherSuites(String cipherSuitesInput) {
        if (cipherSuitesInput == null) {
            return X509Util.getDefaultCipherSuites();
        }
        return cipherSuitesInput.split(",");
    }
}

