/*
 * Decompiled with CFR 0.152.
 */
package org.apache.knox.gateway.cloud.idbroker.abfs;

import java.io.Closeable;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.Charset;
import java.time.Instant;
import java.util.Date;
import java.util.Locale;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nonnull;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.azurebfs.oauth2.AzureADToken;
import org.apache.hadoop.fs.azurebfs.services.AuthType;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.SecretManager;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.hadoop.service.AbstractService;
import org.apache.hadoop.service.Service;
import org.apache.knox.gateway.cloud.idbroker.abfs.AbfsIDBClient;
import org.apache.knox.gateway.cloud.idbroker.abfs.AbfsIDBConstants;
import org.apache.knox.gateway.cloud.idbroker.abfs.AbfsIDBCredentialProvider;
import org.apache.knox.gateway.cloud.idbroker.abfs.AbfsIDBProperty;
import org.apache.knox.gateway.cloud.idbroker.abfs.AbfsIDBTokenIdentifier;
import org.apache.knox.gateway.cloud.idbroker.common.CommonUtils;
import org.apache.knox.gateway.cloud.idbroker.common.KnoxToken;
import org.apache.knox.gateway.cloud.idbroker.common.KnoxTokenMonitor;
import org.apache.knox.gateway.cloud.idbroker.common.OAuthPayload;
import org.apache.knox.gateway.cloud.idbroker.common.Preconditions;
import org.apache.knox.gateway.cloud.idbroker.messages.RequestDTResponseMessage;
import org.apache.knox.gateway.shell.CloudAccessBrokerSession;
import org.apache.knox.gateway.shell.KnoxSession;
import org.apache.knox.gateway.util.Tokens;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class AbfsIDBIntegration
extends AbstractService {
    private static final Logger LOG = LoggerFactory.getLogger(AbfsIDBIntegration.class);
    private URI fsUri;
    private Text service;
    private final Configuration configuration;
    private KnoxTokenMonitor knoxTokenMonitor;
    private final long knoxTokenExpirationOffsetSeconds;
    private final int retryCount;
    private UserGroupInformation owner;
    private AbfsIDBClient idbClient;
    private Token<AbfsIDBTokenIdentifier> deployedToken;
    private KnoxToken knoxToken;
    private AzureADToken adToken;
    private final Lock serviceStartLock = new ReentrantLock(true);
    private final Lock getKnoxTokenLock = new ReentrantLock(true);
    private final Lock getAzureADTokenLock = new ReentrantLock(true);
    private String correlationId = UUID.randomUUID().toString();
    private static final SecretManager<AbfsIDBTokenIdentifier> secretManager = new TokenSecretManager();

    private AbfsIDBIntegration(@Nonnull URI fsUri, @Nonnull Configuration configuration, @Nonnull String origin) throws IOException {
        this("AbfsIDBIntegration", fsUri, configuration, origin);
    }

    AbfsIDBIntegration(@Nonnull String serviceName, @Nonnull URI fsUri, @Nonnull Configuration configuration, @Nonnull String origin) throws IOException {
        super(serviceName);
        this.fsUri = Preconditions.checkNotNull(fsUri, "Filesystem URI");
        this.configuration = Preconditions.checkNotNull(configuration);
        this.owner = UserGroupInformation.getCurrentUser();
        this.service = new Text(fsUri.getScheme() + "://" + fsUri.getAuthority());
        if (LOG.isDebugEnabled() && !this.service.toString().equals(fsUri.toString())) {
            LOG.debug("Truncating service URI from {} to {} [{}]", new Object[]{fsUri, this.service, origin});
        }
        this.knoxTokenExpirationOffsetSeconds = configuration.getLong(AbfsIDBProperty.IDBROKER_DT_EXPIRATION_OFFSET.getPropertyName(), Long.parseLong(AbfsIDBProperty.IDBROKER_DT_EXPIRATION_OFFSET.getDefaultValue()));
        this.retryCount = configuration.getInt(AbfsIDBProperty.IDBROKER_RETRY_COUNT.getPropertyName(), Integer.parseInt(AbfsIDBProperty.IDBROKER_DT_EXPIRATION_OFFSET.getDefaultValue()));
        LOG.debug("Creating AbfsIDBIntegration:\n\tOrigin: {}\n\tService: {}\n\tOwner: {}", new Object[]{origin, this.service, this.owner.getUserName()});
    }

    static AbfsIDBIntegration fromDelegationTokenManager(URI fsUri, Configuration conf) throws IOException {
        AbfsIDBIntegration integration = new AbfsIDBIntegration(fsUri, conf, "DelegationTokenManager");
        integration.init(conf);
        integration.start();
        return integration;
    }

    static AbfsIDBIntegration fromAbfsCredentialProvider(URI fsUri, Configuration conf) throws IOException {
        AbfsIDBIntegration integration = new AbfsIDBIntegration(fsUri, conf, "CredentialProvider");
        integration.init(conf);
        integration.start();
        return integration;
    }

    protected void serviceStart() throws Exception {
        this.serviceStartLock.lock();
        try {
            LOG.debug("Starting IDB integration for ABFS filesystem {}", (Object)this.fsUri);
            super.serviceStart();
            this.idbClient = this.getClient();
            this.initKnoxTokenMonitor();
            this.deployedToken = this.lookupTokenFromOwner();
            if (this.deployedToken != null) {
                AbfsIDBTokenIdentifier id = (AbfsIDBTokenIdentifier)this.deployedToken.decodeIdentifier();
                this.correlationId = id.getTrackingId();
                this.adToken = this.buildADTokenCredentials(id);
                this.knoxToken = this.buildKnoxToken(id);
                LOG.debug("Deployed for {} with token identifier {}", (Object)this.fsUri, (Object)id);
            }
        }
        finally {
            this.serviceStartLock.unlock();
        }
    }

    private void initKnoxTokenMonitor() {
        if (this.knoxTokenMonitor == null && this.idbClient != null && this.idbClient.shouldInitKnoxTokenMonitor()) {
            this.knoxTokenMonitor = new KnoxTokenMonitor();
        }
    }

    private void monitorKnoxToken() {
        this.initKnoxTokenMonitor();
        if (this.knoxTokenMonitor != null) {
            this.knoxTokenMonitor.monitorKnoxToken(this.knoxToken, this.knoxTokenExpirationOffsetSeconds, new GetKnoxTokenCommand());
        }
    }

    private void stopKnoxTokenMonitor() {
        if (this.knoxTokenMonitor != null) {
            this.knoxTokenMonitor.shutdown();
        }
    }

    protected AbfsIDBClient getClient() throws IOException {
        if (this.idbClient == null) {
            this.idbClient = new AbfsIDBClient(this.configuration, this.owner);
        }
        return this.idbClient;
    }

    protected void serviceStop() throws Exception {
        LOG.debug("Stopping IDB integration for ABFS filesystem {}", (Object)this.fsUri);
        this.stopKnoxTokenMonitor();
        super.serviceStop();
    }

    Text getOwnerText() {
        return new Text(this.getOwner().getUserName());
    }

    UserGroupInformation getOwner() {
        return this.owner;
    }

    private void checkStarted() {
        Preconditions.checkState(this.isInState(Service.STATE.STARTED), "Service is in wrong state %s", this.getServiceState());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Token<AbfsIDBTokenIdentifier> getDelegationToken(String renewer) throws IOException {
        this.getKnoxTokenLock.lock();
        try {
            LOG.debug("Delegation token requested");
            if (this.deployedToken != null) {
                LOG.debug("Returning existing delegation token");
                Token<AbfsIDBTokenIdentifier> token = this.deployedToken;
                return token;
            }
            LOG.debug("Requesting new delegation token");
            this.ensureKnoxToken();
            this.ensureADToken();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Knox token expires in {} seconds:\n\tExpiry: {}", (Object)(this.knoxToken.getExpiry() - Instant.now().getEpochSecond()), (Object)Instant.ofEpochSecond(this.knoxToken.getExpiry()).toString());
            }
            String knoxDT = this.knoxToken == null ? "" : this.knoxToken.getAccessToken();
            long expiryTime = this.knoxToken == null ? 0L : this.knoxToken.getExpiry();
            String endpointCertificate = this.knoxToken == null ? "" : this.knoxToken.getEndpointPublicCert();
            boolean managed = this.knoxToken == null ? false : this.knoxToken.isManaged();
            AbfsIDBTokenIdentifier id = new AbfsIDBTokenIdentifier(this.fsUri, this.getOwnerText(), renewer == null ? null : new Text(renewer), "origin", knoxDT, expiryTime, AbfsIDBIntegration.buildOAuthPayloadFromADToken(this.adToken), System.currentTimeMillis(), this.correlationId, this.idbClient.getCredentialsURL(), endpointCertificate, managed);
            LOG.trace("New ABFS DT {}", (Object)id);
            Token token = new Token((TokenIdentifier)id, secretManager);
            token.setService(this.service);
            Token token2 = token;
            return token2;
        }
        finally {
            this.getKnoxTokenLock.unlock();
        }
    }

    private void ensureADToken() throws IOException {
        this.adToken = this.getADToken(true);
    }

    private void ensureKnoxToken() throws IOException {
        if (this.knoxToken == null || StringUtils.isBlank((CharSequence)this.knoxToken.getAccessToken())) {
            if (this.idbClient.shouldExcludeUserFromGettingKnoxToken()) {
                LOG.info("'{}' is excluded from getting Knox Token from IDBroker", (Object)this.idbClient.getOwnerUserName());
            } else {
                LOG.info("There is no Knox Token available, fetching one from IDBroker...");
                this.getNewKnoxToken();
            }
        } else {
            LOG.debug("Using existing Knox Token: " + Tokens.getTokenDisplayText((String)this.knoxToken.getAccessToken()));
            this.maybeReplaceExpiredKnoxTokenFromUGI();
            this.enforceKnoxTokenNotExpired();
        }
        Preconditions.checkNotNull(this.knoxToken, "Failed to retrieve a Knox Token from the IDBroker.");
    }

    private void maybeReplaceExpiredKnoxTokenFromUGI() throws IOException {
        if (this.knoxToken.isExpired()) {
            Token<AbfsIDBTokenIdentifier> token = this.lookupTokenFromOwner();
            if (token != null) {
                AbfsIDBTokenIdentifier id = (AbfsIDBTokenIdentifier)token.decodeIdentifier();
                this.updateAndMonitorKnoxToken(new KnoxToken(id.getOrigin(), id.getAccessToken(), id.getExpiryTime(), id.getCertificate(), id.isManaged()));
                LOG.info("Updated knoxToken from UGI to {}", (Object)Tokens.getTokenDisplayText((String)this.knoxToken.getAccessToken()));
            } else {
                LOG.warn("Token {} expired but no new token was found in UGI", (Object)Tokens.getTokenDisplayText((String)this.knoxToken.getAccessToken()));
            }
        }
    }

    private void updateAndMonitorKnoxToken(KnoxToken newKnoxToken) {
        this.knoxToken = newKnoxToken;
        this.monitorKnoxToken();
    }

    private void enforceKnoxTokenNotExpired() throws IOException {
        long knoxTokenExpirationOffset;
        if (this.knoxTokenMonitor == null && this.knoxToken.isAboutToExpire(knoxTokenExpirationOffset = this.getConfig().getLong(AbfsIDBProperty.IDBROKER_DT_EXPIRATION_OFFSET.getPropertyName(), Long.parseLong(AbfsIDBProperty.IDBROKER_DT_EXPIRATION_OFFSET.getDefaultValue()))) && this.getClient().hasKerberosCredentials()) {
            this.getNewKnoxToken();
        }
    }

    private Token<AbfsIDBTokenIdentifier> lookupTokenFromOwner() throws IOException {
        return CommonUtils.lookupToken(this.owner.getCredentials(), this.service, AbfsIDBConstants.IDB_TOKEN_KIND);
    }

    private AzureADToken buildADTokenCredentials(AbfsIDBTokenIdentifier deployedIdentifier) {
        AzureADToken adToken;
        if (deployedIdentifier != null) {
            LOG.debug("Using existing delegation token for Azure Credentials");
            adToken = AbfsIDBIntegration.buildADTokenFromOAuth(deployedIdentifier.getMarshalledCredentials());
            if (LOG.isTraceEnabled()) {
                if (adToken == null) {
                    LOG.trace("AD Token: null");
                } else {
                    LOG.trace("AD Token:\n\tToken:{}\n\tExpiry:{}", (Object)adToken.getAccessToken(), (Object)adToken.getExpiry().toInstant().toString());
                }
            }
        } else {
            LOG.debug("Delaying token creation until needed");
            adToken = null;
        }
        return adToken;
    }

    private KnoxToken buildKnoxToken(AbfsIDBTokenIdentifier deployedIdentifier) {
        KnoxToken knoxToken = null;
        if (deployedIdentifier != null) {
            LOG.debug("Using existing delegation token for Knox Token");
            knoxToken = new KnoxToken(deployedIdentifier.getOrigin(), deployedIdentifier.getAccessToken(), deployedIdentifier.getExpiryTime(), deployedIdentifier.getCertificate(), deployedIdentifier.isManaged());
            this.monitorKnoxToken();
            if (LOG.isTraceEnabled()) {
                LOG.trace("Knox Token:\n\tToken:{}\n\tExpiry:{}", (Object)knoxToken.getPrintableAccessToken(), (Object)Instant.ofEpochSecond(knoxToken.getExpiry()).toString());
            }
        } else {
            LOG.debug("Delaying Knox token creation until needed");
        }
        return knoxToken;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    AzureADToken getADToken(boolean renewIfNeeded) throws IOException {
        this.getAzureADTokenLock.lock();
        try {
            LOG.trace("Get an AD Token");
            if (this.adToken == null || renewIfNeeded && this.isExpired(this.adToken)) {
                if (LOG.isDebugEnabled()) {
                    if (this.adToken == null) {
                        LOG.debug("No existing AD Token found, getting a new one.");
                    } else if (this.isExpired(this.adToken)) {
                        LOG.debug("Existing AD Token found, but expired, getting a new one.");
                    }
                }
                this.getNewAzureADToken();
                for (int retry = 0; this.isExpired(this.adToken) && retry <= this.retryCount; ++retry) {
                    if (retry == this.retryCount) {
                        LOG.error(String.format(Locale.ROOT, "Reached maximum configured retries %s, token returned from IDBroker is expired, token expiry timestamp %s, current timestamp %s", this.retryCount, this.adToken.getExpiry(), System.currentTimeMillis() / 1000L));
                        throw new IOException(String.format(Locale.ROOT, "Token returned from IDBroker is expired, token expiry timestamp %s, current timestamp %s", this.adToken.getExpiry(), System.currentTimeMillis() / 1000L));
                    }
                    try {
                        TimeUnit.SECONDS.sleep(5L);
                    }
                    catch (InterruptedException e) {
                        throw new IOException(e);
                    }
                    LOG.info("Received token was expired, attempting to get a new AD token, retry count: " + retry);
                    this.getNewAzureADToken();
                }
            } else {
                LOG.debug("Using existing AD Token");
            }
            AzureADToken azureADToken = this.adToken;
            return azureADToken;
        }
        finally {
            this.getAzureADTokenLock.unlock();
        }
    }

    String getUserAgentSuffix() {
        return "correlationId=" + this.correlationId;
    }

    String getCanonicalServiceName() {
        Preconditions.checkNotNull(this.fsUri, "Not bound to a filesystem URI");
        return this.fsUri.toString();
    }

    private static AbfsIDBTokenIdentifier createEmptyIdentifier() {
        return new AbfsIDBTokenIdentifier();
    }

    private static byte[] getSecretManagerPassword() {
        return "non-password".getBytes(Charset.forName("UTF-8"));
    }

    static AzureADToken buildADTokenFromOAuth(@Nonnull OAuthPayload payload) {
        Preconditions.checkNotNull(payload, "no OAuth payload");
        AzureADToken adToken = new AzureADToken();
        adToken.setAccessToken(payload.getToken());
        adToken.setExpiry(new Date(payload.getExpiration()));
        return adToken;
    }

    static OAuthPayload buildOAuthPayloadFromADToken(@Nonnull AzureADToken adToken) {
        Preconditions.checkNotNull(adToken, "no adToken");
        return new OAuthPayload(adToken.getAccessToken(), adToken.getExpiry().getTime());
    }

    public static void enable(Configuration conf) {
        conf.setEnum("fs.azure.account.auth.type", (Enum)AuthType.Custom);
        conf.set("fs.azure.account.oauth.provider.type", AbfsIDBCredentialProvider.class.getName());
        conf.setBoolean("fs.azure.enable.delegation.token", true);
        conf.set("fs.azure.delegation.token.provider.type", "org.apache.knox.gateway.cloud.idbroker.abfs.AbfsIDBDelegationTokenManager");
    }

    boolean isExpired(AzureADToken azureADToken) {
        if (azureADToken == null) {
            return true;
        }
        Date expiry = azureADToken.getExpiry();
        if (expiry != null) {
            expiry = this.calculateExpirationWithOffset(expiry);
        }
        return expiry == null || expiry.toInstant().isBefore(Instant.now());
    }

    private Date calculateExpirationWithOffset(Date expiry) {
        long credentialsExpirationOffset = this.configuration.getLong(AbfsIDBProperty.IDBROKER_CAB_CREDENTIALS_EXPIRATION_OFFSET.getPropertyName(), Long.parseLong(AbfsIDBProperty.IDBROKER_CAB_CREDENTIALS_EXPIRATION_OFFSET.getDefaultValue()));
        Instant expiration = expiry.toInstant().minusSeconds(credentialsExpirationOffset);
        LOG.debug("Credential expiration time with {} seconds offset: {}", (Object)credentialsExpirationOffset, (Object)expiration);
        return Date.from(expiration);
    }

    private void getNewAzureADToken() throws IOException {
        LOG.trace("Getting a new Azure AD Token");
        CloudAccessBrokerSession knoxCredentialsSession = this.getKnoxCredentialsSession();
        Preconditions.checkNotNull(knoxCredentialsSession, "Failed to obtain a session with the IDBroker.");
        this.adToken = (AzureADToken)this.idbClient.fetchCloudCredentials(knoxCredentialsSession);
        if (LOG.isTraceEnabled()) {
            if (this.adToken == null) {
                LOG.trace("AD Token: null");
            } else {
                LOG.trace("AD Token:\n\tToken:{}\n\tExpiry:{}", (Object)this.adToken.getAccessToken(), (Object)this.adToken.getExpiry().toInstant().toString());
            }
        }
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{knoxCredentialsSession});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void getNewKnoxToken() throws IOException {
        RequestDTResponseMessage message;
        LOG.trace("Getting a new Knox Token");
        Pair<KnoxSession, String> sessionDetails = this.getNewKnoxLoginSession();
        KnoxSession knoxLoginSession = (KnoxSession)sessionDetails.getLeft();
        String origin = (String)sessionDetails.getRight();
        Preconditions.checkNotNull(knoxLoginSession, "Failed to obtain a session with the IDBroker.");
        try {
            message = this.idbClient.requestKnoxDelegationToken(knoxLoginSession, origin, this.fsUri);
        }
        catch (Throwable throwable) {
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{knoxLoginSession});
            throw throwable;
        }
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{knoxLoginSession});
        Preconditions.checkNotNull(message, "Failed to request a delegation token from the IDBroker.");
        this.knoxToken = KnoxToken.fromDTResponse(origin, message);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Knox Token:\n\tToken:{}\n\tExpiry:{}", (Object)this.knoxToken.getPrintableAccessToken(), (Object)Instant.ofEpochSecond(this.knoxToken.getExpiry()).toString());
        }
        this.monitorKnoxToken();
    }

    private CloudAccessBrokerSession getKnoxCredentialsSession() throws IOException {
        this.ensureKnoxToken();
        return this.idbClient.createKnoxCABSession(this.knoxToken);
    }

    private Pair<KnoxSession, String> getNewKnoxLoginSession() throws IOException {
        this.checkStarted();
        LOG.debug("Attempting to create a Knox delegation token session using local credentials (kerberos, simple)");
        Pair<KnoxSession, String> sessionDetails = this.idbClient.createKnoxDTSession(this.configuration);
        if (sessionDetails.getLeft() != null) {
            LOG.debug("Created a Knox delegation token session using local credentials (kerberos, simple)");
        }
        if (sessionDetails.getLeft() == null) {
            String message = this.knoxToken == null ? "Authentication with IDBroker failed.  Please ensure you have a Kerberos token by using kinit." : "Authentication with IDBroker failed.  The existing Knox delegation token has expired and must be renewed. However, it cannot be renewed unless a valid Kerberos token is available. Please ensure you have a Kerberos token by using kinit.";
            throw new IllegalStateException(message);
        }
        return sessionDetails;
    }

    private class GetKnoxTokenCommand
    implements KnoxTokenMonitor.GetKnoxTokenCommand {
        private GetKnoxTokenCommand() {
        }

        @Override
        public void execute(KnoxToken knoxToken) throws IOException {
            AbfsIDBIntegration.this.getKnoxTokenLock.lock();
            try {
                AbfsIDBIntegration.this.getNewKnoxToken();
            }
            finally {
                AbfsIDBIntegration.this.getKnoxTokenLock.unlock();
            }
        }
    }

    protected static class TokenSecretManager
    extends SecretManager<AbfsIDBTokenIdentifier> {
        TokenSecretManager() {
        }

        protected byte[] createPassword(AbfsIDBTokenIdentifier identifier) {
            return AbfsIDBIntegration.getSecretManagerPassword();
        }

        public byte[] retrievePassword(AbfsIDBTokenIdentifier identifier) {
            return AbfsIDBIntegration.getSecretManagerPassword();
        }

        public AbfsIDBTokenIdentifier createIdentifier() {
            return AbfsIDBIntegration.createEmptyIdentifier();
        }
    }
}

