/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.util;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.metastore.IMetaStoreClient;
import org.apache.hadoop.hive.metastore.Warehouse;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.metastore.api.StorageDescriptor;
import org.apache.iceberg.AppendFiles;
import org.apache.iceberg.FileFormat;
import org.apache.iceberg.MetricsConfig;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.Schema;
import org.apache.iceberg.Table;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.data.TableMigrationUtil;
import org.apache.impala.analysis.TableName;
import org.apache.impala.catalog.FeFsPartition;
import org.apache.impala.catalog.FeFsTable;
import org.apache.impala.catalog.iceberg.IcebergCatalog;
import org.apache.impala.common.FileSystemUtil;
import org.apache.impala.common.ImpalaRuntimeException;
import org.apache.impala.thrift.TConvertTableRequest;
import org.apache.impala.thrift.TIcebergCatalog;
import org.apache.impala.thrift.TQueryOptions;
import org.apache.impala.thrift.TTableName;
import org.apache.impala.util.IcebergSchemaConverter;
import org.apache.impala.util.IcebergUtil;
import org.apache.impala.util.ThreadNameAnnotator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MigrateTableUtil {
    private static final Logger LOG = LoggerFactory.getLogger(MigrateTableUtil.class);
    private static final long RETRY_TIMEOUT_MS = 3600000L;
    private static final int RETRY_DELAY_MS = 300;

    private MigrateTableUtil() {
    }

    public static void migrateToIcebergTable(IMetaStoreClient hmsClient, TConvertTableRequest request, FeFsTable table, TQueryOptions queryOptions) throws ImpalaRuntimeException {
        LOG.info("Migrating table to Iceberg: " + table.getFullName());
        Schema schema = IcebergSchemaConverter.convertToIcebergSchema(table.getMetaStoreTable());
        PartitionSpec spec = IcebergSchemaConverter.createIcebergPartitionSpec(table.getMetaStoreTable(), schema);
        String fileFormat = MigrateTableUtil.getFileFormat(table.getMetaStoreTable().getSd());
        Preconditions.checkNotNull((Object)fileFormat);
        HashMap props = Maps.newHashMap(request.getProperties());
        props.put("write.format.default", fileFormat);
        if (IcebergUtil.isHiveCatalog(props)) {
            props.put("external.table.purge", "true");
        }
        String location = table.getLocation();
        TIcebergCatalog tCatalog = IcebergUtil.getTIcebergCatalog(props);
        IcebergCatalog catalog = IcebergUtil.getIcebergCatalog(tCatalog, location);
        TTableName name = request.getTable_name();
        TableIdentifier id = TableIdentifier.of((String[])new String[]{name.getDb_name(), name.getTable_name()});
        Table icebergTable = catalog.createTable(id, schema, spec, location, props);
        Preconditions.checkNotNull((Object)icebergTable);
        TableName tableName = TableName.fromThrift(name);
        try {
            if (IcebergUtil.isHiveCatalog(props)) {
                MigrateTableUtil.waitForTableToBeCreated(hmsClient, tableName);
            }
            MigrateTableUtil.importDataFilesInHdfsTable(table, icebergTable, queryOptions);
        }
        catch (Exception e) {
            if (IcebergUtil.isHiveCatalog(props)) {
                catalog.dropTable(tableName.getDb(), tableName.getTbl(), false);
            }
            Path metadataPath = new Path(location, "metadata");
            FileSystemUtil.deleteIfExists(metadataPath);
            throw new ImpalaRuntimeException("Failed to import data into Iceberg table\n", e);
        }
    }

    private static void waitForTableToBeCreated(IMetaStoreClient hmsClient, TableName tableName) throws ImpalaRuntimeException {
        if (MigrateTableUtil.getHmsTableNoThrow(hmsClient, tableName.getDb(), tableName.getTbl()) != null) {
            return;
        }
        long att = 0L;
        try (ThreadNameAnnotator nameAnnotator = new ThreadNameAnnotator("waiting for " + tableName + " to be created");){
            long end;
            long begin = System.currentTimeMillis();
            do {
                try {
                    Thread.sleep(300L);
                    LOG.info("Waiting for " + tableName + " to be created, attempt: " + ++att);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                end = System.currentTimeMillis();
            } while (MigrateTableUtil.getHmsTableNoThrow(hmsClient, tableName.getDb(), tableName.getTbl()) == null && end - begin < 3600000L);
        }
        if (MigrateTableUtil.getHmsTableNoThrow(hmsClient, tableName.getDb(), tableName.getTbl()) == null) {
            throw new ImpalaRuntimeException("Failed to wait for " + tableName + " to be created");
        }
    }

    private static org.apache.hadoop.hive.metastore.api.Table getHmsTableNoThrow(IMetaStoreClient hiveClient, String dbName, String tblName) {
        try {
            return hiveClient.getTable(dbName, tblName);
        }
        catch (Exception e) {
            return null;
        }
    }

    public static String getFileFormat(StorageDescriptor sd) {
        for (String fileFormat : ImmutableList.of((Object)FileFormat.PARQUET.name().toLowerCase(), (Object)FileFormat.ORC.name().toLowerCase(), (Object)FileFormat.AVRO.name().toLowerCase())) {
            if (!sd.getInputFormat().toLowerCase().contains(fileFormat)) continue;
            return fileFormat;
        }
        return null;
    }

    private static void importDataFilesInHdfsTable(FeFsTable hdfsTable, Table icebergTable, TQueryOptions queryOptions) throws ImpalaRuntimeException {
        Params params = Params.of(hdfsTable.getMetaStoreTable().getSd().getInputFormat(), icebergTable.spec(), icebergTable.schema(), MetricsConfig.forTable((Table)icebergTable), MigrateTableUtil.getDegreeOfParallelism(queryOptions), icebergTable.newAppend(), MigrateTableUtil.getDebugAction(queryOptions));
        if (hdfsTable.isPartitioned()) {
            MigrateTableUtil.importDataFiles(hdfsTable, params);
        } else {
            MigrateTableUtil.importDataFiles(hdfsTable.getMetaStoreTable().getSd().getLocation(), params);
        }
        params.append_.commit();
    }

    private static void importDataFiles(FeFsTable hdfsTable, Params params) throws ImpalaRuntimeException {
        List<? extends FeFsPartition> partitions = hdfsTable.loadPartitions(hdfsTable.getPartitionIds());
        for (FeFsPartition feFsPartition : partitions) {
            String partitionName = feFsPartition.getPartitionName();
            LinkedHashMap partitionKeys = Collections.emptyMap();
            try {
                partitionKeys = Warehouse.makeSpecFromName((String)partitionName);
            }
            catch (MetaException e) {
                throw new ImpalaRuntimeException("Unable to create partition keys for " + partitionName, e);
            }
            MigrateTableUtil.importDataFilesImpl(partitionKeys, feFsPartition.getLocationPath(), params);
        }
    }

    private static void importDataFiles(String location, Params params) throws ImpalaRuntimeException {
        MigrateTableUtil.importDataFilesImpl(Collections.emptyMap(), new Path(location), params);
    }

    private static void importDataFilesImpl(Map<String, String> partitionKeys, Path location, Params params) throws ImpalaRuntimeException {
        try {
            LOG.info("Creating Iceberg metadata for folder: " + location.toString() + " using " + params.threadNum_ + " thread(s).");
            if (params.debugAction_.equalsIgnoreCase("CONVERT_TABLE_FAIL_ICEBERG_CALL")) {
                throw new IllegalArgumentException("Exception thrown by debug action.");
            }
            List dataFiles = TableMigrationUtil.listPartition(partitionKeys, (String)location.toString(), (String)params.format_, (PartitionSpec)params.spec_, (Configuration)FileSystemUtil.getConfiguration(), (MetricsConfig)params.metricsConfig_, null, (int)params.threadNum_);
            dataFiles.forEach(arg_0 -> ((AppendFiles)params.append_).appendFile(arg_0));
        }
        catch (Exception e) {
            throw new ImpalaRuntimeException("Unable load data files for location: " + location.toString(), e);
        }
    }

    private static int getDegreeOfParallelism(TQueryOptions queryOptions) {
        int threadNum = Runtime.getRuntime().availableProcessors();
        if (queryOptions.isSetNum_threads_for_table_migration() && queryOptions.num_threads_for_table_migration > 0) {
            threadNum = Math.min(queryOptions.num_threads_for_table_migration, Runtime.getRuntime().availableProcessors());
        }
        return threadNum;
    }

    private static String getDebugAction(TQueryOptions queryOptions) {
        if (!queryOptions.isSetDebug_action()) {
            return "";
        }
        return queryOptions.getDebug_action();
    }

    private static class Params {
        final String format_;
        final PartitionSpec spec_;
        final Schema schema_;
        final MetricsConfig metricsConfig_;
        final int threadNum_;
        final AppendFiles append_;
        final String debugAction_;

        private Params(String format, PartitionSpec spec, Schema schema, MetricsConfig metricsConfig, int threadNum, AppendFiles append, String debugAction) {
            this.format_ = format;
            this.spec_ = spec;
            this.schema_ = schema;
            this.metricsConfig_ = metricsConfig;
            this.threadNum_ = threadNum;
            this.append_ = append;
            this.debugAction_ = debugAction;
        }

        static Params of(String format, PartitionSpec spec, Schema schema, MetricsConfig metricsConfig, int threadNum, AppendFiles append, String debugAction) {
            return new Params(format, spec, schema, metricsConfig, threadNum, append, debugAction);
        }
    }
}

