/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.util;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import com.google.common.primitives.Ints;
import com.google.common.primitives.Longs;
import com.google.flatbuffers.FlatBufferBuilder;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.metastore.api.Table;
import org.apache.iceberg.ContentFile;
import org.apache.iceberg.FileFormat;
import org.apache.iceberg.FileScanTask;
import org.apache.iceberg.PartitionField;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.Schema;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.TableScan;
import org.apache.iceberg.Transaction;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.expressions.Expression;
import org.apache.iceberg.expressions.Expressions;
import org.apache.iceberg.expressions.Literal;
import org.apache.iceberg.expressions.Term;
import org.apache.iceberg.hadoop.HadoopFileIO;
import org.apache.iceberg.io.CloseableIterable;
import org.apache.iceberg.metrics.MetricsReporter;
import org.apache.iceberg.transforms.PartitionSpecVisitor;
import org.apache.iceberg.transforms.Transform;
import org.apache.iceberg.transforms.Transforms;
import org.apache.iceberg.types.Conversions;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.apache.impala.analysis.Analyzer;
import org.apache.impala.analysis.Expr;
import org.apache.impala.analysis.FunctionCallExpr;
import org.apache.impala.analysis.FunctionName;
import org.apache.impala.analysis.FunctionParams;
import org.apache.impala.analysis.IcebergPartitionField;
import org.apache.impala.analysis.IcebergPartitionSpec;
import org.apache.impala.analysis.IcebergPartitionTransform;
import org.apache.impala.analysis.NumericLiteral;
import org.apache.impala.analysis.StatementBase;
import org.apache.impala.analysis.StringLiteral;
import org.apache.impala.analysis.TimeTravelSpec;
import org.apache.impala.catalog.Column;
import org.apache.impala.catalog.FeIcebergTable;
import org.apache.impala.catalog.HdfsFileFormat;
import org.apache.impala.catalog.IcebergColumn;
import org.apache.impala.catalog.IcebergTable;
import org.apache.impala.catalog.IcebergTableLoadingException;
import org.apache.impala.catalog.TableLoadingException;
import org.apache.impala.catalog.iceberg.GroupedContentFiles;
import org.apache.impala.catalog.iceberg.IcebergCatalog;
import org.apache.impala.catalog.iceberg.IcebergCatalogs;
import org.apache.impala.catalog.iceberg.IcebergHadoopCatalog;
import org.apache.impala.catalog.iceberg.IcebergHadoopTables;
import org.apache.impala.catalog.iceberg.IcebergHiveCatalog;
import org.apache.impala.common.AnalysisException;
import org.apache.impala.common.FileSystemUtil;
import org.apache.impala.common.ImpalaRuntimeException;
import org.apache.impala.common.Pair;
import org.apache.impala.fb.FbFileMetadata;
import org.apache.impala.fb.FbIcebergDataFile;
import org.apache.impala.fb.FbIcebergDataFileFormat;
import org.apache.impala.fb.FbIcebergMetadata;
import org.apache.impala.fb.FbIcebergPartitionTransformValue;
import org.apache.impala.thrift.TCompressionCodec;
import org.apache.impala.thrift.THdfsCompression;
import org.apache.impala.thrift.THdfsFileFormat;
import org.apache.impala.thrift.TIcebergCatalog;
import org.apache.impala.thrift.TIcebergFileFormat;
import org.apache.impala.thrift.TIcebergPartition;
import org.apache.impala.thrift.TIcebergPartitionField;
import org.apache.impala.thrift.TIcebergPartitionSpec;
import org.apache.impala.thrift.TIcebergPartitionTransformType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IcebergUtil {
    public static final int ICEBERG_EPOCH_YEAR = 1970;
    private static final int ICEBERG_EPOCH_MONTH = 1;
    private static final int ICEBERG_EPOCH_DAY = 1;
    private static final int ICEBERG_EPOCH_HOUR = 0;
    public static final String HIVE_CATALOG = "hive.catalog";
    private static final Logger LOG = LoggerFactory.getLogger(IcebergUtil.class);
    public static final String ICEBERG_REST_URI = "iceberg_rest_uri";
    public static final String ICEBERG_REST_USER_ID = "iceberg_rest_user_id";
    public static final String ICEBERG_REST_USER_SECRET = "iceberg_rest_user_secret";
    public static final String ICEBERG_REST_WAREHOUSE_LOCATION = "iceberg_rest_warehouse_location";
    public static final ImmutableMap<String, THdfsCompression> PARQUET_CODEC_MAP = ImmutableMap.builder().put((Object)"none", (Object)THdfsCompression.NONE).put((Object)"gzip", (Object)THdfsCompression.GZIP).put((Object)"snappy", (Object)THdfsCompression.SNAPPY).put((Object)"lz4", (Object)THdfsCompression.LZ4).put((Object)"zstd", (Object)THdfsCompression.ZSTD).build();

    public static IcebergCatalog getIcebergCatalog(FeIcebergTable feTable) throws ImpalaRuntimeException {
        return IcebergUtil.getIcebergCatalog(feTable.getIcebergCatalog(), feTable.getIcebergCatalogLocation());
    }

    public static IcebergCatalog getIcebergCatalog(TIcebergCatalog catalog, String location) throws ImpalaRuntimeException {
        switch (catalog) {
            case HADOOP_TABLES: {
                return IcebergHadoopTables.getInstance();
            }
            case HIVE_CATALOG: {
                return IcebergHiveCatalog.getInstance();
            }
            case HADOOP_CATALOG: {
                return new IcebergHadoopCatalog(location);
            }
            case CATALOGS: {
                return IcebergCatalogs.getInstance();
            }
        }
        throw new ImpalaRuntimeException("Unexpected catalog type: " + (Object)((Object)catalog));
    }

    public static org.apache.iceberg.Table loadTable(FeIcebergTable feTable) throws IcebergTableLoadingException {
        return IcebergUtil.loadTable(feTable.getIcebergCatalog(), IcebergUtil.getIcebergTableIdentifier(feTable), feTable.getIcebergCatalogLocation(), feTable.getMetaStoreTable().getParameters());
    }

    public static org.apache.iceberg.Table loadTable(TIcebergCatalog catalog, TableIdentifier tableId, String location, Map<String, String> tableProps) throws IcebergTableLoadingException {
        try {
            IcebergCatalog cat = IcebergUtil.getIcebergCatalog(catalog, location);
            return cat.loadTable(tableId, location, tableProps);
        }
        catch (ImpalaRuntimeException e) {
            throw new IcebergTableLoadingException(String.format("Failed to load Iceberg table: %s at location: %s", tableId, location), e);
        }
    }

    public static TableIdentifier getIcebergTableIdentifier(FeIcebergTable table) {
        return IcebergUtil.getIcebergTableIdentifier(table.getMetaStoreTable());
    }

    public static TableIdentifier getIcebergTableIdentifier(Table msTable) {
        String name = (String)msTable.getParameters().get("iceberg.table_identifier");
        if (name == null || name.isEmpty()) {
            name = (String)msTable.getParameters().get("name");
        }
        if (name == null || name.isEmpty()) {
            return IcebergUtil.getIcebergTableIdentifier(msTable.getDbName(), msTable.getTableName());
        }
        if (!name.contains(".")) {
            return TableIdentifier.of((String[])new String[]{"default", name});
        }
        return TableIdentifier.parse((String)name);
    }

    public static TableIdentifier getIcebergTableIdentifier(String dbName, String tableName) {
        return TableIdentifier.of((String[])new String[]{dbName, tableName});
    }

    public static Transaction getIcebergTransaction(FeIcebergTable feTable) {
        return feTable.getIcebergApiTable().newTransaction();
    }

    public static PartitionSpec createIcebergPartition(Schema schema, TIcebergPartitionSpec partSpec) throws ImpalaRuntimeException {
        if (partSpec == null) {
            return PartitionSpec.unpartitioned();
        }
        List<TIcebergPartitionField> partitionFields = partSpec.getPartition_fields();
        if (partitionFields == null || partitionFields.isEmpty()) {
            return PartitionSpec.unpartitioned();
        }
        PartitionSpec.Builder builder = PartitionSpec.builderFor((Schema)schema);
        for (TIcebergPartitionField partitionField : partitionFields) {
            TIcebergPartitionTransformType transformType = partitionField.getTransform().getTransform_type();
            if (transformType == TIcebergPartitionTransformType.IDENTITY) {
                builder.identity(partitionField.getOrig_field_name());
                continue;
            }
            if (transformType == TIcebergPartitionTransformType.HOUR) {
                builder.hour(partitionField.getOrig_field_name());
                continue;
            }
            if (transformType == TIcebergPartitionTransformType.DAY) {
                builder.day(partitionField.getOrig_field_name());
                continue;
            }
            if (transformType == TIcebergPartitionTransformType.MONTH) {
                builder.month(partitionField.getOrig_field_name());
                continue;
            }
            if (transformType == TIcebergPartitionTransformType.YEAR) {
                builder.year(partitionField.getOrig_field_name());
                continue;
            }
            if (transformType == TIcebergPartitionTransformType.BUCKET) {
                builder.bucket(partitionField.getOrig_field_name(), partitionField.getTransform().getTransform_param());
                continue;
            }
            if (transformType == TIcebergPartitionTransformType.TRUNCATE) {
                builder.truncate(partitionField.getOrig_field_name(), partitionField.getTransform().getTransform_param());
                continue;
            }
            if (transformType == TIcebergPartitionTransformType.VOID) {
                builder.alwaysNull(partitionField.getOrig_field_name());
                continue;
            }
            throw new ImpalaRuntimeException(String.format("Unknown partition transform '%s' for field '%s", new Object[]{transformType, partitionField.getOrig_field_name()}));
        }
        return builder.build();
    }

    public static List<Term> getPartitioningTerms(TIcebergPartitionSpec partSpec) throws ImpalaRuntimeException {
        ArrayList<Term> result = new ArrayList<Term>();
        for (TIcebergPartitionField field : partSpec.getPartition_fields()) {
            result.add(IcebergUtil.getPartitioningTerm(field));
        }
        return result;
    }

    private static Term getPartitioningTerm(TIcebergPartitionField field) throws ImpalaRuntimeException {
        TIcebergPartitionTransformType transformType = field.getTransform().getTransform_type();
        String fieldName = field.getOrig_field_name();
        int transformParam = field.getTransform().getTransform_param();
        switch (transformType) {
            case IDENTITY: {
                return Expressions.transform((String)fieldName, (Transform)Transforms.identity());
            }
            case HOUR: {
                return Expressions.hour((String)fieldName);
            }
            case DAY: {
                return Expressions.day((String)fieldName);
            }
            case MONTH: {
                return Expressions.month((String)fieldName);
            }
            case YEAR: {
                return Expressions.year((String)fieldName);
            }
            case BUCKET: {
                return Expressions.bucket((String)fieldName, (int)transformParam);
            }
            case TRUNCATE: {
                return Expressions.truncate((String)fieldName, (int)transformParam);
            }
            case VOID: {
                return Expressions.transform((String)fieldName, (Transform)Transforms.alwaysNull());
            }
        }
        throw new ImpalaRuntimeException(String.format("Unknown partition transform '%s' for field '%s", new Object[]{transformType, fieldName}));
    }

    public static boolean isHiveCatalog(Table msTable) {
        return IcebergUtil.isHiveCatalog(msTable.getParameters());
    }

    public static boolean isHiveCatalog(Map<String, String> props) {
        TIcebergCatalog tCat = IcebergUtil.getTIcebergCatalog(props);
        if (tCat == TIcebergCatalog.HIVE_CATALOG) {
            return true;
        }
        if (tCat == TIcebergCatalog.CATALOGS) {
            String catName = props.get("iceberg.catalog");
            tCat = IcebergCatalogs.getInstance().getUnderlyingCatalogType(catName);
            return tCat == TIcebergCatalog.HIVE_CATALOG;
        }
        return false;
    }

    public static TIcebergCatalog getTIcebergCatalog(Table msTable) {
        return IcebergUtil.getTIcebergCatalog(msTable.getParameters());
    }

    public static TIcebergCatalog getTIcebergCatalog(Map<String, String> props) {
        return IcebergUtil.getTIcebergCatalog(props.get("iceberg.catalog"));
    }

    public static TIcebergCatalog getTIcebergCatalog(String catalog) {
        if ("hadoop.tables".equalsIgnoreCase(catalog)) {
            return TIcebergCatalog.HADOOP_TABLES;
        }
        if ("hadoop.catalog".equalsIgnoreCase(catalog)) {
            return TIcebergCatalog.HADOOP_CATALOG;
        }
        if (HIVE_CATALOG.equalsIgnoreCase(catalog) || catalog == null) {
            return TIcebergCatalog.HIVE_CATALOG;
        }
        return TIcebergCatalog.CATALOGS;
    }

    public static TIcebergCatalog getUnderlyingCatalog(Table msTable) {
        return IcebergUtil.getUnderlyingCatalog((String)msTable.getParameters().get("iceberg.catalog"));
    }

    public static TIcebergCatalog getUnderlyingCatalog(String catalog) {
        TIcebergCatalog tCat = IcebergUtil.getTIcebergCatalog(catalog);
        if (tCat == TIcebergCatalog.CATALOGS) {
            return IcebergCatalogs.getInstance().getUnderlyingCatalogType(catalog);
        }
        return tCat;
    }

    public static String getIcebergCatalogLocation(Table msTable) {
        return (String)msTable.getParameters().get("iceberg.catalog_location");
    }

    public static TIcebergFileFormat getIcebergFileFormat(Table msTable) {
        Map params = msTable.getParameters();
        TIcebergFileFormat fileFormat = params.containsKey("write.format.default") ? IcebergUtil.getIcebergFileFormat((String)params.get("write.format.default")) : IcebergUtil.getIcebergFileFormat((String)params.get("iceberg.file_format"));
        return fileFormat == null ? TIcebergFileFormat.PARQUET : fileFormat;
    }

    public static TIcebergFileFormat getIcebergFileFormat(String format) {
        if ("PARQUET".equalsIgnoreCase(format) || format == null) {
            return TIcebergFileFormat.PARQUET;
        }
        if ("ORC".equalsIgnoreCase(format)) {
            return TIcebergFileFormat.ORC;
        }
        if ("AVRO".equalsIgnoreCase(format)) {
            return TIcebergFileFormat.AVRO;
        }
        return null;
    }

    public static THdfsCompression getIcebergParquetCompressionCodec(String codec) {
        if (codec == null) {
            return IcebergTable.DEFAULT_PARQUET_COMPRESSION_CODEC;
        }
        return (THdfsCompression)((Object)PARQUET_CODEC_MAP.get((Object)codec.toLowerCase()));
    }

    public static long getIcebergParquetRowGroupSize(String rowGroupSize) {
        if (rowGroupSize == null) {
            return 0L;
        }
        Long rgSize = Longs.tryParse((String)rowGroupSize);
        if (rgSize == null || rgSize < 0x800000L || rgSize > 0x7FF00000L) {
            return 0L;
        }
        return rgSize;
    }

    public static long getIcebergParquetPageSize(String pageSize) {
        if (pageSize == null) {
            return 0L;
        }
        Long pSize = Longs.tryParse((String)pageSize);
        if (pSize == null || pSize < 65536L || pSize > 0x40000000L) {
            return 0L;
        }
        return pSize;
    }

    public static IcebergPartitionTransform getPartitionTransform(PartitionField field, Map<String, Integer> transformParams) throws ImpalaRuntimeException {
        String type = field.transform().toString();
        String transformMappingKey = IcebergUtil.getPartitionTransformMappingKey(field.sourceId(), IcebergUtil.getPartitionTransformType(type));
        return IcebergUtil.getPartitionTransform(type, transformParams.get(transformMappingKey));
    }

    public static IcebergPartitionTransform getPartitionTransform(String transformType, Integer transformParam) throws ImpalaRuntimeException {
        return new IcebergPartitionTransform(IcebergUtil.getPartitionTransformType(transformType), transformParam);
    }

    public static IcebergPartitionTransform getPartitionTransform(String transformType) throws ImpalaRuntimeException {
        return IcebergUtil.getPartitionTransform(transformType, null);
    }

    public static TIcebergPartitionTransformType getPartitionTransformType(String transformType) throws ImpalaRuntimeException {
        Preconditions.checkNotNull((Object)transformType);
        transformType = transformType.toUpperCase();
        if ("IDENTITY".equals(transformType)) {
            return TIcebergPartitionTransformType.IDENTITY;
        }
        if (transformType.startsWith("BUCKET")) {
            return TIcebergPartitionTransformType.BUCKET;
        }
        if (transformType.startsWith("TRUNCATE")) {
            return TIcebergPartitionTransformType.TRUNCATE;
        }
        switch (transformType) {
            case "HOUR": 
            case "HOURS": {
                return TIcebergPartitionTransformType.HOUR;
            }
            case "DAY": 
            case "DAYS": {
                return TIcebergPartitionTransformType.DAY;
            }
            case "MONTH": 
            case "MONTHS": {
                return TIcebergPartitionTransformType.MONTH;
            }
            case "YEAR": 
            case "YEARS": {
                return TIcebergPartitionTransformType.YEAR;
            }
            case "VOID": {
                return TIcebergPartitionTransformType.VOID;
            }
        }
        throw new ImpalaRuntimeException("Unsupported Iceberg partition type: " + transformType);
    }

    private static String getPartitionTransformMappingKey(int sourceId, TIcebergPartitionTransformType transformType) {
        return sourceId + "_" + transformType.toString();
    }

    public static Map<String, Integer> getPartitionTransformParams(PartitionSpec spec) {
        List transformParams = PartitionSpecVisitor.visit((PartitionSpec)spec, (PartitionSpecVisitor)new PartitionSpecVisitor<Pair<String, Integer>>(){

            public Pair<String, Integer> identity(String sourceName, int sourceId) {
                String mappingKey = IcebergUtil.getPartitionTransformMappingKey(sourceId, TIcebergPartitionTransformType.IDENTITY);
                return new Pair<String, Object>(mappingKey, null);
            }

            public Pair<String, Integer> bucket(String sourceName, int sourceId, int numBuckets) {
                String mappingKey = IcebergUtil.getPartitionTransformMappingKey(sourceId, TIcebergPartitionTransformType.BUCKET);
                return new Pair<String, Integer>(mappingKey, numBuckets);
            }

            public Pair<String, Integer> truncate(String sourceName, int sourceId, int width) {
                String mappingKey = IcebergUtil.getPartitionTransformMappingKey(sourceId, TIcebergPartitionTransformType.TRUNCATE);
                return new Pair<String, Integer>(mappingKey, width);
            }

            public Pair<String, Integer> year(String sourceName, int sourceId) {
                String mappingKey = IcebergUtil.getPartitionTransformMappingKey(sourceId, TIcebergPartitionTransformType.YEAR);
                return new Pair<String, Object>(mappingKey, null);
            }

            public Pair<String, Integer> month(String sourceName, int sourceId) {
                String mappingKey = IcebergUtil.getPartitionTransformMappingKey(sourceId, TIcebergPartitionTransformType.MONTH);
                return new Pair<String, Object>(mappingKey, null);
            }

            public Pair<String, Integer> day(String sourceName, int sourceId) {
                String mappingKey = IcebergUtil.getPartitionTransformMappingKey(sourceId, TIcebergPartitionTransformType.DAY);
                return new Pair<String, Object>(mappingKey, null);
            }

            public Pair<String, Integer> hour(String sourceName, int sourceId) {
                String mappingKey = IcebergUtil.getPartitionTransformMappingKey(sourceId, TIcebergPartitionTransformType.HOUR);
                return new Pair<String, Object>(mappingKey, null);
            }

            public Pair<String, Integer> alwaysNull(int fieldId, String sourceName, int sourceId) {
                String mappingKey = IcebergUtil.getPartitionTransformMappingKey(sourceId, TIcebergPartitionTransformType.VOID);
                return new Pair<String, Object>(mappingKey, null);
            }
        });
        HashMap result = Maps.newHashMap();
        for (Pair transformParam : transformParams) {
            result.put(transformParam.first, transformParam.second);
        }
        return result;
    }

    public static THdfsFileFormat toTHdfsFileFormat(TIcebergFileFormat format) {
        switch (format) {
            case ORC: {
                return THdfsFileFormat.ORC;
            }
            case AVRO: {
                return THdfsFileFormat.AVRO;
            }
        }
        return THdfsFileFormat.PARQUET;
    }

    public static HdfsFileFormat toHdfsFileFormat(TIcebergFileFormat format) {
        return HdfsFileFormat.fromThrift(IcebergUtil.toTHdfsFileFormat(format));
    }

    public static HdfsFileFormat toHdfsFileFormat(String format) {
        TIcebergFileFormat icebergFileFormat = IcebergUtil.getIcebergFileFormat(format);
        if (icebergFileFormat == null) {
            throw new IllegalArgumentException("unknown table format " + format);
        }
        return HdfsFileFormat.fromThrift(IcebergUtil.toTHdfsFileFormat(icebergFileFormat));
    }

    public static CloseableIterable<FileScanTask> planFiles(FeIcebergTable table, List<Expression> predicates, TimeTravelSpec timeTravelSpec, MetricsReporter metricsReporter) throws TableLoadingException {
        if (table.snapshotId() == -1L) {
            return CloseableIterable.empty();
        }
        TableScan scan = IcebergUtil.createScanAsOf(table, timeTravelSpec);
        for (Expression predicate : predicates) {
            scan = (TableScan)scan.filter(predicate);
        }
        if (metricsReporter != null) {
            scan = (TableScan)scan.metricsReporter(metricsReporter);
        }
        return scan.planFiles();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static GroupedContentFiles getIcebergFiles(FeIcebergTable table, List<Expression> predicates, TimeTravelSpec timeTravelSpec) throws TableLoadingException {
        try (CloseableIterable<FileScanTask> fileScanTasks = IcebergUtil.planFiles(table, predicates, timeTravelSpec, null);){
            GroupedContentFiles groupedContentFiles = new GroupedContentFiles(fileScanTasks);
            return groupedContentFiles;
        }
        catch (IOException e) {
            throw new TableLoadingException("Error during reading Iceberg manifest files.", e);
        }
    }

    public static long getSnapshotId(FeIcebergTable table, TimeTravelSpec timeTravelSpec) {
        if (timeTravelSpec == null) {
            return table.snapshotId();
        }
        TableScan scan = IcebergUtil.createScanAsOf(table, timeTravelSpec);
        return scan.snapshot().snapshotId();
    }

    private static TableScan createScanAsOf(FeIcebergTable table, TimeTravelSpec timeTravelSpec) {
        TableScan scan = IcebergUtil.newScan(table);
        if (timeTravelSpec == null) {
            scan = scan.useSnapshot(table.snapshotId());
        } else if (timeTravelSpec.getKind() == TimeTravelSpec.Kind.TIME_AS_OF) {
            scan = IcebergUtil.createScanAsOfTime(timeTravelSpec, scan);
        } else {
            Preconditions.checkState((timeTravelSpec.getKind() == TimeTravelSpec.Kind.VERSION_AS_OF ? 1 : 0) != 0);
            scan = scan.useSnapshot(timeTravelSpec.getAsOfVersion());
        }
        return scan;
    }

    private static TableScan createScanAsOfTime(TimeTravelSpec timeTravelSpec, TableScan scan) {
        Preconditions.checkState((timeTravelSpec.getKind() == TimeTravelSpec.Kind.TIME_AS_OF ? 1 : 0) != 0);
        try {
            scan = scan.asOfTime(timeTravelSpec.getAsOfMillis());
        }
        catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Cannot find a snapshot older than " + timeTravelSpec.toTimeString());
        }
        return scan;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static GroupedContentFiles getIcebergFilesFromSnapshot(FeIcebergTable table, List<Expression> predicates, long snapshotId) throws TableLoadingException {
        if (table.snapshotId() == -1L) {
            return new GroupedContentFiles((CloseableIterable<FileScanTask>)CloseableIterable.empty());
        }
        TableScan scan = IcebergUtil.newScan(table);
        scan = scan.useSnapshot(snapshotId);
        for (Expression predicate : predicates) {
            scan = (TableScan)scan.filter(predicate);
        }
        try (CloseableIterable fileScanTasks = scan.planFiles();){
            GroupedContentFiles groupedContentFiles = new GroupedContentFiles((CloseableIterable<FileScanTask>)fileScanTasks);
            return groupedContentFiles;
        }
        catch (IOException e) {
            throw new TableLoadingException("Error during reading Iceberg manifest files.", e);
        }
    }

    private static TableScan newScan(FeIcebergTable table) {
        TableScan scan = table.getIcebergApiTable().newScan();
        return (TableScan)scan.caseSensitive(false);
    }

    public static String getFilePathHash(ContentFile contentFile) {
        return IcebergUtil.getFilePathHash(contentFile.path().toString());
    }

    public static String getFilePathHash(String path) {
        Hasher hasher = Hashing.murmur3_128().newHasher();
        hasher.putUnencodedChars((CharSequence)path);
        return hasher.hash().toString();
    }

    public static FileFormat fbFileFormatToIcebergFileFormat(byte fbFileFormat) throws ImpalaRuntimeException {
        switch (fbFileFormat) {
            case 0: {
                return FileFormat.PARQUET;
            }
        }
        throw new ImpalaRuntimeException(String.format("Unexpected file format: %s", FbIcebergDataFileFormat.name(fbFileFormat)));
    }

    public static PartitionData partitionDataFromDataFile(Types.StructType partitionType, IcebergPartitionSpec spec, FbIcebergDataFile dataFile) throws ImpalaRuntimeException {
        if (dataFile == null || dataFile.rawPartitionFieldsLength() == 0) {
            return null;
        }
        PartitionData data = new PartitionData(spec.getIcebergPartitionFieldsSize());
        int path_i = 0;
        for (int i = 0; i < spec.getIcebergPartitionFieldsSize(); ++i) {
            IcebergPartitionField field = spec.getIcebergPartitionFields().get(i);
            if (field.getTransformType() == TIcebergPartitionTransformType.VOID) continue;
            Preconditions.checkState((path_i < dataFile.rawPartitionFieldsLength() ? 1 : 0) != 0);
            String[] parts = dataFile.rawPartitionFields(path_i).split("=", 2);
            Preconditions.checkArgument((parts.length == 2 && parts[0] != null && field.getFieldName().equals(parts[0]) ? 1 : 0) != 0, (String)"Invalid partition: %s", (Object)dataFile.rawPartitionFields(path_i));
            TIcebergPartitionTransformType transformType = field.getTransformType();
            data.set(i, IcebergUtil.getPartitionValue(((Types.NestedField)partitionType.fields().get(i)).type(), transformType, parts[1]));
            ++path_i;
        }
        return data;
    }

    public static Object getPartitionValue(Type type, TIcebergPartitionTransformType transformType, String stringValue) throws ImpalaRuntimeException {
        String HIVE_NULL = "__HIVE_DEFAULT_PARTITION__";
        if (stringValue == null || stringValue.equals(HIVE_NULL)) {
            return null;
        }
        if (transformType == TIcebergPartitionTransformType.IDENTITY || transformType == TIcebergPartitionTransformType.TRUNCATE || transformType == TIcebergPartitionTransformType.BUCKET || transformType == TIcebergPartitionTransformType.DAY) {
            return Conversions.fromPartitionString((Type)type, (String)stringValue);
        }
        switch (transformType) {
            case YEAR: {
                return IcebergUtil.parseYearToTransformYear(stringValue);
            }
            case MONTH: {
                return IcebergUtil.parseMonthToTransformMonth(stringValue);
            }
            case HOUR: {
                return IcebergUtil.parseHourToTransformHour(stringValue);
            }
            case VOID: {
                return null;
            }
        }
        throw new ImpalaRuntimeException("Unexpected partition transform: " + (Object)((Object)transformType));
    }

    public static Integer getDateTimeTransformValue(TIcebergPartitionTransformType transformType, String stringValue) throws ImpalaRuntimeException {
        try {
            switch (transformType) {
                case YEAR: {
                    return IcebergUtil.parseYearToTransformYear(stringValue);
                }
                case MONTH: {
                    return IcebergUtil.parseMonthToTransformMonth(stringValue);
                }
                case DAY: {
                    return IcebergUtil.parseDayToTransformMonth(stringValue);
                }
                case HOUR: {
                    return IcebergUtil.parseHourToTransformHour(stringValue);
                }
            }
        }
        catch (IllegalStateException | NumberFormatException | DateTimeException e) {
            throw new ImpalaRuntimeException(String.format("Unable to parse value '%s' as '%s' transform value", new Object[]{stringValue, transformType}));
        }
        throw new ImpalaRuntimeException("Unexpected partition transform: " + (Object)((Object)transformType));
    }

    public static boolean isDateTimeTransformType(TIcebergPartitionTransformType transformType) {
        return transformType.equals((Object)TIcebergPartitionTransformType.YEAR) || transformType.equals((Object)TIcebergPartitionTransformType.MONTH) || transformType.equals((Object)TIcebergPartitionTransformType.DAY) || transformType.equals((Object)TIcebergPartitionTransformType.HOUR);
    }

    private static Integer parseYearToTransformYear(String yearStr) {
        int year = Integer.parseInt(yearStr);
        return year - 1970;
    }

    private static Integer parseMonthToTransformMonth(String monthStr) {
        String[] parts = monthStr.split("-", -1);
        Preconditions.checkState((parts.length == 2 ? 1 : 0) != 0);
        int year = Integer.parseInt(parts[0]);
        int month = Integer.parseInt(parts[1]);
        int years = year - 1970;
        int months = month - 1;
        return years * 12 + months;
    }

    private static Integer parseDayToTransformMonth(String monthStr) {
        Literal days = Literal.of((CharSequence)monthStr).to((Type)Types.DateType.get());
        return (Integer)days.value();
    }

    private static Integer parseHourToTransformHour(String hourStr) {
        OffsetDateTime epoch = Instant.ofEpochSecond(0L).atOffset(ZoneOffset.UTC);
        String[] parts = hourStr.split("-", -1);
        Preconditions.checkState((parts.length == 4 ? 1 : 0) != 0);
        int year = Integer.parseInt(parts[0]);
        int month = Integer.parseInt(parts[1]);
        int day = Integer.parseInt(parts[2]);
        int hour = Integer.parseInt(parts[3]);
        OffsetDateTime datetime = OffsetDateTime.of(LocalDateTime.of(year, month, day, hour, 0), ZoneOffset.UTC);
        return (int)ChronoUnit.HOURS.between(epoch, datetime);
    }

    public static TCompressionCodec parseParquetCompressionCodec(boolean onCreateTbl, Map<String, String> tblProperties, StringBuilder errMsg) {
        String codecTblProp = tblProperties.get("write.parquet.compression-codec");
        THdfsCompression codec = IcebergUtil.getIcebergParquetCompressionCodec(codecTblProp);
        if (codec == null) {
            errMsg.append("Invalid parquet compression codec for Iceberg table: ").append(codecTblProp);
            return null;
        }
        TCompressionCodec compressionCodec = new TCompressionCodec();
        if (tblProperties.containsKey("write.parquet.compression-codec")) {
            compressionCodec.setCodec(codec);
        }
        if (onCreateTbl && codec != THdfsCompression.ZSTD) {
            if (tblProperties.containsKey("write.parquet.compression-level")) {
                errMsg.append("Parquet compression level cannot be set for codec ").append((Object)codec).append(". Only ZSTD codec supports compression level table property.");
                return null;
            }
        } else if (tblProperties.containsKey("write.parquet.compression-level")) {
            String clevelTblProp = tblProperties.get("write.parquet.compression-level");
            Integer clevel = Ints.tryParse((String)clevelTblProp);
            if (clevel == null) {
                errMsg.append("Invalid parquet compression level for Iceberg table: ").append(clevelTblProp);
                return null;
            }
            if (clevel < 1 || clevel > 22) {
                errMsg.append("Parquet compression level for Iceberg table should fall in the range of [").append(1).append("..").append(22).append("]");
                return null;
            }
            compressionCodec.setCompression_level(clevel);
        }
        return compressionCodec;
    }

    public static Long parseParquetRowGroupSize(Map<String, String> tblProperties, StringBuilder errMsg) {
        if (tblProperties.containsKey("write.parquet.row-group-size-bytes")) {
            String propVal = tblProperties.get("write.parquet.row-group-size-bytes");
            Long rowGroupSize = Longs.tryParse((String)propVal);
            if (rowGroupSize == null) {
                errMsg.append("Invalid parquet row group size for Iceberg table: ").append(propVal);
                return null;
            }
            if (rowGroupSize < 0x800000L || rowGroupSize > 0x7FF00000L) {
                errMsg.append("Parquet row group size for Iceberg table should ").append("fall in the range of [").append(0x800000L).append("..").append(0x7FF00000L).append("]");
                return null;
            }
            return rowGroupSize;
        }
        return 0L;
    }

    public static boolean isAnyWriteModeSet(Map<String, String> tblProperties) {
        return tblProperties.get("write.delete.mode") != null || tblProperties.get("write.update.mode") != null || tblProperties.get("write.merge.mode") != null;
    }

    public static Long parseParquetPageSize(Map<String, String> tblProperties, String property, String descr, StringBuilder errMsg) {
        if (tblProperties.containsKey(property)) {
            String propVal = tblProperties.get(property);
            Long pageSize = Longs.tryParse((String)propVal);
            if (pageSize == null) {
                errMsg.append("Invalid parquet ").append(descr).append(" for Iceberg table: ").append(propVal);
                return null;
            }
            if (pageSize < 65536L || pageSize > 0x40000000L) {
                errMsg.append("Parquet ").append(descr).append(" for Iceberg table should fall in the range of [").append(65536L).append("..").append(0x40000000L).append("]");
                return null;
            }
            return pageSize;
        }
        return 0L;
    }

    public static Set<Long> currentAncestorIds(org.apache.iceberg.Table table) {
        HashSet<Long> ret = new HashSet<Long>();
        Snapshot snapshot = table.currentSnapshot();
        while (snapshot != null) {
            ret.add(snapshot.snapshotId());
            Long parentId = snapshot.parentId();
            if (parentId == null) break;
            snapshot = table.snapshot(parentId.longValue());
        }
        return ret;
    }

    public static TIcebergPartition createIcebergPartitionInfo(org.apache.iceberg.Table iceApiTbl, ContentFile cf) {
        TIcebergPartition partInfo = new TIcebergPartition(cf.specId(), new ArrayList<String>());
        PartitionSpec spec = (PartitionSpec)iceApiTbl.specs().get(cf.specId());
        Preconditions.checkState((spec.fields().size() == cf.partition().size() ? 1 : 0) != 0);
        ArrayList<String> partitionKeys = new ArrayList<String>();
        for (int i = 0; i < spec.fields().size(); ++i) {
            Object partValue = cf.partition().get(i, Object.class);
            if (partValue != null) {
                partitionKeys.add(partValue.toString());
                continue;
            }
            partitionKeys.add("NULL");
        }
        partInfo.setPartition_values(partitionKeys);
        return partInfo;
    }

    public static FbFileMetadata createIcebergMetadata(org.apache.iceberg.Table iceApiTbl, ContentFile cf, int partId) {
        FlatBufferBuilder fbb = new FlatBufferBuilder(1);
        int iceOffset = IcebergUtil.createIcebergMetadata(iceApiTbl, fbb, cf, partId);
        fbb.finish(FbFileMetadata.createFbFileMetadata(fbb, iceOffset));
        ByteBuffer bb = fbb.dataBuffer().slice();
        ByteBuffer compressedBb = ByteBuffer.allocate(bb.capacity());
        compressedBb.put(bb);
        return FbFileMetadata.getRootAsFbFileMetadata((ByteBuffer)compressedBb.flip());
    }

    private static int createIcebergMetadata(org.apache.iceberg.Table iceApiTbl, FlatBufferBuilder fbb, ContentFile cf, int partId) {
        int partKeysOffset = -1;
        PartitionSpec spec = (PartitionSpec)iceApiTbl.specs().get(cf.specId());
        if (spec != null && !spec.fields().isEmpty()) {
            partKeysOffset = IcebergUtil.createPartitionKeys(fbb, spec, cf);
        }
        int eqFieldIdsOffset = -1;
        List eqFieldIds = cf.equalityFieldIds();
        if (eqFieldIds != null && !eqFieldIds.isEmpty()) {
            eqFieldIdsOffset = FbIcebergMetadata.createEqualityFieldIdsVector(fbb, eqFieldIds.stream().mapToInt(i -> i).sorted().toArray());
        }
        FbIcebergMetadata.startFbIcebergMetadata(fbb);
        int fileFormat = -1;
        if (cf.format() == FileFormat.PARQUET) {
            fileFormat = 0;
        } else if (cf.format() == FileFormat.ORC) {
            fileFormat = 1;
        } else if (cf.format() == FileFormat.AVRO) {
            fileFormat = 2;
        }
        if (fileFormat != -1) {
            FbIcebergMetadata.addFileFormat(fbb, (byte)fileFormat);
        }
        FbIcebergMetadata.addSpecId(fbb, cf.specId());
        FbIcebergMetadata.addRecordCount(fbb, cf.recordCount());
        if (partKeysOffset != -1) {
            FbIcebergMetadata.addPartitionKeys(fbb, partKeysOffset);
        }
        if (cf.dataSequenceNumber() != null) {
            FbIcebergMetadata.addDataSequenceNumber(fbb, cf.dataSequenceNumber());
        } else {
            FbIcebergMetadata.addDataSequenceNumber(fbb, -1L);
        }
        if (eqFieldIdsOffset != -1) {
            FbIcebergMetadata.addEqualityFieldIds(fbb, eqFieldIdsOffset);
        }
        FbIcebergMetadata.addPartId(fbb, partId);
        return FbIcebergMetadata.endFbIcebergMetadata(fbb);
    }

    private static int createPartitionKeys(FlatBufferBuilder fbb, PartitionSpec spec, ContentFile cf) {
        Preconditions.checkState((spec.fields().size() == cf.partition().size() ? 1 : 0) != 0);
        int[] partitionKeyOffsets = new int[spec.fields().size()];
        for (int i = 0; i < spec.fields().size(); ++i) {
            partitionKeyOffsets[i] = IcebergUtil.createPartitionTransformValue(fbb, spec, cf, i);
        }
        return FbIcebergMetadata.createPartitionKeysVector(fbb, partitionKeyOffsets);
    }

    private static int createPartitionTransformValue(FlatBufferBuilder fbb, PartitionSpec spec, ContentFile cf, int fieldIndex) {
        PartitionField field = (PartitionField)spec.fields().get(fieldIndex);
        Pair<Byte, Integer> transform = IcebergUtil.getFbTransform(spec.schema(), field);
        int valueOffset = -1;
        if ((Byte)transform.first != 7) {
            Object partValue = cf.partition().get(fieldIndex, Object.class);
            String partValueString = partValue != null ? partValue.toString() : "__HIVE_DEFAULT_PARTITION__";
            valueOffset = fbb.createString((CharSequence)partValueString);
        }
        FbIcebergPartitionTransformValue.startFbIcebergPartitionTransformValue(fbb);
        FbIcebergPartitionTransformValue.addTransformType(fbb, (Byte)transform.first);
        if (transform.second != null) {
            FbIcebergPartitionTransformValue.addTransformParam(fbb, (Integer)transform.second);
        }
        if (valueOffset != -1) {
            FbIcebergPartitionTransformValue.addTransformValue(fbb, valueOffset);
        }
        FbIcebergPartitionTransformValue.addSourceId(fbb, field.sourceId());
        return FbIcebergPartitionTransformValue.endFbIcebergPartitionTransformValue(fbb);
    }

    private static Pair<Byte, Integer> getFbTransform(Schema schema, PartitionField field) {
        return (Pair)PartitionSpecVisitor.visit((Schema)schema, (PartitionField)field, (PartitionSpecVisitor)new PartitionSpecVisitor<Pair<Byte, Integer>>(){

            public Pair<Byte, Integer> identity(String sourceName, int sourceId) {
                return new Pair<Byte, Object>((byte)0, null);
            }

            public Pair<Byte, Integer> bucket(String sourceName, int sourceId, int numBuckets) {
                return new Pair<Byte, Integer>((byte)5, numBuckets);
            }

            public Pair<Byte, Integer> truncate(String sourceName, int sourceId, int width) {
                return new Pair<Byte, Integer>((byte)6, width);
            }

            public Pair<Byte, Integer> year(String sourceName, int sourceId) {
                return new Pair<Byte, Object>((byte)4, null);
            }

            public Pair<Byte, Integer> month(String sourceName, int sourceId) {
                return new Pair<Byte, Object>((byte)3, null);
            }

            public Pair<Byte, Integer> day(String sourceName, int sourceId) {
                return new Pair<Byte, Object>((byte)2, null);
            }

            public Pair<Byte, Integer> hour(String sourceName, int sourceId) {
                return new Pair<Byte, Object>((byte)1, null);
            }

            public Pair<Byte, Integer> alwaysNull(int fieldId, String sourceName, int sourceId) {
                return new Pair<Byte, Object>((byte)7, null);
            }
        });
    }

    public static TIcebergPartitionTransformType getPartitionTransformType(IcebergColumn column, IcebergPartitionSpec spec) {
        if (!spec.hasPartitionFields()) {
            return TIcebergPartitionTransformType.VOID;
        }
        for (IcebergPartitionField partField : spec.getIcebergPartitionFields()) {
            if (partField.getTransformType() == TIcebergPartitionTransformType.VOID || column.getFieldId() != partField.getSourceId()) continue;
            return partField.getTransformType();
        }
        return TIcebergPartitionTransformType.VOID;
    }

    public static boolean isPartitionColumn(IcebergColumn column, IcebergPartitionSpec spec) {
        return IcebergUtil.getPartitionTransformType(column, spec) != TIcebergPartitionTransformType.VOID;
    }

    public static Map<String, String> composeCatalogProperties() {
        Configuration conf = FileSystemUtil.getConfiguration();
        HashMap<String, String> props = new HashMap<String, String>();
        ArrayList<String> configKeys = new ArrayList<String>(Arrays.asList("io-impl", "io.manifest.cache-enabled", "io.manifest.cache.expiration-interval-ms", "io.manifest.cache.max-total-bytes", "io.manifest.cache.max-content-length", ICEBERG_REST_URI, ICEBERG_REST_USER_ID, ICEBERG_REST_USER_SECRET, ICEBERG_REST_WAREHOUSE_LOCATION));
        for (String key : configKeys) {
            String val = conf.get("iceberg." + key);
            if (val == null) continue;
            props.put(key, val);
        }
        if (!props.containsKey("io-impl")) {
            props.put("io-impl", HadoopFileIO.class.getName());
        }
        return props;
    }

    public static void validateIcebergTableForInsert(FeIcebergTable iceTable) throws AnalysisException {
        for (Types.NestedField field : iceTable.getIcebergSchema().columns()) {
            Types.TimestampType tsType;
            Type iceType = field.type();
            if (!iceType.isPrimitiveType() || !(iceType instanceof Types.TimestampType) || !(tsType = (Types.TimestampType)iceType).shouldAdjustToUTC()) continue;
            throw new AnalysisException("The Iceberg table has a TIMESTAMPTZ column that Impala cannot write.");
        }
        for (Column c : iceTable.getColumns()) {
            if (!c.getType().isComplexType()) continue;
            throw new AnalysisException(String.format("Impala does not support writing tables with complex types. Table '%s' has column '%s' with type: %s", iceTable.getFullName(), c.getName(), c.getType().toSql()));
        }
        if (iceTable.getIcebergFileFormat() != TIcebergFileFormat.PARQUET) {
            throw new AnalysisException(String.format("Impala can only write Parquet data files, while table '%s' expects '%s' data files.", iceTable.getFullName(), iceTable.getIcebergFileFormat().toString()));
        }
    }

    public static void populatePartitionExprs(Analyzer analyzer, List<Expr> widestTypeExprList, List<Column> selectExprTargetColumns, List<Expr> selectListExprs, FeIcebergTable targetTable, List<Expr> outPartitionExprs, List<Integer> outPartitionColPos) throws AnalysisException {
        Preconditions.checkNotNull(outPartitionExprs);
        Preconditions.checkState((boolean)outPartitionExprs.isEmpty());
        Preconditions.checkState((outPartitionColPos == null || outPartitionColPos.isEmpty() ? 1 : 0) != 0);
        IcebergPartitionSpec icebergPartSpec = targetTable.getDefaultPartitionSpec();
        if (!icebergPartSpec.hasPartitionFields()) {
            return;
        }
        String tableName = targetTable.getFullName();
        block0: for (IcebergPartitionField partField : icebergPartSpec.getIcebergPartitionFields()) {
            if (partField.getTransformType() == TIcebergPartitionTransformType.VOID) continue;
            for (int i = 0; i < selectListExprs.size(); ++i) {
                IcebergColumn targetColumn = (IcebergColumn)selectExprTargetColumns.get(i);
                if (targetColumn.getFieldId() != partField.getSourceId()) continue;
                Expr widestTypeExpr = widestTypeExprList != null ? widestTypeExprList.get(i) : null;
                Expr icebergPartitionTransformExpr = IcebergUtil.getIcebergPartitionTransformExpr(analyzer, tableName, partField, targetColumn, selectListExprs.get(i), widestTypeExpr);
                outPartitionExprs.add(icebergPartitionTransformExpr);
                if (outPartitionColPos == null) continue block0;
                outPartitionColPos.add(targetColumn.getPosition());
                continue block0;
            }
        }
    }

    private static Expr getIcebergPartitionTransformExpr(Analyzer analyzer, String targetTableName, IcebergPartitionField partField, IcebergColumn targetColumn, Expr sourceExpr, Expr widestTypeExpr) throws AnalysisException {
        Preconditions.checkState((targetColumn.getFieldId() == partField.getSourceId() ? 1 : 0) != 0);
        Expr compatibleExpr = StatementBase.checkTypeCompatibility(targetTableName, targetColumn, sourceExpr, analyzer, widestTypeExpr);
        Expr ret = IcebergUtil.getIcebergPartitionTransformExpr(partField, compatibleExpr);
        ret.analyze(analyzer);
        return ret;
    }

    private static Expr getIcebergPartitionTransformExpr(IcebergPartitionField partField, Expr compatibleExpr) {
        String funcNameStr = IcebergUtil.transformTypeToFunctionName(partField.getTransformType());
        if (funcNameStr == null || funcNameStr.equals("")) {
            return compatibleExpr;
        }
        FunctionName fnName = new FunctionName("_impala_builtins", funcNameStr);
        ArrayList<Expr> paramList = new ArrayList<Expr>();
        paramList.add(compatibleExpr);
        Integer transformParam = partField.getTransformParam();
        if (transformParam != null) {
            paramList.add(NumericLiteral.create(transformParam.intValue()));
        }
        if (partField.getTransformType() == TIcebergPartitionTransformType.MONTH) {
            paramList.add(new StringLiteral("yyyy-MM"));
        } else if (partField.getTransformType() == TIcebergPartitionTransformType.HOUR) {
            paramList.add(new StringLiteral("yyyy-MM-dd-HH"));
        }
        FunctionCallExpr fnCall = new FunctionCallExpr(fnName, new FunctionParams(paramList));
        fnCall.setIsInternalFnCall(true);
        return fnCall;
    }

    private static String transformTypeToFunctionName(TIcebergPartitionTransformType transformType) {
        switch (transformType) {
            case IDENTITY: {
                return "";
            }
            case HOUR: 
            case MONTH: {
                return "from_timestamp";
            }
            case DAY: {
                return "to_date";
            }
            case YEAR: {
                return "year";
            }
            case BUCKET: {
                return "iceberg_bucket_transform";
            }
            case TRUNCATE: {
                return "iceberg_truncate_transform";
            }
        }
        return "";
    }

    public static class ComputeStatsSnapshotPropertyConverter {
        public static TreeMap<Integer, Long> stringToMap(String snapshotIds) {
            String[] columns;
            TreeMap<Integer, Long> res = new TreeMap<Integer, Long>();
            if (snapshotIds == null) {
                return res;
            }
            for (String column : columns = snapshotIds.split(",")) {
                String[] colIdAndSnapshotId = column.split(":");
                if (colIdAndSnapshotId.length != 2) {
                    return ComputeStatsSnapshotPropertyConverter.logAndReturnEmptyMap(snapshotIds);
                }
                try {
                    String colStr = colIdAndSnapshotId[0];
                    Long snapshotId = Long.parseLong(colIdAndSnapshotId[1]);
                    String[] colRange = colStr.split("-");
                    if (colRange.length == 1) {
                        res.put(Integer.parseInt(colRange[0]), snapshotId);
                        continue;
                    }
                    if (colRange.length == 2) {
                        int rangeStart = Integer.parseInt(colRange[0]);
                        int rangeEnd = Integer.parseInt(colRange[1]);
                        for (int colId = rangeStart; colId <= rangeEnd; ++colId) {
                            res.put(colId, snapshotId);
                        }
                        continue;
                    }
                    return ComputeStatsSnapshotPropertyConverter.logAndReturnEmptyMap(snapshotIds);
                }
                catch (NumberFormatException e) {
                    return ComputeStatsSnapshotPropertyConverter.logAndReturnEmptyMap(snapshotIds);
                }
            }
            return res;
        }

        public static String mapToString(TreeMap<Integer, Long> colAndSnapshotIds) {
            ConversionState state = new ConversionState();
            for (Map.Entry<Integer, Long> entry : colAndSnapshotIds.entrySet()) {
                long snapshotId;
                int col = entry.getKey();
                if (state.canContinueRange(col, snapshotId = entry.getValue().longValue())) {
                    state.extendRange();
                    continue;
                }
                state.flushRange();
                state.initNewRange(col, snapshotId);
            }
            state.flushRange();
            return state.getResult();
        }

        private static TreeMap<Integer, Long> logAndReturnEmptyMap(String snapshotIdsStr) {
            LOG.warn(String.format("Invalid value for table property '%s': \"%s\". Ignoring it.", "impala.computeStatsSnapshotIds", snapshotIdsStr));
            return new TreeMap<Integer, Long>();
        }

        private static class ConversionState {
            private static final int INVALID = -10;
            private int colRangeStart_ = -10;
            private int lastCol_ = -10;
            private long lastSnapshotId_ = -10L;
            private final StringBuilder sb_ = new StringBuilder();

            private ConversionState() {
            }

            private boolean canContinueRange(int col, long snapshotId) {
                return this.lastCol_ != -10 && this.lastSnapshotId_ != -10L && this.lastSnapshotId_ == snapshotId && this.lastCol_ + 1 == col;
            }

            private void extendRange() {
                ++this.lastCol_;
            }

            private void flushRange() {
                if (this.colRangeStart_ != -10) {
                    this.sb_.append(this.colRangeStart_);
                    if (this.lastCol_ != this.colRangeStart_) {
                        this.sb_.append("-").append(this.lastCol_);
                    }
                    this.sb_.append(":").append(this.lastSnapshotId_);
                }
            }

            private void initNewRange(int col, long snapshotId) {
                if (this.colRangeStart_ != -10) {
                    this.sb_.append(",");
                }
                this.colRangeStart_ = col;
                this.lastCol_ = col;
                this.lastSnapshotId_ = snapshotId;
            }

            private String getResult() {
                return this.sb_.toString();
            }
        }
    }

    public static class PartitionData
    implements StructLike {
        private final Object[] values;

        private PartitionData(int size) {
            this.values = new Object[size];
        }

        public int size() {
            return this.values.length;
        }

        public <T> T get(int pos, Class<T> javaClass) {
            return javaClass.cast(this.values[pos]);
        }

        public <T> void set(int pos, T value) {
            if (value instanceof ByteBuffer) {
                ByteBuffer buffer = (ByteBuffer)value;
                byte[] bytes = new byte[buffer.remaining()];
                buffer.duplicate().get(bytes);
                this.values[pos] = bytes;
            } else {
                this.values[pos] = value;
            }
        }

        public boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (other == null || this.getClass() != other.getClass()) {
                return false;
            }
            PartitionData that = (PartitionData)other;
            return Arrays.equals(this.values, that.values);
        }

        public int hashCode() {
            return Arrays.hashCode(this.values);
        }
    }
}

