/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.util;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.Map;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveEntry;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveInfo;
import org.apache.hadoop.hive.metastore.api.Partition;
import org.apache.hadoop.hive.metastore.api.Table;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.impala.analysis.TableName;
import org.apache.impala.catalog.HdfsPartition;
import org.apache.impala.common.FileSystemUtil;
import org.apache.impala.common.ImpalaException;
import org.apache.impala.common.ImpalaRuntimeException;
import org.apache.impala.thrift.THdfsCachingOp;
import org.apache.log4j.Logger;

public class HdfsCachingUtil {
    private static final Logger LOG = Logger.getLogger(HdfsCachingUtil.class);
    public static final String CACHE_DIR_ID_PROP_NAME = "cache_directive_id";
    public static final String CACHE_DIR_REPLICATION_PROP_NAME = "cache_replication";
    private static final int MAX_UNCHANGED_CACHING_REFRESH_INTERVALS = 5;
    private static DistributedFileSystem dfs = null;

    private static DistributedFileSystem getDfs() throws ImpalaRuntimeException {
        if (dfs == null) {
            try {
                dfs = FileSystemUtil.getDistributedFileSystem();
            }
            catch (IOException e) {
                throw new ImpalaRuntimeException("HdfsCachingUtil failed to initialize the DistributedFileSystem: ", e);
            }
        }
        return dfs;
    }

    public static long submitCacheTblDirective(Table table, String poolName, short replication) throws ImpalaRuntimeException {
        long id = HdfsCachingUtil.submitDirective(new Path(table.getSd().getLocation()), poolName, replication);
        table.putToParameters(CACHE_DIR_ID_PROP_NAME, Long.toString(id));
        table.putToParameters(CACHE_DIR_REPLICATION_PROP_NAME, Long.toString(replication));
        return id;
    }

    public static long submitCachePartitionDirective(HdfsPartition.Builder part, String poolName, short replication) throws ImpalaRuntimeException {
        long id = HdfsCachingUtil.submitDirective(new Path(part.getLocation()), poolName, replication);
        part.putToParameters(CACHE_DIR_ID_PROP_NAME, Long.toString(id));
        part.putToParameters(CACHE_DIR_REPLICATION_PROP_NAME, Long.toString(replication));
        return id;
    }

    public static long submitCachePartitionDirective(Partition part, String poolName, short replication) throws ImpalaRuntimeException {
        long id = HdfsCachingUtil.submitDirective(new Path(part.getSd().getLocation()), poolName, replication);
        part.putToParameters(CACHE_DIR_ID_PROP_NAME, Long.toString(id));
        part.putToParameters(CACHE_DIR_REPLICATION_PROP_NAME, Long.toString(replication));
        return id;
    }

    public static void removeTblCacheDirective(Table table) throws ImpalaRuntimeException {
        Long id;
        Map parameters;
        Preconditions.checkNotNull((Object)table);
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Uncaching table: " + table.getDbName() + "." + table.getTableName()));
        }
        if ((parameters = table.getParameters()) == null) {
            LOG.warn((Object)("removePartitionCacheDirective(): table " + table.getTableName() + "has a null parameter map."));
        }
        if ((id = HdfsCachingUtil.getCacheDirectiveId(parameters)) == null) {
            LOG.warn((Object)("removePartitionCacheDirective(): table " + table.getTableName() + "doesn't have a cache directive id."));
            return;
        }
        HdfsCachingUtil.removeDirective(id);
        table.getParameters().remove(CACHE_DIR_ID_PROP_NAME);
        table.getParameters().remove(CACHE_DIR_REPLICATION_PROP_NAME);
    }

    public static void removePartitionCacheDirective(HdfsPartition.Builder part) throws ImpalaException {
        Long id;
        Preconditions.checkNotNull((Object)part);
        Map<String, String> parameters = part.getParameters();
        if (parameters == null) {
            LOG.warn((Object)("removePartitionCacheDirective(): partition " + part.getPartitionName() + "has a null parameter map."));
        }
        if ((id = HdfsCachingUtil.getCacheDirectiveId(parameters)) == null) {
            LOG.warn((Object)("removePartitionCacheDirective(): partition " + part.getPartitionName() + "doesn't have a cache directive id."));
            return;
        }
        HdfsCachingUtil.removeDirective(id);
        part.getParameters().remove(CACHE_DIR_ID_PROP_NAME);
        part.getParameters().remove(CACHE_DIR_REPLICATION_PROP_NAME);
    }

    public static void removePartitionCacheDirective(Map<String, String> partitionParams) throws ImpalaException {
        Long id = HdfsCachingUtil.getCacheDirectiveId(partitionParams);
        if (id == null) {
            return;
        }
        HdfsCachingUtil.removeDirective(id);
        partitionParams.remove(CACHE_DIR_ID_PROP_NAME);
        partitionParams.remove(CACHE_DIR_REPLICATION_PROP_NAME);
    }

    public static Long getCacheDirectiveId(Map<String, String> params) {
        if (params == null) {
            return null;
        }
        String idStr = params.get(CACHE_DIR_ID_PROP_NAME);
        if (idStr == null) {
            return null;
        }
        try {
            return Long.parseLong(idStr);
        }
        catch (NumberFormatException e) {
            return null;
        }
    }

    public static String getCachePool(long directiveId) throws ImpalaRuntimeException {
        CacheDirectiveEntry entry = HdfsCachingUtil.getDirective(directiveId);
        return entry == null ? null : entry.getInfo().getPool();
    }

    public static Short getCacheReplication(long directiveId) throws ImpalaRuntimeException {
        CacheDirectiveEntry entry = HdfsCachingUtil.getDirective(directiveId);
        return entry != null ? entry.getInfo().getReplication() : null;
    }

    public static Short getCachedCacheReplication(Map<String, String> params) {
        Preconditions.checkNotNull(params);
        String replication = params.get(CACHE_DIR_REPLICATION_PROP_NAME);
        if (replication == null) {
            return (short)1;
        }
        try {
            return Short.parseShort(replication);
        }
        catch (NumberFormatException e) {
            return (short)1;
        }
    }

    public static void waitForDirective(long directiveId) throws ImpalaRuntimeException {
        long bytesNeeded = 0L;
        long currentBytesCached = 0L;
        CacheDirectiveEntry cacheDir = HdfsCachingUtil.getDirective(directiveId);
        if (cacheDir == null) {
            return;
        }
        bytesNeeded = cacheDir.getStats().getBytesNeeded();
        currentBytesCached = cacheDir.getStats().getBytesCached();
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)String.format("Waiting on cache directive id: %d. Bytes cached (%d) / needed (%d)", directiveId, currentBytesCached, bytesNeeded));
        }
        if (bytesNeeded == currentBytesCached) {
            return;
        }
        long hdfsRefreshIntervalMs = HdfsCachingUtil.getDfs().getConf().getLong("dfs.namenode.path.based.cache.refresh.interval.ms", 30000L);
        Preconditions.checkState((hdfsRefreshIntervalMs > 0L ? 1 : 0) != 0);
        int unchangedCounter = 0;
        while (unchangedCounter < 5) {
            long previousBytesCached = currentBytesCached;
            cacheDir = HdfsCachingUtil.getDirective(directiveId);
            if (cacheDir == null) {
                return;
            }
            currentBytesCached = cacheDir.getStats().getBytesCached();
            if (currentBytesCached == (bytesNeeded = cacheDir.getStats().getBytesNeeded())) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)String.format("Cache directive id: %d has completed.Bytes cached (%d) / needed (%d)", directiveId, currentBytesCached, bytesNeeded));
                }
                return;
            }
            unchangedCounter = currentBytesCached == previousBytesCached ? ++unchangedCounter : 0;
            try {
                Thread.sleep((long)((double)hdfsRefreshIntervalMs * 1.25));
            }
            catch (InterruptedException interruptedException) {}
        }
        LOG.warn((Object)String.format("No changes in cached bytes in: %d(ms). All data may not be cached. Final stats for cache directive id: %d. Bytes cached (%d)/needed (%d)", hdfsRefreshIntervalMs * 5L, directiveId, currentBytesCached, bytesNeeded));
    }

    private static long submitDirective(Path path, String poolName, short replication) throws ImpalaRuntimeException {
        Preconditions.checkNotNull((Object)path);
        Preconditions.checkState((poolName != null && !poolName.isEmpty() ? 1 : 0) != 0);
        CacheDirectiveInfo info = new CacheDirectiveInfo.Builder().setPool(poolName).setReplication(Short.valueOf(replication)).setPath(path).build();
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Submitting cache directive: " + info.toString()));
        }
        try {
            return HdfsCachingUtil.getDfs().addCacheDirective(info);
        }
        catch (IOException e) {
            throw new ImpalaRuntimeException(e.getMessage(), e);
        }
    }

    public static long modifyCacheDirective(Long id, Table table, String poolName, short replication) throws ImpalaRuntimeException {
        Preconditions.checkNotNull((Object)id);
        HdfsCachingUtil.modifyCacheDirective(id, new Path(table.getSd().getLocation()), poolName, replication);
        table.putToParameters(CACHE_DIR_ID_PROP_NAME, Long.toString(id));
        table.putToParameters(CACHE_DIR_REPLICATION_PROP_NAME, Long.toString(replication));
        return id;
    }

    public static long modifyCacheDirective(Long id, HdfsPartition.Builder part, String poolName, short replication) throws ImpalaRuntimeException {
        Preconditions.checkNotNull((Object)id);
        HdfsCachingUtil.modifyCacheDirective(id, new Path(part.getLocation()), poolName, replication);
        part.putToParameters(CACHE_DIR_ID_PROP_NAME, Long.toString(id));
        part.putToParameters(CACHE_DIR_REPLICATION_PROP_NAME, Long.toString(replication));
        return id;
    }

    private static void modifyCacheDirective(Long id, Path path, String poolName, short replication) throws ImpalaRuntimeException {
        Preconditions.checkNotNull((Object)path);
        Preconditions.checkNotNull((Object)id);
        Preconditions.checkState((poolName != null && !poolName.isEmpty() ? 1 : 0) != 0);
        CacheDirectiveInfo info = new CacheDirectiveInfo.Builder().setId(id).setPool(poolName).setReplication(Short.valueOf(replication)).setPath(path).build();
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Modifying cache directive: " + info.toString()));
        }
        try {
            HdfsCachingUtil.getDfs().modifyCacheDirective(info);
        }
        catch (IOException e) {
            throw new ImpalaRuntimeException(e.getMessage(), e);
        }
    }

    private static void removeDirective(long directiveId) throws ImpalaRuntimeException {
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Removing cache directive id: " + directiveId));
        }
        try {
            HdfsCachingUtil.getDfs().removeCacheDirective(directiveId);
        }
        catch (IOException e) {
            if (e.getMessage().contains("No directive with ID")) {
                return;
            }
            throw new ImpalaRuntimeException(e.getMessage(), e);
        }
    }

    private static CacheDirectiveEntry getDirective(long directiveId) throws ImpalaRuntimeException {
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Getting cache directive id: " + directiveId));
        }
        CacheDirectiveInfo filter = new CacheDirectiveInfo.Builder().setId(Long.valueOf(directiveId)).build();
        try {
            RemoteIterator itr = HdfsCachingUtil.getDfs().listCacheDirectives(filter);
            if (itr.hasNext()) {
                return (CacheDirectiveEntry)itr.next();
            }
        }
        catch (IOException e) {
            throw new ImpalaRuntimeException(e.getMessage(), e);
        }
        throw new ImpalaRuntimeException("HDFS cache directive filter returned empty result. This must not happen");
    }

    public static short getReplicationOrDefault(THdfsCachingOp op) {
        return op.isSetReplication() ? op.getReplication() : (short)1;
    }

    public static boolean isUpdateOp(THdfsCachingOp op, Map<String, String> params) throws ImpalaRuntimeException {
        Long directiveId = Long.parseLong(params.get(CACHE_DIR_ID_PROP_NAME));
        CacheDirectiveEntry entry = HdfsCachingUtil.getDirective(directiveId);
        Preconditions.checkNotNull((Object)entry);
        if (!op.getCache_pool_name().equals(entry.getInfo().getPool())) {
            return false;
        }
        return op.isSetReplication() && op.getReplication() != entry.getInfo().getReplication().shortValue() || !op.isSetReplication() && entry.getInfo().getReplication() != 1;
    }

    public static void validateCachePool(THdfsCachingOp op, Long directiveId, TableName table, HdfsPartition partition) throws ImpalaRuntimeException {
        CacheDirectiveEntry entry = HdfsCachingUtil.getDirective(directiveId);
        Preconditions.checkNotNull((Object)entry);
        if (!op.getCache_pool_name().equals(entry.getInfo().getPool())) {
            throw new ImpalaRuntimeException(String.format("Cannot cache partition in pool '%s' because it is already cached in '%s'. To change the cache pool for this partition, first uncache using: ALTER TABLE %s.%s %sSET UNCACHED", op.getCache_pool_name(), entry.getInfo().getPool(), table.getDb(), table, partition != null ? String.format(" PARTITION(%s) ", partition.getPartitionName().replaceAll("/", ", ")) : ""));
        }
    }

    public static void validateCachePool(THdfsCachingOp op, Long directiveId, TableName table) throws ImpalaRuntimeException {
        HdfsCachingUtil.validateCachePool(op, directiveId, table, null);
    }

    public static boolean validateCacheParams(Map<String, String> params) {
        Long directiveId = HdfsCachingUtil.getCacheDirectiveId(params);
        if (directiveId == null) {
            return false;
        }
        CacheDirectiveEntry entry = null;
        try {
            entry = HdfsCachingUtil.getDirective(directiveId);
        }
        catch (ImpalaRuntimeException e) {
            if (e.getCause() != null && e.getCause() instanceof RemoteException) {
                LOG.error((Object)"Cache directive does not exist", (Throwable)e);
                params.remove(CACHE_DIR_ID_PROP_NAME);
                params.remove(CACHE_DIR_REPLICATION_PROP_NAME);
            } else {
                LOG.error((Object)"IO Exception, possible connectivity issues with HDFS", (Throwable)e);
            }
            return false;
        }
        Preconditions.checkNotNull((Object)entry);
        String replicationFactor = params.get(CACHE_DIR_REPLICATION_PROP_NAME);
        if (replicationFactor != null && Short.parseShort(replicationFactor) != entry.getInfo().getReplication()) {
            LOG.info((Object)("Replication factor for entry in HDFS differs from value in Hive MS: " + entry.getInfo().getPath().toString() + " " + entry.getInfo().getReplication().toString() + " != " + params.get(CACHE_DIR_REPLICATION_PROP_NAME)));
        }
        params.put(CACHE_DIR_REPLICATION_PROP_NAME, String.valueOf(entry.getInfo().getReplication()));
        return true;
    }
}

