/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.errorprone.annotations.Immutable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.common.ValidTxnList;
import org.apache.hadoop.hive.common.ValidWriteIdList;
import org.apache.impala.catalog.CatalogException;
import org.apache.impala.catalog.CatalogServiceCatalog;
import org.apache.impala.catalog.Column;
import org.apache.impala.catalog.FeTable;
import org.apache.impala.catalog.FileDescriptor;
import org.apache.impala.catalog.FileMetadataLoader;
import org.apache.impala.catalog.HdfsPartition;
import org.apache.impala.catalog.HdfsTable;
import org.apache.impala.catalog.ScalarType;
import org.apache.impala.catalog.StructField;
import org.apache.impala.catalog.StructType;
import org.apache.impala.common.FileSystemUtil;
import org.apache.impala.common.Pair;
import org.apache.impala.common.PrintUtils;
import org.apache.impala.compat.MetastoreShim;
import org.apache.impala.thrift.TTransactionalType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AcidUtils {
    private static final Logger LOG = LoggerFactory.getLogger(AcidUtils.class);
    public static final String INSERTONLY_TRANSACTIONAL_PROPERTY = "insert_only";
    public static final String TABLE_IS_TRANSACTIONAL = "transactional";
    public static final String TABLE_TRANSACTIONAL_PROPERTIES = "transactional_properties";
    private static final Pattern BASE_PATTERN = Pattern.compile("base_(?<writeId>\\d+)(?:_v(?<visibilityTxnId>\\d+))?(?:/.*)?");
    private static final String DELTA_STR = "delta_(?<minWriteId>\\d+)_(?<maxWriteId>\\d+)(?:_(?<statementId>\\d+)|_v(?<visibilityTxnId>\\d+))?(?:/.*)?";
    private static final Pattern DELTA_PATTERN = Pattern.compile("delta_(?<minWriteId>\\d+)_(?<maxWriteId>\\d+)(?:_(?<statementId>\\d+)|_v(?<visibilityTxnId>\\d+))?(?:/.*)?");
    private static final Pattern DELETE_DELTA_PATTERN = Pattern.compile("delete_delta_(?<minWriteId>\\d+)_(?<maxWriteId>\\d+)(?:_(?<statementId>\\d+)|_v(?<visibilityTxnId>\\d+))?(?:/.*)?");
    @VisibleForTesting
    static final long SENTINEL_BASE_WRITE_ID = Long.MIN_VALUE;

    private static boolean isInsertOnlyTable(Map<String, String> props) {
        Preconditions.checkNotNull(props);
        if (!AcidUtils.isTransactionalTable(props)) {
            return false;
        }
        String transactionalProp = props.get(TABLE_TRANSACTIONAL_PROPERTIES);
        return transactionalProp != null && INSERTONLY_TRANSACTIONAL_PROPERTY.equalsIgnoreCase(transactionalProp);
    }

    public static boolean isTransactionalTable(FeTable table) {
        Map parameters = table.getMetaStoreTable().getParameters();
        return parameters != null && AcidUtils.isTransactionalTable(parameters);
    }

    public static boolean isTransactionalTable(Map<String, String> props) {
        Preconditions.checkNotNull(props);
        String tableIsTransactional = props.get(TABLE_IS_TRANSACTIONAL);
        if (tableIsTransactional == null) {
            tableIsTransactional = props.get(TABLE_IS_TRANSACTIONAL.toUpperCase());
        }
        return tableIsTransactional != null && tableIsTransactional.equalsIgnoreCase("true");
    }

    public static boolean isFullAcidTable(Map<String, String> props) {
        return AcidUtils.isTransactionalTable(props) && !AcidUtils.isInsertOnlyTable(props);
    }

    public static Column getRowIdColumnType(int position) {
        StructType row__id = new StructType();
        row__id.addField(new StructField("operation", ScalarType.INT, ""));
        row__id.addField(new StructField("originaltransaction", ScalarType.BIGINT, ""));
        row__id.addField(new StructField("bucket", ScalarType.INT, ""));
        row__id.addField(new StructField("rowid", ScalarType.BIGINT, ""));
        row__id.addField(new StructField("currenttransaction", ScalarType.BIGINT, ""));
        return new Column("row__id", row__id, "", position);
    }

    public static void setTransactionalProperties(Map<String, String> props, TTransactionalType defaultTransactionalType) {
        Preconditions.checkNotNull(props);
        if (props.get(TABLE_IS_TRANSACTIONAL) != null || props.get(TABLE_TRANSACTIONAL_PROPERTIES) != null) {
            return;
        }
        switch (defaultTransactionalType) {
            case NONE: {
                break;
            }
            case INSERT_ONLY: {
                props.put(TABLE_IS_TRANSACTIONAL, "true");
                props.put(TABLE_TRANSACTIONAL_PROPERTIES, INSERTONLY_TRANSACTIONAL_PROPERTY);
            }
        }
    }

    public static String getFirstLevelAcidDirPath(Path dataPath, FileSystem fileSystem) throws IOException {
        if (dataPath == null) {
            return null;
        }
        String firstLevelAcidDir = AcidUtils.getAcidSubDir(dataPath);
        if (firstLevelAcidDir != null) {
            return firstLevelAcidDir;
        }
        String acidDirPath = AcidUtils.getFirstLevelAcidDirPath(dataPath.getParent(), fileSystem);
        if (acidDirPath == null) {
            return null;
        }
        if (fileSystem.isDirectory(dataPath)) {
            return acidDirPath + "/" + dataPath.getName();
        }
        return acidDirPath;
    }

    private static String getAcidSubDir(Path dataPath) {
        String dataDir = dataPath.getName();
        if (dataDir.startsWith("base_") || dataDir.startsWith("delta_") || dataDir.startsWith("delete_delta_")) {
            return dataDir;
        }
        return null;
    }

    @VisibleForTesting
    static ParsedBase parseBase(String relPath) {
        Matcher baseMatcher = BASE_PATTERN.matcher(relPath);
        if (baseMatcher.matches()) {
            long writeId = Long.valueOf(baseMatcher.group("writeId"));
            long visibilityTxnId = -1L;
            String visibilityTxnIdStr = baseMatcher.group("visibilityTxnId");
            if (visibilityTxnIdStr != null) {
                visibilityTxnId = Long.valueOf(visibilityTxnIdStr);
            }
            return new ParsedBase(writeId, visibilityTxnId);
        }
        return new ParsedBase(Long.MIN_VALUE, -1L);
    }

    @VisibleForTesting
    static long getBaseWriteId(String relPath) {
        return AcidUtils.parseBase((String)relPath).writeId;
    }

    private static ParsedDelta matcherToParsedDelta(Matcher deltaMatcher) {
        if (!deltaMatcher.matches()) {
            return null;
        }
        long minWriteId = Long.valueOf(deltaMatcher.group("minWriteId"));
        long maxWriteId = Long.valueOf(deltaMatcher.group("maxWriteId"));
        String statementIdStr = deltaMatcher.group("statementId");
        long statementId = statementIdStr != null ? Long.valueOf(statementIdStr) : -1L;
        String visibilityTxnIdStr = deltaMatcher.group("visibilityTxnId");
        long visibilityTxnId = visibilityTxnIdStr != null ? Long.valueOf(visibilityTxnIdStr) : -1L;
        return new ParsedDelta(minWriteId, maxWriteId, statementId, visibilityTxnId);
    }

    private static ParsedDelta parseDelta(String dirPath) {
        return AcidUtils.matcherToParsedDelta(DELTA_PATTERN.matcher(dirPath));
    }

    private static ParsedDelta parseDeleteDelta(String dirPath) {
        return AcidUtils.matcherToParsedDelta(DELETE_DELTA_PATTERN.matcher(dirPath));
    }

    private static String getFirstDirName(String relPath) {
        int slashIdx = relPath.indexOf("/");
        if (slashIdx != -1) {
            return relPath.substring(0, slashIdx);
        }
        return null;
    }

    public static boolean isDeleteDeltaFd(FileDescriptor fd) {
        return fd.getPath().startsWith("delete_delta_");
    }

    public static int filterFdsForAcidState(List<FileDescriptor> fds, ValidWriteIdList validWriteIdList) throws CatalogException {
        Preconditions.checkNotNull(fds);
        if (validWriteIdList == null) {
            return 0;
        }
        WriteListBasedPredicate writeListBasedPredicate = new WriteListBasedPredicate(validWriteIdList, true);
        Iterator<FileDescriptor> it = fds.iterator();
        int numRemoved = 0;
        while (it.hasNext()) {
            if (writeListBasedPredicate.check(it.next().getPath())) continue;
            it.remove();
            ++numRemoved;
        }
        return numRemoved;
    }

    public static List<FileStatus> filterFilesForAcidState(List<FileStatus> stats, Path baseDir, ValidTxnList validTxnList, ValidWriteIdList writeIds, @Nullable FileMetadataLoader.LoadStats loadStats) throws CatalogException {
        WriteListBasedPredicate pred = new WriteListBasedPredicate(validTxnList, writeIds);
        long maxBaseWriteId = Long.MIN_VALUE;
        HashSet<String> deltaDirNames = new HashSet<String>();
        Iterator<FileStatus> it = stats.iterator();
        while (it.hasNext()) {
            FileStatus stat = it.next();
            String relPath = FileSystemUtil.relativizePath(stat.getPath(), baseDir);
            if (!pred.check(relPath)) {
                it.remove();
                if (loadStats == null) continue;
                ++loadStats.uncommittedAcidFilesSkipped;
                continue;
            }
            maxBaseWriteId = Math.max(AcidUtils.getBaseWriteId(relPath), maxBaseWriteId);
            String dirName = AcidUtils.getFirstDirName(relPath);
            if (dirName == null || !dirName.startsWith("delta_") && !dirName.startsWith("delete_delta_")) continue;
            deltaDirNames.add(dirName);
        }
        List<Pair<String, ParsedDelta>> deltas = AcidUtils.getValidDeltaDirsOrdered(deltaDirNames, maxBaseWriteId);
        Set<String> filteredDeltaDirs = AcidUtils.getFilteredDeltaDirs(deltas, maxBaseWriteId, writeIds);
        return AcidUtils.filterFilesForAcidState(stats, baseDir, maxBaseWriteId, filteredDeltaDirs, loadStats);
    }

    private static List<FileStatus> filterFilesForAcidState(List<FileStatus> stats, Path baseDir, long maxBaseWriteId, Set<String> deltaDirs, @Nullable FileMetadataLoader.LoadStats loadStats) throws CatalogException {
        ArrayList<FileStatus> validStats = new ArrayList<FileStatus>(stats);
        Iterator it = validStats.iterator();
        while (it.hasNext()) {
            FileStatus stat = (FileStatus)it.next();
            if (stat.isDirectory()) {
                it.remove();
                continue;
            }
            String relPath = FileSystemUtil.relativizePath(stat.getPath(), baseDir);
            if (relPath.startsWith("delta_") || relPath.startsWith("delete_delta_")) {
                String dirName = AcidUtils.getFirstDirName(relPath);
                if (dirName != null && !deltaDirs.contains(dirName)) {
                    it.remove();
                    if (loadStats != null) {
                        ++loadStats.filesSupersededByAcidState;
                    }
                }
                if (!relPath.endsWith("_flush_length")) continue;
                throw new CatalogException("Found Hive Streaming side-file: " + stat.getPath() + " It means that the contents of the directory are currently being written, therefore Impala is not able to read it. Please try to load the table again once Hive Streaming commits the transaction.");
            }
            long baseWriteId = AcidUtils.getBaseWriteId(relPath);
            if (baseWriteId != Long.MIN_VALUE) {
                if (baseWriteId >= maxBaseWriteId) continue;
                it.remove();
                if (loadStats == null) continue;
                ++loadStats.filesSupersededByAcidState;
                continue;
            }
            if (maxBaseWriteId == Long.MIN_VALUE) continue;
            it.remove();
        }
        return validStats;
    }

    private static List<Pair<String, ParsedDelta>> getValidDeltaDirsOrdered(Set<String> deltaDirNames, long baseWriteId) throws CatalogException {
        ArrayList<Pair<String, ParsedDelta>> deltas = new ArrayList<Pair<String, ParsedDelta>>();
        Iterator<String> it = deltaDirNames.iterator();
        while (it.hasNext()) {
            String dirname = it.next();
            ParsedDelta parsedDelta = AcidUtils.parseDelta(dirname);
            if (parsedDelta == null) {
                parsedDelta = AcidUtils.parseDeleteDelta(dirname);
            }
            if (parsedDelta == null) continue;
            if (parsedDelta.minWriteId <= baseWriteId) {
                Preconditions.checkState((parsedDelta.maxWriteId <= baseWriteId ? 1 : 0) != 0);
                it.remove();
                continue;
            }
            deltas.add(new Pair<String, ParsedDelta>(dirname, parsedDelta));
        }
        deltas.sort(new Comparator<Pair<String, ParsedDelta>>(){

            @Override
            public int compare(Pair<String, ParsedDelta> o1, Pair<String, ParsedDelta> o2) {
                ParsedDelta pd1 = (ParsedDelta)o1.second;
                ParsedDelta pd2 = (ParsedDelta)o2.second;
                if (pd1.minWriteId != pd2.minWriteId) {
                    if (pd1.minWriteId < pd2.minWriteId) {
                        return -1;
                    }
                    return 1;
                }
                if (pd1.maxWriteId != pd2.maxWriteId) {
                    if (pd1.maxWriteId < pd2.maxWriteId) {
                        return 1;
                    }
                    return -1;
                }
                if (pd1.statementId != pd2.statementId) {
                    if (pd1.statementId < pd2.statementId) {
                        return -1;
                    }
                    return 1;
                }
                if (pd1.visibilityTxnId != pd2.visibilityTxnId) {
                    if (pd1.visibilityTxnId < pd2.visibilityTxnId) {
                        return 1;
                    }
                    return -1;
                }
                return ((String)o1.first).compareTo((String)o2.first);
            }
        });
        return deltas;
    }

    private static Set<String> getFilteredDeltaDirs(List<Pair<String, ParsedDelta>> deltas, long baseWriteId, ValidWriteIdList writeIds) {
        long current = baseWriteId;
        long lastStmtId = -1L;
        ParsedDelta prev = null;
        HashSet<String> filteredDeltaDirs = new HashSet<String>();
        for (Pair<String, ParsedDelta> pathDelta : deltas) {
            ParsedDelta next = (ParsedDelta)pathDelta.second;
            if (next.maxWriteId > current) {
                if (writeIds.isWriteIdRangeValid(current + 1L, next.maxWriteId) == ValidWriteIdList.RangeResponse.NONE) continue;
                filteredDeltaDirs.add((String)pathDelta.first);
                current = next.maxWriteId;
                lastStmtId = next.statementId;
                prev = next;
                continue;
            }
            if (next.maxWriteId == current && lastStmtId >= 0L) {
                filteredDeltaDirs.add((String)pathDelta.first);
                prev = next;
                continue;
            }
            if (prev == null || next.maxWriteId != prev.maxWriteId || next.minWriteId != prev.minWriteId || next.statementId != prev.statementId || next.visibilityTxnId != prev.visibilityTxnId) continue;
            filteredDeltaDirs.add((String)pathDelta.first);
            prev = next;
        }
        return filteredDeltaDirs;
    }

    public static int compare(HdfsTable tbl, ValidWriteIdList validWriteIdList, long tableId) {
        Preconditions.checkState((tbl != null && tbl.getMetaStoreTable() != null ? 1 : 0) != 0);
        if (!AcidUtils.isTransactionalTable(tbl)) {
            return 0;
        }
        Preconditions.checkNotNull((Object)tbl.getValidWriteIds());
        if (tableId != -1L && MetastoreShim.getTableId(tbl.getMetaStoreTable()) != tableId) {
            return -1;
        }
        return AcidUtils.compare(tbl.getValidWriteIds(), validWriteIdList);
    }

    @VisibleForTesting
    public static int compare(ValidWriteIdList a, ValidWriteIdList b) {
        Preconditions.checkState((boolean)a.getTableName().equalsIgnoreCase(b.getTableName()));
        int minLen = Math.min(a.getInvalidWriteIds().length, b.getInvalidWriteIds().length);
        for (int i = 0; i < minLen; ++i) {
            if (a.getInvalidWriteIds()[i] == b.getInvalidWriteIds()[i]) continue;
            return a.getInvalidWriteIds()[i] > b.getInvalidWriteIds()[i] ? 1 : -1;
        }
        if (a.getInvalidWriteIds().length == b.getInvalidWriteIds().length) {
            return Long.signum(a.getHighWatermark() - b.getHighWatermark());
        }
        if (a.getInvalidWriteIds().length == minLen) {
            if (a.getHighWatermark() != b.getInvalidWriteIds()[minLen] - 1L) {
                return Long.signum(a.getHighWatermark() - (b.getInvalidWriteIds()[minLen] - 1L));
            }
            if (AcidUtils.allInvalidFrom(b.getInvalidWriteIds(), minLen, b.getHighWatermark())) {
                return 0;
            }
            return -1;
        }
        if (b.getHighWatermark() != a.getInvalidWriteIds()[minLen] - 1L) {
            return Long.signum(a.getInvalidWriteIds()[minLen] - 1L - b.getHighWatermark());
        }
        if (AcidUtils.allInvalidFrom(a.getInvalidWriteIds(), minLen, a.getHighWatermark())) {
            return 0;
        }
        return 1;
    }

    private static boolean allInvalidFrom(long[] invalidIds, int start, long hwm) {
        for (int i = start + 1; i < invalidIds.length; ++i) {
            if (invalidIds[i] == invalidIds[i - 1] + 1L) continue;
            return false;
        }
        return invalidIds[invalidIds.length - 1] == hwm;
    }

    public static List<HdfsPartition.Builder> getPartitionsForRefreshingFileMetadata(CatalogServiceCatalog catalog, HdfsTable hdfsTable) throws CatalogException {
        Stopwatch sw = Stopwatch.createStarted();
        Preconditions.checkState((boolean)hdfsTable.isReadLockedByCurrentThread());
        List<HdfsPartition.Builder> partBuilders = MetastoreShim.getPartitionsForRefreshingFileMetadata(catalog, hdfsTable);
        LOG.debug("Checked the latest compaction id for {}. Time taken: {}", (Object)hdfsTable.getFullName(), (Object)PrintUtils.printTimeMs(sw.stop().elapsed(TimeUnit.MILLISECONDS)));
        return partBuilders;
    }

    @Immutable
    private static final class ParsedDelta {
        final long minWriteId;
        final long maxWriteId;
        final long statementId;
        final long visibilityTxnId;

        ParsedDelta(long minWriteId, long maxWriteId, long statementId, long visibilityTxnId) {
            this.minWriteId = minWriteId;
            this.maxWriteId = maxWriteId;
            this.statementId = statementId;
            this.visibilityTxnId = visibilityTxnId;
        }

        private boolean isCompactedDeltaFile() {
            return this.visibilityTxnId != -1L;
        }
    }

    @Immutable
    private static final class ParsedBase {
        final long writeId;
        final long visibilityTxnId;

        ParsedBase(long writeId, long visibilityTxnId) {
            this.writeId = writeId;
            this.visibilityTxnId = visibilityTxnId;
        }
    }

    private static class WriteListBasedPredicate {
        @Nullable
        private final ValidTxnList validTxnList;
        private final ValidWriteIdList writeIdList;
        private final boolean doStrictCheck;

        WriteListBasedPredicate(ValidWriteIdList writeIdList, boolean strictMode) {
            this.validTxnList = null;
            this.writeIdList = (ValidWriteIdList)Preconditions.checkNotNull((Object)writeIdList);
            this.doStrictCheck = strictMode;
        }

        WriteListBasedPredicate(ValidTxnList validTxnList, ValidWriteIdList writeIdList) {
            this.validTxnList = (ValidTxnList)Preconditions.checkNotNull((Object)validTxnList);
            this.writeIdList = (ValidWriteIdList)Preconditions.checkNotNull((Object)writeIdList);
            this.doStrictCheck = false;
        }

        public boolean check(String dirPath) throws CatalogException {
            ParsedBase parsedBase = AcidUtils.parseBase(dirPath);
            if (parsedBase.writeId != Long.MIN_VALUE) {
                boolean isValid = false;
                isValid = parsedBase.visibilityTxnId != -1L ? this.writeIdList.isValidBase(parsedBase.writeId) && this.isTxnValid(parsedBase.visibilityTxnId) : this.writeIdList.isWriteIdValid(parsedBase.writeId);
                if (this.doStrictCheck && !isValid) {
                    throw new CatalogException("Invalid base file found " + dirPath);
                }
                return isValid;
            }
            ParsedDelta pd = AcidUtils.parseDelta(dirPath);
            if (pd == null) {
                pd = AcidUtils.parseDeleteDelta(dirPath);
            }
            if (pd != null) {
                if (!this.isTxnValid(pd.visibilityTxnId)) {
                    return false;
                }
                ValidWriteIdList.RangeResponse rr = this.writeIdList.isWriteIdRangeValid(pd.minWriteId, pd.maxWriteId);
                if (rr == ValidWriteIdList.RangeResponse.ALL) {
                    return true;
                }
                if (rr == ValidWriteIdList.RangeResponse.NONE) {
                    return false;
                }
                if (!pd.isCompactedDeltaFile()) {
                    return true;
                }
                for (long writeId = pd.minWriteId; writeId <= pd.maxWriteId; ++writeId) {
                    if (this.writeIdList.isWriteIdValid(writeId) || this.writeIdList.isWriteIdAborted(writeId)) continue;
                    if (this.doStrictCheck) {
                        throw new CatalogException("Open writeId " + writeId + " found in compacted delta file " + dirPath);
                    }
                    return false;
                }
                return true;
            }
            return true;
        }

        private boolean isTxnValid(long visibilityTxnId) {
            return this.validTxnList == null || visibilityTxnId == -1L || this.validTxnList.isTxnValid(visibilityTxnId);
        }
    }

    public static class TblTransaction {
        public long txnId;
        public boolean ownsTxn;
        public long writeId;
        public String validWriteIds;
    }
}

