/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.service;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.hive.metastore.api.FieldSchema;
import org.apache.impala.catalog.FeKuduTable;
import org.apache.impala.catalog.KuduColumn;
import org.apache.impala.catalog.KuduTable;
import org.apache.impala.catalog.Table;
import org.apache.impala.catalog.TableNotFoundException;
import org.apache.impala.common.ImpalaRuntimeException;
import org.apache.impala.common.Pair;
import org.apache.impala.common.PrintUtils;
import org.apache.impala.thrift.TAlterTableAddDropRangePartitionParams;
import org.apache.impala.thrift.TColumn;
import org.apache.impala.thrift.TCreateTableParams;
import org.apache.impala.thrift.TKuduPartitionByHashParam;
import org.apache.impala.thrift.TKuduPartitionParam;
import org.apache.impala.thrift.TRangePartition;
import org.apache.impala.thrift.TRangePartitionOperationType;
import org.apache.impala.util.EventSequence;
import org.apache.impala.util.KuduUtil;
import org.apache.kudu.ColumnSchema;
import org.apache.kudu.Schema;
import org.apache.kudu.Type;
import org.apache.kudu.client.AlterTableOptions;
import org.apache.kudu.client.CreateTableOptions;
import org.apache.kudu.client.KuduClient;
import org.apache.kudu.client.KuduException;
import org.apache.kudu.client.PartialRow;
import org.apache.kudu.client.RangePartitionBound;
import org.apache.kudu.client.RangePartitionWithCustomHashSchema;
import org.apache.kudu.util.CharUtil;
import org.apache.kudu.util.DecimalUtil;
import org.apache.log4j.Logger;

public class KuduCatalogOpExecutor {
    public static final Logger LOG = Logger.getLogger(KuduCatalogOpExecutor.class);
    private static final String CHECKED_KUDU_TABLE_EXISTENCE = "Checked table existence in Kudu";
    private static final String CREATED_KUDU_TABLE = "Created table in Kudu";
    private static final String ALTERED_KUDU_TABLE = "Altered table in Kudu";
    public static final String GOT_KUDU_CLIENT = "Got Kudu client";
    private static final String GOT_KUDU_DDL_LOCK = "Got kuduDdlLock";
    public static final String OPENED_KUDU_TABLE = "Opened Kudu table";
    private static final String POPULATED_COLS_FROM_KUDU = "Populated external table cols from Kudu";
    private static final Object kuduDdlLock_ = new Object();

    private static boolean checkTableExistence(KuduClient client, String kuduTableName, EventSequence catalogTimeline) throws KuduException {
        boolean tableExists = client.tableExists(kuduTableName);
        catalogTimeline.markEvent(CHECKED_KUDU_TABLE_EXISTENCE);
        return tableExists;
    }

    private static org.apache.kudu.client.KuduTable createKuduTable(KuduClient client, String name, Schema schema, CreateTableOptions tableOpts, EventSequence catalogTimeline) throws KuduException {
        org.apache.kudu.client.KuduTable table = client.createTable(name, schema, tableOpts);
        catalogTimeline.markEvent(CREATED_KUDU_TABLE);
        return table;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void createSynchronizedTable(EventSequence catalogTimeline, org.apache.hadoop.hive.metastore.api.Table msTbl, TCreateTableParams params) throws ImpalaRuntimeException {
        Preconditions.checkState((boolean)KuduTable.isSynchronizedTable(msTbl));
        Preconditions.checkState((msTbl.getParameters().get("kudu.table_id") == null ? 1 : 0) != 0);
        String kuduTableName = (String)msTbl.getParameters().get("kudu.table_name");
        String masterHosts = (String)msTbl.getParameters().get("kudu.master_addresses");
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)String.format("Creating table '%s' in master '%s'", kuduTableName, masterHosts));
        }
        KuduClient kudu = KuduUtil.getKuduClient(masterHosts, catalogTimeline);
        try {
            Object object = kuduDdlLock_;
            synchronized (object) {
                catalogTimeline.markEvent(GOT_KUDU_DDL_LOCK);
                boolean tableExists = KuduCatalogOpExecutor.checkTableExistence(kudu, kuduTableName, catalogTimeline);
                if (tableExists && params.if_not_exists) {
                    return;
                }
                if (tableExists) {
                    throw new ImpalaRuntimeException(String.format("Table '%s' already exists in Kudu.", kuduTableName));
                }
                Preconditions.checkState((!Strings.isNullOrEmpty((String)kuduTableName) ? 1 : 0) != 0);
                Schema schema = KuduCatalogOpExecutor.createTableSchema(params);
                CreateTableOptions tableOpts = KuduCatalogOpExecutor.buildTableOptions(msTbl, params, schema);
                org.apache.kudu.client.KuduTable table = KuduCatalogOpExecutor.createKuduTable(kudu, kuduTableName, schema, tableOpts, catalogTimeline);
                if (KuduTable.isHMSIntegrationEnabled(masterHosts)) {
                    String tableId = table.getTableId();
                    Preconditions.checkNotNull((Object)tableId);
                    msTbl.getParameters().put("kudu.table_id", tableId);
                }
            }
        }
        catch (Exception e) {
            throw new ImpalaRuntimeException(String.format("Error creating Kudu table '%s'", kuduTableName), e);
        }
    }

    private static ColumnSchema createColumnSchema(TColumn column, boolean isKey, boolean isKeyUnique) throws ImpalaRuntimeException {
        org.apache.impala.catalog.Type type = org.apache.impala.catalog.Type.fromThrift(column.getColumnType());
        Preconditions.checkState((type != null ? 1 : 0) != 0);
        Type kuduType = KuduUtil.fromImpalaType(type);
        ColumnSchema.ColumnSchemaBuilder csb = new ColumnSchema.ColumnSchemaBuilder(column.getColumnName(), kuduType);
        if (isKey && !isKeyUnique) {
            csb.nonUniqueKey(true);
        } else {
            csb.key(isKey);
        }
        if (column.isSetIs_nullable()) {
            Preconditions.checkArgument((!isKey || !column.isIs_nullable() ? 1 : 0) != 0);
            csb.nullable(column.isIs_nullable());
        } else {
            csb.nullable(!isKey);
        }
        if (column.isSetDefault_value()) {
            csb.defaultValue(KuduUtil.getKuduDefaultValue(column.getDefault_value(), type, column.getColumnName()));
        }
        if (column.isSetBlock_size()) {
            csb.desiredBlockSize(column.getBlock_size());
        }
        if (column.isSetEncoding()) {
            csb.encoding(KuduUtil.fromThrift(column.getEncoding()));
        }
        if (column.isSetCompression()) {
            csb.compressionAlgorithm(KuduUtil.fromThrift(column.getCompression()));
        }
        if (type.isDecimal()) {
            csb.typeAttributes(DecimalUtil.typeAttributes((int)type.getPrecision(), (int)type.getDecimalDigits()));
        }
        if (kuduType == Type.VARCHAR) {
            csb.typeAttributes(CharUtil.typeAttributes((int)type.getColumnSize()));
        }
        if (column.isSetComment() && !column.getComment().isEmpty()) {
            csb.comment(column.getComment());
        }
        return csb.build();
    }

    private static Schema createTableSchema(TCreateTableParams params) throws ImpalaRuntimeException {
        List<String> keyColNames = params.getPrimary_key_column_names();
        Preconditions.checkState((!keyColNames.isEmpty() ? 1 : 0) != 0);
        ImmutableList colNames = ImmutableList.copyOf((Iterable)Iterables.transform(params.getColumns(), TColumn::getColumnName));
        List leadingColNames = colNames.subList(0, keyColNames.size());
        if (!leadingColNames.equals(keyColNames)) {
            throw new ImpalaRuntimeException(String.format("Kudu %s columns must be specified as the first columns in the table (expected leading columns (%s) but found (%s))", KuduUtil.getPrimaryKeyString(params.is_primary_key_unique), PrintUtils.joinQuoted(keyColNames), PrintUtils.joinQuoted(leadingColNames)));
        }
        ArrayList<ColumnSchema> colSchemas = new ArrayList<ColumnSchema>(params.getColumnsSize());
        for (TColumn column : params.getColumns()) {
            boolean isKey = colSchemas.size() < keyColNames.size();
            boolean isKeyUnique = isKey ? params.is_primary_key_unique : false;
            colSchemas.add(KuduCatalogOpExecutor.createColumnSchema(column, isKey, isKeyUnique));
        }
        return new Schema(colSchemas);
    }

    private static CreateTableOptions buildTableOptions(org.apache.hadoop.hive.metastore.api.Table msTbl, TCreateTableParams params, Schema schema) throws ImpalaRuntimeException {
        CreateTableOptions tableOpts = new CreateTableOptions();
        List<TKuduPartitionParam> partitionParams = params.getPartition_by();
        if (partitionParams != null) {
            boolean hasRangePartitioning = false;
            for (TKuduPartitionParam partParam : partitionParams) {
                if (partParam.isSetBy_hash_param()) {
                    Preconditions.checkState((!partParam.isSetBy_range_param() ? 1 : 0) != 0);
                    tableOpts.addHashPartitions(partParam.getBy_hash_param().getColumns(), partParam.getBy_hash_param().getNum_partitions());
                    continue;
                }
                Preconditions.checkState((boolean)partParam.isSetBy_range_param());
                hasRangePartitioning = true;
                List<String> rangePartitionColumns = partParam.getBy_range_param().getColumns();
                tableOpts.setRangePartitionColumns(rangePartitionColumns);
                for (TRangePartition rangePartition : partParam.getBy_range_param().getRange_partitions()) {
                    List<Pair<PartialRow, RangePartitionBound>> rangeBounds = KuduCatalogOpExecutor.getRangePartitionBounds(rangePartition, schema, rangePartitionColumns);
                    Preconditions.checkState((rangeBounds.size() == 2 ? 1 : 0) != 0);
                    Pair<PartialRow, RangePartitionBound> lowerBound = rangeBounds.get(0);
                    Pair<PartialRow, RangePartitionBound> upperBound = rangeBounds.get(1);
                    if (rangePartition.isSetHash_specs()) {
                        RangePartitionWithCustomHashSchema rangePart = KuduCatalogOpExecutor.getRangePartitionWithCustomHashSchema(rangePartition, rangeBounds);
                        tableOpts.addRangePartition(rangePart);
                        continue;
                    }
                    tableOpts.addRangePartition((PartialRow)lowerBound.first, (PartialRow)upperBound.first, (RangePartitionBound)lowerBound.second, (RangePartitionBound)upperBound.second);
                }
            }
            if (!hasRangePartitioning) {
                tableOpts.setRangePartitionColumns(Collections.emptyList());
            }
        } else {
            tableOpts.setRangePartitionColumns(Collections.emptyList());
        }
        String replication = (String)msTbl.getParameters().get("kudu.num_tablet_replicas");
        if (!Strings.isNullOrEmpty((String)replication)) {
            int parsedReplicas = -1;
            try {
                parsedReplicas = Integer.parseInt(replication);
                Preconditions.checkState((parsedReplicas > 0 ? 1 : 0) != 0, (Object)("Invalid number of replicas table property:" + replication));
            }
            catch (Exception e) {
                throw new ImpalaRuntimeException(String.format("Invalid number of table replicas specified: '%s'", replication));
            }
            tableOpts.setNumReplicas(parsedReplicas);
        }
        tableOpts.setOwner(msTbl.getOwner());
        if (params.getComment() != null) {
            tableOpts.setComment(params.getComment());
        }
        return tableOpts;
    }

    public static void dropTable(org.apache.hadoop.hive.metastore.api.Table msTbl, boolean ifExists, int kudu_table_reserve_seconds, EventSequence catalogTimeline) throws ImpalaRuntimeException, TableNotFoundException {
        Preconditions.checkState((boolean)KuduTable.isSynchronizedTable(msTbl));
        String tableName = (String)msTbl.getParameters().get("kudu.table_name");
        String masterHosts = (String)msTbl.getParameters().get("kudu.master_addresses");
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)String.format("Dropping table '%s' from master '%s'", tableName, masterHosts));
        }
        KuduClient kudu = KuduUtil.getKuduClient(masterHosts, catalogTimeline);
        try {
            Preconditions.checkState((!Strings.isNullOrEmpty((String)tableName) ? 1 : 0) != 0);
            if (kudu.tableExists(tableName)) {
                kudu.deleteTable(tableName, kudu_table_reserve_seconds);
                catalogTimeline.markEvent("Deleted table in Kudu");
            } else if (!ifExists) {
                throw new TableNotFoundException(String.format("Table '%s' does not exist in Kudu master(s) '%s'.", tableName, masterHosts));
            }
        }
        catch (Exception e) {
            throw new ImpalaRuntimeException(String.format("Error dropping table '%s'", tableName), e);
        }
    }

    public static void populateExternalTableColsFromKudu(EventSequence catalogTimeline, org.apache.hadoop.hive.metastore.api.Table msTbl) throws ImpalaRuntimeException {
        org.apache.hadoop.hive.metastore.api.Table msTblCopy = msTbl.deepCopy();
        List cols = msTblCopy.getSd().getCols();
        Preconditions.checkState((boolean)Table.isExternalTable(msTbl));
        Preconditions.checkState((msTblCopy.getParameters().get("kudu.table_id") == null ? 1 : 0) != 0);
        String kuduTableName = (String)msTblCopy.getParameters().get("kudu.table_name");
        Preconditions.checkState((!Strings.isNullOrEmpty((String)kuduTableName) ? 1 : 0) != 0);
        String masterHosts = (String)msTblCopy.getParameters().get("kudu.master_addresses");
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)String.format("Loading schema of table '%s' from master '%s'", kuduTableName, masterHosts));
        }
        KuduClient kudu = KuduUtil.getKuduClient(masterHosts, catalogTimeline);
        try {
            if (!KuduCatalogOpExecutor.checkTableExistence(kudu, kuduTableName, catalogTimeline)) {
                throw new ImpalaRuntimeException(String.format("Table does not exist in Kudu: '%s'", kuduTableName));
            }
            org.apache.kudu.client.KuduTable kuduTable = kudu.openTable(kuduTableName);
            catalogTimeline.markEvent(OPENED_KUDU_TABLE);
            cols.clear();
            HashSet lowerCaseColNames = Sets.newHashSet();
            for (ColumnSchema colSchema : kuduTable.getSchema().getColumns()) {
                if (!lowerCaseColNames.add(colSchema.getName().toLowerCase())) {
                    throw new ImpalaRuntimeException(String.format("Error loading Kudu table: Impala does not support column names that differ only in casing '%s'", colSchema.getName()));
                }
                org.apache.impala.catalog.Type type = KuduUtil.toImpalaType(colSchema.getType(), colSchema.getTypeAttributes());
                String comment = !colSchema.getComment().isEmpty() ? colSchema.getComment() : null;
                cols.add(new FieldSchema(colSchema.getName(), type.toSql().toLowerCase(), comment));
            }
        }
        catch (Exception e) {
            throw new ImpalaRuntimeException(String.format("Error loading schema of table '%s'", kuduTableName), e);
        }
        List newCols = msTbl.getSd().getCols();
        newCols.clear();
        newCols.addAll(cols);
        catalogTimeline.markEvent(POPULATED_COLS_FROM_KUDU);
    }

    public static void validateKuduTblExists(org.apache.hadoop.hive.metastore.api.Table msTbl) throws ImpalaRuntimeException {
        Preconditions.checkArgument((boolean)KuduTable.isKuduTable(msTbl));
        Map properties = msTbl.getParameters();
        String masterHosts = (String)properties.get("kudu.master_addresses");
        Preconditions.checkState((!Strings.isNullOrEmpty((String)masterHosts) ? 1 : 0) != 0);
        String kuduTableName = (String)properties.get("kudu.table_name");
        Preconditions.checkState((!Strings.isNullOrEmpty((String)kuduTableName) ? 1 : 0) != 0);
        KuduClient kudu = KuduUtil.getKuduClient(masterHosts);
        try {
            kudu.tableExists(kuduTableName);
        }
        catch (Exception e) {
            throw new ImpalaRuntimeException(String.format("Kudu table '%s' does not exist on master '%s'", kuduTableName, masterHosts), e);
        }
    }

    public static void renameTable(KuduTable tbl, String newName, EventSequence catalogTimeline) throws ImpalaRuntimeException {
        Preconditions.checkState((!Strings.isNullOrEmpty((String)newName) ? 1 : 0) != 0);
        AlterTableOptions alterTableOptions = new AlterTableOptions();
        alterTableOptions.renameTable(newName);
        String errMsg = String.format("Error renaming Kudu table %s to %s", tbl.getKuduTableName(), newName);
        KuduClient client = KuduUtil.getKuduClient(tbl.getKuduMasterHosts(), catalogTimeline);
        try {
            client.alterTable(tbl.getKuduTableName(), alterTableOptions);
            if (!client.isAlterTableDone(newName)) {
                throw new ImpalaRuntimeException(errMsg + ": Kudu operation timed out");
            }
            catalogTimeline.markEvent(ALTERED_KUDU_TABLE);
        }
        catch (KuduException e) {
            throw new ImpalaRuntimeException(errMsg, e);
        }
    }

    private static RangePartitionWithCustomHashSchema getRangePartitionWithCustomHashSchema(TRangePartition rangePartition, List<Pair<PartialRow, RangePartitionBound>> rangeBounds) {
        Pair<PartialRow, RangePartitionBound> lowerBound = rangeBounds.get(0);
        Pair<PartialRow, RangePartitionBound> upperBound = rangeBounds.get(1);
        RangePartitionWithCustomHashSchema rangePart = new RangePartitionWithCustomHashSchema((PartialRow)lowerBound.first, (PartialRow)upperBound.first, (RangePartitionBound)lowerBound.second, (RangePartitionBound)upperBound.second);
        Iterator<TKuduPartitionParam> specIter = rangePartition.getHash_specsIterator();
        while (specIter.hasNext()) {
            TKuduPartitionParam param = specIter.next();
            TKuduPartitionByHashParam hash = param.getBy_hash_param();
            int seed = 0;
            rangePart.addHashPartitions(hash.getColumns(), hash.getNum_partitions(), seed);
        }
        return rangePart;
    }

    public static void addDropRangePartition(KuduTable tbl, TAlterTableAddDropRangePartitionParams params, EventSequence catalogTimeline) throws ImpalaRuntimeException {
        block6: {
            TRangePartition rangePartition = params.getRange_partition_spec();
            List<Pair<PartialRow, RangePartitionBound>> rangeBounds = KuduCatalogOpExecutor.getRangePartitionBounds(rangePartition, tbl);
            Preconditions.checkState((rangeBounds.size() == 2 ? 1 : 0) != 0);
            Pair<PartialRow, RangePartitionBound> lowerBound = rangeBounds.get(0);
            Pair<PartialRow, RangePartitionBound> upperBound = rangeBounds.get(1);
            AlterTableOptions alterTableOptions = new AlterTableOptions();
            TRangePartitionOperationType type = params.getType();
            if (type == TRangePartitionOperationType.ADD) {
                if (rangePartition.isSetHash_specs()) {
                    RangePartitionWithCustomHashSchema rangePart = KuduCatalogOpExecutor.getRangePartitionWithCustomHashSchema(rangePartition, rangeBounds);
                    alterTableOptions.addRangePartition(rangePart);
                } else {
                    alterTableOptions.addRangePartition((PartialRow)lowerBound.first, (PartialRow)upperBound.first, (RangePartitionBound)lowerBound.second, (RangePartitionBound)upperBound.second);
                }
            } else {
                alterTableOptions.dropRangePartition((PartialRow)lowerBound.first, (PartialRow)upperBound.first, (RangePartitionBound)lowerBound.second, (RangePartitionBound)upperBound.second);
            }
            String errMsg = String.format("Error %s range partition in table %s", type == TRangePartitionOperationType.ADD ? "adding" : "dropping", tbl.getName());
            try {
                KuduCatalogOpExecutor.alterKuduTable(tbl, alterTableOptions, errMsg, catalogTimeline);
            }
            catch (ImpalaRuntimeException e) {
                if (params.isIgnore_errors()) break block6;
                throw e;
            }
        }
    }

    private static List<Pair<PartialRow, RangePartitionBound>> getRangePartitionBounds(TRangePartition rangePartition, KuduTable tbl) throws ImpalaRuntimeException {
        List<String> rangePartitioningColNames = FeKuduTable.Utils.getRangePartitioningColNames(tbl);
        ArrayList rangePartitioningKuduColNames = Lists.newArrayListWithCapacity((int)rangePartitioningColNames.size());
        for (String colName : rangePartitioningColNames) {
            rangePartitioningKuduColNames.add(((KuduColumn)tbl.getColumn(colName)).getKuduName());
        }
        return KuduCatalogOpExecutor.getRangePartitionBounds(rangePartition, tbl.getKuduSchema(), rangePartitioningKuduColNames);
    }

    private static List<Pair<PartialRow, RangePartitionBound>> getRangePartitionBounds(TRangePartition rangePartition, Schema schema, List<String> rangePartitioningColNames) throws ImpalaRuntimeException {
        Preconditions.checkNotNull((Object)schema);
        Preconditions.checkState((!rangePartitioningColNames.isEmpty() ? 1 : 0) != 0);
        Preconditions.checkState((rangePartition.isSetLower_bound_values() || rangePartition.isSetUpper_bound_values() ? 1 : 0) != 0);
        ArrayList rangeBounds = Lists.newArrayListWithCapacity((int)2);
        Pair<PartialRow, RangePartitionBound> lowerBound = KuduUtil.buildRangePartitionBound(schema, rangePartitioningColNames, rangePartition.getLower_bound_values(), rangePartition.isIs_lower_bound_inclusive());
        rangeBounds.add(lowerBound);
        Pair<PartialRow, RangePartitionBound> upperBound = KuduUtil.buildRangePartitionBound(schema, rangePartitioningColNames, rangePartition.getUpper_bound_values(), rangePartition.isIs_upper_bound_inclusive());
        rangeBounds.add(upperBound);
        return rangeBounds;
    }

    public static void addColumn(KuduTable tbl, List<TColumn> columns, EventSequence catalogTimeline) throws ImpalaRuntimeException {
        AlterTableOptions alterTableOptions = new AlterTableOptions();
        for (TColumn column : columns) {
            alterTableOptions.addColumn(KuduCatalogOpExecutor.createColumnSchema(column, false, false));
        }
        String errMsg = "Error adding columns to Kudu table " + tbl.getName();
        KuduCatalogOpExecutor.alterKuduTable(tbl, alterTableOptions, errMsg, catalogTimeline);
    }

    public static void dropColumn(KuduTable tbl, String colName, EventSequence catalogTimeline) throws ImpalaRuntimeException {
        Preconditions.checkState((!Strings.isNullOrEmpty((String)colName) ? 1 : 0) != 0);
        KuduColumn col = (KuduColumn)tbl.getColumn(colName);
        AlterTableOptions alterTableOptions = new AlterTableOptions();
        alterTableOptions.dropColumn(col.getKuduName());
        String errMsg = String.format("Error dropping column %s from Kudu table %s", colName, tbl.getName());
        KuduCatalogOpExecutor.alterKuduTable(tbl, alterTableOptions, errMsg, catalogTimeline);
    }

    public static void alterColumn(KuduTable tbl, String colName, TColumn newCol, EventSequence catalogTimeline) throws ImpalaRuntimeException {
        String newColName;
        Preconditions.checkState((!Strings.isNullOrEmpty((String)colName) ? 1 : 0) != 0);
        Preconditions.checkNotNull((Object)newCol);
        Preconditions.checkState((!newCol.isIs_key() ? 1 : 0) != 0);
        Preconditions.checkState((!newCol.isSetIs_nullable() ? 1 : 0) != 0);
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)String.format("Altering column '%s' to '%s'", colName, newCol.toString()));
        }
        KuduColumn col = (KuduColumn)tbl.getColumn(colName);
        String kuduColName = col.getKuduName();
        AlterTableOptions alterTableOptions = new AlterTableOptions();
        if (newCol.isSetDefault_value()) {
            org.apache.impala.catalog.Type type = org.apache.impala.catalog.Type.fromThrift(newCol.getColumnType());
            Object defaultValue = KuduUtil.getKuduDefaultValue(newCol.getDefault_value(), type, newCol.getColumnName());
            if (defaultValue == null) {
                alterTableOptions.removeDefault(kuduColName);
            } else {
                alterTableOptions.changeDefault(kuduColName, defaultValue);
            }
        }
        if (newCol.isSetBlock_size()) {
            alterTableOptions.changeDesiredBlockSize(kuduColName, newCol.getBlock_size());
        }
        if (newCol.isSetEncoding()) {
            alterTableOptions.changeEncoding(kuduColName, KuduUtil.fromThrift(newCol.getEncoding()));
        }
        if (newCol.isSetCompression()) {
            alterTableOptions.changeCompressionAlgorithm(kuduColName, KuduUtil.fromThrift(newCol.getCompression()));
        }
        if (!(newColName = newCol.getColumnName()).toLowerCase().equals(colName.toLowerCase())) {
            alterTableOptions.renameColumn(kuduColName, newColName);
        }
        if (newCol.isSetComment()) {
            alterTableOptions.changeComment(kuduColName, newCol.getComment());
        }
        String errMsg = String.format("Error altering column %s in Kudu table %s", colName, tbl.getName());
        KuduCatalogOpExecutor.alterKuduTable(tbl, alterTableOptions, errMsg, catalogTimeline);
    }

    public static void alterSetOwner(KuduTable tbl, String newOwner, EventSequence catalogTimeline) throws ImpalaRuntimeException {
        AlterTableOptions alterTableOptions = new AlterTableOptions();
        alterTableOptions.setOwner(newOwner);
        String errMsg = String.format("Error setting the owner of Kudu table %s to %s", tbl.getName(), newOwner);
        KuduCatalogOpExecutor.alterKuduTable(tbl, alterTableOptions, errMsg, catalogTimeline);
    }

    public static void alterKuduTable(KuduTable tbl, AlterTableOptions ato, String errMsg, EventSequence catalogTimeline) throws ImpalaRuntimeException {
        KuduClient client = KuduUtil.getKuduClient(tbl.getKuduMasterHosts(), catalogTimeline);
        try {
            client.alterTable(tbl.getKuduTableName(), ato);
            if (!client.isAlterTableDone(tbl.getKuduTableName())) {
                throw new ImpalaRuntimeException(errMsg + ": Kudu operation timed out");
            }
            catalogTimeline.markEvent(ALTERED_KUDU_TABLE);
        }
        catch (KuduException e) {
            throw new ImpalaRuntimeException(errMsg, e);
        }
    }
}

