/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.common;

import com.google.common.base.Preconditions;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.apache.iceberg.Schema;
import org.apache.iceberg.expressions.Expression;
import org.apache.iceberg.expressions.Expressions;
import org.apache.iceberg.expressions.UnboundPredicate;
import org.apache.iceberg.expressions.UnboundTerm;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.apache.impala.analysis.Analyzer;
import org.apache.impala.analysis.BinaryPredicate;
import org.apache.impala.analysis.BoolLiteral;
import org.apache.impala.analysis.CompoundPredicate;
import org.apache.impala.analysis.DateLiteral;
import org.apache.impala.analysis.Expr;
import org.apache.impala.analysis.InPredicate;
import org.apache.impala.analysis.IsNullPredicate;
import org.apache.impala.analysis.LiteralExpr;
import org.apache.impala.analysis.NumericLiteral;
import org.apache.impala.analysis.SlotDescriptor;
import org.apache.impala.analysis.SlotRef;
import org.apache.impala.analysis.StringLiteral;
import org.apache.impala.catalog.Column;
import org.apache.impala.catalog.IcebergColumn;
import org.apache.impala.catalog.PrimitiveType;
import org.apache.impala.common.AnalysisException;
import org.apache.impala.common.ImpalaRuntimeException;
import org.apache.impala.common.InternalException;
import org.apache.impala.util.ExprUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IcebergPredicateConverter {
    private static final Logger LOG = LoggerFactory.getLogger(IcebergPredicateConverter.class);
    private final Schema schema_;
    private final Analyzer analyzer_;

    public IcebergPredicateConverter(Schema schema, Analyzer analyzer) {
        this.schema_ = schema;
        this.analyzer_ = analyzer;
    }

    public Expression convert(Expr expr) throws ImpalaRuntimeException {
        if (expr instanceof BinaryPredicate) {
            return this.convert((BinaryPredicate)expr);
        }
        if (expr instanceof InPredicate) {
            return this.convert((InPredicate)expr);
        }
        if (expr instanceof IsNullPredicate) {
            return this.convert((IsNullPredicate)expr);
        }
        if (expr instanceof CompoundPredicate) {
            return this.convert((CompoundPredicate)expr);
        }
        throw new ImpalaRuntimeException(String.format("Unsupported expression: %s", expr.toSql()));
    }

    protected Expression convert(BinaryPredicate predicate) throws ImpalaRuntimeException {
        Term term = this.getTerm((Expr)predicate.getChild(0));
        IcebergColumn column = term.referencedColumn_;
        LiteralExpr literal = this.getSecondChildAsLiteralExpr(predicate);
        this.checkNullLiteral(literal);
        Expression.Operation op = this.getOperation(predicate);
        Object value = this.getIcebergValue(column, literal);
        List<Object> literals = Collections.singletonList(value);
        return Expressions.predicate((Expression.Operation)op, term.term_, literals);
    }

    protected UnboundPredicate<Object> convert(InPredicate predicate) throws ImpalaRuntimeException {
        Term term = this.getTerm((Expr)predicate.getChild(0));
        IcebergColumn column = term.referencedColumn_;
        ArrayList<Object> values = new ArrayList<Object>();
        for (int i = 1; i < predicate.getChildren().size(); ++i) {
            if (!Expr.IS_LITERAL.apply(predicate.getChild(i))) {
                throw new ImpalaRuntimeException(String.format("Expression is not a literal: %s", predicate.getChild(i)));
            }
            LiteralExpr literal = (LiteralExpr)predicate.getChild(i);
            this.checkNullLiteral(literal);
            Object value = this.getIcebergValue(column, literal);
            values.add(value);
        }
        if (predicate.isNotIn()) {
            return Expressions.notIn(term.term_, values);
        }
        return Expressions.in(term.term_, values);
    }

    protected UnboundPredicate<Object> convert(IsNullPredicate predicate) throws ImpalaRuntimeException {
        Term term = this.getTerm((Expr)predicate.getChild(0));
        if (predicate.isNotNull()) {
            return Expressions.notNull(term.term_);
        }
        return Expressions.isNull(term.term_);
    }

    protected Expression convert(CompoundPredicate predicate) throws ImpalaRuntimeException {
        Expression.Operation op = this.getOperation(predicate);
        Expr leftExpr = (Expr)predicate.getChild(0);
        Expression left = this.convert(leftExpr);
        if (op.equals((Object)Expression.Operation.NOT)) {
            return Expressions.not((Expression)left);
        }
        Expr rightExpr = (Expr)predicate.getChild(1);
        Expression right = this.convert(rightExpr);
        return op.equals((Object)Expression.Operation.AND) ? Expressions.and((Expression)left, (Expression)right) : Expressions.or((Expression)left, (Expression)right);
    }

    protected void checkNullLiteral(LiteralExpr literal) throws ImpalaRuntimeException {
        if (Expr.IS_NULL_LITERAL.apply((Object)literal)) {
            throw new ImpalaRuntimeException("Expression can't be NULL literal: " + literal);
        }
    }

    protected Object getIcebergValue(IcebergColumn column, LiteralExpr literal) throws ImpalaRuntimeException {
        PrimitiveType primitiveType = literal.getType().getPrimitiveType();
        switch (primitiveType) {
            case BOOLEAN: {
                return ((BoolLiteral)literal).getValue();
            }
            case TINYINT: 
            case SMALLINT: 
            case INT: {
                return ((NumericLiteral)literal).getIntValue();
            }
            case BIGINT: {
                return ((NumericLiteral)literal).getLongValue();
            }
            case FLOAT: {
                return Float.valueOf((float)((NumericLiteral)literal).getDoubleValue());
            }
            case DOUBLE: {
                return ((NumericLiteral)literal).getDoubleValue();
            }
            case STRING: 
            case DATETIME: 
            case CHAR: {
                return ((StringLiteral)literal).getUnescapedValue();
            }
            case TIMESTAMP: {
                return this.getIcebergTsValue(literal, column, this.schema_);
            }
            case DATE: {
                return ((DateLiteral)literal).getValue();
            }
            case DECIMAL: {
                return this.getIcebergDecimalValue(column, (NumericLiteral)literal);
            }
        }
        throw new ImpalaRuntimeException(String.format("Unable to parse Iceberg value '%s' for type %s", new Object[]{literal.getStringValue(), primitiveType}));
    }

    protected Expression.Operation getIcebergOperator(BinaryPredicate.Operator op) throws ImpalaRuntimeException {
        switch (op) {
            case EQ: {
                return Expression.Operation.EQ;
            }
            case NE: {
                return Expression.Operation.NOT_EQ;
            }
            case LE: {
                return Expression.Operation.LT_EQ;
            }
            case GE: {
                return Expression.Operation.GT_EQ;
            }
            case LT: {
                return Expression.Operation.LT;
            }
            case GT: {
                return Expression.Operation.GT;
            }
        }
        throw new ImpalaRuntimeException(String.format("Unsupported Impala operator: %s", op.getName()));
    }

    protected Expression.Operation getIcebergOperator(CompoundPredicate.Operator op) throws ImpalaRuntimeException {
        switch (op) {
            case AND: {
                return Expression.Operation.AND;
            }
            case OR: {
                return Expression.Operation.OR;
            }
            case NOT: {
                return Expression.Operation.NOT;
            }
        }
        throw new ImpalaRuntimeException(String.format("Unsupported Impala operator: %s", new Object[]{op}));
    }

    protected BigDecimal getIcebergDecimalValue(IcebergColumn column, NumericLiteral literal) throws ImpalaRuntimeException {
        org.apache.impala.catalog.Type colType = column.getType();
        int scale = colType.getDecimalDigits();
        BigDecimal literalValue = literal.getValue();
        if (literalValue.scale() > scale) {
            throw new ImpalaRuntimeException(String.format("Invalid scale %d for type: %s", literalValue.scale(), colType.toSql()));
        }
        if (literalValue.scale() < scale) {
            return literalValue.setScale(scale);
        }
        return literalValue;
    }

    protected Long getIcebergTsValue(LiteralExpr literal, IcebergColumn column, Schema iceSchema) throws ImpalaRuntimeException {
        try {
            Type iceType = iceSchema.findType(column.getFieldId());
            Preconditions.checkState((boolean)(iceType instanceof Types.TimestampType));
            Types.TimestampType tsType = (Types.TimestampType)iceType;
            if (tsType.shouldAdjustToUTC()) {
                return ExprUtil.localTimestampToUnixTimeMicros(this.analyzer_, literal);
            }
            return ExprUtil.utcTimestampToUnixTimeMicros(this.analyzer_, literal);
        }
        catch (InternalException ex) {
            LOG.warn("Exception occurred during timestamp conversion: %s\nThis means timestamp predicate is not pushed to Iceberg, let Impala backend handle it.", (Throwable)ex);
        }
        catch (AnalysisException analysisException) {
            // empty catch block
        }
        throw new ImpalaRuntimeException(String.format("Unable to parse timestamp value from: %s", literal.getStringValue()));
    }

    protected Column getColumnFromSlotRef(SlotRef slotRef) throws ImpalaRuntimeException {
        SlotDescriptor desc = slotRef.getDesc();
        Column column = desc.getColumn();
        if (column == null) {
            throw new ImpalaRuntimeException("Expressions with complex types can't be converted to Iceberg expressions: " + slotRef);
        }
        return column;
    }

    protected LiteralExpr getSecondChildAsLiteralExpr(Expr expr) throws ImpalaRuntimeException {
        if (!(expr.getChild(1) instanceof LiteralExpr)) {
            throw new ImpalaRuntimeException(String.format("Invalid child expression: %s", expr));
        }
        return (LiteralExpr)expr.getChild(1);
    }

    protected Expression.Operation getOperation(Expr expr) throws ImpalaRuntimeException {
        Expression.Operation op;
        if (expr instanceof BinaryPredicate) {
            op = this.getIcebergOperator(((BinaryPredicate)expr).getOp());
        } else if (expr instanceof CompoundPredicate) {
            op = this.getIcebergOperator(((CompoundPredicate)expr).getOp());
        } else {
            throw new ImpalaRuntimeException(String.format("Invalid expression type: %s", expr.getType()));
        }
        return op;
    }

    protected Term getTerm(Expr expr) throws ImpalaRuntimeException {
        if (!(expr instanceof SlotRef)) {
            throw new ImpalaRuntimeException(String.format("Unable to create term from expression: %s", expr.toSql()));
        }
        Column column = this.getColumnFromSlotRef((SlotRef)expr);
        if (!(column instanceof IcebergColumn)) {
            throw new ImpalaRuntimeException(String.format("Invalid column type %s for column: %s", column.getType(), column));
        }
        return new Term((UnboundTerm<Object>)Expressions.ref((String)column.getName()), (IcebergColumn)column);
    }

    public static class Term {
        public final UnboundTerm<Object> term_;
        public final IcebergColumn referencedColumn_;

        public Term(UnboundTerm<Object> term, IcebergColumn referencedColumn) {
            this.term_ = term;
            this.referencedColumn_ = referencedColumn;
        }
    }
}

