/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.common;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.Stack;
import java.util.StringTokenizer;
import java.util.UUID;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.ozone.BasicRootedOzoneClientAdapterImpl;
import org.apache.hadoop.fs.ozone.BasicRootedOzoneFileSystem;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.client.HdfsAdmin;
import org.apache.hadoop.hdfs.protocol.EncryptionZone;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicy;
import org.apache.hadoop.ozone.client.ObjectStore;
import org.apache.hadoop.ozone.client.OzoneBucket;
import org.apache.hadoop.ozone.client.OzoneVolume;
import org.apache.impala.catalog.HdfsCompression;
import org.apache.impala.common.Pair;
import org.apache.impala.service.BackendConfig;
import org.apache.impala.thrift.TTableName;
import org.apache.impala.util.DebugUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemUtil {
    private static Configuration CONF;
    private static final Logger LOG;
    public static final String SCHEME_ABFS = "abfs";
    public static final String SCHEME_ABFSS = "abfss";
    public static final String SCHEME_ADL = "adl";
    public static final String SCHEME_FILE = "file";
    public static final String SCHEME_HDFS = "hdfs";
    public static final String SCHEME_S3A = "s3a";
    public static final String SCHEME_O3FS = "o3fs";
    public static final String SCHEME_OFS = "ofs";
    public static final String SCHEME_ALLUXIO = "alluxio";
    public static final String SCHEME_GCS = "gs";
    public static final String SCHEME_COS = "cosn";
    public static final String SCHEME_OSS = "oss";
    public static final String SCHEME_SFS = "sfs";
    public static final String SCHEME_OBS = "obs";
    public static final String NO_ERASURE_CODE_LABEL = "NONE";
    private static final Set<String> SCHEME_SUPPORT_STORAGE_IDS;
    private static final Set<String> SCHEME_WRITEABLE_BY_IMPALA;
    private static final Set<String> SCHEME_SUPPORTED_AS_DEFAULT_FS;
    private static final Set<String> SCHEME_VALID_FOR_LOAD_INPATH;
    private static Set<String> BLOCK_LOCATIONS_FOR_FS_SCHEMES;
    private static Set<String> NO_BLOCK_LOCATIONS_FOR_AUTHORITIES;
    public static final String DOT = ".";
    public static final String HIVE_TEMP_FILE_PREFIX = "_tmp.";
    public static final String HIVE_NEW_TEMP_FILE_PREFIX = "-tmp.";
    public static final String SPARK_TEMP_FILE_PREFIX = "_spark_metadata";
    public static final String IMPALA_STAGING_DIR_PREFIX = "_impala_insert_staging";
    private static final List<String> TMP_DIR_PREFIX_LIST;

    @VisibleForTesting
    public static void setConfiguration(Configuration conf) {
        CONF = conf;
        FileSystemUtil.initConfigurationDependendentStaticFields();
    }

    private static void initConfigurationDependendentStaticFields() {
        FileSystemUtil.resetConfigurationDependendentStaticFields();
        String PRELOAD_BLOCK_LOCATIONS_CONFIGURATION_PREFIX = "impala.preload-block-locations-for-scheduling";
        String AUTHORITY = ".authority.";
        String SCHEME = ".scheme.";
        Map preloadBlockLocations = CONF.getPropsWithPrefix("impala.preload-block-locations-for-scheduling");
        for (Map.Entry noBlocksEntry : preloadBlockLocations.entrySet()) {
            if (Boolean.parseBoolean((String)noBlocksEntry.getValue())) continue;
            String key = (String)noBlocksEntry.getKey();
            if (key.isEmpty()) {
                BLOCK_LOCATIONS_FOR_FS_SCHEMES.clear();
                continue;
            }
            if (((String)noBlocksEntry.getKey()).startsWith(".scheme.")) {
                BLOCK_LOCATIONS_FOR_FS_SCHEMES.remove(key.substring(".scheme.".length()));
                continue;
            }
            if (!((String)noBlocksEntry.getKey()).startsWith(".authority.")) continue;
            NO_BLOCK_LOCATIONS_FOR_AUTHORITIES.add(key.substring(".authority.".length()));
        }
    }

    private static void resetConfigurationDependendentStaticFields() {
        BLOCK_LOCATIONS_FOR_FS_SCHEMES = new HashSet<String>(SCHEME_SUPPORT_STORAGE_IDS);
        NO_BLOCK_LOCATIONS_FOR_AUTHORITIES = new HashSet<String>();
    }

    public static int deleteAllVisibleFiles(Path directory) throws IOException {
        FileSystem fs = directory.getFileSystem(CONF);
        Preconditions.checkState((boolean)fs.getFileStatus(directory).isDirectory());
        int numFilesDeleted = 0;
        for (FileStatus fStatus : fs.listStatus(directory)) {
            if (!fStatus.isFile() || FileSystemUtil.isHiddenFile(fStatus.getPath().getName())) continue;
            if (LOG.isTraceEnabled()) {
                LOG.trace("Removing: " + fStatus.getPath());
            }
            fs.delete(fStatus.getPath(), false);
            ++numFilesDeleted;
        }
        return numFilesDeleted;
    }

    public static int getTotalNumVisibleFiles(Path directory) throws IOException {
        FileSystem fs = directory.getFileSystem(CONF);
        Preconditions.checkState((boolean)fs.getFileStatus(directory).isDirectory());
        int numFiles = 0;
        for (FileStatus fStatus : fs.listStatus(directory)) {
            if (!fStatus.isFile() || FileSystemUtil.isHiddenFile(fStatus.getPath().getName())) continue;
            ++numFiles;
        }
        return numFiles;
    }

    private static boolean arePathsInSameHdfsEncryptionZone(FileSystem fs, Path p1, Path p2) throws IOException {
        if (!FileSystemUtil.isDistributedFileSystem(p1) || !FileSystemUtil.isDistributedFileSystem(p2)) {
            return false;
        }
        HdfsAdmin hdfsAdmin = new HdfsAdmin(fs.getUri(), CONF);
        EncryptionZone z1 = hdfsAdmin.getEncryptionZoneForPath(p1);
        EncryptionZone z2 = hdfsAdmin.getEncryptionZoneForPath(p2);
        if (z1 == null && z2 == null) {
            return true;
        }
        if (z1 == null || z2 == null) {
            return false;
        }
        return z1.equals((Object)z2);
    }

    private static String getOzoneReplication(ObjectStore os, Path p) throws IOException {
        String path = Path.getPathWithoutSchemeAndAuthority((Path)p).toString();
        StringTokenizer tokens = new StringTokenizer(path, "/");
        if (!tokens.hasMoreTokens()) {
            return null;
        }
        OzoneVolume volume = os.getVolume(tokens.nextToken());
        if (!tokens.hasMoreTokens()) {
            return null;
        }
        OzoneBucket bucket = volume.getBucket(tokens.nextToken());
        if (!tokens.hasMoreTokens()) {
            return bucket.getReplicationConfig().getReplication();
        }
        String keyName = tokens.nextToken("").substring(1);
        return bucket.getKey(keyName).getReplicationConfig().getReplication();
    }

    public static String getErasureCodingPolicy(Path p) {
        if (FileSystemUtil.isDistributedFileSystem(p)) {
            try {
                DistributedFileSystem dfs = (DistributedFileSystem)p.getFileSystem(CONF);
                ErasureCodingPolicy policy = dfs.getErasureCodingPolicy(p);
                if (policy != null) {
                    return policy.getName();
                }
            }
            catch (IOException e) {
                LOG.warn("Unable to retrieve erasure coding policy for {}", (Object)p, (Object)e);
            }
        } else if (FileSystemUtil.isOzoneFileSystem(p)) {
            try {
                FileSystem fs = p.getFileSystem(CONF);
                if (!fs.getFileStatus(p).isErasureCoded()) {
                    return NO_ERASURE_CODE_LABEL;
                }
                if (fs instanceof BasicRootedOzoneFileSystem) {
                    BasicRootedOzoneFileSystem ofs = (BasicRootedOzoneFileSystem)fs;
                    Preconditions.checkState((boolean)(ofs.getAdapter() instanceof BasicRootedOzoneClientAdapterImpl));
                    BasicRootedOzoneClientAdapterImpl adapter = (BasicRootedOzoneClientAdapterImpl)ofs.getAdapter();
                    String replication = FileSystemUtil.getOzoneReplication(adapter.getObjectStore(), p);
                    if (replication != null) {
                        return replication;
                    }
                } else {
                    LOG.debug("Retrieving erasure code policy not supported for {}", (Object)p);
                }
            }
            catch (IOException e) {
                LOG.warn("Unable to retrieve erasure coding policy for {}", (Object)p, (Object)e);
            }
        }
        return NO_ERASURE_CODE_LABEL;
    }

    public static int relocateAllVisibleFiles(Path sourceDir, Path destDir) throws IOException {
        return FileSystemUtil.relocateAllVisibleFiles(sourceDir, destDir, null);
    }

    public static int relocateAllVisibleFiles(Path sourceDir, Path destDir, List<Path> loadedFiles) throws IOException {
        FileSystem destFs = destDir.getFileSystem(CONF);
        FileSystem sourceFs = sourceDir.getFileSystem(CONF);
        Preconditions.checkState((boolean)destFs.isDirectory(destDir));
        Preconditions.checkState((boolean)sourceFs.isDirectory(sourceDir));
        UUID uuid = UUID.randomUUID();
        int numFilesMoved = 0;
        for (FileStatus fStatus : sourceFs.listStatus(sourceDir)) {
            if (fStatus.isDirectory()) {
                if (!LOG.isTraceEnabled()) continue;
                LOG.trace("Skipping copy of directory: " + fStatus.getPath());
                continue;
            }
            if (FileSystemUtil.isHiddenFile(fStatus.getPath().getName())) continue;
            Path destFile = new Path(destDir, fStatus.getPath().getName());
            if (destFs.exists(destFile)) {
                destFile = new Path(destDir, FileSystemUtil.appendToBaseFileName(destFile.getName(), uuid.toString()));
            }
            FileSystemUtil.relocateFile(fStatus.getPath(), destFile, false);
            if (loadedFiles != null) {
                loadedFiles.add(destFile);
            }
            ++numFilesMoved;
        }
        return numFilesMoved;
    }

    public static Pair<String, String> volumeBucketPair(Path p) {
        String path = Path.getPathWithoutSchemeAndAuthority((Path)p).toString();
        StringTokenizer tokens = new StringTokenizer(path, "/");
        String volume = "";
        String bucket = "";
        if (tokens.hasMoreTokens()) {
            volume = tokens.nextToken();
            if (tokens.hasMoreTokens()) {
                bucket = tokens.nextToken();
            }
        }
        return Pair.create(volume, bucket);
    }

    public static boolean isSameBucket(Path source, Path dest) throws IOException {
        if (!FileSystemUtil.isPathOnFileSystem(source, dest.getFileSystem(CONF))) {
            return false;
        }
        if (!FileSystemUtil.hasScheme(source, SCHEME_OFS)) {
            return true;
        }
        return FileSystemUtil.volumeBucketPair(source).equals(FileSystemUtil.volumeBucketPair(dest));
    }

    public static Path relocateFile(Path sourceFile, Path dest, boolean renameIfAlreadyExists) throws IOException {
        boolean doRename;
        Path destFile;
        FileSystem destFs = dest.getFileSystem(CONF);
        Path path = destFile = destFs.isDirectory(dest) ? new Path(dest, sourceFile.getName()) : dest;
        if (renameIfAlreadyExists && destFs.exists(destFile)) {
            Path destDir = destFs.isDirectory(dest) ? dest : dest.getParent();
            destFile = new Path(destDir, FileSystemUtil.appendToBaseFileName(destFile.getName(), UUID.randomUUID().toString()));
        }
        boolean sameBucket = FileSystemUtil.isSameBucket(sourceFile, dest);
        boolean destIsDfs = FileSystemUtil.isDistributedFileSystem(destFs);
        boolean sameEncryptionZone = FileSystemUtil.arePathsInSameHdfsEncryptionZone(destFs, sourceFile, destFile);
        boolean bl = doRename = destIsDfs && sameBucket && sameEncryptionZone;
        if (!doRename) {
            boolean bl2 = doRename = !destIsDfs && sameBucket;
        }
        if (doRename) {
            LOG.trace("Moving '{}' to '{}'", (Object)sourceFile, (Object)destFile);
            if (!destFs.rename(sourceFile, destFile)) {
                throw new IOException(String.format("Failed to move '%s' to '%s'", sourceFile, destFile));
            }
            return destFile;
        }
        Preconditions.checkState((!doRename ? 1 : 0) != 0);
        if (destIsDfs && sameBucket) {
            Preconditions.checkState((!sameEncryptionZone ? 1 : 0) != 0);
            LOG.trace("Copying source '{}' to '{}' because HDFS encryption zones are different.", (Object)sourceFile, (Object)destFile);
        } else {
            LOG.trace("Copying '{}' to '{}' between filesystems.", (Object)sourceFile, (Object)destFile);
        }
        FileSystem sourceFs = sourceFile.getFileSystem(CONF);
        FileUtil.copy((FileSystem)sourceFs, (Path)sourceFile, (FileSystem)destFs, (Path)destFile, (boolean)true, (boolean)true, (Configuration)CONF);
        return destFile;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String readFile(Path file) throws IOException {
        FileSystem fs = file.getFileSystem(CONF);
        FSDataInputStream fileStream = fs.open(file);
        try {
            String string = IOUtils.toString((InputStream)fileStream);
            return string;
        }
        finally {
            IOUtils.closeQuietly((InputStream)fileStream);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String readMagicString(Path file) throws IOException {
        FileSystem fs = file.getFileSystem(CONF);
        FSDataInputStream fileStream = fs.open(file);
        byte[] buffer = new byte[4];
        try {
            IOUtils.read((InputStream)fileStream, (byte[])buffer, (int)0, (int)4);
            String string = new String(buffer);
            return string;
        }
        finally {
            IOUtils.closeQuietly((InputStream)fileStream);
        }
    }

    private static String appendToBaseFileName(String baseFileName, String appendStr) {
        StringBuilder sb = new StringBuilder(baseFileName);
        int extensionIdx = baseFileName.lastIndexOf(46);
        if (extensionIdx != -1) {
            sb.replace(extensionIdx, extensionIdx + 1, "_" + appendStr + DOT);
        } else {
            sb.append("_" + appendStr);
        }
        return sb.toString();
    }

    public static boolean containsVisibleSubdirectory(Path directory) throws FileNotFoundException, IOException {
        FileSystem fs = directory.getFileSystem(CONF);
        for (FileStatus fStatus : fs.listStatus(directory)) {
            String pathName = fStatus.getPath().getName();
            if (!fStatus.isDirectory() || FileSystemUtil.isHiddenFile(pathName)) continue;
            return true;
        }
        return false;
    }

    public static Path makeTmpSubdirectory(Path directory) throws IOException {
        FileSystem fs = directory.getFileSystem(CONF);
        Path tmpDir = new Path(directory, ".tmp_" + UUID.randomUUID().toString());
        fs.mkdirs(tmpDir);
        return tmpDir;
    }

    public static boolean isHiddenFile(String fileName) {
        String lcFileName = fileName.toLowerCase();
        return lcFileName.startsWith(DOT) || lcFileName.startsWith("_") || lcFileName.endsWith(".copying") || lcFileName.endsWith(".tmp");
    }

    public static boolean isValidDataFile(FileStatus fileStatus) {
        String fileName = fileStatus.getPath().getName();
        return !fileStatus.isDirectory() && !FileSystemUtil.isHiddenFile(fileName) && HdfsCompression.fromFileName(fileName) != HdfsCompression.LZO_INDEX;
    }

    public static boolean supportsStorageIds(FileSystem fs) {
        return BLOCK_LOCATIONS_FOR_FS_SCHEMES.contains(fs.getScheme()) && !NO_BLOCK_LOCATIONS_FOR_AUTHORITIES.contains(fs.getUri().getAuthority());
    }

    public static boolean hasRecursiveListFiles(FileSystem fs) {
        return FileSystemUtil.isS3AFileSystem(fs);
    }

    public static boolean isS3AFileSystem(FileSystem fs) {
        return FileSystemUtil.hasScheme(fs, SCHEME_S3A);
    }

    public static boolean isS3AFileSystem(Path path) {
        return FileSystemUtil.hasScheme(path, SCHEME_S3A);
    }

    public static boolean isGCSFileSystem(FileSystem fs) {
        return FileSystemUtil.hasScheme(fs, SCHEME_GCS);
    }

    public static boolean isCOSFileSystem(FileSystem fs) {
        return FileSystemUtil.hasScheme(fs, SCHEME_COS);
    }

    public static boolean isOSSFileSystem(FileSystem fs) {
        return FileSystemUtil.hasScheme(fs, SCHEME_OSS);
    }

    public static boolean isOBSFileSystem(FileSystem fs) {
        return FileSystemUtil.hasScheme(fs, SCHEME_OBS);
    }

    public static boolean isADLFileSystem(FileSystem fs) {
        return FileSystemUtil.hasScheme(fs, SCHEME_ADL);
    }

    public static boolean isADLFileSystem(Path path) {
        return FileSystemUtil.hasScheme(path, SCHEME_ADL);
    }

    public static boolean isABFSFileSystem(FileSystem fs) {
        return FileSystemUtil.hasScheme(fs, SCHEME_ABFS) || FileSystemUtil.hasScheme(fs, SCHEME_ABFSS);
    }

    public static boolean isABFSFileSystem(Path path) {
        return FileSystemUtil.hasScheme(path, SCHEME_ABFS) || FileSystemUtil.hasScheme(path, SCHEME_ABFSS);
    }

    public static boolean isLocalFileSystem(FileSystem fs) {
        return FileSystemUtil.hasScheme(fs, SCHEME_FILE);
    }

    public static boolean isLocalFileSystem(Path path) {
        return FileSystemUtil.hasScheme(path, SCHEME_FILE);
    }

    public static boolean isDistributedFileSystem(FileSystem fs) {
        return FileSystemUtil.hasScheme(fs, SCHEME_HDFS);
    }

    public static boolean isDistributedFileSystem(Path path) {
        return FileSystemUtil.hasScheme(path, SCHEME_HDFS);
    }

    public static boolean isOzoneFileSystem(FileSystem fs) {
        return FileSystemUtil.hasScheme(fs, SCHEME_O3FS) || FileSystemUtil.hasScheme(fs, SCHEME_OFS);
    }

    public static boolean isOzoneFileSystem(Path path) {
        return FileSystemUtil.hasScheme(path, SCHEME_O3FS) || FileSystemUtil.hasScheme(path, SCHEME_OFS);
    }

    private static boolean hasScheme(FileSystem fs, String scheme) {
        return scheme.equals(fs.getScheme());
    }

    private static boolean hasScheme(Path path, String scheme) {
        return scheme.equals(path.toUri().getScheme());
    }

    public static FileSystem getDefaultFileSystem() throws IOException {
        Path path = new Path(FileSystem.getDefaultUri((Configuration)CONF));
        FileSystem fs = path.getFileSystem(CONF);
        return fs;
    }

    public static FileSystem getFileSystemForPath(Path p) throws IOException {
        return p.getFileSystem(CONF);
    }

    public static FileSystem getFileSystemForPath(String p) throws IOException {
        Path location = new Path(p);
        URI defaultUri = FileSystem.getDefaultUri((Configuration)CONF);
        URI locationUri = location.toUri();
        if (locationUri.getScheme() == null || locationUri.getScheme().equalsIgnoreCase(defaultUri.getScheme())) {
            return FileSystemUtil.getDefaultFileSystem();
        }
        return location.getFileSystem(CONF);
    }

    public static DistributedFileSystem getDistributedFileSystem() throws IOException {
        FileSystem fs = FileSystemUtil.getDefaultFileSystem();
        Preconditions.checkState((boolean)(fs instanceof DistributedFileSystem));
        return (DistributedFileSystem)fs;
    }

    public static Path createFullyQualifiedPath(Path location) {
        URI defaultUri = FileSystem.getDefaultUri((Configuration)CONF);
        URI locationUri = location.toUri();
        if (locationUri.getScheme() == null || locationUri.getScheme().equalsIgnoreCase(defaultUri.getScheme())) {
            return location.makeQualified(defaultUri, location);
        }
        return location;
    }

    public static boolean isPathOnFileSystem(Path path, FileSystem fs) {
        Preconditions.checkState((!path.equals((Object)Path.getPathWithoutSchemeAndAuthority((Path)path)) ? 1 : 0) != 0, (Object)String.format("Path '%s' is not qualified.", path));
        try {
            Path qp = fs.makeQualified(path);
            return path.equals((Object)qp);
        }
        catch (IllegalArgumentException e) {
            LOG.debug(String.format("Path '%s' is not on file system '%s'", path, fs));
            return false;
        }
    }

    public static void copyToLocal(Path source, Path dest) throws IOException {
        FileSystem fs = source.getFileSystem(CONF);
        fs.copyToLocalFile(source, dest);
    }

    public static String copyFileFromUriToLocal(String srcUri) throws IOException {
        Preconditions.checkNotNull((Object)srcUri);
        String localLibPath = BackendConfig.INSTANCE.getBackendCfg().local_library_path;
        String fileExt = FilenameUtils.getExtension((String)srcUri);
        String localPath = localLibPath != null && !localLibPath.isEmpty() ? localLibPath + "/" + UUID.randomUUID().toString() + DOT + fileExt : "/tmp/" + UUID.randomUUID().toString() + DOT + fileExt;
        try {
            Path remoteFilePath = new Path(srcUri);
            Path localFilePath = new Path("file://" + localPath);
            FileSystemUtil.copyToLocal(remoteFilePath, localFilePath);
        }
        catch (Exception e) {
            String errorMsg = "Failed to copy " + srcUri + " to local path: " + localPath;
            LOG.error(errorMsg, (Throwable)e);
            throw new IOException(String.format("%s, %s", errorMsg, e.getMessage()));
        }
        return localPath;
    }

    public static void deleteIfExists(Path path) {
        try {
            FileSystem fs = path.getFileSystem(CONF);
            if (!fs.exists(path)) {
                return;
            }
            fs.delete(path);
        }
        catch (IOException e) {
            LOG.warn("Encountered an exception deleting file at path " + path.toString(), (Throwable)e);
        }
    }

    public static boolean isPathCacheable(Path path) {
        return FileSystemUtil.isDistributedFileSystem(path);
    }

    public static Configuration getConfiguration() {
        return CONF;
    }

    public static boolean isImpalaWritableFilesystem(String location) throws IOException {
        Path path = new Path(location);
        String scheme = path.toUri().getScheme();
        return SCHEME_WRITEABLE_BY_IMPALA.contains(scheme);
    }

    public static boolean isValidDefaultFileSystem(FileSystem fs) {
        return SCHEME_SUPPORTED_AS_DEFAULT_FS.contains(fs.getScheme());
    }

    public static boolean isValidLoadDataInpath(FileSystem fs) {
        return SCHEME_VALID_FOR_LOAD_INPATH.contains(fs.getScheme());
    }

    public static String getValidLoadDataInpathSchemes() {
        return String.join((CharSequence)", ", SCHEME_VALID_FOR_LOAD_INPATH);
    }

    public static RemoteIterator<? extends FileStatus> listStatus(FileSystem fs, Path p, boolean recursive, String debugAction) throws IOException {
        try {
            if (recursive) {
                if (FileSystemUtil.isS3AFileSystem(fs)) {
                    return FileSystemUtil.listFiles(fs, p, true, debugAction);
                }
                DebugUtils.executeDebugAction(debugAction, "catalogd_refresh_hdfs_listing_delay");
                return new FilterIterator(p, new RecursingIterator(fs, p, debugAction, FileSystemUtil::listStatusIterator));
            }
            DebugUtils.executeDebugAction(debugAction, "catalogd_refresh_hdfs_listing_delay");
            return new FilterIterator(p, FileSystemUtil.listStatusIterator(fs, p));
        }
        catch (FileNotFoundException e) {
            if (LOG.isWarnEnabled()) {
                LOG.warn("Path does not exist: " + p.toString(), (Throwable)e);
            }
            return null;
        }
    }

    public static RemoteIterator<? extends FileStatus> listFiles(FileSystem fs, Path p, boolean recursive, String debugAction) throws IOException {
        try {
            DebugUtils.executeDebugAction(debugAction, "catalogd_refresh_hdfs_listing_delay");
            Object baseIterator = FileSystemUtil.hasRecursiveListFiles(fs) ? fs.listFiles(p, recursive) : (recursive ? new RecursingIterator<LocatedFileStatus>(fs, p, debugAction, FileSystemUtil::listLocatedStatusIterator) : FileSystemUtil.listLocatedStatusIterator(fs, p));
            return new FilterIterator(p, (RemoteIterator<? extends FileStatus>)baseIterator);
        }
        catch (FileNotFoundException e) {
            if (LOG.isWarnEnabled()) {
                LOG.warn("Path does not exist: " + p.toString(), (Throwable)e);
            }
            return null;
        }
    }

    public static RemoteIterator<FileStatus> listStatusIterator(FileSystem fs, Path p) throws IOException {
        RemoteIterator iterator = fs.listStatusIterator(p);
        iterator.hasNext();
        return iterator;
    }

    public static RemoteIterator<LocatedFileStatus> listLocatedStatusIterator(FileSystem fs, Path p) throws IOException {
        RemoteIterator iterator = fs.listLocatedStatus(p);
        iterator.hasNext();
        return iterator;
    }

    public static boolean isDir(Path p) throws IOException {
        FileSystem fs = FileSystemUtil.getFileSystemForPath(p);
        return fs.isDirectory(p);
    }

    public static boolean isFile(Path p) throws IOException, FileNotFoundException {
        FileSystem fs = FileSystemUtil.getFileSystemForPath(p);
        return fs.getFileStatus(p).isFile();
    }

    public static String relativizePath(Path path, Path startPath) {
        URI relUri = startPath.toUri().relativize(path.toUri());
        if (!FileSystemUtil.isRelative(relUri)) {
            throw new RuntimeException("FileSystem returned an unexpected path " + path + " for a file within " + startPath);
        }
        return relUri.getPath();
    }

    public static String relativizePathNoThrow(Path path, Path startPath) {
        URI relUri = startPath.toUri().relativize(path.toUri());
        if (FileSystemUtil.isRelative(relUri)) {
            return relUri.getPath();
        }
        return null;
    }

    private static boolean isRelative(URI uri) {
        return !uri.isAbsolute() && !uri.getPath().startsWith("/");
    }

    @VisibleForTesting
    static boolean isInIgnoredDirectory(Path parent, FileStatus fileStatus) {
        Path currentPath;
        Preconditions.checkNotNull((Object)fileStatus);
        Path path = currentPath = fileStatus.isDirectory() ? fileStatus.getPath() : fileStatus.getPath().getParent();
        while (currentPath != null && !currentPath.equals((Object)parent)) {
            if (FileSystemUtil.isIgnoredDir(currentPath)) {
                LOG.debug("Ignoring {} since it is either in a hidden directory or a temporary staging directory {}", (Object)fileStatus.getPath(), (Object)currentPath);
                return true;
            }
            currentPath = currentPath.getParent();
        }
        return false;
    }

    @VisibleForTesting
    static boolean isIgnoredDir(Path path) {
        String filename = path.getName();
        for (String prefix : TMP_DIR_PREFIX_LIST) {
            if (!filename.startsWith(prefix)) continue;
            return true;
        }
        return false;
    }

    public static Set<TTableName> loadWarmupTableNames(String configFile) {
        if (configFile == null || configFile.isEmpty()) {
            return Collections.emptySet();
        }
        LinkedHashSet<TTableName> warmupTables = new LinkedHashSet<TTableName>();
        Path path = new Path(configFile);
        try {
            String line;
            FileSystem fs = path.getFileSystem(new Configuration());
            BufferedReader reader = new BufferedReader(new InputStreamReader((InputStream)fs.open(path)));
            while ((line = reader.readLine()) != null) {
                if ((line = line.trim()).isEmpty() || line.startsWith("#") || line.startsWith("//")) continue;
                String[] names = line.split("\\.");
                if (names.length != 2) {
                    LOG.warn("Skipped illegal table name in warmup list: " + line);
                    continue;
                }
                warmupTables.add(new TTableName(names[0].toLowerCase(), names[1].toLowerCase()));
            }
            return warmupTables;
        }
        catch (IOException e) {
            LOG.warn("Failed to load the list of warmup tables", (Throwable)e);
            return Collections.emptySet();
        }
    }

    static {
        LOG = LoggerFactory.getLogger(FileSystemUtil.class);
        SCHEME_SUPPORT_STORAGE_IDS = ImmutableSet.builder().add((Object)SCHEME_HDFS).add((Object)SCHEME_O3FS).add((Object)SCHEME_OFS).add((Object)SCHEME_ALLUXIO).build();
        SCHEME_WRITEABLE_BY_IMPALA = ImmutableSet.builder().add((Object)SCHEME_ABFS).add((Object)SCHEME_ABFSS).add((Object)SCHEME_ADL).add((Object)SCHEME_FILE).add((Object)SCHEME_HDFS).add((Object)SCHEME_S3A).add((Object)SCHEME_O3FS).add((Object)SCHEME_OFS).add((Object)SCHEME_GCS).add((Object)SCHEME_COS).add((Object)SCHEME_OSS).add((Object)SCHEME_OBS).build();
        SCHEME_SUPPORTED_AS_DEFAULT_FS = ImmutableSet.builder().add((Object)SCHEME_ABFS).add((Object)SCHEME_ABFSS).add((Object)SCHEME_ADL).add((Object)SCHEME_HDFS).add((Object)SCHEME_S3A).add((Object)SCHEME_OFS).add((Object)SCHEME_GCS).add((Object)SCHEME_COS).add((Object)SCHEME_OSS).add((Object)SCHEME_OBS).build();
        SCHEME_VALID_FOR_LOAD_INPATH = ImmutableSet.builder().add((Object)SCHEME_ABFS).add((Object)SCHEME_ABFSS).add((Object)SCHEME_ADL).add((Object)SCHEME_HDFS).add((Object)SCHEME_S3A).add((Object)SCHEME_O3FS).add((Object)SCHEME_OFS).add((Object)SCHEME_GCS).add((Object)SCHEME_COS).add((Object)SCHEME_OSS).add((Object)SCHEME_OBS).build();
        FileSystemUtil.setConfiguration(new Configuration());
        TMP_DIR_PREFIX_LIST = new ArrayList<String>();
        if (BackendConfig.INSTANCE == null || BackendConfig.INSTANCE.getIgnoredDirPrefixList() == null) {
            TMP_DIR_PREFIX_LIST.add(DOT);
            TMP_DIR_PREFIX_LIST.add(HIVE_TEMP_FILE_PREFIX);
            TMP_DIR_PREFIX_LIST.add(HIVE_NEW_TEMP_FILE_PREFIX);
            TMP_DIR_PREFIX_LIST.add(SPARK_TEMP_FILE_PREFIX);
            TMP_DIR_PREFIX_LIST.add(IMPALA_STAGING_DIR_PREFIX);
            LOG.warn("BackendConfig.INSTANCE uninitialized. Use hard-coded prefix-list.");
        } else {
            String s = BackendConfig.INSTANCE.getIgnoredDirPrefixList();
            for (String prefix : s.split(",")) {
                if (prefix.isEmpty()) continue;
                TMP_DIR_PREFIX_LIST.add(prefix);
            }
        }
        LOG.info("Prefix list of ignored dirs: " + TMP_DIR_PREFIX_LIST);
    }

    private static class RecursingIterator<T extends FileStatus>
    implements RemoteIterator<T> {
        private final BiFunctionWithException<FileSystem, Path, RemoteIterator<T>> newIterFunc_;
        private final FileSystem fs_;
        private final String debugAction_;
        private final Stack<RemoteIterator<T>> iters_ = new Stack();
        private RemoteIterator<T> curIter_;
        private T curFile_;

        private RecursingIterator(FileSystem fs, Path startPath, String debugAction, BiFunctionWithException<FileSystem, Path, RemoteIterator<T>> newIterFunc) throws IOException {
            this.fs_ = (FileSystem)Preconditions.checkNotNull((Object)fs);
            this.debugAction_ = debugAction;
            this.newIterFunc_ = (BiFunctionWithException)Preconditions.checkNotNull(newIterFunc);
            Preconditions.checkNotNull((Object)startPath);
            this.curIter_ = newIterFunc.apply(fs, startPath);
            LOG.trace("listed start path: {}", (Object)startPath);
            DebugUtils.executeDebugAction(debugAction, "catalogd_pause_after_hdfs_remote_iterator_creation");
        }

        public boolean hasNext() throws IOException {
            while (this.curFile_ == null) {
                if (this.curIter_.hasNext()) {
                    FileStatus fileStat = (FileStatus)this.curIter_.next();
                    try {
                        this.handleFileStat(fileStat);
                    }
                    catch (FileNotFoundException e) {
                        LOG.warn("Ignoring non-existing sub dir", (Throwable)e);
                    }
                    continue;
                }
                if (!this.iters_.empty()) {
                    this.curIter_ = this.iters_.pop();
                    continue;
                }
                return false;
            }
            return true;
        }

        private void handleFileStat(T fileStatus) throws IOException {
            LOG.trace("handleFileStat: {}", (Object)fileStatus.getPath());
            if (FileSystemUtil.isIgnoredDir(fileStatus.getPath())) {
                LOG.debug("Ignoring {} since it is either a hidden directory or a temporary staging directory", (Object)fileStatus.getPath());
                this.curFile_ = null;
                return;
            }
            if (fileStatus.isFile()) {
                this.curFile_ = fileStatus;
                return;
            }
            RemoteIterator<T> subIter = this.newIterFunc_.apply(this.fs_, fileStatus.getPath());
            this.iters_.push(this.curIter_);
            this.curIter_ = subIter;
            this.curFile_ = fileStatus;
            LOG.trace("listed sub dir: {}", (Object)fileStatus.getPath());
            DebugUtils.executeDebugAction(this.debugAction_, "catalogd_pause_after_hdfs_remote_iterator_creation");
        }

        public T next() throws IOException {
            if (this.hasNext()) {
                T result = this.curFile_;
                this.curFile_ = null;
                return result;
            }
            throw new NoSuchElementException("No more entries");
        }
    }

    @FunctionalInterface
    public static interface BiFunctionWithException<T, U, R> {
        public R apply(T var1, U var2) throws IOException;
    }

    static class FilterIterator
    implements RemoteIterator<FileStatus> {
        private final RemoteIterator<? extends FileStatus> baseIterator_;
        private FileStatus curFile_ = null;
        private final Path startPath_;

        FilterIterator(Path startPath, RemoteIterator<? extends FileStatus> baseIterator) {
            this.startPath_ = (Path)Preconditions.checkNotNull((Object)startPath);
            this.baseIterator_ = (RemoteIterator)Preconditions.checkNotNull(baseIterator);
        }

        public boolean hasNext() throws IOException {
            while (this.curFile_ == null) {
                if (!this.baseIterator_.hasNext()) {
                    return false;
                }
                FileStatus next = (FileStatus)this.baseIterator_.next();
                if (FileSystemUtil.isInIgnoredDirectory(this.startPath_, next)) continue;
                this.curFile_ = next;
                return true;
            }
            return true;
        }

        public FileStatus next() throws IOException {
            if (this.hasNext()) {
                FileStatus next = this.curFile_;
                this.curFile_ = null;
                return next;
            }
            throw new NoSuchElementException("No more entries");
        }
    }

    public static enum FsType {
        ADLS,
        HDFS,
        LOCAL,
        S3,
        OZONE,
        ALLUXIO,
        GCS,
        COS,
        OSS,
        SFS,
        OBS;

        private static final Map<String, FsType> SCHEME_TO_FS_MAPPING;

        public static FsType getFsType(String scheme) {
            if (scheme.startsWith("sfs+")) {
                return SFS;
            }
            return SCHEME_TO_FS_MAPPING.get(scheme);
        }

        static {
            SCHEME_TO_FS_MAPPING = ImmutableMap.builder().put((Object)FileSystemUtil.SCHEME_ABFS, (Object)ADLS).put((Object)FileSystemUtil.SCHEME_ABFSS, (Object)ADLS).put((Object)FileSystemUtil.SCHEME_ADL, (Object)ADLS).put((Object)FileSystemUtil.SCHEME_FILE, (Object)LOCAL).put((Object)FileSystemUtil.SCHEME_HDFS, (Object)HDFS).put((Object)FileSystemUtil.SCHEME_S3A, (Object)S3).put((Object)FileSystemUtil.SCHEME_O3FS, (Object)OZONE).put((Object)FileSystemUtil.SCHEME_OFS, (Object)OZONE).put((Object)FileSystemUtil.SCHEME_ALLUXIO, (Object)ALLUXIO).put((Object)FileSystemUtil.SCHEME_GCS, (Object)GCS).put((Object)FileSystemUtil.SCHEME_COS, (Object)COS).put((Object)FileSystemUtil.SCHEME_OSS, (Object)OSS).put((Object)FileSystemUtil.SCHEME_OBS, (Object)OBS).build();
        }
    }
}

