/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.analysis.paimon;

import com.google.common.collect.Sets;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.apache.impala.analysis.AnalysisUtils;
import org.apache.impala.analysis.Analyzer;
import org.apache.impala.analysis.ColumnDef;
import org.apache.impala.analysis.CreateTableStmt;
import org.apache.impala.analysis.ShowStatsStmt;
import org.apache.impala.catalog.HdfsFileFormat;
import org.apache.impala.catalog.paimon.FePaimonTable;
import org.apache.impala.catalog.paimon.ImpalaTypeUtils;
import org.apache.impala.catalog.paimon.PaimonUtil;
import org.apache.impala.common.AnalysisException;
import org.apache.impala.thrift.TBucketType;
import org.apache.impala.thrift.TPaimonCatalog;
import org.apache.impala.thrift.TShowStatsParams;
import org.apache.paimon.CoreOptions;
import org.apache.paimon.options.OptionsUtils;

public class PaimonAnalyzer {
    public static void analyzeCreateTableStmt(CreateTableStmt stmt, Analyzer analyzer) throws AnalysisException {
        AnalysisUtils.throwIfNotNull(stmt.getCachingOp(), "A Paimon table cannot be cached in HDFS.");
        AnalysisUtils.throwIfTrue(stmt.getTblProperties().containsKey(CoreOptions.PARTITION.key()), "Can't specify 'partition' table property in Paimon DDL, use PARTITIONED BY clause instead.");
        AnalysisUtils.throwIfTrue(stmt.geTBucketInfo().bucket_type != TBucketType.NONE, "CLUSTERED BY clause is not support by PAIMON now, use property bucket-key instead.");
        AnalysisUtils.throwIfTrue(stmt.getTblProperties().containsKey(CoreOptions.PRIMARY_KEY.key()) && !stmt.getPrimaryKeys().isEmpty(), "Can't specify both PRIMARY KEY clause and 'primary-key' table property in Paimon DDL.");
        PaimonAnalyzer.analyzePaimonColumns(stmt, analyzer);
        PaimonAnalyzer.analyzePaimonFormat(stmt, analyzer);
    }

    public static void analyzeShowStatStmt(ShowStatsStmt statsOp, FePaimonTable table, Analyzer analyzer) throws AnalysisException {
        TShowStatsParams params = statsOp.toThrift();
        switch (params.getOp()) {
            case TABLE_STATS: 
            case COLUMN_STATS: {
                return;
            }
            case PARTITIONS: {
                if (PaimonUtil.hasPartition(table.getPaimonApiTable())) break;
                throw new AnalysisException("Table is not partitioned: " + table.getFullName());
            }
            default: {
                throw new AnalysisException(statsOp.toSql() + " is not supported for Paimon Table");
            }
        }
    }

    private static void putPaimonProperty(CreateTableStmt stmt, String key, String value) {
        stmt.putGeneratedProperty(key, value);
    }

    private static void analyzePaimonFormat(CreateTableStmt stmt, Analyzer analyzer) throws AnalysisException {
        Map<String, String> tblProperties = stmt.getTblProperties();
        Map paimonTblProperties = OptionsUtils.convertToPropertiesPrefixKey(tblProperties, (String)"");
        boolean isExternal = stmt.isExternal();
        if (!isExternal && Boolean.parseBoolean(tblProperties.get("external.table.purge"))) {
            throw new AnalysisException(String.format("Table property '%s' cannot be set to true with a managed paimon table.", "external.table.purge"));
        }
        if (stmt.isExternal() && Boolean.parseBoolean(tblProperties.getOrDefault("external.table.purge", "false"))) {
            throw new AnalysisException(" External table with purge is not supported.");
        }
        String handler = tblProperties.get("storage_handler");
        if (handler != null && !handler.equals("org.apache.paimon.hive.PaimonStorageHandler")) {
            throw new AnalysisException("Invalid storage handler specified for Paimon format: " + handler);
        }
        stmt.putGeneratedProperty("storage_handler", "org.apache.paimon.hive.PaimonStorageHandler");
        if (!paimonTblProperties.containsKey(CoreOptions.DELETION_VECTORS_ENABLED.key()) && !paimonTblProperties.containsKey(CoreOptions.FULL_COMPACTION_DELTA_COMMITS.key())) {
            PaimonAnalyzer.putPaimonProperty(stmt, CoreOptions.DELETION_VECTORS_ENABLED.key(), "true");
        }
        String fileformat = (String)paimonTblProperties.get(CoreOptions.FILE_FORMAT.key());
        HdfsFileFormat hdfsFileFormat = PaimonUtil.getPaimonFileFormat(fileformat);
        if (fileformat != null && hdfsFileFormat == null) {
            throw new AnalysisException("Unsupported fileformat for Paimon table: " + fileformat);
        }
        TPaimonCatalog catalog = PaimonUtil.getTPaimonCatalog(tblProperties);
        switch (catalog) {
            case HIVE_CATALOG: {
                PaimonAnalyzer.validateTableInHiveCatalog(stmt, tblProperties);
                break;
            }
            case HADOOP_CATALOG: {
                PaimonAnalyzer.validateTableInHadoopCatalog(stmt, tblProperties);
                break;
            }
            default: {
                throw new AnalysisException(String.format("Unknown Paimon catalog type: %s", new Object[]{catalog}));
            }
        }
        if (!isExternal && !catalog.equals((Object)TPaimonCatalog.HIVE_CATALOG) && "false".equalsIgnoreCase(tblProperties.get("external.table.purge"))) {
            analyzer.addWarning("The table property 'external.table.purge' will be set to 'TRUE' on newly created managed Paimon tables.");
        }
    }

    private static void validateTableInHiveCatalog(CreateTableStmt stmt, Map<String, String> tblProperties) throws AnalysisException {
        if (tblProperties.get("paimon.catalog_location") != null) {
            throw new AnalysisException(String.format("%s cannot be set for Paimon table stored in hive.catalog", "paimon.catalog_location"));
        }
        if (tblProperties.get("paimon.table_identifier") != null) {
            throw new AnalysisException(String.format("%s cannot be set for Paimon table stored in hive.catalog", "paimon.table_identifier"));
        }
        AnalysisUtils.throwIfTrue(stmt.isExternal() && stmt.getLocation() == null, "Location must be set for external Paimon table stored in hive catalog");
    }

    private static void validateTableInHadoopCatalog(CreateTableStmt stmt, Map<String, String> tblProperties) throws AnalysisException {
        String catalogLoc = tblProperties.get("paimon.catalog_location");
        if (catalogLoc == null || catalogLoc.isEmpty()) {
            throw new AnalysisException(String.format("Table property '%s' is necessary for Paimon table with 'hadoop.catalog'.", "paimon.catalog_location"));
        }
        AnalysisUtils.throwIfTrue(stmt.isExternal() && tblProperties.get("paimon.table_identifier") == null, String.format("Table property '%s' is necessary for Paimon table with 'hadoop.catalog'.", "paimon.table_identifier"));
    }

    private static void throwIfColumnTypeIsNotSupported(ColumnDef columnDef) throws AnalysisException {
        if (!ImpalaTypeUtils.isSupportedColumnType(columnDef.getType())) {
            throw new AnalysisException("Tables stored by Paimon do not support the column " + columnDef.getColName() + " type: " + columnDef.getType().toSql());
        }
    }

    private static void analyzePaimonColumns(CreateTableStmt stmt, Analyzer analyzer) throws AnalysisException {
        HashSet colSets = Sets.newHashSet();
        for (ColumnDef col : stmt.getColumnDefs()) {
            PaimonAnalyzer.throwIfColumnTypeIsNotSupported(col);
            colSets.add(col.getColName().toLowerCase());
        }
        for (ColumnDef col : stmt.getPartitionColumnDefs()) {
            PaimonAnalyzer.throwIfColumnTypeIsNotSupported(col);
            colSets.add(col.getColName().toLowerCase());
        }
        if (stmt.getTblProperties().containsKey(CoreOptions.PRIMARY_KEY.key())) {
            List<String> colNames = PaimonUtil.extractColumnNames(stmt.getTblProperties().get(CoreOptions.PRIMARY_KEY.key()));
            for (String col : colNames) {
                AnalysisUtils.throwIfTrue(!colSets.contains(col), String.format("Invalid col name %s specified in 'primary-key' table properties.", col));
            }
        }
        if (stmt.getTblProperties().containsKey(CoreOptions.BUCKET_KEY.key())) {
            List<String> parts = PaimonUtil.extractColumnNames(stmt.getTblProperties().get(CoreOptions.BUCKET_KEY.key()));
            for (String col : parts) {
                AnalysisUtils.throwIfTrue(!colSets.contains(col), String.format("Invalid col name %s specified in 'bucket-key' table properties.", col));
            }
        }
        if (stmt.getColumnDefs().isEmpty()) {
            AnalysisUtils.throwIfTrue(!stmt.isExternal(), "Managed table does not support inferring schema from underlying paimon table");
        }
    }
}

