/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.util;

import com.google.common.base.Preconditions;
import com.google.common.collect.Range;
import com.google.common.collect.RangeMap;
import com.google.common.collect.TreeRangeMap;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.impala.compat.MetastoreShim;
import org.apache.impala.testutil.HiveJdbcClientPool;
import org.apache.impala.testutil.TestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RandomHiveQueryRunner {
    private final Random random_;
    private final String tblNamePrefix_;
    private final String dbNamePrefix_;
    private final int numClients_;
    private final int numQueriesPerClient_;
    private final AtomicInteger[] partitionIdGenerators_;
    private final ExecutorService executorService_;
    private final HiveJdbcClientPool hiveJdbcClientPool_;
    private ArrayList<Future<Void>> futures_;
    private final AtomicBoolean isStarted_ = new AtomicBoolean(false);
    private static final Logger LOG = LoggerFactory.getLogger(RandomHiveQueryRunner.class);
    private int totalQueryWeight_;
    private final RangeMap<Integer, QueryType> rangeMap = TreeRangeMap.create();
    private final List<QueryType> skippedQueryTypes;

    public RandomHiveQueryRunner(Random random, String dbNamePrefix, String tblNamePrefix, int numClients, int numQueriesPerClient, List<QueryType> skippedQueryTypes) throws SQLException, ClassNotFoundException {
        this.random_ = (Random)Preconditions.checkNotNull((Object)random);
        this.numClients_ = numClients;
        this.numQueriesPerClient_ = numQueriesPerClient;
        this.partitionIdGenerators_ = new AtomicInteger[numClients];
        for (int i = 0; i < numClients; ++i) {
            this.partitionIdGenerators_[i] = new AtomicInteger(0);
        }
        this.dbNamePrefix_ = (String)Preconditions.checkNotNull((Object)dbNamePrefix);
        this.tblNamePrefix_ = (String)Preconditions.checkNotNull((Object)tblNamePrefix);
        this.executorService_ = Executors.newFixedThreadPool(numClients, new ThreadFactoryBuilder().setNameFormat("hive-query-executor-%d").build());
        this.hiveJdbcClientPool_ = HiveJdbcClientPool.create(numClients);
        this.totalQueryWeight_ = 0;
        this.skippedQueryTypes = skippedQueryTypes;
        for (QueryType qt : QueryType.values()) {
            this.rangeMap.put(Range.closedOpen((Comparable)Integer.valueOf(this.totalQueryWeight_), (Comparable)Integer.valueOf(this.totalQueryWeight_ + qt.weight_)), (Object)qt);
            this.totalQueryWeight_ += qt.weight_;
        }
    }

    private QueryType getNextQueryType() {
        return (QueryType)((Object)this.rangeMap.get((Comparable)Integer.valueOf(this.random_.nextInt(this.totalQueryWeight_))));
    }

    private TestHiveQuery getNext(int clientId) {
        QueryType type = this.getNextQueryType();
        if (this.skippedQueryTypes != null && this.skippedQueryTypes.contains((Object)type)) {
            LOG.info("Skipping this query type {}", (Object)type);
            return null;
        }
        String dbName = this.dbNamePrefix_ + clientId;
        String tblName = this.tblNamePrefix_ + clientId;
        switch (type) {
            case CREATE_DB: {
                return CreateDbQuery.create(dbName);
            }
            case DROP_DB: {
                return DropDbQuery.create(dbName);
            }
            case CREATE_TABLE: {
                return CreateTblQuery.create(dbName, tblName, "c1 int, c2 string");
            }
            case CREATE_TABLE_AS_SELECT: {
                return CreateTblAsSelect.create(dbName, tblName, "functional", "alltypes");
            }
            case DROP_TABLE: {
                return DropTblQuery.create(dbName, tblName);
            }
            case ALTER_TABLE_ADD_COL: {
                return AlterTblAddColQuery.create(dbName, tblName, RandomHiveQueryRunner.getRandomColName(6) + " string");
            }
            case ALTER_TABLE_ADD_PROPERTY: {
                return AlterTblAddProperty.create(dbName, tblName, RandomHiveQueryRunner.getRandomColName(6), RandomHiveQueryRunner.getRandomColName(6));
            }
            case ADD_PARTITION: {
                tblName = tblName + "_part";
                return AlterTableAddPartition.create(dbName, tblName, "part=" + this.partitionIdGenerators_[clientId].getAndIncrement());
            }
            case DROP_PARTITION: {
                tblName = tblName + "_part";
                int lastPartitionId = this.partitionIdGenerators_[clientId].get();
                int bound = 1;
                if (lastPartitionId > 0) {
                    bound = lastPartitionId;
                }
                return AlterTableDropPartition.create(dbName, tblName, "part=" + this.random_.nextInt(bound));
            }
            case DYN_PARTITION_INSERT: {
                tblName = tblName + "_alltypes_part";
                return DynamicPartitionInsert.create(dbName, tblName, "functional", "alltypes", "year,month", false);
            }
            case DYN_PARTITION_INSERT_OVERWRITE: {
                tblName = tblName + "_alltypes_part";
                return DynamicPartitionInsert.create(dbName, tblName, "functional", "alltypes", "year,month", true);
            }
            case INSERT_PARTITION: {
                tblName = tblName + "_alltypes_part";
                return InsertTblOrPartition.create(dbName, tblName, "functional", "alltypes", false, "year,month", 100);
            }
            case INSERT_OVERWRITE_PARTITION: {
                tblName = tblName + "_alltypes_part";
                return InsertTblOrPartition.create(dbName, tblName, "functional", "alltypes", true, "year,month", 100);
            }
            case INSERT_TABLE: {
                tblName = tblName + "_alltypes_part";
                return InsertTblOrPartition.create(dbName, tblName, "functional", "alltypesnopart", false, null, 100);
            }
            case INSERT_OVERWRITE_TABLE: {
                tblName = tblName + "_alltypes_part";
                return InsertTblOrPartition.create(dbName, tblName, "functional", "alltypesnopart", true, null, 100);
            }
        }
        throw new RuntimeException(String.format("Invalid statement type %s", new Object[]{type}));
    }

    public void start() throws Exception {
        if (!this.isStarted_.compareAndSet(false, true)) {
            throw new Exception("Random hive query generator is already started");
        }
        this.futures_ = new ArrayList();
        for (int i = 0; i < this.numClients_; ++i) {
            this.futures_.add(this.executorService_.submit(() -> {
                int clientId = Integer.parseInt(Thread.currentThread().getName().substring("hive-query-executor-".length()));
                int queryNumber = 1;
                while (queryNumber <= this.numQueriesPerClient_) {
                    TestHiveQuery query = this.getNext(clientId);
                    if (query == null) continue;
                    try {
                        LOG.info("Client {} running hive query set {}: {}", new Object[]{clientId, queryNumber, query});
                        query.run(this.hiveJdbcClientPool_);
                        ++queryNumber;
                    }
                    catch (Exception e) {
                        throw new ExecutionException(String.format("Client %s errored out while executing query set %s %s or its dependent queries. Exception message is: %s", clientId, queryNumber, query, e.getMessage()), e);
                    }
                }
                return null;
            }));
        }
        this.executorService_.shutdown();
    }

    public void checkForErrors() throws ExecutionException {
        Preconditions.checkState((boolean)this.isStarted_.get());
        Preconditions.checkState((boolean)this.executorService_.isTerminated());
        try {
            for (Future<Void> result : this.futures_) {
                result.get();
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public boolean isTerminated() {
        return this.executorService_.isTerminated();
    }

    private static String getRandomColName(int size) {
        return "col_" + TestUtils.getRandomString(size);
    }

    public void shutdownNow() {
        try {
            this.executorService_.shutdownNow();
        }
        finally {
            if (this.hiveJdbcClientPool_ != null) {
                this.hiveJdbcClientPool_.close();
            }
        }
    }

    static class DynamicPartitionInsert
    extends SourceTableBasedQuery {
        private final String dyPartitionSpec_;
        private final boolean overwrite_;

        private DynamicPartitionInsert(String dbName, String tblName, String srcDbName, String srcTblName, String dyPartitionSpec, boolean overwrite) {
            super(dbName, tblName, srcDbName, srcTblName, overwrite ? QueryType.DYN_PARTITION_INSERT_OVERWRITE : QueryType.DYN_PARTITION_INSERT);
            this.dyPartitionSpec_ = (String)Preconditions.checkNotNull((Object)dyPartitionSpec);
            this.overwrite_ = overwrite;
        }

        static DynamicPartitionInsert create(String dbName, String tblName, String srcDbName, String srcTblName, String dyPartitionSpec, boolean overwrite) {
            DynamicPartitionInsert query = new DynamicPartitionInsert(dbName, tblName, srcDbName, srcTblName, dyPartitionSpec, overwrite);
            query.addDependentQuery(new CreateDbQuery(dbName, true));
            query.addDependentQuery(new DropTblQuery(dbName, tblName, true));
            query.addDependentQuery(new CreateTblAsSelect(dbName, tblName, srcDbName, srcTblName, true));
            query.addDependentQuery(new SetConfigStmt("hive.exec.dynamic.partition.mode", "nonstrict"));
            query.addDependentQuery(new SetConfigStmt("hive.exec.max.dynamic.partitions", "10000"));
            query.addDependentQuery(new SetConfigStmt("hive.exec.max.dynamic.partitions.pernode", "10000"));
            if (MetastoreShim.getMajorVersion() >= 3L) {
                query.addDependentQuery(new SetConfigStmt("tez.session.am.dag.submit.timeout.secs", "2"));
            }
            return query;
        }

        @Override
        protected String getQuery() {
            StringBuilder sb = new StringBuilder("insert");
            if (this.overwrite_) {
                sb.append(" overwrite table ");
            } else {
                sb.append(" into table ");
            }
            sb.append(String.format("%s.%s partition (%s)", this.dbName_, this.tblName_, this.dyPartitionSpec_));
            sb.append(" select * from ");
            sb.append(String.format("%s.%s", this.srcDbName_, this.srcTblName_));
            return sb.toString();
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            DynamicPartitionInsert that = (DynamicPartitionInsert)o;
            return this.overwrite_ == that.overwrite_ && Objects.equals(this.dyPartitionSpec_, that.dyPartitionSpec_);
        }

        @Override
        public int hashCode() {
            return Objects.hash(super.hashCode(), this.dyPartitionSpec_, this.overwrite_);
        }
    }

    private static class InsertTblOrPartition
    extends SourceTableBasedQuery {
        private final boolean overwrite_;
        private final String partitionSpec_;
        private final int limit_;

        private InsertTblOrPartition(String dbName, String tblName, String srcDbName, String srcTblName, boolean overwrite, String partitionSpec, int limit) {
            super(dbName, tblName, srcDbName, srcTblName, InsertTblOrPartition.getQueryType(overwrite, partitionSpec));
            this.overwrite_ = overwrite;
            this.partitionSpec_ = partitionSpec;
            Preconditions.checkArgument((limit > 0 ? 1 : 0) != 0);
            this.limit_ = limit;
        }

        private static QueryType getQueryType(boolean overwrite, String partitionSpec) {
            if (partitionSpec != null) {
                return overwrite ? QueryType.INSERT_OVERWRITE_PARTITION : QueryType.INSERT_PARTITION;
            }
            return overwrite ? QueryType.INSERT_OVERWRITE_TABLE : QueryType.INSERT_TABLE;
        }

        static InsertTblOrPartition create(String dbName, String tblName, String srcDbName, String srcTblName, boolean overwrite, String partitionSpec, int limit) {
            InsertTblOrPartition query = new InsertTblOrPartition(dbName, tblName, srcDbName, srcTblName, overwrite, partitionSpec, limit);
            query.addDependentQuery(new CreateDbQuery(dbName, true));
            query.addDependentQuery(new DropTblQuery(dbName, tblName, true));
            query.addDependentQuery(new CreateTblAsSelect(dbName, tblName, srcDbName, srcTblName, true));
            query.addDependentQuery(new SetConfigStmt("hive.exec.dynamic.partition.mode", "nonstrict"));
            query.addDependentQuery(new SetConfigStmt("hive.exec.max.dynamic.partitions", "10000"));
            query.addDependentQuery(new SetConfigStmt("hive.exec.max.dynamic.partitions.pernode", "10000"));
            if (MetastoreShim.getMajorVersion() >= 3L) {
                query.addDependentQuery(new SetConfigStmt("tez.session.am.dag.submit.timeout.secs", "2"));
            }
            return query;
        }

        @Override
        protected String getQuery() {
            StringBuilder sb = new StringBuilder("insert ");
            if (this.overwrite_) {
                sb.append("overwrite table ");
            } else {
                sb.append("into table ");
            }
            sb.append(String.format("%s.%s ", this.dbName_, this.tblName_));
            if (this.partitionSpec_ != null) {
                sb.append(String.format("partition (%s) ", this.partitionSpec_));
            }
            sb.append(String.format("select * from %s.%s limit %s", this.srcDbName_, this.srcTblName_, this.limit_));
            return sb.toString();
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            InsertTblOrPartition that = (InsertTblOrPartition)o;
            return this.overwrite_ == that.overwrite_ && this.limit_ == that.limit_ && Objects.equals(this.partitionSpec_, that.partitionSpec_);
        }

        @Override
        public int hashCode() {
            return Objects.hash(super.hashCode(), this.overwrite_, this.partitionSpec_, this.limit_);
        }
    }

    private static abstract class SourceTableBasedQuery
    extends TestHiveQuery {
        protected final String tblName_;
        protected final String dbName_;
        protected final String srcDbName_;
        protected final String srcTblName_;

        private SourceTableBasedQuery(String dbName, String tblName, String srcDbName, String srcTblName, QueryType queryType) {
            super(queryType);
            this.dbName_ = (String)Preconditions.checkNotNull((Object)dbName);
            this.tblName_ = (String)Preconditions.checkNotNull((Object)tblName);
            this.srcDbName_ = (String)Preconditions.checkNotNull((Object)srcDbName);
            this.srcTblName_ = (String)Preconditions.checkNotNull((Object)srcTblName);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            SourceTableBasedQuery that = (SourceTableBasedQuery)o;
            return this.tblName_.equals(that.tblName_) && this.dbName_.equals(that.dbName_) && this.srcDbName_.equals(that.srcDbName_) && this.srcTblName_.equals(that.srcTblName_);
        }

        public int hashCode() {
            return Objects.hash(this.tblName_, this.dbName_, this.srcDbName_, this.srcTblName_);
        }
    }

    private static class SetConfigStmt
    extends TestHiveQuery {
        private final String key_;
        private final String val_;

        private SetConfigStmt(String key, String value) {
            super(QueryType.SET_CONFIG);
            this.key_ = (String)Preconditions.checkNotNull((Object)key);
            this.val_ = (String)Preconditions.checkNotNull((Object)value);
        }

        @Override
        protected String getQuery() {
            return String.format("set %s = %s", this.key_, this.val_);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            SetConfigStmt that = (SetConfigStmt)o;
            return this.key_.equals(that.key_) && this.val_.equals(that.val_);
        }

        public int hashCode() {
            return Objects.hash(this.key_, this.val_);
        }
    }

    private static class AlterTableDropPartition
    extends AlterTblQuery {
        private final boolean ifExists_;

        private AlterTableDropPartition(String dbName, String tblName, String partitionSpec, boolean ifExists) {
            super(dbName, tblName, partitionSpec, QueryType.DROP_PARTITION);
            this.ifExists_ = ifExists;
        }

        static AlterTableDropPartition create(String dbName, String tblName, String partitionSpec) {
            Preconditions.checkNotNull((Object)partitionSpec);
            AlterTableDropPartition query = new AlterTableDropPartition(dbName, tblName, partitionSpec, false);
            query.addDependentQuery(new CreateDbQuery(dbName, true));
            query.addDependentQuery(new CreatePartitionedTblQuery(dbName, tblName, RandomHiveQueryRunner.getRandomColName(6) + " string", AlterTableDropPartition.getPartitionColFromSpec(partitionSpec), true));
            query.addDependentQuery(new AlterTableAddPartition(dbName, tblName, partitionSpec, true));
            return query;
        }

        @Override
        protected String getQuery() {
            StringBuilder sb = new StringBuilder(String.format("alter table %s.%s drop ", this.dbName_, this.tblName_));
            if (this.ifExists_) {
                sb.append(" if exists ");
            }
            sb.append(String.format(" partition (%s)", this.partitionSpec_));
            return sb.toString();
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            AlterTableDropPartition that = (AlterTableDropPartition)o;
            return this.ifExists_ == that.ifExists_;
        }

        @Override
        public int hashCode() {
            return Objects.hash(super.hashCode(), this.ifExists_);
        }
    }

    private static class AlterTableAddPartition
    extends AlterTblQuery {
        private final boolean ifNotExists_;

        private AlterTableAddPartition(String dbName, String tblName, String partitionSpec, boolean ifNotExists) {
            super(dbName, tblName, partitionSpec, QueryType.ADD_PARTITION);
            this.ifNotExists_ = ifNotExists;
        }

        static AlterTableAddPartition create(String dbName, String tblName, String partitionSpec) {
            Preconditions.checkNotNull((Object)partitionSpec);
            AlterTableAddPartition alterTableAddPartition = new AlterTableAddPartition(dbName, tblName, partitionSpec, false);
            alterTableAddPartition.addDependentQuery(new CreateDbQuery(dbName, true));
            alterTableAddPartition.addDependentQuery(new CreatePartitionedTblQuery(dbName, tblName, RandomHiveQueryRunner.getRandomColName(6) + " string", AlterTableAddPartition.getPartitionColFromSpec(partitionSpec), true));
            alterTableAddPartition.addDependentQuery(new AlterTableDropPartition(dbName, tblName, partitionSpec, true));
            return alterTableAddPartition;
        }

        @Override
        protected String getQuery() {
            StringBuilder sb = new StringBuilder();
            sb.append(String.format("alter table %s.%s add ", this.dbName_, this.tblName_));
            if (this.ifNotExists_) {
                sb.append("if not exists ");
            }
            sb.append(String.format(" partition (%s)", this.partitionSpec_));
            return sb.toString();
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            AlterTableAddPartition that = (AlterTableAddPartition)o;
            return this.ifNotExists_ == that.ifNotExists_;
        }

        @Override
        public int hashCode() {
            return Objects.hash(super.hashCode(), this.ifNotExists_);
        }
    }

    private static class AlterTblAddProperty
    extends AlterTblQuery {
        private final String key_;
        private final String value_;

        private AlterTblAddProperty(String dbName, String tblName, String key, String val) {
            super(dbName, tblName, QueryType.ALTER_TABLE_ADD_PROPERTY);
            this.key_ = (String)Preconditions.checkNotNull((Object)key);
            this.value_ = (String)Preconditions.checkNotNull((Object)val);
        }

        static AlterTblAddProperty create(String dbName, String tblName, String key, String val) {
            AlterTblAddProperty query = new AlterTblAddProperty(dbName, tblName, key, val);
            query.addDependentQuery(new CreateDbQuery(dbName, true));
            query.addDependentQuery(new CreateTblQuery(dbName, tblName, RandomHiveQueryRunner.getRandomColName(6) + " string", true));
            return query;
        }

        @Override
        protected String getQuery() {
            return String.format("alter table %s.%s set tblproperties ('%s'='%s')", this.dbName_, this.tblName_, this.key_, this.value_);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            AlterTblAddProperty that = (AlterTblAddProperty)o;
            return this.key_.equals(that.key_) && this.value_.equals(that.value_);
        }

        @Override
        public int hashCode() {
            return Objects.hash(super.hashCode(), this.key_, this.value_);
        }
    }

    private static class AlterTblAddColQuery
    extends AlterTblQuery {
        private final String colSpec_;

        private AlterTblAddColQuery(String dbName, String tblName, String colSpec) {
            super(dbName, tblName, QueryType.ALTER_TABLE_ADD_COL);
            this.colSpec_ = (String)Preconditions.checkNotNull((Object)colSpec);
        }

        static AlterTblAddColQuery create(String dbName, String tblName, String colSpec) {
            AlterTblAddColQuery query = new AlterTblAddColQuery(dbName, tblName, colSpec);
            query.addDependentQuery(new CreateDbQuery(dbName, true));
            query.addDependentQuery(new CreateTblQuery(dbName, tblName, RandomHiveQueryRunner.getRandomColName(6) + " string", true));
            return query;
        }

        @Override
        protected String getQuery() {
            return String.format("alter table %s.%s add columns (%s)", this.dbName_, this.tblName_, this.colSpec_);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            AlterTblAddColQuery that = (AlterTblAddColQuery)o;
            return Objects.equals(this.colSpec_, that.colSpec_);
        }

        @Override
        public int hashCode() {
            return Objects.hash(super.hashCode(), this.colSpec_);
        }
    }

    private static abstract class AlterTblQuery
    extends TestHiveQuery {
        protected final String tblName_;
        protected final String dbName_;
        protected final String partitionSpec_;

        private AlterTblQuery(String dbName, String tblName, QueryType queryType) {
            super(queryType);
            this.dbName_ = (String)Preconditions.checkNotNull((Object)dbName);
            this.tblName_ = (String)Preconditions.checkNotNull((Object)tblName);
            this.partitionSpec_ = null;
            this.addDependentQuery(new CreateDbQuery(this.dbName_, true));
            this.addDependentQuery(new CreateTblQuery(this.dbName_, this.tblName_, RandomHiveQueryRunner.getRandomColName(6) + " string", true));
        }

        private AlterTblQuery(String dbName, String tblName, String partitionSpec, QueryType queryType) {
            super(queryType);
            this.dbName_ = (String)Preconditions.checkNotNull((Object)dbName);
            this.tblName_ = (String)Preconditions.checkNotNull((Object)tblName);
            this.partitionSpec_ = partitionSpec;
        }

        static String getPartitionColFromSpec(String partitionSpec) {
            Preconditions.checkNotNull((Object)partitionSpec);
            if (partitionSpec.contains("=")) {
                String[] parts = partitionSpec.split("=");
                return parts[0].trim() + " int";
            }
            return partitionSpec;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof AlterTblQuery)) {
                return false;
            }
            AlterTblQuery that = (AlterTblQuery)o;
            return this.tblName_.equals(that.tblName_) && this.dbName_.equals(that.dbName_) && Objects.equals(this.partitionSpec_, that.partitionSpec_);
        }

        public int hashCode() {
            return Objects.hash(this.tblName_, this.dbName_, this.partitionSpec_);
        }
    }

    private static class DropTblQuery
    extends TestHiveQuery {
        private final String tblName_;
        private final String dbName_;
        private static final String dummyTblSpecForDepQueries_ = "c1 int";
        private final boolean ifExists_;

        private DropTblQuery(String dbName, String tblName, boolean ifExists) {
            super(QueryType.DROP_TABLE);
            this.dbName_ = (String)Preconditions.checkNotNull((Object)dbName);
            this.tblName_ = (String)Preconditions.checkNotNull((Object)tblName);
            this.ifExists_ = ifExists;
        }

        static DropTblQuery create(String dbName, String tblName) {
            DropTblQuery query = new DropTblQuery(dbName, tblName, false);
            query.addDependentQuery(new CreateDbQuery(dbName, true));
            query.addDependentQuery(new CreateTblQuery(dbName, tblName, dummyTblSpecForDepQueries_, true));
            return query;
        }

        @Override
        protected String getQuery() {
            StringBuilder sb = new StringBuilder("drop table ");
            if (this.ifExists_) {
                sb.append("if exists ");
            }
            sb.append(String.format("%s.%s ", this.dbName_, this.tblName_));
            return sb.toString();
        }

        /*
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null) return false;
            if (this.getClass() != o.getClass()) {
                return false;
            }
            DropTblQuery that = (DropTblQuery)o;
            if (this.ifExists_ != that.ifExists_) return false;
            if (!this.tblName_.equals(that.tblName_)) return false;
            if (!this.dbName_.equals(that.dbName_)) return false;
            if (!dummyTblSpecForDepQueries_.equals(dummyTblSpecForDepQueries_)) return false;
            return true;
        }

        public int hashCode() {
            return Objects.hash(this.tblName_, this.dbName_, dummyTblSpecForDepQueries_, this.ifExists_);
        }
    }

    private static class CreateTblAsSelect
    extends TestHiveQuery {
        private final String tblName_;
        private final String dbName_;
        private final String srcDbName_;
        private final String srcTblName_;
        private final boolean ifNotExists_;

        private CreateTblAsSelect(String dbName, String tblName, String srcDbName, String srcTblName, boolean ifNotExists) {
            super(QueryType.CREATE_TABLE_AS_SELECT);
            this.dbName_ = (String)Preconditions.checkNotNull((Object)dbName);
            this.tblName_ = (String)Preconditions.checkNotNull((Object)tblName);
            this.srcDbName_ = (String)Preconditions.checkNotNull((Object)srcDbName);
            this.srcTblName_ = (String)Preconditions.checkNotNull((Object)srcTblName);
            this.ifNotExists_ = ifNotExists;
        }

        static CreateTblAsSelect create(String dbName, String tblName, String srcDbName, String srcTblName) {
            CreateTblAsSelect query = new CreateTblAsSelect(dbName, tblName, srcDbName, srcTblName, false);
            query.addDependentQuery(new CreateDbQuery(dbName, true));
            query.addDependentQuery(new DropTblQuery(dbName, tblName, true));
            return query;
        }

        @Override
        protected String getQuery() {
            StringBuilder sb = new StringBuilder("create table ");
            if (this.ifNotExists_) {
                sb.append("if not exists ");
            }
            sb.append(String.format("%s.%s ", this.dbName_, this.tblName_));
            sb.append(" like ");
            sb.append(String.format(" %s.%s ", this.srcDbName_, this.srcTblName_));
            return sb.toString();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CreateTblAsSelect that = (CreateTblAsSelect)o;
            return this.ifNotExists_ == that.ifNotExists_ && this.tblName_.equals(that.tblName_) && this.dbName_.equals(that.dbName_) && this.srcDbName_.equals(that.srcDbName_) && this.srcTblName_.equals(that.srcTblName_);
        }

        public int hashCode() {
            return Objects.hash(this.tblName_, this.dbName_, this.srcDbName_, this.srcTblName_, this.ifNotExists_);
        }
    }

    private static class CreatePartitionedTblQuery
    extends CreateTblQuery {
        private final String partitionSpec_;

        private CreatePartitionedTblQuery(String dbName, String tblName, String tblSpec, String partitionSpec, boolean ifNotExists) {
            super(dbName, tblName, tblSpec, ifNotExists);
            this.partitionSpec_ = partitionSpec;
        }

        static CreatePartitionedTblQuery create(String dbName, String tblName, String tblSpec, String partitionSpec) {
            CreatePartitionedTblQuery query = new CreatePartitionedTblQuery(dbName, tblName, tblSpec, partitionSpec, false);
            query.addDependentQuery(new CreateDbQuery(dbName, true));
            query.addDependentQuery(new DropTblQuery(dbName, tblName, true));
            return query;
        }

        @Override
        protected String getQuery() {
            return super.getQuery() + " partitioned by (" + this.partitionSpec_ + ")";
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            CreatePartitionedTblQuery that = (CreatePartitionedTblQuery)o;
            return Objects.equals(this.partitionSpec_, that.partitionSpec_);
        }

        @Override
        public int hashCode() {
            return Objects.hash(super.hashCode(), this.partitionSpec_);
        }
    }

    private static class CreateTblQuery
    extends TestHiveQuery {
        private final String tblName_;
        private final String dbName_;
        private final String tblSpec_;
        private final boolean ifNotExists_;

        private CreateTblQuery(String dbName, String tblName, String tblSpec, boolean ifNotExists) {
            super(QueryType.CREATE_TABLE);
            this.dbName_ = (String)Preconditions.checkNotNull((Object)dbName);
            this.tblName_ = (String)Preconditions.checkNotNull((Object)tblName);
            this.tblSpec_ = (String)Preconditions.checkNotNull((Object)tblSpec);
            this.ifNotExists_ = ifNotExists;
        }

        static CreateTblQuery create(String dbName, String tblName, String tblSpec) {
            CreateTblQuery query = new CreateTblQuery(dbName, tblName, tblSpec, false);
            query.addDependentQuery(new CreateDbQuery(dbName, true));
            query.addDependentQuery(new DropTblQuery(dbName, tblName, true));
            return query;
        }

        @Override
        protected String getQuery() {
            StringBuilder sb = new StringBuilder("create table ");
            if (this.ifNotExists_) {
                sb.append("if not exists ");
            }
            sb.append(String.format("%s.%s ", this.dbName_, this.tblName_));
            sb.append("(" + this.tblSpec_ + ")");
            return sb.toString();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CreateTblQuery that = (CreateTblQuery)o;
            return this.ifNotExists_ == that.ifNotExists_ && this.tblName_.equals(that.tblName_) && this.dbName_.equals(that.dbName_) && this.tblSpec_.equals(that.tblSpec_);
        }

        public int hashCode() {
            return Objects.hash(this.tblName_, this.dbName_, this.tblSpec_, this.ifNotExists_);
        }
    }

    private static class DropDbQuery
    extends TestHiveQuery {
        private final String dbName_;
        private final boolean ifExists_;
        private final boolean cascade_;

        private DropDbQuery(String dbName, boolean ifExists, boolean cascade) {
            super(QueryType.DROP_DB);
            this.dbName_ = (String)Preconditions.checkNotNull((Object)dbName);
            this.ifExists_ = ifExists;
            this.cascade_ = cascade;
        }

        static DropDbQuery create(String dbName) {
            DropDbQuery query = new DropDbQuery(dbName, false, true);
            query.addDependentQuery(new CreateDbQuery(dbName, true));
            return query;
        }

        @Override
        public String getQuery() {
            StringBuilder sb = new StringBuilder("drop database ");
            if (this.ifExists_) {
                sb.append("if exists ");
            }
            sb.append(this.dbName_);
            if (this.cascade_) {
                sb.append(" cascade");
            }
            return sb.toString();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            DropDbQuery that = (DropDbQuery)o;
            return this.ifExists_ == that.ifExists_ && this.cascade_ == that.cascade_ && Objects.equals(this.dbName_, that.dbName_);
        }

        public int hashCode() {
            return Objects.hash(this.dbName_, this.ifExists_, this.cascade_);
        }
    }

    private static class CreateDbQuery
    extends TestHiveQuery {
        private final String dbName_;
        private final boolean ifNotExists_;

        private CreateDbQuery(String dbName, boolean ifNotExists) {
            super(QueryType.CREATE_DB);
            this.dbName_ = (String)Preconditions.checkNotNull((Object)dbName);
            this.ifNotExists_ = ifNotExists;
        }

        static CreateDbQuery create(String dbName_) {
            CreateDbQuery query = new CreateDbQuery(dbName_, false);
            query.addDependentQuery(new DropDbQuery(dbName_, true, true));
            return query;
        }

        @Override
        public String getQuery() {
            StringBuilder sb = new StringBuilder("create database ");
            if (this.ifNotExists_) {
                sb.append("if not exists ");
            }
            sb.append(this.dbName_);
            return sb.toString();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CreateDbQuery that = (CreateDbQuery)o;
            return this.ifNotExists_ == that.ifNotExists_ && Objects.equals(this.dbName_, that.dbName_);
        }

        public int hashCode() {
            return Objects.hash(this.dbName_, this.ifNotExists_);
        }
    }

    private static abstract class TestHiveQuery {
        private List<TestHiveQuery> dependentQueries_ = new ArrayList<TestHiveQuery>();
        protected final QueryType queryType_;

        protected abstract String getQuery();

        TestHiveQuery(QueryType queryType) {
            this.queryType_ = queryType;
        }

        private void runInternal(HiveJdbcClientPool.HiveJdbcClient client) throws SQLException {
            for (TestHiveQuery dependentQuery : this.dependentQueries_) {
                dependentQuery.runInternal(client);
            }
            client.executeSql(this.getQuery());
        }

        public void run(HiveJdbcClientPool pool) throws Exception {
            try (HiveJdbcClientPool.HiveJdbcClient hiveJdbcClient = pool.getClient();){
                this.runInternal(hiveJdbcClient);
            }
            catch (Exception e) {
                LOG.error("Unexpected error received while running the hive query", (Throwable)e);
                throw e;
            }
        }

        protected void addDependentQuery(TestHiveQuery dependentQuery) {
            if (!this.dependentQueries_.contains(dependentQuery)) {
                this.dependentQueries_.add(dependentQuery);
            }
        }

        public String toString() {
            StringBuilder sb = new StringBuilder("\n" + this.getQuery());
            for (int i = 0; i < this.dependentQueries_.size(); ++i) {
                sb.append("\n   " + this.dependentQueries_.get(i).getQuery());
            }
            return sb.toString();
        }
    }

    static enum QueryType {
        CREATE_DB(5),
        DROP_DB(5),
        CREATE_TABLE(10),
        CREATE_TABLE_AS_SELECT(10),
        DROP_TABLE(10),
        ALTER_TABLE_ADD_COL(5),
        ALTER_TABLE_ADD_PROPERTY(15),
        ADD_PARTITION(25),
        DROP_PARTITION(25),
        DYN_PARTITION_INSERT(30),
        DYN_PARTITION_INSERT_OVERWRITE(20),
        INSERT_PARTITION(20),
        INSERT_OVERWRITE_PARTITION(15),
        INSERT_TABLE(25),
        INSERT_OVERWRITE_TABLE(20),
        SET_CONFIG(0);

        private final int weight_;

        private QueryType(int weight) {
            Preconditions.checkArgument((weight >= 0 ? 1 : 0) != 0);
            this.weight_ = weight;
        }
    }
}

