/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.testutil;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.DecimalFormat;
import java.util.ArrayList;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.impala.common.ImpalaRuntimeException;
import org.apache.log4j.Appender;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Layout;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;

public class ImpalaJdbcClient {
    private static final Logger LOG = Logger.getLogger(ImpalaJdbcClient.class);
    private static final String SASL_AUTH_SPEC = ";auth=none";
    private static final String NOSASL_AUTH_SPEC = ";auth=noSasl";
    private static final String LDAP_AUTH_SPEC = ";user=%s;password=%s";
    private static final String HTTP_TRANSPORT_SPEC = ";transportMode=http";
    private static final int HS2_BINARY_PORT = 21050;
    private static final int HS2_HTTP_PORT = 28000;
    private final String driverName_;
    private final String connString_;
    private Connection conn_;
    private Statement stmt_;

    private ImpalaJdbcClient(String driverName, String connString) {
        this.driverName_ = driverName;
        this.connString_ = connString;
    }

    private void validateConnection() throws SQLException {
        if (this.conn_ == null) {
            throw new RuntimeException("Connection not initialized.");
        }
        if (this.conn_.isClosed()) {
            throw new RuntimeException("Connection not open.");
        }
        Preconditions.checkNotNull((Object)this.stmt_);
        if (this.stmt_.isClosed()) {
            this.stmt_ = this.conn_.createStatement();
        }
    }

    public void connect() throws ClassNotFoundException, SQLException, ImpalaRuntimeException {
        LOG.info((Object)("Using JDBC Driver Name: " + this.driverName_));
        LOG.info((Object)("Connecting to: " + this.connString_));
        Class.forName(this.driverName_);
        Driver d = DriverManager.getDriver(this.connString_);
        String driverClassName = d.getClass().getName();
        if (!driverClassName.equals(this.driverName_)) {
            throw new ImpalaRuntimeException(String.format("Specified driver is %s, but for the given connection string %s the selected driver is %s. You might want to specify a different connection string with option -c.", this.driverName_, this.connString_, driverClassName));
        }
        this.conn_ = DriverManager.getConnection(this.connString_);
        this.stmt_ = this.conn_.createStatement();
    }

    public void close() throws SQLException {
        if (this.stmt_ != null) {
            this.stmt_.close();
        }
        if (this.conn_ != null) {
            this.conn_.close();
        }
    }

    public ResultSet execQuery(String query) throws SQLException {
        this.validateConnection();
        LOG.info((Object)("Executing: " + query));
        return this.stmt_.executeQuery(query);
    }

    public void execStatement(String query) throws SQLException {
        this.validateConnection();
        LOG.info((Object)("Executing statement: " + query));
        this.stmt_.execute(query);
    }

    public void changeDatabase(String db_name) throws SQLException {
        this.validateConnection();
        LOG.info((Object)("Using: " + db_name));
        this.stmt_.execute("use " + db_name);
    }

    public Connection getConnection() {
        return this.conn_;
    }

    public Statement getStatement() {
        return this.stmt_;
    }

    public static ImpalaJdbcClient createClient(String driver, String connString) {
        return new ImpalaJdbcClient(driver, connString);
    }

    public static ImpalaJdbcClient createClientUsingHiveJdbcDriver() {
        return ImpalaJdbcClient.createClient("org.apache.hive.jdbc.HiveDriver", ImpalaJdbcClient.getNoAuthConnectionStr("binary"));
    }

    public static ImpalaJdbcClient createClientUsingHiveJdbcDriver(String connString) {
        return ImpalaJdbcClient.createClient("org.apache.hive.jdbc.HiveDriver", connString);
    }

    public static ImpalaJdbcClient createHttpClientUsingHiveJdbcDriver() {
        return ImpalaJdbcClient.createClient("org.apache.hive.jdbc.HiveDriver", ImpalaJdbcClient.getNoAuthConnectionStr("http"));
    }

    public static String getNoAuthConnectionStr(String connType) {
        return ImpalaJdbcClient.getConnectionStr(connType, NOSASL_AUTH_SPEC);
    }

    public static String getLdapConnectionStr(String connType, String username, String password) {
        return ImpalaJdbcClient.getConnectionStr(connType, String.format(LDAP_AUTH_SPEC, username, password));
    }

    private static String getConnectionStr(String connType, String authStr) {
        String connString = "jdbc:hive2://localhost:%d/default" + authStr;
        if (connType == "binary") {
            return String.format(connString, 21050, "default");
        }
        Preconditions.checkState((connType == "http" ? 1 : 0) != 0);
        return String.format(connString + HTTP_TRANSPORT_SPEC, 28000, "default");
    }

    private static ClientExecOptions parseOptions(String[] args) throws ParseException {
        String query;
        Options options = new Options();
        options.addOption("i", true, "host:port of target machine impalad is listening on");
        options.addOption("c", true, "Full connection string to use. Overrides host/port value");
        options.addOption("t", true, "SASL/NOSASL, whether to use SASL transport or not");
        options.addOption("q", true, "Query String");
        options.addOption("d", true, "Driver name, default: org.apache.hive.jdbc.HiveDriver");
        options.addOption("help", false, "Help");
        BasicParser optionParser = new BasicParser();
        CommandLine cmdArgs = optionParser.parse(options, args);
        String transportOption = cmdArgs.getOptionValue("t");
        if (transportOption == null) {
            LOG.error((Object)"Must specify '-t' option, whether to use SASL transport or not.");
            LOG.error((Object)"Using the wrong type of transport will cause the program to hang.");
            LOG.error((Object)("Usage: " + options.toString()));
            System.exit(1);
        }
        if (!transportOption.equalsIgnoreCase("SASL") && !transportOption.equalsIgnoreCase("NOSASL")) {
            LOG.error((Object)("Invalid argument " + transportOption + " to '-t' option."));
            LOG.error((Object)("Usage: " + options.toString()));
            System.exit(1);
        }
        boolean useSasl = transportOption.equalsIgnoreCase("SASL");
        String connStr = cmdArgs.getOptionValue("c", null);
        if (connStr == null) {
            String hostPort = cmdArgs.getOptionValue("i", "localhost:21050");
            connStr = "jdbc:hive2://" + hostPort + "/";
            connStr = useSasl ? connStr + SASL_AUTH_SPEC : connStr + NOSASL_AUTH_SPEC;
        }
        if ((query = cmdArgs.getOptionValue("q")) == null) {
            LOG.error((Object)"Must specify a query to execute.");
            LOG.error((Object)("Usage: " + options.toString()));
            System.exit(1);
        }
        String driver = cmdArgs.getOptionValue("d", "org.apache.hive.jdbc.HiveDriver");
        return new ClientExecOptions(connStr, query, driver);
    }

    private static String formatColumnValue(String colVal, String columnType) throws NumberFormatException {
        columnType = columnType.toLowerCase();
        if (colVal == null) {
            return columnType.equals("string") ? "'NULL'" : "NULL";
        }
        if (columnType.equals("string")) {
            return "'" + colVal + "'";
        }
        if (columnType.equals("float") || columnType.equals("double")) {
            DecimalFormat df = new DecimalFormat("#.##################################");
            double doubleVal = Double.parseDouble(colVal);
            return df.format(doubleVal);
        }
        return colVal;
    }

    private static void execQuery(ImpalaJdbcClient client, String queryString) throws SQLException, NumberFormatException {
        String[] queries;
        for (String query : queries = queryString.trim().split(";")) {
            if ((query = query.trim()).indexOf(" ") > -1 && query.substring(0, query.indexOf(" ")).equalsIgnoreCase("use")) {
                String[] split_query = query.split(" ");
                String db_name = split_query[split_query.length - 1];
                client.changeDatabase(db_name);
                client.getStatement().close();
                continue;
            }
            long startTime = System.currentTimeMillis();
            ResultSet res = client.execQuery(query);
            ResultSetMetaData meta = res.getMetaData();
            ArrayList arrayList = Lists.newArrayList();
            LOG.info((Object)"----[START]----");
            int rowCount = 0;
            while (res.next()) {
                arrayList.clear();
                for (int i = 1; i <= meta.getColumnCount(); ++i) {
                    String colVal = ImpalaJdbcClient.formatColumnValue(res.getString(i), meta.getColumnTypeName(i));
                    arrayList.add(colVal);
                }
                LOG.info((Object)Joiner.on((String)",").join((Iterable)arrayList));
                ++rowCount;
            }
            LOG.info((Object)"----[END]----");
            long endTime = System.currentTimeMillis();
            float seconds = (float)(endTime - startTime) / 1000.0f;
            LOG.info((Object)("Returned " + rowCount + " row(s) in " + seconds + "s"));
            client.getStatement().close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] args) throws SQLException, ClassNotFoundException, ParseException, ImpalaRuntimeException {
        PatternLayout layout = new PatternLayout("%m%n");
        ConsoleAppender consoleAppender = new ConsoleAppender((Layout)layout);
        LOG.addAppender((Appender)consoleAppender);
        LOG.setAdditivity(false);
        ClientExecOptions execOptions = ImpalaJdbcClient.parseOptions(args);
        try (ImpalaJdbcClient client = ImpalaJdbcClient.createClient(execOptions.getDriver(), execOptions.getConnStr());){
            client.connect();
            ImpalaJdbcClient.execQuery(client, execOptions.getQuery());
        }
    }

    private static class ClientExecOptions {
        private final String connStr;
        private final String query;
        private final String driver;

        public ClientExecOptions(String connStr, String query, String driver) {
            this.connStr = connStr;
            this.query = query;
            this.driver = driver;
        }

        public String getQuery() {
            return this.query;
        }

        public String getConnStr() {
            return this.connStr;
        }

        public String getDriver() {
            return this.driver;
        }
    }
}

