/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.analysis;

import com.google.common.base.Preconditions;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.impala.analysis.AnalysisContext;
import org.apache.impala.analysis.InsertStmt;
import org.apache.impala.analysis.JoinOperator;
import org.apache.impala.analysis.ParseNode;
import org.apache.impala.analysis.QueryStmt;
import org.apache.impala.analysis.TableName;
import org.apache.impala.analysis.ToSqlOptions;
import org.apache.impala.analysis.ToSqlUtils;
import org.apache.impala.authorization.Privilege;
import org.apache.impala.common.AnalysisException;
import org.apache.impala.common.FrontendTestBase;
import org.apache.impala.service.BackendConfig;
import org.junit.Assert;
import org.junit.Test;

public class ToSqlTest
extends FrontendTestBase {
    private static final String[] joinConditions_ = new String[]{"USING (id)", "ON (a.id = b.id)"};
    private static final String[] leftSemiJoinTypes_ = new String[]{"LEFT SEMI JOIN", "LEFT ANTI JOIN"};
    private static final String[] rightSemiJoinTypes_ = new String[]{"RIGHT SEMI JOIN", "RIGHT ANTI JOIN"};
    private static final String[] joinTypes_ = new String[JoinOperator.values().length - 3];
    private static final String[] nonSemiJoinTypes_;

    private void testToSql(String query) {
        this.testToSql(query, query);
    }

    private void testToSql(AnalysisContext ctx, String query) {
        this.testToSql(ctx, query, query, false);
    }

    private void testToSql(String query, String expected) {
        this.testToSql(query, System.getProperty("user.name"), expected);
    }

    private void testToSql(String query, String expected, ToSqlOptions options) {
        String defaultDb = System.getProperty("user.name");
        this.testToSql(this.createAnalysisCtx(defaultDb), query, defaultDb, expected, false, options);
    }

    private void testToSql(AnalysisContext ctx, String query, String expected, boolean ignoreWhiteSpace) {
        this.testToSql(ctx, query, System.getProperty("user.name"), expected, ignoreWhiteSpace);
    }

    private void testToSql(String query, String defaultDb, String expected) {
        this.testToSql(query, defaultDb, expected, false);
    }

    private void testToSql(AnalysisContext ctx, String query, String defaultDb, String expected, boolean ignoreWhiteSpace) {
        this.testToSql(ctx, query, defaultDb, expected, ignoreWhiteSpace, ToSqlOptions.DEFAULT);
    }

    private void testToSql(String query, String defaultDb, String expected, boolean ignoreWhitespace) {
        this.testToSql(this.createAnalysisCtx(defaultDb), query, defaultDb, expected, ignoreWhitespace, ToSqlOptions.DEFAULT);
    }

    private void testToSql(AnalysisContext ctx, String query, String defaultDb, String expected, boolean ignoreWhitespace, ToSqlOptions options) {
        String actual = null;
        try {
            ParseNode node = this.AnalyzesOk(query, ctx);
            actual = node instanceof QueryStmt && !options.showRewritten() ? ((QueryStmt)node).getOrigSqlString() : node.toSql(options);
            if (ignoreWhitespace) {
                actual = actual.replace('\n', ' ').replaceAll(" +", " ").trim();
            }
            Assert.assertEquals((Object)expected, (Object)actual);
        }
        catch (Exception e) {
            e.printStackTrace();
            Assert.fail((String)("Failed to analyze query: " + query + "\n" + e.getMessage()));
        }
        this.AnalyzesOk(actual, ctx);
    }

    private void runTestTemplate(String sql, String expectedSql, String[] ... testDims) {
        Object[] testVector = new Object[testDims.length];
        this.runTestTemplate(sql, expectedSql, 0, testVector, testDims);
    }

    private void runTestTemplate(String sql, String expectedSql, int dim, Object[] testVector, String[] ... testDims) {
        if (dim >= testDims.length) {
            this.testToSql(String.format(sql, testVector), String.format(expectedSql, testVector));
            return;
        }
        for (String s : testDims[dim]) {
            testVector[dim] = s;
            this.runTestTemplate(sql, expectedSql, dim + 1, testVector, testDims);
        }
    }

    private void TblsTestToSql(String query, TableName tbl, String expectedSql) {
        Preconditions.checkState((boolean)tbl.isFullyQualified());
        Preconditions.checkState((boolean)query.contains("$TBL"));
        String uqQuery = query.replace("$TBL", tbl.getTbl());
        this.testToSql(uqQuery, tbl.getDb(), expectedSql);
        this.AnalyzesOk(uqQuery, this.createAnalysisCtx(tbl.getDb()));
        String fqQuery = query.replace("$TBL", tbl.toString());
        this.testToSql(fqQuery, expectedSql);
    }

    @Test
    public void selectListTest() {
        this.testToSql("select 1234, 1234.0, 1234.0 + 1, 1234.0 + 1.0, 1 + 1, \"abc\" from functional.alltypes", "SELECT 1234, 1234.0, 1234.0 + 1, 1234.0 + 1.0, 1 + 1, 'abc' FROM functional.alltypes");
        this.testToSql("select 1234 i, 1234.0 as j, (1234.0 + 1) k, (1234.0 + 1.0) as l from functional.alltypes", "SELECT 1234 i, 1234.0 j, (1234.0 + 1) k, (1234.0 + 1.0) l FROM functional.alltypes");
        this.testToSql("select 1234 i, 1234.0 as j, (1234.0 + 1) k, (1234.0 + 1.0) as l", "SELECT 1234 i, 1234.0 j, (1234.0 + 1) k, (1234.0 + 1.0) l");
        this.testToSql("select null, 1234 < 5678, 1234.0 < 5678.0, 1234 < null from functional.alltypes", "SELECT NULL, 1234 < 5678, 1234.0 < 5678.0, 1234 < NULL FROM functional.alltypes");
    }

    private boolean isCollectionTableRef(String tableName) {
        return tableName.split("\\.").length > 0;
    }

    private void testAllTableAliases(String[] tables, String[] columns) throws AnalysisException {
        for (String tbl : tables) {
            TableName tblName = new TableName("functional", tbl);
            String uqAlias = tbl.substring(tbl.lastIndexOf(".") + 1);
            String fqAlias = "functional." + tbl;
            boolean isCollectionTblRef = this.isCollectionTableRef(tbl);
            for (String col : columns) {
                String quotedCol = ToSqlUtils.identSql((String)col);
                this.TblsTestToSql(String.format("select %s from $TBL", col), tblName, String.format("SELECT %s FROM %s", quotedCol, fqAlias));
                this.TblsTestToSql(String.format("select %s.%s from $TBL", uqAlias, col), tblName, String.format("SELECT %s.%s FROM %s", uqAlias, quotedCol, fqAlias));
                if (!isCollectionTblRef) {
                    this.TblsTestToSql(String.format("select %s.%s from $TBL", fqAlias, col), tblName, String.format("SELECT %s.%s FROM %s", fqAlias, col, fqAlias));
                }
                this.TblsTestToSql(String.format("select %s from $TBL a", col), tblName, String.format("SELECT %s FROM %s a", quotedCol, fqAlias));
                this.TblsTestToSql(String.format("select a.%s from $TBL a", col), tblName, String.format("SELECT a.%s FROM %s a", quotedCol, fqAlias));
            }
        }
        for (String t1 : tables) {
            for (String t2 : tables) {
                if (t1 == t2 || this.isCollectionTableRef(t1) && this.isCollectionTableRef(t2)) continue;
                for (String col : columns) {
                    this.testToSql(String.format("select functional.%s.%s, functional.%s.%s from functional.%s, functional.%s", t1, col, t2, col, t1, t2), String.format("SELECT functional.%s.%s, functional.%s.%s FROM functional.%s, functional.%s", t1, col, t2, col, t1, t2));
                }
            }
        }
    }

    private void testChildTableRefs(String childTable, String childColumn) {
        TableName tbl = new TableName("functional", "allcomplextypes");
        childColumn = ToSqlUtils.identSql((String)childColumn);
        this.TblsTestToSql(String.format("select %s from $TBL, allcomplextypes.%s", childColumn, childTable), tbl, String.format("SELECT %s FROM %s, functional.allcomplextypes.%s", childColumn, tbl.toSql(), childTable));
        this.TblsTestToSql(String.format("select %s from $TBL, functional.allcomplextypes.%s", childColumn, childTable), tbl, String.format("SELECT %s FROM %s, functional.allcomplextypes.%s", childColumn, tbl.toSql(), childTable));
        this.TblsTestToSql(String.format("select %s from $TBL a, a.%s", childColumn, childTable), tbl, String.format("SELECT %s FROM %s a, a.%s", childColumn, tbl.toSql(), childTable));
        this.TblsTestToSql(String.format("select b.%s from $TBL a, a.%s b, a.int_map_col c", childColumn, childTable), tbl, String.format("SELECT b.%s FROM %s a, a.%s b, a.int_map_col c", childColumn, tbl.toSql(), childTable));
        this.TblsTestToSql(String.format("select c.%s from $TBL a, a.int_array_col b, a.%s c", childColumn, childTable), tbl, String.format("SELECT c.%s FROM %s a, a.int_array_col b, a.%s c", childColumn, tbl.toSql(), childTable));
        for (String joinType : joinTypes_) {
            this.TblsTestToSql(String.format("select 1 from $TBL %s allcomplextypes.%s", joinType, childTable), tbl, String.format("SELECT 1 FROM %s %s functional.allcomplextypes.%s", tbl.toSql(), joinType, childTable));
            this.TblsTestToSql(String.format("select 1 from $TBL a %s a.%s", joinType, childTable), tbl, String.format("SELECT 1 FROM %s a %s a.%s", tbl.toSql(), joinType, childTable));
        }
        this.TblsTestToSql(String.format("select %s from $TBL a, functional.allcomplextypes.%s", childColumn, childTable), tbl, String.format("SELECT %s FROM %s a, functional.allcomplextypes.%s", childColumn, tbl.toSql(), childTable));
        this.TblsTestToSql(String.format("select %s from $TBL.%s, functional.allcomplextypes", childColumn, childTable), tbl, String.format("SELECT %s FROM %s.%s, functional.allcomplextypes", childColumn, tbl.toSql(), childTable));
    }

    @Test
    public void TestCreateTable() throws AnalysisException {
        this.testToSql("create table p (a int) partitioned by (day string) sort by (a) comment 'This is a test'", "default", "CREATE TABLE default.p ( a INT ) PARTITIONED BY ( day STRING ) SORT BY LEXICAL ( a ) COMMENT 'This is a test' STORED AS TEXTFILE", true);
        this.testToSql("create table p (a int, b int) partitioned by (day string) sort by (a ,b) ", "default", "CREATE TABLE default.p ( a INT, b INT ) PARTITIONED BY ( day STRING ) SORT BY LEXICAL ( a, b ) STORED AS TEXTFILE", true);
        this.testToSql("create table p (a int) partitioned by (day string) sort by lexical (a) comment 'This is a test'", "default", "CREATE TABLE default.p ( a INT ) PARTITIONED BY ( day STRING ) SORT BY LEXICAL ( a ) COMMENT 'This is a test' STORED AS TEXTFILE", true);
        this.testToSql("create table p (a int, b int) partitioned by (day string) sort by lexical (a, b)", "default", "CREATE TABLE default.p ( a INT, b INT ) PARTITIONED BY ( day STRING ) SORT BY LEXICAL ( a, b ) STORED AS TEXTFILE", true);
        this.testToSql("create table p (a int, b int) partitioned by (day string) sort by zorder(a ,b) ", "default", "CREATE TABLE default.p ( a INT, b INT ) PARTITIONED BY ( day STRING ) SORT BY ZORDER ( a, b ) STORED AS TEXTFILE", true);
        String kuduMasters = "127.0.0.1";
        BackendConfig.INSTANCE.getBackendCfg().setKudu_master_hosts(kuduMasters);
        this.testToSql(String.format("create table p (a bigint primary key, b timestamp default '1987-05-19') partition by hash(a) partitions 3 stored as kudu tblproperties ('kudu.master_addresses'='%s')", kuduMasters), "default", String.format("CREATE TABLE default.p ( a BIGINT PRIMARY KEY, b TIMESTAMP DEFAULT '1987-05-19' ) PARTITION BY HASH (a) PARTITIONS 3 STORED AS KUDU TBLPROPERTIES ('kudu.master_addresses'='%s', 'storage_handler'='org.apache.hadoop.hive.kudu.KuduStorageHandler')", kuduMasters), true);
        this.testToSql("create table pk(id int, year string, primary key (id, year))", "default", "CREATE TABLE default.pk ( id INT, year STRING, PRIMARY KEY (id, year) ) STORED AS TEXTFILE", true);
        this.addTestDb("test_pk_fk", "Test DB for PK/FK tests");
        AnalysisContext ctx = this.createAnalysisCtx("test_pk_fk");
        this.testToSql(ctx, "create table fk(seq int, id int, year string, a int, FOREIGN KEY(id, year) REFERENCES functional.parent_table(id, year), FOREIGN KEY (a) REFERENCES functional.parent_table_2(a))", "CREATE TABLE test_pk_fk.fk ( seq INT, id INT, year STRING, a INT, FOREIGN KEY(id, year) REFERENCES functional.parent_table(id, year), FOREIGN KEY(a) REFERENCES functional.parent_table_2(a) ) STORED AS TEXTFILE", true);
        this.testToSql("CREATE TABLE test_create_managed_paimon_table (user_id BIGINT, item_id BIGINT, behavior STRING) STORED AS PAIMON", "default", "CREATE TABLE default.test_create_managed_paimon_table ( user_id BIGINT, item_id BIGINT, behavior STRING ) STORED AS PAIMON TBLPROPERTIES ('deletion-vectors.enabled'='true', 'storage_handler'='org.apache.paimon.hive.PaimonStorageHandler')", true);
        this.testToSql("CREATE TABLE  test_create_managed_part_paimon_table ( user_id BIGINT, item_id BIGINT, behavior STRING) PARTITIONED BY (dt STRING, hh STRING ) STORED AS PAIMON", "default", "CREATE TABLE default.test_create_managed_part_paimon_table ( user_id BIGINT, item_id BIGINT, behavior STRING ) PARTITIONED BY ( dt STRING, hh STRING ) STORED AS PAIMON TBLPROPERTIES ('deletion-vectors.enabled'='true', 'storage_handler'='org.apache.paimon.hive.PaimonStorageHandler')", true);
        this.testToSql("CREATE TABLE test_create_managed_part_pk_paimon_table (user_id BIGINT, item_id BIGINT, behavior STRING) PARTITIONED BY (dt STRING, hh STRING) STORED AS PAIMON TBLPROPERTIES ('primary-key'='user_id')", "default", "CREATE TABLE default.test_create_managed_part_pk_paimon_table ( user_id BIGINT, item_id BIGINT, behavior STRING ) PARTITIONED BY ( dt STRING, hh STRING ) STORED AS PAIMON TBLPROPERTIES ('deletion-vectors.enabled'='true', 'primary-key'='user_id', 'storage_handler'='org.apache.paimon.hive.PaimonStorageHandler')", true);
        this.testToSql("CREATE TABLE test_create_managed_bucket_paimon_table (user_id BIGINT, item_id BIGINT, behavior STRING) STORED AS PAIMON TBLPROPERTIES ('bucket' = '4', 'bucket-key'='behavior')", "default", "CREATE TABLE default.test_create_managed_bucket_paimon_table ( user_id BIGINT, item_id BIGINT, behavior STRING ) STORED AS PAIMON TBLPROPERTIES ('bucket'='4', 'bucket-key'='behavior', 'deletion-vectors.enabled'='true', 'storage_handler'='org.apache.paimon.hive.PaimonStorageHandler')", true);
    }

    @Test
    public void TestCreateTableAsSelect() throws AnalysisException {
        this.testToSql("create table p partitioned by (int_col) as select double_col, int_col from functional.alltypes", "default", "CREATE TABLE default.p PARTITIONED BY ( int_col ) STORED AS TEXTFILE AS SELECT double_col, int_col FROM functional.alltypes", true);
        this.testToSql("create table p partitioned by (int_col) comment 'This is a test' as select double_col, int_col from functional.alltypes", "default", "CREATE TABLE default.p PARTITIONED BY ( int_col ) COMMENT 'This is a test' STORED AS TEXTFILE AS SELECT double_col, int_col FROM functional.alltypes", true);
        this.testToSql("create table p partitioned by (int_col) sort by (string_col) as select double_col, string_col, int_col from functional.alltypes", "default", "CREATE TABLE default.p PARTITIONED BY ( int_col ) SORT BY LEXICAL ( string_col ) STORED AS TEXTFILE AS SELECT double_col, string_col, int_col FROM functional.alltypes", true);
        this.testToSql("create table p partitioned by (int_col) sort by lexical (string_col) as select double_col, string_col, int_col from functional.alltypes", "default", "CREATE TABLE default.p PARTITIONED BY ( int_col ) SORT BY LEXICAL ( string_col ) STORED AS TEXTFILE AS SELECT double_col, string_col, int_col FROM functional.alltypes", true);
        this.testToSql("create table p partitioned by (string_col) sort by zorder (int_col, bool_col) as select int_col, bool_col, string_col from functional.alltypes", "default", "CREATE TABLE default.p PARTITIONED BY ( string_col ) SORT BY ZORDER ( int_col, bool_col ) STORED AS TEXTFILE AS SELECT int_col, bool_col, string_col FROM functional.alltypes", true);
        String kuduMasters = "127.0.0.1";
        BackendConfig.INSTANCE.getBackendCfg().setKudu_master_hosts(kuduMasters);
        this.testToSql(String.format("create table p primary key (a,b) partition by hash(a) partitions 3, range (b) (partition value = 1) stored as kudu tblproperties ('kudu.master_addresses'='%s') as select int_col a, bigint_col b from functional.alltypes", kuduMasters), "default", String.format("CREATE TABLE default.p PRIMARY KEY (a, b) PARTITION BY HASH (a) PARTITIONS 3, RANGE (b) (PARTITION VALUE = 1) STORED AS KUDU TBLPROPERTIES ('kudu.master_addresses'='%s', 'storage_handler'='org.apache.hadoop.hive.kudu.KuduStorageHandler') AS SELECT int_col a, bigint_col b FROM functional.alltypes", kuduMasters), true);
    }

    @Test
    public void TestCreateTableLike() throws AnalysisException {
        this.testToSql("create table p like functional.alltypes", "default", "CREATE TABLE p LIKE functional.alltypes");
        this.testToSql("create table p sort by (id) like functional.alltypes", "default", "CREATE TABLE p SORT BY LEXICAL (id) LIKE functional.alltypes");
        this.testToSql("create table p sort by LEXICAL (id) like functional.alltypes", "default", "CREATE TABLE p SORT BY LEXICAL (id) LIKE functional.alltypes");
        this.testToSql("create table p sort by zorder (bool_col, int_col) like functional.alltypes", "default", "CREATE TABLE p SORT BY ZORDER (bool_col,int_col) LIKE functional.alltypes");
    }

    @Test
    public void TestCreateTableLikeFile() throws AnalysisException {
        this.testToSql("create table if not exists p like parquet '/test-warehouse/schemas/alltypestiny.parquet'", "default", "CREATE TABLE IF NOT EXISTS default.p LIKE PARQUET 'hdfs://localhost:20500/test-warehouse/schemas/alltypestiny.parquet' STORED AS TEXTFILE", true);
        this.testToSql("create table if not exists p like parquet '/test-warehouse/schemas/alltypestiny.parquet' sort by (int_col, id)", "default", "CREATE TABLE IF NOT EXISTS default.p LIKE PARQUET 'hdfs://localhost:20500/test-warehouse/schemas/alltypestiny.parquet' SORT BY LEXICAL ( int_col, id ) STORED AS TEXTFILE", true);
        this.testToSql("create table if not exists p like parquet '/test-warehouse/schemas/alltypestiny.parquet' sort by lexical (int_col, id)", "default", "CREATE TABLE IF NOT EXISTS default.p LIKE PARQUET 'hdfs://localhost:20500/test-warehouse/schemas/alltypestiny.parquet' SORT BY LEXICAL ( int_col, id ) STORED AS TEXTFILE", true);
        this.testToSql("create table if not exists p like parquet '/test-warehouse/schemas/alltypestiny.parquet' sort by zorder (int_col, id)", "default", "CREATE TABLE IF NOT EXISTS default.p LIKE PARQUET 'hdfs://localhost:20500/test-warehouse/schemas/alltypestiny.parquet' SORT BY ZORDER ( int_col, id ) STORED AS TEXTFILE", true);
    }

    @Test
    public void TestCreateBucketTable() throws AnalysisException {
        this.testToSql("create table bucketed_table ( a int ) CLUSTERED BY ( a ) into 24 buckets", "default", "CREATE TABLE default.bucketed_table ( a INT ) CLUSTERED BY ( a ) INTO 24 BUCKETS STORED AS TEXTFILE", true);
        this.testToSql("create table bucketed_table1 ( a int ) partitioned by ( dt string ) CLUSTERED BY ( a ) into 24 buckets", "default", "CREATE TABLE default.bucketed_table1 ( a INT ) PARTITIONED BY ( dt STRING ) CLUSTERED BY ( a ) INTO 24 BUCKETS STORED AS TEXTFILE", true);
        this.testToSql("create table bucketed_table2 ( a int, s string ) partitioned by ( dt string ) CLUSTERED BY ( a ) sort by (s) into 24 buckets", "default", "CREATE TABLE default.bucketed_table2 ( a INT, s STRING ) PARTITIONED BY ( dt STRING ) CLUSTERED BY ( a ) SORT BY LEXICAL ( s ) INTO 24 BUCKETS STORED AS TEXTFILE", true);
    }

    @Test
    public void TestCreateView() throws AnalysisException {
        this.testToSql("create view foo_new as select int_col, string_col from functional.alltypes", "default", "CREATE VIEW foo_new AS SELECT int_col, string_col FROM functional.alltypes");
        this.testToSql("create view if not exists foo as select * from functional.alltypes", "default", "CREATE VIEW IF NOT EXISTS foo AS SELECT * FROM functional.alltypes");
        this.testToSql("create view functional.foo (a, b) as select int_col x, double_col y from functional.alltypes", "default", "CREATE VIEW functional.foo(a, b) AS SELECT int_col x, double_col y FROM functional.alltypes");
        this.testToSql("create view foo (aaa, bbb) as select * from functional.complex_view", "default", "CREATE VIEW foo(aaa, bbb) AS SELECT * FROM functional.complex_view");
        this.testToSql("create view foo as select trim('abc'), 17 * 7", "default", "CREATE VIEW foo AS SELECT trim('abc'), 17 * 7");
        this.testToSql("create view foo (cnt) as select count(distinct x.int_col) from functional.alltypessmall x inner join functional.alltypessmall y on (x.id = y.id) group by x.bigint_col", "default", "CREATE VIEW foo(cnt) AS SELECT count(DISTINCT x.int_col) FROM functional.alltypessmall x INNER JOIN functional.alltypessmall y ON (x.id = y.id) GROUP BY x.bigint_col");
        this.testToSql("create view foo (a, b) as values(1, 'a'), (2, 'b')", "default", "CREATE VIEW foo(a, b) AS VALUES((1, 'a'), (2, 'b'))");
        this.testToSql("create view foo (a, b) as select 1, 'a' union all select 2, 'b'", "default", "CREATE VIEW foo(a, b) AS SELECT 1, 'a' UNION ALL SELECT 2, 'b'");
        this.testToSql("create view test_view_with_subquery as select * from functional.alltypestiny t where exists (select * from functional.alltypessmall s where s.id = t.id)", "default", "CREATE VIEW test_view_with_subquery AS SELECT * FROM functional.alltypestiny t WHERE EXISTS (SELECT * FROM functional.alltypessmall s WHERE s.id = t.id)");
        this.testToSql("create view test_properties tblproperties ('a'='aa', 'b'='bb') as select int_col, string_col from functional.alltypes", "default", "CREATE VIEW test_properties TBLPROPERTIES ('a'='aa', 'b'='bb') AS SELECT int_col, string_col FROM functional.alltypes");
    }

    @Test
    public void TestAlterView() throws AnalysisException {
        this.testToSql("alter view functional.alltypes_view as select * from functional.alltypesagg", "default", "ALTER VIEW functional.alltypes_view AS SELECT * FROM functional.alltypesagg");
        this.testToSql("alter view functional.alltypes_view (a, b) as select int_col, string_col from functional.alltypes", "default", "ALTER VIEW functional.alltypes_view(a, b) AS SELECT int_col, string_col FROM functional.alltypes");
        this.testToSql("alter view functional.alltypes_view (a, b) as select int_col x, string_col y from functional.alltypes", "default", "ALTER VIEW functional.alltypes_view(a, b) AS SELECT int_col x, string_col y FROM functional.alltypes");
        this.testToSql("alter view functional.alltypes_view as select trim('abc'), 17 * 7", "default", "ALTER VIEW functional.alltypes_view AS SELECT trim('abc'), 17 * 7");
        this.testToSql("alter view functional.alltypes_view (aaa, bbb) as select * from functional.complex_view", "default", "ALTER VIEW functional.alltypes_view(aaa, bbb) AS SELECT * FROM functional.complex_view");
        this.testToSql("alter view functional.complex_view (abc, xyz) as select year, month from functional.alltypes_view", "default", "ALTER VIEW functional.complex_view(abc, xyz) AS SELECT `year`, `month` FROM functional.alltypes_view");
        this.testToSql("alter view functional.alltypes_view (cnt) as select count(distinct x.int_col) from functional.alltypessmall x inner join functional.alltypessmall y on (x.id = y.id) group by x.bigint_col", "default", "ALTER VIEW functional.alltypes_view(cnt) AS SELECT count(DISTINCT x.int_col) FROM functional.alltypessmall x INNER JOIN functional.alltypessmall y ON (x.id = y.id) GROUP BY x.bigint_col");
        this.testToSql("alter view functional.alltypes_view set tblproperties ('a'='b')", "functional", "ALTER VIEW functional.alltypes_view SET TBLPROPERTIES ('a'='b')");
        this.testToSql("alter view functional.alltypes_view unset tblproperties ('a', 'c')", "functional", "ALTER VIEW functional.alltypes_view UNSET TBLPROPERTIES ('a', 'c')");
    }

    @Test
    public void TestTableAliases() throws AnalysisException {
        String[] tables = new String[]{"alltypes", "alltypes_view"};
        String[] columns = new String[]{"int_col", "*"};
        this.testAllTableAliases(tables, columns);
        this.testToSql("select * from functional.alltypes cross join functional_parquet.alltypes", "SELECT * FROM functional.alltypes CROSS JOIN functional_parquet.alltypes");
    }

    @Test
    public void TestAlterTableExecute() throws AnalysisException {
        this.testToSql("alter table iceberg_partitioned execute rollback('2022-01-04 10:00:00')", "functional_parquet", "ALTER TABLE functional_parquet.iceberg_partitioned EXECUTE rollback('2022-01-04 10:00:00')");
        this.testToSql("alter table iceberg_partitioned execute expire_snapshots('2022-01-04 10:00:00')", "functional_parquet", "ALTER TABLE functional_parquet.iceberg_partitioned EXECUTE expire_snapshots('2022-01-04 10:00:00')");
        this.testToSql("alter table iceberg_partitioned execute remove_orphan_files('2022-01-04 10:00:00')", "functional_parquet", "ALTER TABLE functional_parquet.iceberg_partitioned EXECUTE remove_orphan_files('2022-01-04 10:00:00')");
    }

    @Test
    public void TestStructFields() throws AnalysisException {
        String[] tables = new String[]{"allcomplextypes"};
        String[] columns = new String[]{"id", "int_struct_col.f1", "nested_struct_col.f2.f12.f21"};
        this.testAllTableAliases(tables, columns);
    }

    @Test
    public void TestCollectionTableRefs() throws AnalysisException {
        this.testAllTableAliases(new String[]{"allcomplextypes.int_array_col"}, new String[]{"item", "*"});
        this.testAllTableAliases(new String[]{"allcomplextypes.struct_array_col"}, new String[]{"f1", "f2", "*"});
        this.testAllTableAliases(new String[]{"allcomplextypes.int_map_col"}, new String[]{"key", "value", "*"});
        this.testAllTableAliases(new String[]{"allcomplextypes.struct_map_col"}, new String[]{"key", "f1", "f2", "*"});
        this.testAllTableAliases(new String[]{"allcomplextypes.complex_nested_struct_col.f2.f12"}, new String[]{"key", "f21", "*"});
        this.testChildTableRefs("int_array_col", "item");
        this.testChildTableRefs("int_map_col", "key");
        this.testChildTableRefs("complex_nested_struct_col.f2.f12", "f21");
    }

    @Test
    public void TestIdentifierQuoting() {
        this.testToSql("select 1 as `abc`, 2.0 as 'xyz'", "SELECT 1 abc, 2.0 xyz");
        this.testToSql("select 1 as _c0, 2.0 as $abc", "SELECT 1 `_c0`, 2.0 `$abc`");
        this.testToSql("select 1 as `???`, 2.0 as '^^^'", "SELECT 1 `???`, 2.0 `^^^`");
        this.testToSql("select `end`.`alter`, `end`.`table` from (select 1 as `alter`, 2 as `table`) `end`", "SELECT `end`.`alter`, `end`.`table` FROM (SELECT 1 `alter`, 2 `table`) `end`");
        this.testToSql("select a from (select 1 as a) as _t", "SELECT a FROM (SELECT 1 a) `_t`");
        this.testToSql("with _t as (select 1 as a) select * from _t", "WITH `_t` AS (SELECT 1 a) SELECT * FROM `_t`");
        this.testToSql("select `1 + 10`, `trim('abc')` from (select 1 + 10, trim('abc')) as t", "SELECT `1 + 10`, `trim('abc')` FROM (SELECT 1 + 10, trim('abc')) t");
    }

    @Test
    public void normalizeStringLiteralTest() {
        this.testToSql("select \"'\"", "SELECT '\\''");
        this.testToSql("select \"\\'\"", "SELECT '\\''");
        this.testToSql("select \"\\\\'\"", "SELECT '\\\\\\''");
        this.testToSql("select '\"'", "SELECT '\"'");
        this.testToSql("select '\\\"'", "SELECT '\"'");
        this.testToSql("select '\\''", "SELECT '\\''");
        this.testToSql("select '\\\\\\''", "SELECT '\\\\\\''");
        this.testToSql("select regexp_replace(string_col, \"\\\\'\", \"'\") from functional.alltypes", "SELECT regexp_replace(string_col, '\\\\\\'', '\\'') FROM functional.alltypes");
        this.testToSql("select * from functional.alltypes where '123' = \"123\"", "SELECT * FROM functional.alltypes WHERE '123' = '123'");
    }

    @Test
    public void whereTest() {
        this.testToSql("select id from functional.alltypes where tinyint_col < 40 OR int_col = 4 AND float_col > 1.4", "SELECT id FROM functional.alltypes WHERE tinyint_col < 40 OR int_col = 4 AND float_col > 1.4");
        this.testToSql("select id from functional.alltypes where string_col = \"abc\"", "SELECT id FROM functional.alltypes WHERE string_col = 'abc'");
        this.testToSql("select id from functional.alltypes where string_col = 'abc'", "SELECT id FROM functional.alltypes WHERE string_col = 'abc'");
        this.testToSql("select id from functional.alltypes where 5 between smallint_col and int_col", "SELECT id FROM functional.alltypes WHERE 5 BETWEEN smallint_col AND int_col");
        this.testToSql("select id from functional.alltypes where 5 not between smallint_col and int_col", "SELECT id FROM functional.alltypes WHERE 5 NOT BETWEEN smallint_col AND int_col");
        this.testToSql("select id from functional.alltypes where 5 in (smallint_col, int_col)", "SELECT id FROM functional.alltypes WHERE 5 IN (smallint_col, int_col)");
        this.testToSql("select id from functional.alltypes where 5 not in (smallint_col, int_col)", "SELECT id FROM functional.alltypes WHERE 5 NOT IN (smallint_col, int_col)");
    }

    @Test
    public void joinTest() {
        this.testToSql("select * from functional.alltypes a, functional.alltypes b where a.id = b.id", "SELECT * FROM functional.alltypes a, functional.alltypes b WHERE a.id = b.id");
        this.testToSql("select * from functional.alltypes a cross join functional.alltypes b", "SELECT * FROM functional.alltypes a CROSS JOIN functional.alltypes b");
        this.runTestTemplate("select * from functional.alltypes a %s functional.alltypes b %s", "SELECT * FROM functional.alltypes a %s functional.alltypes b %s", joinTypes_, joinConditions_);
    }

    private void planHintsTestForInsertAndUpsert(String prefix, String suffix) {
        for (InsertStmt.HintLocation loc : InsertStmt.HintLocation.values()) {
            this.testToSql(this.InjectInsertHint("insert%s into functional.alltypes(int_col, bool_col) partition(year, month)%sselect int_col, bool_col, year, month from functional.alltypes", String.format(" %snoshuffle%s", prefix, suffix), loc), this.InjectInsertHint("INSERT%s INTO TABLE functional.alltypes(int_col, bool_col) PARTITION (`year`, `month`)%s SELECT int_col, bool_col, `year`, `month` FROM functional.alltypes", " \n-- +noshuffle\n", loc));
            this.testToSql(this.InjectInsertHint("insert%s into functional.alltypes(int_col, bool_col) partition(year, month)%sselect int_col, bool_col, year, month from functional.alltypes", String.format(" %sshuffle,clustered%s", prefix, suffix), loc), this.InjectInsertHint("INSERT%s INTO TABLE functional.alltypes(int_col, bool_col) PARTITION (`year`, `month`)%s SELECT int_col, bool_col, `year`, `month` FROM functional.alltypes", " \n-- +shuffle,clustered\n", loc));
            this.testToSql(this.InjectInsertHint("upsert%s into functional_kudu.alltypes(id, int_col)%sselect id, int_col from functional_kudu.alltypes", String.format(" %snoshuffle%s", prefix, suffix), loc), this.InjectInsertHint("UPSERT%s INTO TABLE functional_kudu.alltypes(id, int_col)%s SELECT id, int_col FROM functional_kudu.alltypes", " \n-- +noshuffle\n", loc));
            this.testToSql(this.InjectInsertHint("upsert%s into functional_kudu.alltypes(id, int_col)%sselect id, int_col from functional_kudu.alltypes", String.format(" %sshuffle,clustered%s", prefix, suffix), loc), this.InjectInsertHint("UPSERT%s INTO TABLE functional_kudu.alltypes(id, int_col)%s SELECT id, int_col FROM functional_kudu.alltypes", " \n-- +shuffle,clustered\n", loc));
        }
    }

    @Test
    public void planHintsTest() {
        for (String[] hintStyle : this.hintStyles_) {
            String prefix = hintStyle[0];
            String suffix = hintStyle[1];
            this.planHintsTestForInsertAndUpsert(prefix, suffix);
            this.testToSql(String.format("select * from functional.alltypes a join %sbroadcast%s functional.alltypes b on a.id = b.id", prefix, suffix), "SELECT * FROM functional.alltypes a INNER JOIN \n-- +broadcast\n functional.alltypes b ON a.id = b.id");
            this.testToSql(String.format("select * from functional.alltypes atp %sschedule_random_replica%s", prefix, suffix), "SELECT * FROM functional.alltypes atp\n-- +schedule_random_replica\n");
            this.testToSql(String.format("select * from functional.alltypes %sschedule_random_replica%s", prefix, suffix), "SELECT * FROM functional.alltypes\n-- +schedule_random_replica\n");
            this.testToSql(String.format("select * from functional.alltypes %sschedule_random_replica,schedule_disk_local%s", prefix, suffix), "SELECT * FROM functional.alltypes\n-- +schedule_random_replica,schedule_disk_local\n");
            this.testToSql(String.format("select c1 from (select atp.tinyint_col as c1 from functional.alltypes atp %sschedule_random_replica%s) s1", prefix, suffix), "SELECT c1 FROM (SELECT atp.tinyint_col c1 FROM functional.alltypes atp\n-- +schedule_random_replica\n) s1");
            if (prefix.contains("[")) {
                prefix = "";
                suffix = "";
            }
            this.testToSql(String.format("select %sstraight_join%s * from functional.alltypes", prefix, suffix), "SELECT \n-- +straight_join\n * FROM functional.alltypes");
            this.testToSql(String.format("select distinct %sstraight_join%s * from functional.alltypes", prefix, suffix), "SELECT DISTINCT \n-- +straight_join\n * FROM functional.alltypes");
            this.testToSql(String.format("select distinct %sstraight_join%s * from functional.alltypes", prefix, suffix), "SELECT DISTINCT \n-- +straight_join\n * FROM functional.alltypes", ToSqlOptions.DEFAULT);
            this.testToSql(String.format("select distinct %sstraight_join%s * from functional.alltypes where bool_col = false and id <= 5 and id >= 2", prefix, suffix), "SELECT DISTINCT /* +straight_join */ * FROM functional.alltypes WHERE bool_col = FALSE AND id <= 5 AND id >= 2", ToSqlOptions.REWRITTEN);
        }
    }

    @Test
    public void aggregationTest() {
        this.testToSql("select COUNT(*), count(id), COUNT(id), SUM(id), AVG(id) from functional.alltypes group by tinyint_col", "SELECT count(*), count(id), count(id), sum(id), avg(id) FROM functional.alltypes GROUP BY tinyint_col");
        this.testToSql("select avg(float_col / id) from functional.alltypes group by tinyint_col", "SELECT avg(float_col / id) FROM functional.alltypes GROUP BY tinyint_col");
        this.testToSql("select avg(double_col) from functional.alltypes group by int_col, tinyint_col, bigint_col", "SELECT avg(double_col) FROM functional.alltypes GROUP BY int_col, tinyint_col, bigint_col");
        this.testToSql("select avg(id) from functional.alltypes group by tinyint_col having count(tinyint_col) > 10", "SELECT avg(id) FROM functional.alltypes GROUP BY tinyint_col HAVING count(tinyint_col) > 10");
        this.testToSql("select sum(id) from functional.alltypes group by tinyint_col having avg(tinyint_col) > 10 AND count(tinyint_col) > 5", "SELECT sum(id) FROM functional.alltypes GROUP BY tinyint_col HAVING avg(tinyint_col) > 10 AND count(tinyint_col) > 5");
        this.testToSql("select int_col, string_col, sum(id) from functional.alltypes group by rollup(int_col, string_col)", "SELECT int_col, string_col, sum(id) FROM functional.alltypes GROUP BY ROLLUP(int_col, string_col)");
        this.testToSql("select int_col, string_col, sum(id) from functional.alltypes group by int_col, string_col with rollup", "SELECT int_col, string_col, sum(id) FROM functional.alltypes GROUP BY ROLLUP(int_col, string_col)");
        this.testToSql("select int_col, string_col, sum(id) from functional.alltypes group by cube(int_col, string_col)", "SELECT int_col, string_col, sum(id) FROM functional.alltypes GROUP BY CUBE(int_col, string_col)");
        this.testToSql("select int_col, string_col, sum(id) from functional.alltypes group by int_col, string_col with cube", "SELECT int_col, string_col, sum(id) FROM functional.alltypes GROUP BY CUBE(int_col, string_col)");
        this.testToSql("select int_col, string_col, sum(id) from functional.alltypes group by grouping sets((int_col, string_col), (int_col), ())", "SELECT int_col, string_col, sum(id) FROM functional.alltypes GROUP BY GROUPING SETS((int_col, string_col), (int_col), ())");
    }

    @Test
    public void orderByTest() {
        this.testToSql("select id, string_col from functional.alltypes order by string_col ASC, float_col DESC, int_col ASC", "SELECT id, string_col FROM functional.alltypes ORDER BY string_col ASC, float_col DESC, int_col ASC");
        this.testToSql("select id, string_col from functional.alltypes order by string_col DESC, float_col ASC, int_col DESC", "SELECT id, string_col FROM functional.alltypes ORDER BY string_col DESC, float_col ASC, int_col DESC");
        this.testToSql("select id, string_col from functional.alltypes order by string_col ASC NULLS FIRST, float_col DESC NULLS LAST, int_col DESC", "SELECT id, string_col FROM functional.alltypes ORDER BY string_col ASC NULLS FIRST, float_col DESC NULLS LAST, int_col DESC");
        this.testToSql("select id, string_col from functional.alltypes order by string_col ASC NULLS FIRST, float_col DESC NULLS LAST, int_col DESC LIMIT 10 OFFSET 5", "SELECT id, string_col FROM functional.alltypes ORDER BY string_col ASC NULLS FIRST, float_col DESC NULLS LAST, int_col DESC LIMIT 10 OFFSET 5");
        this.testToSql("select id, string_col from functional.alltypes order by string_col ASC NULLS FIRST, float_col DESC NULLS LAST, int_col DESC LIMIT 10 OFFSET 0", "SELECT id, string_col FROM functional.alltypes ORDER BY string_col ASC NULLS FIRST, float_col DESC NULLS LAST, int_col DESC LIMIT 10");
        this.testToSql("select id, string_col from functional.alltypes order by string_col DESC NULLS FIRST, float_col ASC NULLS LAST, int_col DESC", "SELECT id, string_col FROM functional.alltypes ORDER BY string_col DESC, float_col ASC, int_col DESC");
    }

    @Test
    public void allTest() {
        this.testToSql("select bigint_col, avg(double_col), sum(tinyint_col) from functional.alltypes where double_col > 2.5 AND string_col != \"abc\"group by bigint_col, int_col having count(int_col) > 10 OR sum(bigint_col) > 20 order by 2 DESC NULLS LAST, 3 ASC", "SELECT bigint_col, avg(double_col), sum(tinyint_col) FROM functional.alltypes WHERE double_col > 2.5 AND string_col != 'abc' GROUP BY bigint_col, int_col HAVING count(int_col) > 10 OR sum(bigint_col) > 20 ORDER BY 2 DESC NULLS LAST, 3 ASC");
    }

    @Test
    public void unionTest() {
        this.testToSql("select bool_col, rank() over(order by id) from functional.alltypes union select bool_col, int_col from functional.alltypessmall union select bool_col, bigint_col from functional.alltypes", "SELECT bool_col, rank() OVER (ORDER BY id ASC) FROM functional.alltypes UNION SELECT bool_col, int_col FROM functional.alltypessmall UNION SELECT bool_col, bigint_col FROM functional.alltypes");
        this.testToSql("select bool_col, int_col from functional.alltypes union all select bool_col, int_col from functional.alltypessmall union all select bool_col, int_col from functional.alltypessmall union all select bool_col, int_col from functional.alltypessmall union all select bool_col, bigint_col from functional.alltypes", "SELECT bool_col, int_col FROM functional.alltypes UNION ALL SELECT bool_col, int_col FROM functional.alltypessmall UNION ALL SELECT bool_col, int_col FROM functional.alltypessmall UNION ALL SELECT bool_col, int_col FROM functional.alltypessmall UNION ALL SELECT bool_col, bigint_col FROM functional.alltypes");
        this.testToSql("(select bool_col, int_col from functional.alltypes) union all (select bool_col, int_col from functional.alltypessmall) union all (select bool_col, bigint_col from functional.alltypes order by 1 nulls first limit 1) order by int_col nulls first, bool_col limit 5 + 5", "SELECT bool_col, int_col FROM functional.alltypes UNION ALL SELECT bool_col, int_col FROM functional.alltypessmall UNION ALL SELECT bool_col, bigint_col FROM functional.alltypes ORDER BY 1 ASC NULLS FIRST LIMIT 1 ORDER BY int_col ASC NULLS FIRST, bool_col ASC LIMIT 5 + 5");
        this.testToSql("select bool_col, int_col from functional.alltypes union all select bool_col, int_col from functional.alltypessmall union all (select bool_col, bigint_col from functional.alltypes) order by int_col nulls first, bool_col limit 10", "SELECT bool_col, int_col FROM functional.alltypes UNION ALL SELECT bool_col, int_col FROM functional.alltypessmall UNION ALL (SELECT bool_col, bigint_col FROM functional.alltypes) ORDER BY int_col ASC NULLS FIRST, bool_col ASC LIMIT 10");
        this.testToSql("select bool_col, int_col from functional.alltypes union all (select bool_col, int_col from functional.alltypessmall union distinct (select bool_col, bigint_col from functional.alltypes)) order by int_col, bool_col limit 10", "SELECT bool_col, int_col FROM functional.alltypes UNION ALL (SELECT bool_col, int_col FROM functional.alltypessmall UNION SELECT bool_col, bigint_col FROM functional.alltypes) ORDER BY int_col ASC, bool_col ASC LIMIT 10");
    }

    @Test
    public void valuesTest() {
        this.testToSql("values(1, 'a', 1.0)", "VALUES(1, 'a', 1.0)");
        this.testToSql("values(1 as x, 'a' y, 1.0 as z)", "VALUES(1 x, 'a' y, 1.0 z)");
        this.testToSql("values(1, 'a'), (2, 'b'), (3, 'c')", "VALUES((1, 'a'), (2, 'b'), (3, 'c'))");
        this.testToSql("values(1 x, 'a' as y), (2 as y, 'b'), (3, 'c' x)", "VALUES((1 x, 'a' y), (2 y, 'b'), (3, 'c' x))");
        this.testToSql("select * from (values(1, 'a'), (2, 'b')) as t", "SELECT * FROM (VALUES((1, 'a'), (2, 'b'))) t");
        this.testToSql("values(1, 'a'), (2, 'b') union all values(3, 'c')", "VALUES((1, 'a'), (2, 'b')) UNION ALL (VALUES(3, 'c'))");
        this.testToSql("insert into table functional.alltypessmall partition (`year`=2009, `month`=4) values(1, true, 1, 1, 10, 10, 10.0, 10.0, 'a', 'a', cast (0 as timestamp))", "INSERT INTO TABLE functional.alltypessmall PARTITION (`year`=2009, `month`=4) VALUES(1, TRUE, 1, 1, 10, 10, 10.0, 10.0, 'a', 'a', CAST(0 AS TIMESTAMP))");
        this.testToSql("insert into table functional.date_tbl partition (date_part='9999-12-31') values(112, DATE '1970-01-01')", "INSERT INTO TABLE functional.date_tbl PARTITION (date_part='9999-12-31') VALUES(112, DATE '1970-01-01')");
        this.testToSql("upsert into table functional_kudu.testtbl values(1, 'a', 1)", "UPSERT INTO TABLE functional_kudu.testtbl VALUES(1, 'a', 1)");
        this.testToSql("insert into table functional.binary_tbl values(1, 'a', cast('a' as binary))", "INSERT INTO TABLE functional.binary_tbl VALUES(1, 'a', CAST('a' AS BINARY))");
    }

    @Test
    public void inlineViewTest() {
        this.testToSql("select t.* from (select a.* from functional.alltypes a, functional.alltypes b where a.id = b.id) t", "SELECT t.* FROM (SELECT a.* FROM functional.alltypes a, functional.alltypes b WHERE a.id = b.id) t");
        this.testToSql("select t.* from (select a.* from functional.alltypes a cross join functional.alltypes b) t", "SELECT t.* FROM (SELECT a.* FROM functional.alltypes a CROSS JOIN functional.alltypes b) t");
        this.runTestTemplate("select t.* from (select a.* from functional.alltypes a %s functional.alltypes b %s) t", "SELECT t.* FROM (SELECT a.* FROM functional.alltypes a %s functional.alltypes b %s) t", nonSemiJoinTypes_, joinConditions_);
        this.runTestTemplate("select t.* from (select a.* from functional.alltypes a %s functional.alltypes b %s) t", "SELECT t.* FROM (SELECT a.* FROM functional.alltypes a %s functional.alltypes b %s) t", leftSemiJoinTypes_, joinConditions_);
        this.runTestTemplate("select t.* from (select b.* from functional.alltypes a %s functional.alltypes b %s) t", "SELECT t.* FROM (SELECT b.* FROM functional.alltypes a %s functional.alltypes b %s) t", rightSemiJoinTypes_, joinConditions_);
        this.testToSql("select t1.int_col, t2.int_col from (select int_col, rank() over (order by int_col) from functional.alltypes) t1 inner join (select int_col from functional.alltypes) t2 on (t1.int_col = t2.int_col)", "SELECT t1.int_col, t2.int_col FROM (SELECT int_col, rank() OVER (ORDER BY int_col ASC) FROM functional.alltypes) t1 INNER JOIN (SELECT int_col FROM functional.alltypes) t2 ON (t1.int_col = t2.int_col)");
        this.testToSql("select count(t1.string_col), sum(t2.float_col) from (select id, string_col from functional.alltypes) t1 inner join (select id, float_col from functional.alltypes) t2 on (t1.id = t2.id) group by t1.id, t2.id having count(t2.float_col) > 2", "SELECT count(t1.string_col), sum(t2.float_col) FROM (SELECT id, string_col FROM functional.alltypes) t1 INNER JOIN (SELECT id, float_col FROM functional.alltypes) t2 ON (t1.id = t2.id) GROUP BY t1.id, t2.id HAVING count(t2.float_col) > 2");
        this.testToSql("select t1.id, t2.id from (select id, string_col from functional.alltypes) t1 inner join (select id, float_col from functional.alltypes) t2 on (t1.id = t2.id) order by t1.id, t2.id nulls first", "SELECT t1.id, t2.id FROM (SELECT id, string_col FROM functional.alltypes) t1 INNER JOIN (SELECT id, float_col FROM functional.alltypes) t2 ON (t1.id = t2.id) ORDER BY t1.id ASC, t2.id ASC NULLS FIRST");
        this.testToSql("select t1.id, t2.id from (select id, string_col from functional.alltypes) t1, (select id, float_col from functional.alltypes) t2 where t1.id = t2.id and t1.string_col = 'abc' and t2.float_col < 10", "SELECT t1.id, t2.id FROM (SELECT id, string_col FROM functional.alltypes) t1, (SELECT id, float_col FROM functional.alltypes) t2 WHERE t1.id = t2.id AND t1.string_col = 'abc' AND t2.float_col < 10");
        this.testToSql("select cnt from functional.allcomplextypes t, (select count(*) cnt from t.int_array_col) v", "SELECT cnt FROM functional.allcomplextypes t, (SELECT count(*) cnt FROM t.int_array_col) v");
        this.testToSql("select avg from functional.allcomplextypes t, (select avg(a1.item) avg from t.int_array_col a1, t.int_array_col a2) v", "SELECT avg FROM functional.allcomplextypes t, (SELECT avg(a1.item) avg FROM t.int_array_col a1, t.int_array_col a2) v");
        this.testToSql("select key, item from functional.allcomplextypes t, (select a1.key, value.item from t.array_map_col a1, a1.value) v", "SELECT `key`, item FROM functional.allcomplextypes t, (SELECT a1.`key`, value.item FROM t.array_map_col a1, a1.value) v");
        this.testToSql("select item from functional.allcomplextypes t, (select * from t.int_array_col union all select * from t.int_array_col) v", "SELECT item FROM functional.allcomplextypes t, (SELECT * FROM t.int_array_col UNION ALL SELECT * FROM t.int_array_col) v");
        this.testToSql("with w as (select c from functional.allcomplextypes t, (select count(a1.key) c from t.array_map_col a1) v1) select * from w", "WITH w AS (SELECT c FROM functional.allcomplextypes t, (SELECT count(a1.`key`) c FROM t.array_map_col a1) v1) SELECT * FROM w");
    }

    @Test
    public void TestUpdate() {
        this.testToSql("update functional_kudu.dimtbl set name = '10' where name < '11'", "UPDATE functional_kudu.dimtbl SET name = '10' FROM functional_kudu.dimtbl WHERE name < '11'");
        this.testToSql("update functional_kudu.dimtbl set name = '10', zip=cast(99 as int) where name < '11'", "UPDATE functional_kudu.dimtbl SET name = '10', zip = CAST(99 AS INT) FROM functional_kudu.dimtbl WHERE name < '11'");
        this.testToSql("update a set name = '10' FROM functional_kudu.dimtbl a", "UPDATE a SET name = '10' FROM functional_kudu.dimtbl a");
        this.testToSql("update a set a.name = 'oskar' from functional_kudu.dimtbl a join functional.alltypes b on a.id = b.id where zip > 94549", "UPDATE a SET a.name = 'oskar' FROM functional_kudu.dimtbl a INNER JOIN functional.alltypes b ON a.id = b.id WHERE zip > 94549");
    }

    @Test
    public void TestDelete() {
        this.testToSql("delete functional_kudu.testtbl where zip = 10", "DELETE FROM functional_kudu.testtbl WHERE zip = 10");
        this.testToSql("delete from functional_kudu.testtbl where zip = 10", "DELETE FROM functional_kudu.testtbl WHERE zip = 10");
        this.testToSql("delete a from functional_kudu.testtbl a where zip = 10", "DELETE a FROM functional_kudu.testtbl a WHERE zip = 10");
    }

    @Test
    public void subqueryTest() {
        this.testToSql("select * from functional.alltypes where id in (select id from functional.alltypestiny)", "SELECT * FROM functional.alltypes WHERE id IN (SELECT id FROM functional.alltypestiny)");
        this.testToSql("select * from functional.alltypes where id not in (select id from functional.alltypestiny)", "SELECT * FROM functional.alltypes WHERE id NOT IN (SELECT id FROM functional.alltypestiny)");
        this.testToSql("select * from functional.alltypes where bigint_col = (select count(*) from functional.alltypestiny)", "SELECT * FROM functional.alltypes WHERE bigint_col = (SELECT count(*) FROM functional.alltypestiny)");
        this.testToSql("select * from functional.alltypes where exists (select * from functional.alltypestiny)", "SELECT * FROM functional.alltypes WHERE EXISTS (SELECT * FROM functional.alltypestiny)");
        this.testToSql("select * from functional.alltypes where not exists (select * from functional.alltypestiny)", "SELECT * FROM functional.alltypes WHERE NOT EXISTS (SELECT * FROM functional.alltypestiny)");
        this.testToSql("select * from functional.alltypes where id in (select id from functional.alltypestiny where int_col = (select avg(int_col) from functional.alltypesagg))", "SELECT * FROM functional.alltypes WHERE id IN (SELECT id FROM functional.alltypestiny WHERE int_col = (SELECT avg(int_col) FROM functional.alltypesagg))");
        this.testToSql("select * from (select id from functional.alltypes where int_col in (select int_col from functional.alltypestiny)) t where t.id < 10", "SELECT * FROM (SELECT id FROM functional.alltypes WHERE int_col IN (SELECT int_col FROM functional.alltypestiny)) t WHERE t.id < 10");
        this.testToSql("with t as (select * from functional.alltypes where id in (select id from functional.alltypestiny)) select * from t", "WITH t AS (SELECT * FROM functional.alltypes WHERE id IN (SELECT id FROM functional.alltypestiny)) SELECT * FROM t");
        this.testToSql("with t as (select * from functional.alltypes s where id in (select id from functional.alltypestiny t where s.id = t.id)) select * from t t1, t t2 where t1.id = t2.id", "WITH t AS (SELECT * FROM functional.alltypes s WHERE id IN (SELECT id FROM functional.alltypestiny t WHERE s.id = t.id)) SELECT * FROM t t1, t t2 WHERE t1.id = t2.id");
    }

    @Test
    public void withClauseTest() {
        this.testToSql("with t as (select * from functional.alltypes) select * from t", "WITH t AS (SELECT * FROM functional.alltypes) SELECT * FROM t");
        this.testToSql("with t(c1) as (select * from functional.alltypes) select * from t", "WITH t(c1) AS (SELECT * FROM functional.alltypes) SELECT * FROM t");
        this.testToSql("with t(`table`, col, `create`) as (select * from functional.alltypes) select * from t", "WITH t(`table`, col, `create`) AS (SELECT * FROM functional.alltypes) SELECT * FROM t");
        this.testToSql("with t(c1, c2) as (select * from functional.alltypes) select * from t", "WITH t(c1, c2) AS (SELECT * FROM functional.alltypes) SELECT * FROM t");
        this.testToSql("with t as (select sum(int_col) over(partition by tinyint_col, bool_col order by float_col rows between unbounded preceding and current row) as x from functional.alltypes) select t1.x, t2.x from t t1 join t t2 on (t1.x = t2.x)", "WITH t AS (SELECT sum(int_col) OVER (PARTITION BY tinyint_col, bool_col ORDER BY float_col ASC ROWS BETWEEN UNBOUNDED PRECEDING AND CURRENT ROW) x FROM functional.alltypes) SELECT t1.x, t2.x FROM t t1 INNER JOIN t t2 ON (t1.x = t2.x)");
        this.testToSql("with t as (select * from functional.alltypes) select * from t a inner join t b on (a.int_col = b.int_col)", "WITH t AS (SELECT * FROM functional.alltypes) SELECT * FROM t a INNER JOIN t b ON (a.int_col = b.int_col)");
        this.testToSql("with t(c1, c2) as (select * from functional.alltypes) select a.c1, a.c2 from t a inner join t b on (a.c1 = b.c2)", "WITH t(c1, c2) AS (SELECT * FROM functional.alltypes) SELECT a.c1, a.c2 FROM t a INNER JOIN t b ON (a.c1 = b.c2)");
        this.testToSql("with t as (select * from functional.alltypes) select * from t a inner join t b using(int_col)", "WITH t AS (SELECT * FROM functional.alltypes) SELECT * FROM t a INNER JOIN t b USING (int_col)");
        this.testToSql("with t1 as (select * from functional.alltypes)select * from t1 union all select * from t1", "WITH t1 AS (SELECT * FROM functional.alltypes) SELECT * FROM t1 UNION ALL SELECT * FROM t1");
        this.testToSql("with t1 as (select * from functional.alltypes) values(1, 2), (3, 4)", "WITH t1 AS (SELECT * FROM functional.alltypes) VALUES((1, 2), (3, 4))");
        this.testToSql("with t1 as (select * from functional.alltypes) insert into functional.alltypes partition(year, month) select * from t1", "WITH t1 AS (SELECT * FROM functional.alltypes) INSERT INTO TABLE functional.alltypes PARTITION (`year`, `month`) SELECT * FROM t1");
        this.testToSql("with t1 as (select * from functional.alltypes) upsert into functional_kudu.testtbl select bigint_col, string_col, int_col from t1", "WITH t1 AS (SELECT * FROM functional.alltypes) UPSERT INTO TABLE functional_kudu.testtbl SELECT bigint_col, string_col, int_col FROM t1");
        this.testToSql("with t as (select a.* from functional.alltypes a, functional.alltypes b where a.id = b.id) select * from t", "WITH t AS (SELECT a.* FROM functional.alltypes a, functional.alltypes b WHERE a.id = b.id) SELECT * FROM t");
        this.testToSql("with t as (select a.* from functional.alltypes a cross join functional.alltypes b) select * from t", "WITH t AS (SELECT a.* FROM functional.alltypes a CROSS JOIN functional.alltypes b) SELECT * FROM t");
        this.runTestTemplate("with t as (select a.* from functional.alltypes a %s functional.alltypes b %s) select * from t", "WITH t AS (SELECT a.* FROM functional.alltypes a %s functional.alltypes b %s) SELECT * FROM t", nonSemiJoinTypes_, joinConditions_);
        this.runTestTemplate("with t as (select a.* from functional.alltypes a %s functional.alltypes b %s) select * from t", "WITH t AS (SELECT a.* FROM functional.alltypes a %s functional.alltypes b %s) SELECT * FROM t", leftSemiJoinTypes_, joinConditions_);
        this.runTestTemplate("with t as (select b.* from functional.alltypes a %s functional.alltypes b %s) select * from t", "WITH t AS (SELECT b.* FROM functional.alltypes a %s functional.alltypes b %s) SELECT * FROM t", rightSemiJoinTypes_, joinConditions_);
        this.testToSql("with t as (select int_col x, bigint_col y from functional.alltypestiny order by id nulls first limit 2) select * from t t1 left outer join t t2 on t1.y = t2.x full outer join t t3 on t2.y = t3.x order by t1.x nulls first limit 5 * 2", "WITH t AS (SELECT int_col x, bigint_col y FROM functional.alltypestiny ORDER BY id ASC NULLS FIRST LIMIT 2) SELECT * FROM t t1 LEFT OUTER JOIN t t2 ON t1.y = t2.x FULL OUTER JOIN t t3 ON t2.y = t3.x ORDER BY t1.x ASC NULLS FIRST LIMIT 5 * 2");
    }

    @Test
    public void insertTest() {
        this.testToSql("insert into table functional.alltypesnopart select id, bool_col, tinyint_col, smallint_col, int_col, bigint_col, float_col, double_col, date_string_col, string_col, timestamp_col from functional.alltypes", "INSERT INTO TABLE functional.alltypesnopart SELECT id, bool_col, tinyint_col, smallint_col, int_col, bigint_col, float_col, double_col, date_string_col, string_col, timestamp_col FROM functional.alltypes");
        this.testToSql("insert overwrite table functional.alltypesnopart select id, bool_col, tinyint_col, smallint_col, int_col, bigint_col, float_col, double_col, date_string_col, string_col, timestamp_col from functional.alltypes", "INSERT OVERWRITE TABLE functional.alltypesnopart SELECT id, bool_col, tinyint_col, smallint_col, int_col, bigint_col, float_col, double_col, date_string_col, string_col, timestamp_col FROM functional.alltypes");
        this.testToSql("insert into table functional.alltypessmall partition (year=2009, month=4)select id, bool_col, tinyint_col, smallint_col, int_col, bigint_col, float_col, double_col, date_string_col, string_col, timestamp_col from functional.alltypes", "INSERT INTO TABLE functional.alltypessmall PARTITION (`year`=2009, `month`=4) SELECT id, bool_col, tinyint_col, smallint_col, int_col, bigint_col, float_col, double_col, date_string_col, string_col, timestamp_col FROM functional.alltypes");
        this.testToSql("insert into table functional.date_tbl partition (date_part='2009-10-30') select id, cast(timestamp_col as date) from functional.alltypes", "INSERT INTO TABLE functional.date_tbl PARTITION (date_part='2009-10-30') SELECT id, CAST(timestamp_col AS DATE) FROM functional.alltypes");
        this.testToSql("insert into table functional.alltypessmall partition (year, month)select id, bool_col, tinyint_col, smallint_col, int_col, bigint_col, float_col, double_col, date_string_col, string_col, timestamp_col, year, month from functional.alltypes", "INSERT INTO TABLE functional.alltypessmall PARTITION (`year`, `month`) SELECT id, bool_col, tinyint_col, smallint_col, int_col, bigint_col, float_col, double_col, date_string_col, string_col, timestamp_col, `year`, `month` FROM functional.alltypes");
        this.testToSql("insert into table functional.date_tbl partition (date_part) select id, cast(timestamp_col as date) date_col, cast(timestamp_col as date) date_part from functional.alltypes", "INSERT INTO TABLE functional.date_tbl PARTITION (date_part) SELECT id, CAST(timestamp_col AS DATE) date_col, CAST(timestamp_col AS DATE) date_part FROM functional.alltypes");
        this.testToSql("insert into table functional.alltypessmall partition (year=2009, month)select id, bool_col, tinyint_col, smallint_col, int_col, bigint_col, float_col, double_col, date_string_col, string_col, timestamp_col, month from functional.alltypes", "INSERT INTO TABLE functional.alltypessmall PARTITION (`year`=2009, `month`) SELECT id, bool_col, tinyint_col, smallint_col, int_col, bigint_col, float_col, double_col, date_string_col, string_col, timestamp_col, `month` FROM functional.alltypes");
        this.testToSql("insert into table functional.alltypesnopart(id, bool_col, tinyint_col)  values(1, true, 0)", "INSERT INTO TABLE functional.alltypesnopart(id, bool_col, tinyint_col) VALUES(1, TRUE, 0)");
        this.testToSql("insert into table functional.alltypes(id, year, month)  values(1, 1990, 12)", "INSERT INTO TABLE functional.alltypes(id, `year`, `month`) VALUES(1, 1990, 12)");
        this.testToSql("insert into table functional.alltypesnopart()", "INSERT INTO TABLE functional.alltypesnopart()");
        this.testToSql("insert into table functional.alltypes(id)  partition (year=2009, month) values(1, 12)", "INSERT INTO TABLE functional.alltypes(id) PARTITION (`year`=2009, `month`) VALUES(1, 12)");
    }

    @Test
    public void upsertTest() {
        this.testToSql("upsert into functional_kudu.testtbl values (1, 'a', 1)", "UPSERT INTO TABLE functional_kudu.testtbl VALUES(1, 'a', 1)");
        this.testToSql("upsert into functional_kudu.testtbl select bigint_col, string_col, int_col from functional.alltypes", "UPSERT INTO TABLE functional_kudu.testtbl SELECT bigint_col, string_col, int_col FROM functional.alltypes");
        this.testToSql("with x as (select bigint_col, string_col, int_col from functional.alltypes) upsert into table functional_kudu.testtbl select * from x", "WITH x AS (SELECT bigint_col, string_col, int_col FROM functional.alltypes) UPSERT INTO TABLE functional_kudu.testtbl SELECT * FROM x");
        this.testToSql("upsert into table functional_kudu.testtbl (zip, id, name) values (1, 1, 'a')", "UPSERT INTO TABLE functional_kudu.testtbl(zip, id, name) VALUES(1, 1, 'a')");
    }

    @Test
    public void alterTableAddPartitionTest() {
        this.testToSql("alter table functional.alltypes add partition (year=2050, month=1)", "ALTER TABLE functional.alltypes ADD PARTITION (year=2050, month=1)");
        this.testToSql("alter table functional.alltypes add partition (year=2050, month=1) partition (year=2050, month=2)", "ALTER TABLE functional.alltypes ADD PARTITION (year=2050, month=1) PARTITION (year=2050, month=2)");
        this.testToSql("alter table functional.alltypes add if not exists partition (year=2050, month=1) partition (year=2050, month=2)", "ALTER TABLE functional.alltypes ADD IF NOT EXISTS PARTITION (year=2050, month=1) PARTITION (year=2050, month=2)");
        this.testToSql("alter table functional.alltypes add if not exists partition (year=2050, month=1) location 'hdfs://localhost:20500/y2050m1' partition (year=2050, month=2) location '/y2050m2'", "ALTER TABLE functional.alltypes ADD IF NOT EXISTS PARTITION (year=2050, month=1) LOCATION 'hdfs://localhost:20500/y2050m1' PARTITION (year=2050, month=2) LOCATION 'hdfs://localhost:20500/y2050m2'");
        this.testToSql("alter table functional.alltypes add if not exists partition (year=2050, month=1) location 'hdfs://localhost:20500/y2050m1' cached in 'testPool' with replication=3 partition (year=2050, month=2) location '/y2050m2' uncached", "ALTER TABLE functional.alltypes ADD IF NOT EXISTS PARTITION (year=2050, month=1) LOCATION 'hdfs://localhost:20500/y2050m1' CACHED IN 'testPool' WITH REPLICATION = 3 PARTITION (year=2050, month=2) LOCATION 'hdfs://localhost:20500/y2050m2' UNCACHED");
    }

    @Test
    public void testAnalyticExprs() {
        this.testToSql("select sum(int_col) over (partition by id order by tinyint_col rows between unbounded preceding and current row) from functional.alltypes", "SELECT sum(int_col) OVER (PARTITION BY id ORDER BY tinyint_col ASC ROWS BETWEEN UNBOUNDED PRECEDING AND CURRENT ROW) FROM functional.alltypes");
        this.testToSql("select last_value(tinyint_col ignore nulls) over (order by tinyint_col) from functional.alltypesagg", "SELECT last_value(tinyint_col IGNORE NULLS) OVER (ORDER BY tinyint_col ASC) FROM functional.alltypesagg");
    }

    @Test
    public void testExprs() {
        this.testToSql("select count(*), (count(*)), avg(int_col), (avg(int_col)), sum(int_col), (sum(int_col)), min(int_col), (min(int_col)), max(int_col), (max(int_col)) from functional.alltypes", "SELECT count(*), (count(*)), avg(int_col), (avg(int_col)), sum(int_col), (sum(int_col)), min(int_col), (min(int_col)), max(int_col), (max(int_col)) FROM functional.alltypes");
        this.testToSql("select 1 * 1, (1 * 1), 2 / 2, (2 / 2), 3 % 3, (3 % 3), 4 DIV 4, (4 DIV 4), 5 + 5, (5 + 5), 6 - 6, (6 - 6), 7 & 7, (7 & 7), 8 | 8, (8 | 8), 9 ^ 9, (9 ^ 9), ~10, (~10)", "SELECT 1 * 1, (1 * 1), 2 / 2, (2 / 2), 3 % 3, (3 % 3), 4 DIV 4, (4 DIV 4), 5 + 5, (5 + 5), 6 - 6, (6 - 6), 7 & 7, (7 & 7), 8 | 8, (8 | 8), 9 ^ 9, (9 ^ 9), ~10, (~10)");
        this.testToSql("select (((1 + 2) * (3 - 4) + 6) / 7)", "SELECT (((1 + 2) * (3 - 4) + 6) / 7)");
        this.testToSql("select case when true then 1 end, (case when true then 1 end)", "SELECT CASE WHEN TRUE THEN 1 END, (CASE WHEN TRUE THEN 1 END)");
        this.testToSql("select case when true then 1 when false then 2 else 3 end, (case when true then 1 when false then 2 else 3 end)", "SELECT CASE WHEN TRUE THEN 1 WHEN FALSE THEN 2 ELSE 3 END, (CASE WHEN TRUE THEN 1 WHEN FALSE THEN 2 ELSE 3 END)");
        this.testToSql("select case true when true then 1 when false then 2 else 3 end, (case true when true then 1 when false then 2 else 3 end)", "SELECT CASE TRUE WHEN TRUE THEN 1 WHEN FALSE THEN 2 ELSE 3 END, (CASE TRUE WHEN TRUE THEN 1 WHEN FALSE THEN 2 ELSE 3 END)");
        this.testToSql("select decode(1, 2, 3), (decode(4, 5, 6))", "SELECT decode(1, 2, 3), (decode(4, 5, 6))");
        this.testToSql("select decode(1, 2, 3, 4, 5, 6), (decode(1, 2, 3, 4, 5, 6))", "SELECT decode(1, 2, 3, 4, 5, 6), (decode(1, 2, 3, 4, 5, 6))");
        this.testToSql("select cast(NULL as INT), (cast(NULL as INT))", "SELECT CAST(NULL AS INT), (CAST(NULL AS INT))");
        this.testToSql("select pi(), (pi()), trim('a'), (trim('a'))", "SELECT pi(), (pi()), trim('a'), (trim('a'))");
        this.testToSql("select trim('aa' from 'abc'), (trim('aa' from 'abc')), trim(leading 'aa' from 'abc'), trim(string_col from 'string') from functional.alltypes", "SELECT trim('aa' FROM 'abc'), (trim('aa' FROM 'abc')), trim(LEADING 'aa' FROM 'abc'), trim(string_col FROM 'string') FROM functional.alltypes");
        this.testToSql("select 10, (10), 20.0, (20.0), NULL, (NULL), 'abc', ('abc')", "SELECT 10, (10), 20.0, (20.0), NULL, (NULL), 'abc', ('abc')");
        this.testToSql("select 5 between 10 and 20, (5 between 10 and 20)", "SELECT 5 BETWEEN 10 AND 20, (5 BETWEEN 10 AND 20)");
        this.testToSql("select 5 not between 10 and 20, (5 not between 10 and 20)", "SELECT 5 NOT BETWEEN 10 AND 20, (5 NOT BETWEEN 10 AND 20)");
        this.testToSql("select 'a' = 'b', ('a' = 'b'), 'a' != 'b', ('a' != 'b'), 1 < 2, (1 < 2), 1 <= 2, (1 <= 2), 1 > 2, (1 > 2), 1 >= 2, (1 >= 2)", "SELECT 'a' = 'b', ('a' = 'b'), 'a' != 'b', ('a' != 'b'), 1 < 2, (1 < 2), 1 <= 2, (1 <= 2), 1 > 2, (1 > 2), 1 >= 2, (1 >= 2)");
        this.testToSql("select true and false, (true and false), false or true, (false or true), !true, (!true), not false, (not false)", "SELECT TRUE AND FALSE, (TRUE AND FALSE), FALSE OR TRUE, (FALSE OR TRUE), NOT TRUE, (NOT TRUE), NOT FALSE, (NOT FALSE)");
        this.testToSql("select ((true and (false or false) or true) and (false or true))", "SELECT ((TRUE AND (FALSE OR FALSE) OR TRUE) AND (FALSE OR TRUE))");
        this.testToSql("select true || false, 'abc' || 'xyz'", "SELECT TRUE OR FALSE, concat('abc', 'xyz')");
        this.testToSql("select 5 in (4, 6, 7, 5), (5 in (4, 6, 7, 5)),5 not in (4, 6, 7, 5), (5 not In (4, 6, 7, 5))", "SELECT 5 IN (4, 6, 7, 5), (5 IN (4, 6, 7, 5)), 5 NOT IN (4, 6, 7, 5), (5 NOT IN (4, 6, 7, 5))");
        this.testToSql("select 5 is null, (5 is null), 10 is not null, (10 is not null)", "SELECT 5 IS NULL, (5 IS NULL), 10 IS NOT NULL, (10 IS NOT NULL)");
        this.testToSql("select (true is true)", "SELECT (istrue(TRUE))");
        this.testToSql("select (true is not true)", "SELECT (isnottrue(TRUE))");
        this.testToSql("select (true is false)", "SELECT (isfalse(TRUE))");
        this.testToSql("select (true is unknown)", "SELECT (TRUE IS NULL)");
        this.testToSql("select (true is not unknown)", "SELECT (TRUE IS NOT NULL)");
        this.testToSql("select not(true is true)", "SELECT NOT (istrue(TRUE))");
        this.testToSql("select (false is false)", "SELECT (isfalse(FALSE))");
        this.testToSql("select (null is unknown)", "SELECT (NULL IS NULL)");
        this.testToSql("select (1 > 1 is true is unknown)", "SELECT (istrue(1 > 1) IS NULL)");
        this.testToSql("select 'a' LIKE '%b.', ('a' LIKE '%b.'), 'a' ILIKE '%b.', ('a' ILIKE '%b.'), 'b' RLIKE '.c%', ('b' RLIKE '.c%'),'d' IREGEXP '.e%', ('d' IREGEXP '.e%'),'d' REGEXP '.e%', ('d' REGEXP '.e%')", "SELECT 'a' LIKE '%b.', ('a' LIKE '%b.'), 'a' ILIKE '%b.', ('a' ILIKE '%b.'), 'b' RLIKE '.c%', ('b' RLIKE '.c%'), 'd' IREGEXP '.e%', ('d' IREGEXP '.e%'), 'd' REGEXP '.e%', ('d' REGEXP '.e%')");
        this.testToSql("select bool_col, (bool_col), int_col, (int_col) string_col, (string_col), timestamp_col, (timestamp_col) from functional.alltypes", "SELECT bool_col, (bool_col), int_col, (int_col) string_col, (string_col), timestamp_col, (timestamp_col) FROM functional.alltypes");
        this.testToSql("select timestamp_col + interval 10 years, (timestamp_col + interval 10 years) from functional.alltypes", "SELECT timestamp_col + INTERVAL 10 years, (timestamp_col + INTERVAL 10 years) FROM functional.alltypes");
        this.testToSql("select timestamp_col - interval 20 months, (timestamp_col - interval 20 months) from functional.alltypes", "SELECT timestamp_col - INTERVAL 20 months, (timestamp_col - INTERVAL 20 months) FROM functional.alltypes");
        this.testToSql("select interval 30 weeks + timestamp_col, (interval 30 weeks + timestamp_col) from functional.alltypes", "SELECT INTERVAL 30 weeks + timestamp_col, (INTERVAL 30 weeks + timestamp_col) FROM functional.alltypes");
        this.testToSql("select date_add(timestamp_col, interval 40 days), (date_add(timestamp_col, interval 40 days)) from functional.alltypes", "SELECT DATE_ADD(timestamp_col, INTERVAL 40 days), (DATE_ADD(timestamp_col, INTERVAL 40 days)) FROM functional.alltypes");
        this.testToSql("select date_sub(timestamp_col, interval 40 hours), (date_sub(timestamp_col, interval 40 hours)) from functional.alltypes", "SELECT DATE_SUB(timestamp_col, INTERVAL 40 hours), (DATE_SUB(timestamp_col, INTERVAL 40 hours)) FROM functional.alltypes");
        this.testToSql("select unnest(arr1), unnest(arr2) from functional_parquet.complextypes_arrays", "SELECT UNNEST(arr1), UNNEST(arr2) FROM functional_parquet.complextypes_arrays");
        this.testToSql("select unnest(arr1) a1, unnest(arr2) a2 from functional_parquet.complextypes_arrays", "SELECT UNNEST(arr1) a1, UNNEST(arr2) a2 FROM functional_parquet.complextypes_arrays");
        this.testToSql("select unnest(arr1) as a1, unnest(arr2) as a2 from functional_parquet.complextypes_arrays", "SELECT UNNEST(arr1) a1, UNNEST(arr2) a2 FROM functional_parquet.complextypes_arrays");
        this.testToSql("select unnest(t.arr1) as a1, unnest(t.arr2) as a2 from functional_parquet.complextypes_arrays t", "SELECT UNNEST(t.arr1) a1, UNNEST(t.arr2) a2 FROM functional_parquet.complextypes_arrays t");
        this.testToSql("select arr1.item, arr2.item from functional_parquet.complextypes_arrays t, unnest(t.arr1, t.arr2)", "SELECT arr1.item, arr2.item FROM functional_parquet.complextypes_arrays t, UNNEST(t.arr1, t.arr2)");
        this.testToSql("select arr1.item, arr2.item from functional_parquet.complextypes_arrays t, unnest(t.arr1, t.arr2), functional_parquet.alltypes", "SELECT arr1.item, arr2.item FROM functional_parquet.complextypes_arrays t, UNNEST(t.arr1, t.arr2), functional_parquet.alltypes");
        this.testToSql("select arr1.item, arr2.item from functional_parquet.complextypes_arrays t, functional_parquet.alltypes, unnest(t.arr1, t.arr2)", "SELECT arr1.item, arr2.item FROM functional_parquet.complextypes_arrays t, functional_parquet.alltypes, UNNEST(t.arr1, t.arr2)");
    }

    @Test
    public void testDecimal() {
        this.testToSql("select cast(1 as decimal)", "SELECT CAST(1 AS DECIMAL(9,0))");
    }

    @Test
    public void testSet() {
        this.testToSql("set a = 1", "SET a='1'");
        this.testToSql("set `a b` = \"x y\"", "SET `a b`='x y'");
        this.testToSql("set", "SET");
    }

    @Test
    public void testTableSample() {
        this.testToSql("select * from functional.alltypes tablesample system(10)", "SELECT * FROM functional.alltypes TABLESAMPLE SYSTEM(10)");
        this.testToSql("select * from functional.alltypes tablesample system(10) repeatable(20)", "SELECT * FROM functional.alltypes TABLESAMPLE SYSTEM(10) REPEATABLE(20)");
        this.testToSql("select * from functional.alltypes a tablesample system(10) /* +schedule_random */", "SELECT * FROM functional.alltypes a TABLESAMPLE SYSTEM(10)\n-- +schedule_random\n");
        this.testToSql("with t as (select * from functional.alltypes tablesample system(5)) select * from t", "WITH t AS (SELECT * FROM functional.alltypes TABLESAMPLE SYSTEM(5)) SELECT * FROM t");
    }

    @Test
    public void testCreateDropRole() {
        String testRole = "test_role";
        AnalysisContext ctx = this.createAnalysisCtx(this.createAuthorizationFactory());
        this.testToSql(ctx, String.format("CREATE ROLE %s", testRole));
        try {
            catalog_.addRole(testRole);
            this.testToSql(ctx, String.format("DROP ROLE %s", testRole));
        }
        finally {
            catalog_.removeRole(testRole);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testGrantRevokePrivStmt() {
        AnalysisContext ctx = this.createAnalysisCtx(this.createAuthorizationFactory());
        List<String> principalTypes = Arrays.asList("USER", "ROLE", "GROUP");
        String testRole = System.getProperty("user.name");
        String testUri = "hdfs://localhost:20500/test-warehouse";
        String testStorageHandlerUri = "kudu://localhost/tbl";
        for (String pt : principalTypes) {
            try {
                catalog_.addRole(testRole);
                List privileges = Arrays.stream(Privilege.values()).filter(p -> p != Privilege.OWNER && p != Privilege.VIEW_METADATA && p != Privilege.ANY).collect(Collectors.toList());
                for (Privilege p2 : privileges) {
                    this.testToSql(ctx, String.format("GRANT %s ON SERVER server1 TO %s %s", p2, pt, testRole));
                    this.testToSql(ctx, String.format("GRANT %s ON SERVER TO %s", p2, testRole), String.format("GRANT %s ON SERVER server1 TO ROLE %s", p2, testRole), false);
                    this.testToSql(ctx, String.format("GRANT %s ON SERVER server1 TO %s %s WITH GRANT OPTION", p2, pt, testRole));
                    this.testToSql(ctx, String.format("REVOKE %s ON SERVER server1 FROM %s %s", p2, pt, testRole));
                    this.testToSql(ctx, String.format("REVOKE %s ON SERVER FROM %s %s", p2, pt, testRole), String.format("REVOKE %s ON SERVER server1 FROM %s %s", p2, pt, testRole), false);
                    this.testToSql(ctx, String.format("REVOKE GRANT OPTION FOR %s ON SERVER server1 FROM %s %s", p2, pt, testRole));
                    this.testToSql(ctx, String.format("GRANT %s ON DATABASE functional TO %s %s", p2, pt, testRole));
                    this.testToSql(ctx, String.format("GRANT %s ON DATABASE functional TO %s %s WITH GRANT OPTION", p2, pt, testRole));
                    this.testToSql(ctx, String.format("REVOKE %s ON DATABASE functional FROM %s %s", p2, pt, testRole));
                    this.testToSql(ctx, String.format("REVOKE GRANT OPTION FOR %s ON DATABASE functional FROM %s %s", p2, pt, testRole));
                }
                privileges = Arrays.stream(Privilege.values()).filter(p -> p != Privilege.OWNER && p != Privilege.CREATE && p != Privilege.VIEW_METADATA && p != Privilege.ANY).collect(Collectors.toList());
                for (Privilege p2 : privileges) {
                    this.testToSql(ctx, String.format("GRANT %s ON TABLE functional.alltypes TO %s %s", p2, pt, testRole));
                    this.testToSql(ctx, String.format("GRANT %s ON TABLE functional.alltypes TO %s %s WITH GRANT OPTION", p2, pt, testRole));
                    this.testToSql(ctx, String.format("REVOKE %s ON TABLE functional.alltypes FROM %s %s", p2, pt, testRole));
                    this.testToSql(ctx, String.format("REVOKE GRANT OPTION FOR %s ON TABLE functional.alltypes FROM %s %s", p2, pt, testRole));
                }
                privileges = Arrays.stream(Privilege.values()).filter(p -> p != Privilege.OWNER && (p == Privilege.CREATE || p == Privilege.DROP || p == Privilege.SELECT)).collect(Collectors.toList());
                for (Privilege p2 : privileges) {
                    this.testToSql(ctx, String.format("GRANT %s ON USER_DEFINED_FN functional.identity TO %s %s", p2, pt, testRole));
                    this.testToSql(ctx, String.format("GRANT %s ON USER_DEFINED_FN functional.identity TO %s %s WITH GRANT OPTION", p2, pt, testRole));
                    this.testToSql(ctx, String.format("REVOKE %s ON USER_DEFINED_FN functional.identity FROM %s %s", p2, pt, testRole));
                    this.testToSql(ctx, String.format("REVOKE GRANT OPTION FOR %s ON USER_DEFINED_FN functional.identity FROM %s %s", p2, pt, testRole));
                }
                this.testToSql(ctx, String.format("GRANT ALL ON URI '%s' TO %s %s", testUri, pt, testRole));
                this.testToSql(ctx, String.format("GRANT ALL ON URI '%s' TO %s %s WITH GRANT OPTION", testUri, pt, testRole));
                this.testToSql(ctx, String.format("REVOKE ALL ON URI '%s' FROM %s %s", testUri, pt, testRole));
                this.testToSql(ctx, String.format("REVOKE GRANT OPTION FOR ALL ON URI '%s' FROM %s %s", testUri, pt, testRole));
                this.testToSql(ctx, String.format("GRANT RWSTORAGE ON STORAGEHANDLER_URI '%s' TO %s %s", testStorageHandlerUri, pt, testRole));
                this.testToSql(ctx, String.format("GRANT RWSTORAGE ON STORAGEHANDLER_URI '%s' TO %s %s WITH GRANT OPTION", testStorageHandlerUri, pt, testRole));
                this.testToSql(ctx, String.format("REVOKE RWSTORAGE ON STORAGEHANDLER_URI '%s' FROM %s %s", testStorageHandlerUri, pt, testRole));
                this.testToSql(ctx, String.format("REVOKE GRANT OPTION FOR RWSTORAGE ON STORAGEHANDLER_URI '%s' FROM %s %s", testStorageHandlerUri, pt, testRole));
                this.testToSql(ctx, String.format("GRANT SELECT (id) ON TABLE functional.alltypes TO %s %s", pt, testRole));
                this.testToSql(ctx, String.format("GRANT SELECT (id) ON TABLE functional.alltypes TO %s %s WITH GRANT OPTION", pt, testRole));
                this.testToSql(ctx, String.format("REVOKE SELECT (id) ON TABLE functional.alltypes FROM %s %s", pt, testRole));
                this.testToSql(ctx, String.format("REVOKE GRANT OPTION FOR SELECT (id) ON TABLE functional.alltypes FROM %s %s", pt, testRole));
            }
            finally {
                catalog_.removeRole(testRole);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testGrantRevokeRoleStmt() {
        AnalysisContext ctx = this.createAnalysisCtx(this.createAuthorizationFactory());
        String testRole = "test_role";
        String testGroup = "test_group";
        try {
            catalog_.addRole(testRole);
            this.testToSql(ctx, String.format("GRANT ROLE %s TO GROUP %s", testRole, testGroup));
            this.testToSql(ctx, String.format("REVOKE ROLE %s FROM GROUP %s", testRole, testGroup));
        }
        finally {
            catalog_.removeRole(testRole);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testShowGrantPrincipalStmt() {
        AnalysisContext ctx = this.createAnalysisCtx(this.createAuthorizationFactory());
        String testRole = "test_role";
        String testUser = System.getProperty("user.name");
        String testUri = "hdfs://localhost:20500/test-warehouse";
        try {
            catalog_.addRole(testRole);
            this.testToSql(ctx, String.format("SHOW GRANT ROLE %s", testRole));
            this.testToSql(ctx, String.format("SHOW GRANT USER %s", testUser));
            this.testToSql(ctx, String.format("SHOW GRANT ROLE %s ON SERVER", testRole));
            this.testToSql(ctx, String.format("SHOW GRANT ROLE %s ON DATABASE functional", testRole));
            this.testToSql(ctx, String.format("SHOW GRANT ROLE %s ON TABLE functional.alltypes", testRole));
            this.testToSql(ctx, String.format("SHOW GRANT ROLE %s ON URI '%s'", testRole, testUri));
        }
        finally {
            catalog_.removeRole(testRole);
        }
    }

    @Test
    public void testShowRolesStmt() {
        AnalysisContext ctx = this.createAnalysisCtx(this.createAuthorizationFactory());
        String testGroup = "test_group";
        this.testToSql(ctx, "SHOW CURRENT ROLES");
        this.testToSql(ctx, "SHOW ROLES");
        this.testToSql(ctx, String.format("SHOW ROLE GRANT GROUP %s", testGroup));
    }

    @Test
    public void testInvalidate() {
        this.testToSql("INVALIDATE METADATA");
        this.testToSql("INVALIDATE METADATA functional.alltypes");
    }

    @Test
    public void testRefresh() {
        this.testToSql("REFRESH functional.alltypes");
        this.testToSql("REFRESH functional.alltypes PARTITION (year=2009, month=1)");
        this.testToSql("REFRESH functional.alltypes PARTITION (year=2009, month=1) PARTITION (year=2010, month=2)");
        this.testToSql("REFRESH FUNCTIONS functional");
        this.testToSql(this.createAnalysisCtx(this.createAuthorizationFactory()), "REFRESH AUTHORIZATION");
    }

    @Test
    public void testAdminFn() {
        this.testToSql(":shutdown()");
        this.testToSql(":shutdown('hostname')");
        this.testToSql(":shutdown('hostname', 1000)");
        this.testToSql(":shutdown(1000)");
    }

    @Test
    public void testShowTables() {
        this.testToSql("SHOW TABLES", "default", "SHOW TABLES");
        this.testToSql("SHOW TABLES IN functional");
        this.testToSql("SHOW TABLES LIKE 'alltypes*'", "functional", "SHOW TABLES LIKE 'alltypes*'");
        this.testToSql("SHOW TABLES IN functional LIKE 'alltypes*'");
    }

    @Test
    public void testShowMetadataTables() {
        String q1 = "SHOW METADATA TABLES IN iceberg_query_metadata";
        this.testToSql(q1, "functional_parquet", q1);
        String q2 = "SHOW METADATA TABLES IN iceberg_query_metadata LIKE '*file*'";
        this.testToSql(q2, "functional_parquet", q2);
        this.testToSql("SHOW METADATA TABLES IN functional_parquet.iceberg_query_metadata");
        this.testToSql("SHOW METADATA TABLES IN functional_parquet.iceberg_query_metadata LIKE '*file*'");
    }

    @Test
    public void testShowViews() {
        this.testToSql("SHOW VIEWS", "default", "SHOW VIEWS");
        this.testToSql("SHOW VIEWS IN functional");
        this.testToSql("SHOW VIEWS LIKE 'alltypes*'", "functional", "SHOW VIEWS LIKE 'alltypes*'");
        this.testToSql("SHOW VIEWS IN functional LIKE 'alltypes*'");
    }

    static {
        int numNonSemiJoinTypes = JoinOperator.values().length - 3 - leftSemiJoinTypes_.length - rightSemiJoinTypes_.length;
        nonSemiJoinTypes_ = new String[numNonSemiJoinTypes];
        int i = 0;
        int j = 0;
        for (JoinOperator op : JoinOperator.values()) {
            if (op.isCrossJoin() || op.isNullAwareLeftAntiJoin() || op.isIcebergDeleteJoin()) continue;
            ToSqlTest.joinTypes_[i++] = op.toString();
            if (op.isSemiJoin()) continue;
            ToSqlTest.nonSemiJoinTypes_[j++] = op.toString();
        }
    }
}

