/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.analysis;

import com.google.common.base.Preconditions;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import org.apache.impala.analysis.Analyzer;
import org.apache.impala.analysis.DescriptorTable;
import org.apache.impala.analysis.Expr;
import org.apache.impala.analysis.FunctionName;
import org.apache.impala.analysis.ParseNode;
import org.apache.impala.analysis.ResetMetadataStmt;
import org.apache.impala.analysis.SelectStmt;
import org.apache.impala.analysis.SlotDescriptor;
import org.apache.impala.analysis.TableName;
import org.apache.impala.analysis.TupleDescriptor;
import org.apache.impala.analysis.TupleId;
import org.apache.impala.catalog.Function;
import org.apache.impala.catalog.ScalarType;
import org.apache.impala.catalog.Type;
import org.apache.impala.common.AnalysisException;
import org.apache.impala.common.FrontendTestBase;
import org.apache.impala.common.RuntimeEnv;
import org.apache.impala.compat.MetastoreShim;
import org.apache.impala.util.FunctionUtils;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AnalyzerTest
extends FrontendTestBase {
    protected static final Logger LOG = LoggerFactory.getLogger(AnalyzerTest.class);
    protected static Map<ScalarType, String> typeToLiteralValue_ = new HashMap<ScalarType, String>();

    @Before
    public void setUpTest() throws Exception {
        RuntimeEnv.INSTANCE.reset();
        RuntimeEnv.INSTANCE.setTestEnv(true);
    }

    @AfterClass
    public static void cleanUpClass() throws Exception {
        RuntimeEnv.INSTANCE.reset();
    }

    protected void TblsAnalyzeOk(String query, TableName tbl) {
        Preconditions.checkState((boolean)tbl.isFullyQualified());
        Preconditions.checkState((boolean)query.contains("$TBL"));
        String uqQuery = query.replace("$TBL", tbl.getTbl());
        this.AnalyzesOk(uqQuery, this.createAnalysisCtx(tbl.getDb()));
        String fqQuery = query.replace("$TBL", tbl.toString());
        this.AnalyzesOk(fqQuery);
    }

    protected void TblsAnalysisError(String query, TableName tbl, String expectedError) {
        Preconditions.checkState((boolean)tbl.isFullyQualified());
        Preconditions.checkState((boolean)query.contains("$TBL"));
        String uqQuery = query.replace("$TBL", tbl.getTbl());
        this.AnalysisError(uqQuery, this.createAnalysisCtx(tbl.getDb()), expectedError);
        String fqQuery = query.replace("$TBL", tbl.toString());
        this.AnalysisError(fqQuery, expectedError);
    }

    @Test
    public void TestCompressedText() throws AnalysisException {
        this.AnalyzesOk("SELECT count(*) FROM functional_text_bzip.tinyinttable");
        this.AnalyzesOk("SELECT count(*) FROM functional_text_def.tinyinttable");
        this.AnalyzesOk("SELECT count(*) FROM functional_text_gzip.tinyinttable");
        this.AnalyzesOk("SELECT count(*) FROM functional_text_snap.tinyinttable");
    }

    @Test
    public void TestMemLayout() throws AnalysisException {
        this.testSelectStar();
        this.testNonNullable();
        this.testMixedNullable();
        this.testNonMaterializedSlots();
    }

    private void testSelectStar() throws AnalysisException {
        SelectStmt stmt = (SelectStmt)this.AnalyzesOk("select * from functional.AllTypes, functional.date_tbl");
        Analyzer analyzer = stmt.getAnalyzer();
        DescriptorTable descTbl = analyzer.getDescTbl();
        TupleDescriptor tupleDesc = descTbl.getTupleDesc(new TupleId(0));
        tupleDesc.materializeSlots();
        TupleDescriptor dateTblTupleDesc = descTbl.getTupleDesc(new TupleId(1));
        dateTblTupleDesc.materializeSlots();
        descTbl.computeMemLayout();
        Assert.assertEquals((double)89.0, (double)tupleDesc.getAvgSerializedSize(), (double)0.0);
        this.checkLayoutParams("functional.alltypes.timestamp_col", 16, 0, 80, 0, analyzer);
        this.checkLayoutParams("functional.alltypes.date_string_col", 12, 16, 80, 1, analyzer);
        this.checkLayoutParams("functional.alltypes.string_col", 12, 28, 80, 2, analyzer);
        this.checkLayoutParams("functional.alltypes.bigint_col", 8, 40, 80, 3, analyzer);
        this.checkLayoutParams("functional.alltypes.double_col", 8, 48, 80, 4, analyzer);
        this.checkLayoutParams("functional.alltypes.id", 4, 56, 80, 5, analyzer);
        this.checkLayoutParams("functional.alltypes.int_col", 4, 60, 80, 6, analyzer);
        this.checkLayoutParams("functional.alltypes.float_col", 4, 64, 80, 7, analyzer);
        this.checkLayoutParams("functional.alltypes.year", 4, 68, 81, 0, analyzer);
        this.checkLayoutParams("functional.alltypes.month", 4, 72, 81, 1, analyzer);
        this.checkLayoutParams("functional.alltypes.smallint_col", 2, 76, 81, 2, analyzer);
        this.checkLayoutParams("functional.alltypes.bool_col", 1, 78, 81, 3, analyzer);
        this.checkLayoutParams("functional.alltypes.tinyint_col", 1, 79, 81, 4, analyzer);
        Assert.assertEquals((double)12.0, (double)dateTblTupleDesc.getAvgSerializedSize(), (double)0.0);
        this.checkLayoutParams("functional.date_tbl.id_col", 4, 0, 12, 0, analyzer);
        this.checkLayoutParams("functional.date_tbl.date_col", 4, 4, 12, 1, analyzer);
        this.checkLayoutParams("functional.date_tbl.date_part", 4, 8, 12, 2, analyzer);
    }

    private void testNonNullable() throws AnalysisException {
        SelectStmt stmt = (SelectStmt)this.AnalyzesOk("select count(int_col), count(*) from functional.AllTypes");
        DescriptorTable descTbl = stmt.getAnalyzer().getDescTbl();
        TupleDescriptor aggDesc = descTbl.getTupleDesc(new TupleId(1));
        aggDesc.materializeSlots();
        descTbl.computeMemLayout();
        Assert.assertEquals((double)16.0, (double)aggDesc.getAvgSerializedSize(), (double)0.0);
        Assert.assertEquals((long)16L, (long)aggDesc.getByteSize());
        this.checkLayoutParams((SlotDescriptor)aggDesc.getSlots().get(0), 8, 0, 0, -1);
        this.checkLayoutParams((SlotDescriptor)aggDesc.getSlots().get(1), 8, 8, 0, -1);
    }

    private void testMixedNullable() throws AnalysisException {
        SelectStmt stmt = (SelectStmt)this.AnalyzesOk("select sum(int_col), count(*) from functional.AllTypes");
        DescriptorTable descTbl = stmt.getAnalyzer().getDescTbl();
        TupleDescriptor aggDesc = descTbl.getTupleDesc(new TupleId(1));
        aggDesc.materializeSlots();
        descTbl.computeMemLayout();
        Assert.assertEquals((double)16.0, (double)aggDesc.getAvgSerializedSize(), (double)0.0);
        Assert.assertEquals((long)17L, (long)aggDesc.getByteSize());
        this.checkLayoutParams((SlotDescriptor)aggDesc.getSlots().get(0), 8, 0, 16, 0);
        this.checkLayoutParams((SlotDescriptor)aggDesc.getSlots().get(1), 8, 8, 0, -1);
    }

    private void testNonMaterializedSlots() throws AnalysisException {
        SelectStmt stmt = (SelectStmt)this.AnalyzesOk("select * from functional.alltypes, functional.date_tbl");
        Analyzer analyzer = stmt.getAnalyzer();
        DescriptorTable descTbl = analyzer.getDescTbl();
        TupleDescriptor tupleDesc = descTbl.getTupleDesc(new TupleId(0));
        tupleDesc.materializeSlots();
        List slots = tupleDesc.getSlots();
        ((SlotDescriptor)slots.get(0)).setIsMaterialized(false);
        ((SlotDescriptor)slots.get(7)).setIsMaterialized(false);
        ((SlotDescriptor)slots.get(9)).setIsMaterialized(false);
        TupleDescriptor dateTblTupleDesc = descTbl.getTupleDesc(new TupleId(1));
        dateTblTupleDesc.materializeSlots();
        slots = dateTblTupleDesc.getSlots();
        ((SlotDescriptor)slots.get(0)).setIsMaterialized(false);
        ((SlotDescriptor)slots.get(1)).setIsMaterialized(false);
        descTbl.computeMemLayout();
        Assert.assertEquals((double)64.0, (double)tupleDesc.getAvgSerializedSize(), (double)0.0);
        this.checkLayoutParams("functional.alltypes.id", 0, -1, 0, 0, analyzer);
        this.checkLayoutParams("functional.alltypes.double_col", 0, -1, 0, 0, analyzer);
        this.checkLayoutParams("functional.alltypes.string_col", 0, -1, 0, 0, analyzer);
        this.checkLayoutParams("functional.alltypes.timestamp_col", 16, 0, 56, 0, analyzer);
        this.checkLayoutParams("functional.alltypes.date_string_col", 12, 16, 56, 1, analyzer);
        this.checkLayoutParams("functional.alltypes.bigint_col", 8, 28, 56, 2, analyzer);
        this.checkLayoutParams("functional.alltypes.int_col", 4, 36, 56, 3, analyzer);
        this.checkLayoutParams("functional.alltypes.float_col", 4, 40, 56, 4, analyzer);
        this.checkLayoutParams("functional.alltypes.year", 4, 44, 56, 5, analyzer);
        this.checkLayoutParams("functional.alltypes.month", 4, 48, 56, 6, analyzer);
        this.checkLayoutParams("functional.alltypes.smallint_col", 2, 52, 56, 7, analyzer);
        this.checkLayoutParams("functional.alltypes.bool_col", 1, 54, 57, 0, analyzer);
        this.checkLayoutParams("functional.alltypes.tinyint_col", 1, 55, 57, 1, analyzer);
        Assert.assertEquals((double)4.0, (double)dateTblTupleDesc.getAvgSerializedSize(), (double)0.0);
        this.checkLayoutParams("functional.date_tbl.id_col", 0, -1, 0, 0, analyzer);
        this.checkLayoutParams("functional.date_tbl.date_col", 0, -1, 0, 0, analyzer);
        this.checkLayoutParams("functional.date_tbl.date_part", 4, 0, 4, 0, analyzer);
    }

    private void checkLayoutParams(SlotDescriptor d, int byteSize, int byteOffset, int nullIndicatorByte, int nullIndicatorBit) {
        Assert.assertEquals((long)byteSize, (long)d.getByteSize());
        Assert.assertEquals((long)byteOffset, (long)d.getByteOffset());
        Assert.assertEquals((long)nullIndicatorByte, (long)d.getNullIndicatorByte());
        Assert.assertEquals((long)nullIndicatorBit, (long)d.getNullIndicatorBit());
    }

    private void checkLayoutParams(String colAlias, int byteSize, int byteOffset, int nullIndicatorByte, int nullIndicatorBit, Analyzer analyzer) {
        List<String> colAliasRawPath = Arrays.asList(colAlias.split("\\."));
        SlotDescriptor d = analyzer.getSlotDescriptor(colAliasRawPath);
        this.checkLayoutParams(d, byteSize, byteOffset, nullIndicatorByte, nullIndicatorBit);
    }

    protected void checkExprType(String query, Type type) {
        SelectStmt select = (SelectStmt)this.AnalyzesOk(query);
        Assert.assertEquals((Object)((Expr)select.getResultExprs().get(0)).getType(), (Object)type);
    }

    @Test
    public void TestUnsupportedTypes() {
        this.AnalysisError("select * from functional.unsupported_timestamp_partition", "Failed to load metadata for table: 'functional.unsupported_timestamp_partition'");
        this.AnalysisError("select * from functional.unsupported_binary_partition", "Failed to load metadata for table: 'functional.unsupported_binary_partition'");
        this.AnalyzesOk("describe functional_hbase.allcomplextypes");
        for (ScalarType t : Type.getUnsupportedTypes()) {
            this.AnalysisError(String.format("create table new_table (new_col %s)", t.toSql()), String.format("Unsupported data type: %s", t.toSql()));
            this.AnalysisError(String.format("create table new_table (new_col int) PARTITIONED BY (p_col %s)", t.toSql()), String.format("Unsupported data type: %s", t.toSql()));
            this.AnalysisError(String.format("alter table functional.alltypes add columns (new_col %s)", t.toSql()), String.format("Unsupported data type: %s", t.toSql()));
            this.AnalysisError(String.format("alter table functional.alltypes change column int_col new_col %s", t.toSql()), String.format("Unsupported data type: %s", t.toSql()));
            String udfSuffix = " LOCATION '/test-warehouse/libTestUdfs.so' SYMBOL='_Z8IdentityPN10impala_udf15FunctionContextERKNS_10BooleanValE'";
            this.AnalysisError(String.format("create function foo(VARCHAR(5)) RETURNS %s LOCATION '/test-warehouse/libTestUdfs.so' SYMBOL='_Z8IdentityPN10impala_udf15FunctionContextERKNS_10BooleanValE'", t.toSql()), String.format("Unsupported data type: %s", t.toSql()));
            this.AnalysisError(String.format("create function foo(%s) RETURNS int LOCATION '/test-warehouse/libTestUdfs.so' SYMBOL='_Z8IdentityPN10impala_udf15FunctionContextERKNS_10BooleanValE'", t.toSql()), String.format("Unsupported data type: %s", t.toSql()));
            String udaSuffix = " LOCATION '/test-warehouse/libTestUdas.so' UPDATE_FN='AggUpdate'";
            this.AnalysisError(String.format("create aggregate function foo(string, double) RETURNS %s LOCATION '/test-warehouse/libTestUdas.so' UPDATE_FN='AggUpdate'", t.toSql()), String.format("Unsupported data type: %s", t.toSql()));
            this.AnalysisError(String.format("create aggregate function foo(%s, double) RETURNS int LOCATION '/test-warehouse/libTestUdas.so' UPDATE_FN='AggUpdate'", t.toSql()), String.format("Unsupported data type: %s", t.toSql()));
            this.AnalysisError(String.format("select cast('abc' as %s)", t.toSql()), String.format("Unsupported data type: %s", t.toSql()));
        }
    }

    @Test
    public void TestVirtualColumnInputFileName() {
        this.AnalyzesOk("select input__file__name from functional.alltypes");
        this.AnalyzesOk("select input__file__name, id from functional_parquet.alltypes");
        this.AnalyzesOk("select input__file__name, * from functional_orc_def.alltypes");
        this.AnalyzesOk("select input__file__name, * from (select input__file__name, * from functional_avro.alltypes) v");
        this.AnalyzesOk("select id, input__file__name from functional_parquet.iceberg_partitioned");
        this.AnalyzesOk("select input__file__name, * from functional_parquet.complextypestbl c, c.int_array");
        this.AnalyzesOk("select c.input__file__name, c.int_array.* from functional_parquet.complextypestbl c, c.int_array");
        this.AnalysisError("select id, nested_struct.input__file__name from functional_parquet.complextypestbl", "Could not resolve column/field reference: 'nested_struct.input__file__name'");
        this.AnalysisError("select c.int_array.input__file__name, c.int_array.* from functional_parquet.complextypestbl c, c.int_array", "Could not resolve column/field reference: 'c.int_array.input__file__name'");
        this.AnalysisError("select id, nested_struct.input__file__name from functional_parquet.complextypestbl", "Could not resolve column/field reference: 'nested_struct.input__file__name'");
        this.AnalysisError("select input__file__name from functional_kudu.alltypes", "Could not resolve column/field reference: 'input__file__name'");
        this.AnalysisError("select input__file__name from functional_hbase.alltypes", "Could not resolve column/field reference: 'input__file__name'");
    }

    @Test
    public void TestCopyTestCase() {
        this.AnalyzesOk("copy testcase to 'hdfs:///tmp' select * from functional.alltypes");
        this.AnalyzesOk("copy testcase to 'hdfs:///tmp' select * from functional.alltypes union select * from functional.alltypes");
        this.AnalyzesOk("copy testcase to 'hdfs:///tmp' select * from functional.alltypes_view");
        this.AnalyzesOk("copy testcase to 'hdfs:///tmp' select * from functional.alltypes_view union all select * from functional.alltypes");
        this.AnalyzesOk("copy testcase to 'hdfs:///tmp' with v as (select 1) select * from v");
        this.AnalysisError("copy testcase to 'hdfs:///foo' select 1", "Path does not exist: hdfs://localhost:20500/foo");
        this.AnalysisError("copy testcase from 'hdfs:///tmp/file-doesnot-exist'", "Path does not exist");
    }

    @Test
    public void TestBinaryHBaseTable() {
        this.AnalyzesOk("select * from functional_hbase.alltypessmallbinary");
    }

    @Test
    public void TestUnsupportedSerde() {
        this.AnalysisError("select * from functional.bad_serde", "Failed to load metadata for table: 'functional.bad_serde'");
    }

    @Test
    public void TestResetMetadata() {
        BiConsumer<ParseNode, ResetMetadataStmt.Action> assertAction = (parseNode, action) -> {
            Preconditions.checkArgument((boolean)(parseNode instanceof ResetMetadataStmt));
            Assert.assertEquals((Object)action, (Object)((ResetMetadataStmt)parseNode).getAction());
        };
        assertAction.accept(this.AnalyzesOk("invalidate metadata"), ResetMetadataStmt.Action.INVALIDATE_METADATA_ALL);
        assertAction.accept(this.AnalyzesOk("invalidate metadata functional.alltypessmall"), ResetMetadataStmt.Action.INVALIDATE_METADATA_TABLE);
        assertAction.accept(this.AnalyzesOk("invalidate metadata functional.alltypes_view"), ResetMetadataStmt.Action.INVALIDATE_METADATA_TABLE);
        assertAction.accept(this.AnalyzesOk("invalidate metadata functional.bad_serde"), ResetMetadataStmt.Action.INVALIDATE_METADATA_TABLE);
        assertAction.accept(this.AnalyzesOk("refresh functional.alltypessmall"), ResetMetadataStmt.Action.REFRESH_TABLE);
        assertAction.accept(this.AnalyzesOk("refresh functional.alltypes_view"), ResetMetadataStmt.Action.REFRESH_TABLE);
        assertAction.accept(this.AnalyzesOk("refresh functional.bad_serde"), ResetMetadataStmt.Action.REFRESH_TABLE);
        assertAction.accept(this.AnalyzesOk("refresh functional.alltypessmall partition (year=2009, month=1)"), ResetMetadataStmt.Action.REFRESH_PARTITION);
        assertAction.accept(this.AnalyzesOk("refresh functional.alltypessmall partition (year=2009, month=NULL)"), ResetMetadataStmt.Action.REFRESH_PARTITION);
        assertAction.accept(this.AnalyzesOk("refresh functional.alltypessmall partition (year=2009, month=1) partition (year=2010, month=2) partition (year=2010, month=4)"), ResetMetadataStmt.Action.REFRESH_PARTITION);
        assertAction.accept(this.AnalyzesOk("refresh authorization", this.createAnalysisCtx(this.createAuthorizationFactory())), ResetMetadataStmt.Action.REFRESH_AUTHORIZATION);
        assertAction.accept(this.AnalyzesOk("invalidate metadata functional.unknown_table"), ResetMetadataStmt.Action.INVALIDATE_METADATA_TABLE);
        assertAction.accept(this.AnalyzesOk("invalidate metadata unknown_db.unknown_table"), ResetMetadataStmt.Action.INVALIDATE_METADATA_TABLE);
        this.AnalysisError("refresh functional.unknown_table", "Table does not exist: functional.unknown_table");
        this.AnalysisError("refresh unknown_db.unknown_table", "Database does not exist: unknown_db");
        this.AnalysisError("refresh functional.alltypessmall partition (year=2009, int_col=10)", "Column 'int_col' is not a partition column in table: functional.alltypessmall");
        this.AnalysisError("refresh functional.alltypessmall partition (year=2009)", "Items in partition spec must exactly match the partition columns in the table definition: functional.alltypessmall (1 vs 2)");
        this.AnalysisError("refresh functional.alltypessmall partition (year=2009, year=2009)", "Duplicate partition key name: year");
        this.AnalysisError("refresh functional.alltypessmall partition (year=2009, month='foo')", "Value of partition spec (column=month) has incompatible type: 'STRING'. Expected type: 'INT'");
        this.AnalysisError("refresh functional.alltypessmall partition (year=2009) partition (month=1)", "Items in partition spec must exactly match the partition columns in the table definition: functional.alltypessmall (1 vs 2)");
        this.AnalysisError("refresh functional.zipcode_incomes partition (year=2009, month=1)", "Table is not partitioned: functional.zipcode_incomes");
        this.AnalysisError("refresh functional_parquet.iceberg_partitioned partition(action='view')", "Partition clause in this statement is not supported for Iceberg tables.");
    }

    @Test
    public void TestExplain() {
        this.AnalysisError("explain insert into table functional.alltypessmall partition (year=2009, month=4, year=10)select id, bool_col, tinyint_col, smallint_col, int_col, bigint_col, float_col, double_col, date_string_col, string_col, timestamp_col from functional.alltypes", "Duplicate column 'year' in partition clause");
        this.AnalysisError("explain select id from (select id+2 from functional_hbase.alltypessmall) a", "Could not resolve column/field reference: 'id'");
        this.AnalysisError("explain upsert into table functional.alltypes select * from functional.alltypes", "UPSERT is only supported for Kudu tables");
        this.AnalyzesOk("explain select * from functional.AllTypes");
        this.AnalyzesOk("explain insert into table functional.alltypessmall partition (year=2009, month=4)select id, bool_col, tinyint_col, smallint_col, int_col, int_col, float_col, float_col, date_string_col, string_col, timestamp_col from functional.alltypes");
        this.AnalyzesOk("explain upsert into table functional_kudu.testtbl select * from functional_kudu.testtbl");
    }

    @Test
    public void TestLimitAndOffset() {
        this.AnalyzesOk("select * from functional.AllTypes limit 10 * 10 + 10 - 10 % 10");
        this.AnalyzesOk("select * from functional.AllTypes limit 1 ^ 0 | 3 & 3");
        this.AnalyzesOk("select * from functional.AllTypes order by id limit 10 offset 1+2*3%4");
        this.AnalyzesOk("select t5.id from (select id from functional.AllTypes order by id limit 10 offset 2) t5");
        this.AnalyzesOk("with t5 as (select id from functional.AllTypes order by id limit 10 offset 2) select * from t5");
        this.AnalyzesOk("select id, bool_col from functional.AllTypes limit CAST(10.0 AS INT)");
        this.AnalyzesOk("select id, bool_col from functional.AllTypes limit CAST(NOT FALSE AS INT)");
        this.AnalyzesOk("select * from functional.AllTypes order by id limit 10 offset CAST(1.0 AS INT)");
        this.AnalysisError("select * from functional.AllTypes limit 10 - 20", "LIMIT must be a non-negative integer: 10 - 20 = -10");
        this.AnalysisError("select * from functional.AllTypes order by id limit 10 offset 10 - 20", "OFFSET must be a non-negative integer: 10 - 20 = -10");
        this.AnalysisError("select * from functional.AllTypes limit 10.0", "LIMIT expression must be an integer type but is 'DECIMAL(3,1)': 10.0");
        this.AnalysisError("select * from functional.AllTypes limit NOT FALSE", "LIMIT expression must be an integer type but is 'BOOLEAN': NOT FALSE");
        this.AnalysisError("select * from functional.AllTypes limit CAST(\"asdf\" AS INT)", "LIMIT expression evaluates to NULL: CAST('asdf' AS INT)");
        this.AnalysisError("select * from functional.AllTypes order by id limit 10 OFFSET 10.0", "OFFSET expression must be an integer type but is 'DECIMAL(3,1)': 10.0");
        this.AnalysisError("select * from functional.AllTypes order by id limit 10 offset CAST('asdf' AS INT)", "OFFSET expression evaluates to NULL: CAST('asdf' AS INT)");
        this.AnalysisError("select id, bool_col from functional.AllTypes limit id < 10", "LIMIT expression must be a constant expression: id < 10");
        this.AnalysisError("select id, bool_col from functional.AllTypes order by id limit 10 offset id < 10", "OFFSET expression must be a constant expression: id < 10");
        this.AnalysisError("select id, bool_col from functional.AllTypes limit count(*)", "LIMIT expression must be a constant expression: count(*)");
        this.AnalysisError("select id, bool_col from functional.AllTypes order by id limit 10 offset count(*)", "OFFSET expression must be a constant expression: count(*)");
        this.AnalysisError("SELECT a FROM test LIMIT 10 OFFSET 5", "OFFSET requires an ORDER BY clause: LIMIT 10 OFFSET 5");
        this.AnalysisError("SELECT x.id FROM (SELECT id FROM alltypesagg LIMIT 5 OFFSET 5) x ORDER BY x.id LIMIT 100 OFFSET 4", "OFFSET requires an ORDER BY clause: LIMIT 5 OFFSET 5");
        this.AnalysisError("SELECT a FROM test OFFSET 5", "OFFSET requires an ORDER BY clause: OFFSET 5");
        this.AnalyzesOk("SELECT id FROM functional.Alltypes ORDER BY bool_col OFFSET 5");
    }

    @Test
    public void TestAnalyzeShowCreateTable() {
        this.AnalyzesOk("show create table functional.AllTypes");
        this.AnalyzesOk("show create table functional.alltypes_view");
        this.AnalysisError("show create table functional.not_a_table", "Table does not exist: functional.not_a_table");
        this.AnalysisError("show create table doesnt_exist", "Table does not exist: default.doesnt_exist");
    }

    @Test
    public void TestAnalyzeTransactional() {
        Assume.assumeTrue((MetastoreShim.getMajorVersion() > 2L ? 1 : 0) != 0);
        String fullAcidErrorMsg = "%s not supported on full transactional (ACID) table: functional_orc_def.full_transactional_table";
        String transactionalErrorMsg = "%s not supported on transactional (ACID) table: %s";
        String insertOnlyTbl = "functional.insert_only_transactional_table";
        String fullTxnTbl = "functional_orc_def.full_transactional_table";
        this.AnalyzesOk("create table test as select * from functional_orc_def.full_transactional_table");
        this.AnalyzesOk("create table test as select * from functional.insert_only_transactional_table");
        this.AnalyzesOk("create table test like functional_orc_def.full_transactional_table");
        this.AnalyzesOk("create table test like functional.insert_only_transactional_table");
        this.AnalyzesOk("insert into functional.testtbl select 1,'test',* from functional_orc_def.full_transactional_table");
        this.AnalyzesOk("insert into functional.testtbl select *,'test',1 from functional.insert_only_transactional_table");
        this.AnalyzesOk("insert into functional.insert_only_transactional_table select * from functional.insert_only_transactional_table");
        this.AnalyzesOk("compute stats functional_orc_def.full_transactional_table");
        this.AnalyzesOk("compute stats functional.insert_only_transactional_table");
        this.AnalyzesOk("select * from functional_orc_def.full_transactional_table");
        this.AnalyzesOk("select * from functional.insert_only_transactional_table");
        this.AnalyzesOk("drop table functional_orc_def.full_transactional_table");
        this.AnalyzesOk("drop table functional.insert_only_transactional_table");
        this.AnalysisError("truncate table functional_orc_def.full_transactional_table", String.format(fullAcidErrorMsg, "TRUNCATE"));
        this.AnalyzesOk("truncate table functional.insert_only_transactional_table");
        this.AnalysisError("alter table functional_orc_def.full_transactional_table add columns (col2 string)", String.format(transactionalErrorMsg, "ALTER TABLE", fullTxnTbl));
        this.AnalysisError("alter table functional.insert_only_transactional_table add columns (col2 string)", String.format(transactionalErrorMsg, "ALTER TABLE", insertOnlyTbl));
        this.AnalysisError("drop stats functional_orc_def.full_transactional_table", String.format(transactionalErrorMsg, "DROP STATS", fullTxnTbl));
        this.AnalysisError("drop stats functional.insert_only_transactional_table", String.format(transactionalErrorMsg, "DROP STATS", insertOnlyTbl));
        this.AnalyzesOk("describe functional.insert_only_transactional_table");
        this.AnalyzesOk("describe functional_orc_def.full_transactional_table");
        this.AnalyzesOk("show column stats functional_orc_def.full_transactional_table");
        this.AnalyzesOk("show column stats functional.insert_only_transactional_table");
        this.AnalyzesOk("refresh functional.insert_only_transactional_table");
        this.AnalyzesOk("refresh functional_orc_def.full_transactional_table");
        this.AnalysisError("refresh functional.insert_only_transactional_table partition (j=1)", "Refreshing partitions is not allowed on transactional tables. Try to refresh the whole table instead.");
        this.AnalysisError("refresh functional.insert_only_transactional_table partition (j=1) partition (j=2)", "Refreshing partitions is not allowed on transactional tables. Try to refresh the whole table instead.");
        this.AnalysisError("refresh functional_orc_def.full_transactional_table partition (j=1)", "Refreshing partitions is not allowed on transactional tables. Try to refresh the whole table instead.");
    }

    @Test
    public void TestAnalyzeMaterializedView() {
        Assume.assumeTrue((MetastoreShim.getMajorVersion() > 2L ? 1 : 0) != 0);
        this.AnalysisError("alter table functional.materialized_view set tblproperties ('foo'='bar')", "Write not supported. Table functional.materialized_view  access type is: READONLY");
        this.AnalysisError("insert into table functional.materialized_view select * from functional.insert_only_transactional_table", "Impala does not support INSERTing into views: functional.materialized_view");
        this.AnalysisError("drop table functional.materialized_view ", "Write not supported. Table functional.materialized_view  access type is: READONLY");
        this.AnalyzesOk("Select * from functional.materialized_view");
    }

    private Function createFunction(boolean hasVarArgs, Type ... args) {
        return new Function(new FunctionName("test"), args, (Type)Type.INVALID, hasVarArgs);
    }

    @Test
    public void TestFunctionMatching() {
        Function[] fns = new Function[]{this.createFunction(false, new Type[0]), this.createFunction(false, new Type[]{Type.INT}), this.createFunction(true, new Type[]{Type.INT}), this.createFunction(false, new Type[]{Type.TINYINT}), this.createFunction(true, new Type[]{Type.TINYINT}), this.createFunction(false, new Type[]{Type.DOUBLE}), this.createFunction(true, new Type[]{Type.DOUBLE}), this.createFunction(false, new Type[]{Type.DOUBLE, Type.DOUBLE}), this.createFunction(true, new Type[]{Type.DOUBLE, Type.DOUBLE}), this.createFunction(false, new Type[]{Type.SMALLINT, Type.TINYINT}), this.createFunction(false, new Type[]{Type.INT, Type.DOUBLE, Type.DOUBLE, Type.DOUBLE}), this.createFunction(true, new Type[]{Type.INT, Type.STRING, Type.INT}), this.createFunction(false, new Type[]{Type.TINYINT, Type.STRING, Type.TINYINT, Type.INT, Type.TINYINT}), this.createFunction(false, new Type[]{Type.TINYINT, Type.STRING, Type.BIGINT, Type.INT, Type.TINYINT}), this.createFunction(false, new Type[]{Type.DATE, Type.STRING, Type.DATE}), this.createFunction(true, new Type[]{Type.TIMESTAMP}), this.createFunction(true, new Type[]{Type.DATE}), this.createFunction(true, new Type[]{Type.STRING}), this.createFunction(true, new Type[]{Type.BINARY})};
        Assert.assertFalse((boolean)fns[1].compare(fns[0], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((boolean)fns[1].compare(fns[2], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((boolean)fns[1].compare(fns[3], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((boolean)fns[1].compare(fns[4], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[1].compare(fns[5], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[1].compare(fns[6], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[1].compare(fns[7], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[1].compare(fns[8], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((boolean)fns[1].compare(fns[2], Function.CompareMode.IS_INDISTINGUISHABLE));
        Assert.assertTrue((boolean)fns[3].compare(fns[4], Function.CompareMode.IS_INDISTINGUISHABLE));
        Assert.assertTrue((boolean)fns[5].compare(fns[6], Function.CompareMode.IS_INDISTINGUISHABLE));
        Assert.assertFalse((boolean)fns[5].compare(fns[7], Function.CompareMode.IS_INDISTINGUISHABLE));
        Assert.assertFalse((boolean)fns[5].compare(fns[8], Function.CompareMode.IS_INDISTINGUISHABLE));
        Assert.assertTrue((boolean)fns[6].compare(fns[7], Function.CompareMode.IS_INDISTINGUISHABLE));
        Assert.assertTrue((boolean)fns[6].compare(fns[8], Function.CompareMode.IS_INDISTINGUISHABLE));
        Assert.assertTrue((boolean)fns[7].compare(fns[8], Function.CompareMode.IS_INDISTINGUISHABLE));
        Assert.assertFalse((boolean)fns[1].compare(fns[3], Function.CompareMode.IS_INDISTINGUISHABLE));
        Assert.assertFalse((boolean)fns[1].compare(fns[4], Function.CompareMode.IS_INDISTINGUISHABLE));
        Assert.assertFalse((boolean)fns[9].compare(fns[4], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((boolean)fns[2].compare(fns[9], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((boolean)fns[8].compare(fns[10], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[10].compare(fns[8], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((boolean)fns[11].compare(fns[12], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[11].compare(fns[13], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[15].compare(fns[14], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((boolean)fns[15].compare(fns[14], Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[15].compare(fns[16], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((boolean)fns[15].compare(fns[16], Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[15].compare(fns[17], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((boolean)fns[15].compare(fns[17], Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[16].compare(fns[14], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((boolean)fns[16].compare(fns[14], Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[16].compare(fns[15], Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[16].compare(fns[17], Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((boolean)fns[16].compare(fns[17], Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[17].compare(fns[14], Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[17].compare(fns[15], Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[17].compare(fns[16], Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        Assert.assertFalse((boolean)fns[18].compare(fns[17], Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        for (int i = 0; i < fns.length; ++i) {
            for (int j = 0; j < fns.length; ++j) {
                if (i == j) {
                    Assert.assertTrue((boolean)fns[i].compare(fns[i], Function.CompareMode.IS_IDENTICAL));
                    Assert.assertTrue((boolean)fns[i].compare(fns[i], Function.CompareMode.IS_INDISTINGUISHABLE));
                    Assert.assertTrue((boolean)fns[i].compare(fns[i], Function.CompareMode.IS_SUPERTYPE_OF));
                    continue;
                }
                Assert.assertFalse((boolean)fns[i].compare(fns[j], Function.CompareMode.IS_IDENTICAL));
                if (fns[i].compare(fns[j], Function.CompareMode.IS_INDISTINGUISHABLE)) {
                    Assert.assertTrue((fns[i].compare(fns[j], Function.CompareMode.IS_SUPERTYPE_OF) || fns[j].compare(fns[i], Function.CompareMode.IS_SUPERTYPE_OF) ? 1 : 0) != 0);
                    Assert.assertTrue((boolean)fns[j].compare(fns[i], Function.CompareMode.IS_INDISTINGUISHABLE));
                    continue;
                }
                if (!fns[i].compare(fns[j], Function.CompareMode.IS_INDISTINGUISHABLE)) continue;
            }
        }
    }

    @Test
    public void testFunctionMatchScore() {
        Function fnDate = this.createFunction(false, new Type[]{Type.DATE, Type.DATE, Type.INT});
        Function fnTimestamp = this.createFunction(false, new Type[]{Type.TIMESTAMP, Type.TIMESTAMP, Type.INT});
        Function fn = this.createFunction(false, new Type[]{Type.STRING, Type.DATE, Type.TINYINT});
        Assert.assertEquals((long)-1L, (long)fnDate.calcMatchScore(fn, Function.CompareMode.IS_SUPERTYPE_OF));
        int fnDateScore = fnDate.calcMatchScore(fn, Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF);
        Assert.assertTrue((fnDateScore >= 0 ? 1 : 0) != 0);
        Assert.assertEquals((long)-1L, (long)fnTimestamp.calcMatchScore(fn, Function.CompareMode.IS_SUPERTYPE_OF));
        int fnTimestampScore = fnTimestamp.calcMatchScore(fn, Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF);
        Assert.assertTrue((fnTimestampScore >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((fnDateScore > fnTimestampScore ? 1 : 0) != 0);
        fn = this.createFunction(false, new Type[]{Type.STRING, Type.TIMESTAMP, Type.TINYINT});
        Assert.assertEquals((long)-1L, (long)fnDate.calcMatchScore(fn, Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        Assert.assertEquals((long)-1L, (long)fnTimestamp.calcMatchScore(fn, Function.CompareMode.IS_SUPERTYPE_OF));
        Assert.assertTrue((fnTimestamp.calcMatchScore(fn, Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF) >= 0 ? 1 : 0) != 0);
        fn = this.createFunction(false, new Type[]{Type.STRING, Type.STRING, Type.TINYINT});
        Assert.assertEquals((long)-1L, (long)fnDate.calcMatchScore(fn, Function.CompareMode.IS_SUPERTYPE_OF));
        fnDateScore = fnDate.calcMatchScore(fn, Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF);
        Assert.assertTrue((fnDateScore >= 0 ? 1 : 0) != 0);
        Assert.assertEquals((long)-1L, (long)fnTimestamp.calcMatchScore(fn, Function.CompareMode.IS_SUPERTYPE_OF));
        fnTimestampScore = fnTimestamp.calcMatchScore(fn, Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF);
        Assert.assertTrue((fnTimestampScore >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((fnDateScore == fnTimestampScore ? 1 : 0) != 0);
    }

    @Test
    public void testResolveFunction() {
        Function[] fns = new Function[]{this.createFunction(false, new Type[]{Type.TIMESTAMP, Type.TIMESTAMP, Type.INT}), this.createFunction(false, new Type[]{Type.DATE, Type.DATE, Type.INT})};
        Function fnStr = this.createFunction(false, new Type[]{Type.STRING, Type.STRING, Type.TINYINT});
        Assert.assertEquals((Object)fns[0], (Object)FunctionUtils.resolveFunction(Arrays.asList(fns), (Function)fnStr, (Function.CompareMode)Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        Function fnTimestamp = this.createFunction(false, new Type[]{Type.TIMESTAMP, Type.STRING, Type.TINYINT});
        Assert.assertEquals((Object)fns[0], (Object)FunctionUtils.resolveFunction(Arrays.asList(fns), (Function)fnTimestamp, (Function.CompareMode)Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        fnTimestamp = this.createFunction(false, new Type[]{Type.TIMESTAMP, Type.DATE, Type.TINYINT});
        Assert.assertEquals((Object)fns[0], (Object)FunctionUtils.resolveFunction(Arrays.asList(fns), (Function)fnTimestamp, (Function.CompareMode)Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        fnTimestamp = this.createFunction(false, new Type[]{Type.DATE, Type.TIMESTAMP, Type.TINYINT});
        Assert.assertEquals((Object)fns[0], (Object)FunctionUtils.resolveFunction(Arrays.asList(fns), (Function)fnTimestamp, (Function.CompareMode)Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        Function fnDate = this.createFunction(false, new Type[]{Type.DATE, Type.STRING, Type.TINYINT});
        Assert.assertEquals((Object)fns[1], (Object)FunctionUtils.resolveFunction(Arrays.asList(fns), (Function)fnDate, (Function.CompareMode)Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
        fnDate = this.createFunction(false, new Type[]{Type.STRING, Type.DATE, Type.TINYINT});
        Assert.assertEquals((Object)fns[1], (Object)FunctionUtils.resolveFunction(Arrays.asList(fns), (Function)fnDate, (Function.CompareMode)Function.CompareMode.IS_NONSTRICT_SUPERTYPE_OF));
    }

    @Test
    public void testAnalyzeBucketed() {
        this.AnalyzesOk("select count(*) from functional.bucketed_table");
        this.AnalyzesOk("select count(*) from functional.bucketed_ext_table");
        this.AnalyzesOk("drop stats functional.bucketed_table");
        this.AnalyzesOk("describe functional.bucketed_table");
        this.AnalyzesOk("show column stats functional.bucketed_table");
        this.AnalyzesOk("create table test as select * from functional.bucketed_table");
        this.AnalyzesOk("compute stats functional.bucketed_table");
        this.AnalyzesOk("drop table functional.bucketed_table");
        String errorMsgBucketed = "functional.bucketed_table is a bucketed table. Only read operations are supported on such tables.";
        String errorMsgExtBucketed = "functional.bucketed_ext_table is a bucketed table. Only read operations are supported on such tables.";
        String errorMsgInsertOnlyBucketed = "functional.insert_only_transactional_bucketed_table is a bucketed table. Only read operations are supported on such tables.";
        String errorMsg = "Table bucketed_ext_table write not supported";
        if (MetastoreShim.getMajorVersion() > 2L) {
            this.AnalyzesOk("select count(*) from functional.insert_only_transactional_bucketed_table");
            this.AnalysisError("insert into functional.insert_only_transactional_bucketed_table select * from functional.insert_only_transactional_bucketed_table", errorMsgInsertOnlyBucketed);
            this.AnalysisError("insert into functional.bucketed_ext_table select * from functional.bucketed_ext_table", errorMsgExtBucketed);
        } else {
            this.AnalysisError("insert into functional.bucketed_ext_table select * from functional.bucketed_ext_table", errorMsgExtBucketed);
        }
        this.AnalysisError("insert into functional.bucketed_table select * from functional.bucketed_table", errorMsgBucketed);
        this.AnalysisError("create table test like functional.bucketed_table", errorMsgBucketed);
        this.AnalysisError("truncate table functional.bucketed_table", errorMsgBucketed);
        this.AnalysisError("alter table functional.bucketed_table add columns(col3 int)", errorMsgBucketed);
    }

    static {
        typeToLiteralValue_.put(Type.BOOLEAN, "true");
        typeToLiteralValue_.put(Type.TINYINT, "1");
        typeToLiteralValue_.put(Type.SMALLINT, "128");
        typeToLiteralValue_.put(Type.INT, "32768");
        typeToLiteralValue_.put(Type.BIGINT, "2147483648");
        typeToLiteralValue_.put(Type.FLOAT, "cast(1.0 as float)");
        typeToLiteralValue_.put(Type.DOUBLE, "cast(3.4028235E38 as double)");
        typeToLiteralValue_.put(Type.TIMESTAMP, "cast('2012-12-21 00:00:00.000' as timestamp)");
        typeToLiteralValue_.put(Type.DATE, "cast('2012-12-21' as date)");
        typeToLiteralValue_.put(Type.STRING, "'Hello, World!'");
        typeToLiteralValue_.put(Type.NULL, "NULL");
    }
}

