/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.service;

import com.google.common.base.Preconditions;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.iceberg.AppendFiles;
import org.apache.iceberg.BaseTable;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.DataFiles;
import org.apache.iceberg.DeleteFile;
import org.apache.iceberg.DeleteFiles;
import org.apache.iceberg.ExpireSnapshots;
import org.apache.iceberg.FileMetadata;
import org.apache.iceberg.ManageSnapshots;
import org.apache.iceberg.Metrics;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.ReplacePartitions;
import org.apache.iceberg.RewriteFiles;
import org.apache.iceberg.RowDelta;
import org.apache.iceberg.Schema;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.Table;
import org.apache.iceberg.Transaction;
import org.apache.iceberg.UpdatePartitionSpec;
import org.apache.iceberg.UpdateProperties;
import org.apache.iceberg.UpdateSchema;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.exceptions.ValidationException;
import org.apache.iceberg.expressions.Expression;
import org.apache.iceberg.expressions.Expressions;
import org.apache.iceberg.expressions.Term;
import org.apache.iceberg.types.Type;
import org.apache.impala.analysis.IcebergPartitionSpec;
import org.apache.impala.catalog.FeIcebergTable;
import org.apache.impala.catalog.IcebergTable;
import org.apache.impala.catalog.TableLoadingException;
import org.apache.impala.catalog.TableNotFoundException;
import org.apache.impala.catalog.events.MetastoreEvents;
import org.apache.impala.catalog.iceberg.GroupedContentFiles;
import org.apache.impala.catalog.iceberg.IcebergCatalog;
import org.apache.impala.catalog.iceberg.IcebergHiveCatalog;
import org.apache.impala.common.ImpalaRuntimeException;
import org.apache.impala.fb.FbIcebergColumnStats;
import org.apache.impala.fb.FbIcebergDataFile;
import org.apache.impala.thrift.TAlterTableDropPartitionParams;
import org.apache.impala.thrift.TAlterTableExecuteExpireSnapshotsParams;
import org.apache.impala.thrift.TAlterTableExecuteRollbackParams;
import org.apache.impala.thrift.TColumn;
import org.apache.impala.thrift.TIcebergCatalog;
import org.apache.impala.thrift.TIcebergOperationParam;
import org.apache.impala.thrift.TIcebergPartitionSpec;
import org.apache.impala.thrift.TRollbackType;
import org.apache.impala.util.IcebergSchemaConverter;
import org.apache.impala.util.IcebergUtil;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IcebergCatalogOpExecutor {
    public static final Logger LOG = LoggerFactory.getLogger(IcebergCatalogOpExecutor.class);

    public static Table createTable(TIcebergCatalog catalog, TableIdentifier identifier, String location, List<TColumn> columns, TIcebergPartitionSpec partitionSpec, List<String> primaryKeyColumnNames, String owner, Map<String, String> tableProperties) throws ImpalaRuntimeException {
        Schema schema = IcebergCatalogOpExecutor.createIcebergSchema(columns, primaryKeyColumnNames);
        PartitionSpec spec = IcebergUtil.createIcebergPartition(schema, partitionSpec);
        IcebergCatalog icebergCatalog = IcebergUtil.getIcebergCatalog(catalog, location);
        if (icebergCatalog instanceof IcebergHiveCatalog) {
            tableProperties.put("hive.metastore.table.owner", owner);
        }
        Table iceTable = icebergCatalog.createTable(identifier, schema, spec, location, tableProperties);
        LOG.info("Create iceberg table successful.");
        return iceTable;
    }

    public static void populateExternalTableCols(org.apache.hadoop.hive.metastore.api.Table msTbl, Table iceTbl) throws TableLoadingException {
        try {
            msTbl.getSd().setCols(IcebergSchemaConverter.convertToHiveSchema(iceTbl.schema()));
        }
        catch (ImpalaRuntimeException e) {
            throw new TableLoadingException(e.getMessage());
        }
    }

    public static void dropTable(FeIcebergTable feTable, boolean ifExists) throws TableNotFoundException, ImpalaRuntimeException {
        Preconditions.checkState((boolean)IcebergTable.isSynchronizedTable(feTable.getMetaStoreTable()));
        IcebergCatalog iceCatalog = IcebergUtil.getIcebergCatalog(feTable);
        if (!iceCatalog.dropTable(feTable, IcebergTable.isSynchronizedTable(feTable.getMetaStoreTable())) && !ifExists) {
            throw new TableNotFoundException(String.format("Table '%s' does not exist in Iceberg catalog.", feTable.getFullName()));
        }
    }

    public static void addColumns(Transaction txn, List<TColumn> columns) throws TableLoadingException, ImpalaRuntimeException {
        UpdateSchema schema = txn.updateSchema();
        for (TColumn column : columns) {
            Type type = IcebergSchemaConverter.fromImpalaColumnType(column.getColumnType());
            schema.addColumn(column.getColumnName(), type, column.getComment());
        }
        schema.commit();
    }

    public static void alterColumn(Transaction txn, String colName, TColumn newCol) throws TableLoadingException, ImpalaRuntimeException {
        UpdateSchema schema = txn.updateSchema();
        Type type = IcebergSchemaConverter.fromImpalaColumnType(newCol.getColumnType());
        Preconditions.checkState((boolean)type.isPrimitiveType());
        schema.updateColumn(colName, type.asPrimitiveType());
        if (!colName.equals(newCol.getColumnName())) {
            schema.renameColumn(colName, newCol.getColumnName());
        }
        if (newCol.getComment() != null && !newCol.getComment().isEmpty()) {
            schema.updateColumnDoc(colName, newCol.getComment());
        }
        schema.commit();
    }

    public static void alterTableSetPartitionSpec(FeIcebergTable feTable, TIcebergPartitionSpec partSpec, Transaction transaction) throws ImpalaRuntimeException {
        try {
            if (!feTable.getPrimaryKeyColumnNames().isEmpty()) {
                throw new ImpalaRuntimeException("Not allowed to do partition evolution on Iceberg tables with primary keys.");
            }
        }
        catch (TException tEx) {
            throw new ImpalaRuntimeException(tEx.getMessage());
        }
        BaseTable iceTable = (BaseTable)feTable.getIcebergApiTable();
        UpdatePartitionSpec updatePartitionSpec = transaction.updateSpec();
        iceTable.spec().fields().forEach(partitionField -> updatePartitionSpec.removeField(partitionField.name()));
        List<Term> partitioningTerms = IcebergUtil.getPartitioningTerms(partSpec);
        partitioningTerms.forEach(arg_0 -> ((UpdatePartitionSpec)updatePartitionSpec).addField(arg_0));
        updatePartitionSpec.commit();
    }

    public static String alterTableExecuteExpireSnapshots(Transaction txn, TAlterTableExecuteExpireSnapshotsParams params) {
        ExpireSnapshots expireApi = txn.expireSnapshots();
        Preconditions.checkState((boolean)params.isSetOlder_than_millis());
        expireApi.expireOlderThan(params.older_than_millis);
        expireApi.commit();
        return "Snapshots have been expired.";
    }

    public static String alterTableExecuteRollback(Transaction iceTxn, FeIcebergTable tbl, TAlterTableExecuteRollbackParams params) {
        TRollbackType kind = params.getKind();
        ManageSnapshots manageSnapshots = iceTxn.manageSnapshots();
        switch (kind) {
            case TIME_ID: {
                Preconditions.checkState((boolean)params.isSetTimestamp_millis());
                long timestampMillis = params.getTimestamp_millis();
                LOG.info("Rollback iceberg table to snapshot before timestamp {}", (Object)timestampMillis);
                manageSnapshots.rollbackToTime(timestampMillis);
                break;
            }
            case VERSION_ID: {
                Preconditions.checkState((boolean)params.isSetSnapshot_id());
                long snapshotId = params.getSnapshot_id();
                LOG.info("Rollback iceberg table to snapshot id {}", (Object)snapshotId);
                manageSnapshots.rollbackTo(snapshotId);
                break;
            }
            default: {
                throw new IllegalStateException("Bad kind of execute rollback " + (Object)((Object)kind));
            }
        }
        manageSnapshots.commit();
        return "Rollback executed.";
    }

    public static long alterTableDropPartition(Transaction iceTxn, TAlterTableDropPartitionParams params) {
        DeleteFiles deleteFiles = iceTxn.newDelete();
        if (params.iceberg_drop_partition_request.is_truncate) {
            deleteFiles.deleteFromRowFilter((Expression)Expressions.alwaysTrue());
        } else {
            for (String path : params.iceberg_drop_partition_request.paths) {
                deleteFiles.deleteFile((CharSequence)path);
            }
        }
        deleteFiles.commit();
        return params.iceberg_drop_partition_request.num_partitions;
    }

    public static void dropColumn(Transaction txn, String colName) {
        UpdateSchema schema = txn.updateSchema();
        schema.deleteColumn(colName);
        schema.commit();
    }

    public static void renameTable(FeIcebergTable feTable, TableIdentifier tableId) throws ImpalaRuntimeException {
        IcebergCatalog catalog = IcebergUtil.getIcebergCatalog(feTable);
        catalog.renameTable(feTable, tableId);
    }

    public static void setTblProperties(Transaction txn, Map<String, String> properties) {
        UpdateProperties updateProps = txn.updateProperties();
        for (Map.Entry<String, String> entry : properties.entrySet()) {
            updateProps.set(entry.getKey(), entry.getValue());
        }
        updateProps.commit();
    }

    public static void unsetTblProperties(Transaction txn, List<String> removeProperties) {
        UpdateProperties updateProps = txn.updateProperties();
        for (String prop : removeProperties) {
            updateProps.remove(prop);
        }
        updateProps.commit();
    }

    private static Schema createIcebergSchema(List<TColumn> columns, List<String> primaryKeyColumnNames) throws ImpalaRuntimeException {
        return IcebergSchemaConverter.genIcebergSchema(columns, primaryKeyColumnNames);
    }

    public static void execute(FeIcebergTable feIcebergTable, Transaction txn, TIcebergOperationParam icebergOp) throws ImpalaRuntimeException {
        switch (icebergOp.operation) {
            case INSERT: {
                IcebergCatalogOpExecutor.appendFiles(feIcebergTable, txn, icebergOp);
                break;
            }
            case DELETE: {
                IcebergCatalogOpExecutor.deleteRows(feIcebergTable, txn, icebergOp);
                break;
            }
            case UPDATE: {
                IcebergCatalogOpExecutor.updateRows(feIcebergTable, txn, icebergOp);
                break;
            }
            case OPTIMIZE: {
                IcebergCatalogOpExecutor.rewriteTable(feIcebergTable, txn, icebergOp);
                break;
            }
            default: {
                throw new ImpalaRuntimeException("Unknown Iceberg operation: " + (Object)((Object)icebergOp.operation));
            }
        }
    }

    private static void deleteRows(FeIcebergTable feIcebergTable, Transaction txn, TIcebergOperationParam icebergOp) throws ImpalaRuntimeException {
        List<ByteBuffer> deleteFilesFb = icebergOp.getIceberg_delete_files_fb();
        RowDelta rowDelta = txn.newRowDelta();
        for (ByteBuffer buf : deleteFilesFb) {
            DeleteFile deleteFile = IcebergCatalogOpExecutor.createDeleteFile(feIcebergTable, buf);
            rowDelta.addDeletes(deleteFile);
        }
        try {
            rowDelta.validateFromSnapshot(icebergOp.getInitial_snapshot_id());
            rowDelta.validateNoConflictingDataFiles();
            rowDelta.validateDataFilesExist(icebergOp.getData_files_referenced_by_position_deletes());
            rowDelta.validateDeletedFiles();
            rowDelta.commit();
        }
        catch (ValidationException e) {
            throw new ImpalaRuntimeException(e.getMessage(), e);
        }
    }

    private static void updateRows(FeIcebergTable feIcebergTable, Transaction txn, TIcebergOperationParam icebergOp) throws ImpalaRuntimeException {
        List<ByteBuffer> deleteFilesFb = icebergOp.getIceberg_delete_files_fb();
        List<ByteBuffer> dataFilesFb = icebergOp.getIceberg_data_files_fb();
        RowDelta rowDelta = txn.newRowDelta();
        for (ByteBuffer buf : deleteFilesFb) {
            DeleteFile deleteFile = IcebergCatalogOpExecutor.createDeleteFile(feIcebergTable, buf);
            rowDelta.addDeletes(deleteFile);
        }
        for (ByteBuffer buf : dataFilesFb) {
            DataFile dataFile = IcebergCatalogOpExecutor.createDataFile(feIcebergTable, buf);
            rowDelta.addRows(dataFile);
        }
        try {
            rowDelta.validateFromSnapshot(icebergOp.getInitial_snapshot_id());
            rowDelta.validateNoConflictingDataFiles();
            rowDelta.validateNoConflictingDeleteFiles();
            rowDelta.validateDataFilesExist(icebergOp.getData_files_referenced_by_position_deletes());
            rowDelta.validateDeletedFiles();
            rowDelta.commit();
        }
        catch (ValidationException e) {
            throw new ImpalaRuntimeException(e.getMessage(), e);
        }
    }

    private static DataFile createDataFile(FeIcebergTable feIcebergTable, ByteBuffer buf) throws ImpalaRuntimeException {
        FbIcebergDataFile dataFile = FbIcebergDataFile.getRootAsFbIcebergDataFile(buf);
        PartitionSpec partSpec = (PartitionSpec)feIcebergTable.getIcebergApiTable().specs().get(dataFile.specId());
        IcebergPartitionSpec impPartSpec = feIcebergTable.getPartitionSpec(dataFile.specId());
        Metrics metrics = IcebergCatalogOpExecutor.buildDataFileMetrics(dataFile);
        DataFiles.Builder builder = DataFiles.builder((PartitionSpec)partSpec).withMetrics(metrics).withPath(dataFile.path()).withFormat(IcebergUtil.fbFileFormatToIcebergFileFormat(dataFile.format())).withRecordCount(dataFile.recordCount()).withFileSizeInBytes(dataFile.fileSizeInBytes());
        IcebergUtil.PartitionData partitionData = IcebergUtil.partitionDataFromDataFile(partSpec.partitionType(), impPartSpec, dataFile);
        if (partitionData != null) {
            builder.withPartition((StructLike)partitionData);
        }
        return builder.build();
    }

    private static DeleteFile createDeleteFile(FeIcebergTable feIcebergTable, ByteBuffer buf) throws ImpalaRuntimeException {
        FbIcebergDataFile deleteFile = FbIcebergDataFile.getRootAsFbIcebergDataFile(buf);
        PartitionSpec partSpec = (PartitionSpec)feIcebergTable.getIcebergApiTable().specs().get(deleteFile.specId());
        IcebergPartitionSpec impPartSpec = feIcebergTable.getPartitionSpec(deleteFile.specId());
        Metrics metrics = IcebergCatalogOpExecutor.buildDataFileMetrics(deleteFile);
        FileMetadata.Builder builder = FileMetadata.deleteFileBuilder((PartitionSpec)partSpec).ofPositionDeletes().withMetrics(metrics).withPath(deleteFile.path()).withFormat(IcebergUtil.fbFileFormatToIcebergFileFormat(deleteFile.format())).withRecordCount(deleteFile.recordCount()).withFileSizeInBytes(deleteFile.fileSizeInBytes());
        IcebergUtil.PartitionData partitionData = IcebergUtil.partitionDataFromDataFile(partSpec.partitionType(), impPartSpec, deleteFile);
        if (partitionData != null) {
            builder.withPartition((StructLike)partitionData);
        }
        return builder.build();
    }

    public static void appendFiles(FeIcebergTable feIcebergTable, Transaction txn, TIcebergOperationParam icebergOp) throws ImpalaRuntimeException {
        List<ByteBuffer> dataFilesFb = icebergOp.getIceberg_data_files_fb();
        BatchWrite batchWrite = icebergOp.isIs_overwrite() ? new DynamicOverwrite(txn, icebergOp.getInitial_snapshot_id()) : new Append(txn);
        for (ByteBuffer buf : dataFilesFb) {
            DataFile dataFile = IcebergCatalogOpExecutor.createDataFile(feIcebergTable, buf);
            batchWrite.addFile(dataFile);
        }
        try {
            batchWrite.commit();
        }
        catch (ValidationException e) {
            throw new ImpalaRuntimeException(e.getMessage(), e);
        }
    }

    private static Metrics buildDataFileMetrics(FbIcebergDataFile dataFile) {
        HashMap<Integer, Long> columnSizes = new HashMap<Integer, Long>();
        HashMap<Integer, Long> valueCounts = new HashMap<Integer, Long>();
        HashMap<Integer, Long> nullValueCounts = new HashMap<Integer, Long>();
        HashMap<Integer, ByteBuffer> lowerBounds = new HashMap<Integer, ByteBuffer>();
        HashMap<Integer, ByteBuffer> upperBounds = new HashMap<Integer, ByteBuffer>();
        for (int i = 0; i < dataFile.perColumnStatsLength(); ++i) {
            int fieldId;
            FbIcebergColumnStats stats = dataFile.perColumnStats(i);
            if (stats == null || (fieldId = stats.fieldId()) == -1) continue;
            columnSizes.put(fieldId, stats.totalCompressedByteSize());
            valueCounts.put(fieldId, stats.valueCount());
            nullValueCounts.put(fieldId, stats.nullCount());
            if (stats.lowerBoundLength() > 0) {
                lowerBounds.put(fieldId, stats.lowerBoundAsByteBuffer());
            }
            if (stats.upperBoundLength() <= 0) continue;
            upperBounds.put(fieldId, stats.upperBoundAsByteBuffer());
        }
        return new Metrics(Long.valueOf(dataFile.recordCount()), columnSizes, valueCounts, nullValueCounts, null, lowerBounds, upperBounds);
    }

    private static void rewriteTable(FeIcebergTable feIcebergTable, Transaction txn, TIcebergOperationParam icebergOp) throws ImpalaRuntimeException {
        GroupedContentFiles contentFiles;
        try {
            contentFiles = IcebergUtil.getIcebergFilesFromSnapshot(feIcebergTable, Collections.emptyList(), icebergOp.getInitial_snapshot_id());
        }
        catch (TableLoadingException e) {
            throw new ImpalaRuntimeException(e.getMessage(), e);
        }
        RewriteFiles rewrite = txn.newRewrite();
        for (DataFile dataFile : contentFiles.dataFilesWithDeletes) {
            rewrite.deleteFile(dataFile);
        }
        for (DataFile dataFile : contentFiles.dataFilesWithoutDeletes) {
            rewrite.deleteFile(dataFile);
        }
        for (DeleteFile deleteFile : contentFiles.positionDeleteFiles) {
            rewrite.deleteFile(deleteFile);
        }
        for (DeleteFile deleteFile : contentFiles.equalityDeleteFiles) {
            rewrite.deleteFile(deleteFile);
        }
        List<ByteBuffer> dataFilesToAdd = icebergOp.getIceberg_data_files_fb();
        for (ByteBuffer buf : dataFilesToAdd) {
            DataFile dataFile = IcebergCatalogOpExecutor.createDataFile(feIcebergTable, buf);
            rewrite.addFile(dataFile);
        }
        try {
            rewrite.validateFromSnapshot(icebergOp.getInitial_snapshot_id());
            rewrite.commit();
        }
        catch (ValidationException validationException) {
            throw new ImpalaRuntimeException(validationException.getMessage(), validationException);
        }
    }

    public static void truncateTable(Transaction txn) {
        DeleteFiles delete = txn.newDelete();
        delete.deleteFromRowFilter((Expression)Expressions.alwaysTrue());
        delete.commit();
    }

    public static void addCatalogVersionToTxn(Transaction txn, String serviceId, long version) {
        UpdateProperties updateProps = txn.updateProperties();
        updateProps.set(MetastoreEvents.MetastoreEventPropertyKey.CATALOG_SERVICE_ID.getKey(), serviceId);
        updateProps.set(MetastoreEvents.MetastoreEventPropertyKey.CATALOG_VERSION.getKey(), String.valueOf(version));
        updateProps.commit();
    }

    private static class DynamicOverwrite
    implements BatchWrite {
        private final ReplacePartitions replace;
        final long initialSnapshotId;

        public DynamicOverwrite(Transaction txn, long initialSnapshotId) {
            this.replace = txn.newReplacePartitions();
            this.initialSnapshotId = initialSnapshotId;
        }

        @Override
        public void addFile(DataFile file) {
            this.replace.addFile(file);
        }

        @Override
        public void commit() {
            this.replace.validateFromSnapshot(this.initialSnapshotId);
            this.replace.validateNoConflictingData();
            this.replace.validateNoConflictingDeletes();
            this.replace.commit();
        }
    }

    private static class Append
    implements BatchWrite {
        private final AppendFiles append;

        public Append(Transaction txn) {
            this.append = txn.newAppend();
        }

        @Override
        public void addFile(DataFile file) {
            this.append.appendFile(file);
        }

        @Override
        public void commit() {
            this.append.commit();
        }
    }

    private static interface BatchWrite {
        public void addFile(DataFile var1);

        public void commit();
    }
}

