/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.extdatasource.jdbc.dao;

import com.google.common.base.Preconditions;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.TimeZone;
import org.apache.hadoop.conf.Configuration;
import org.apache.impala.extdatasource.jdbc.exception.JdbcDatabaseAccessException;
import org.apache.impala.extdatasource.thrift.TColumnDesc;
import org.apache.impala.extdatasource.util.SerializationUtils;
import org.apache.impala.thrift.TColumnData;
import org.apache.impala.thrift.TColumnType;
import org.apache.impala.thrift.TScalarType;
import org.apache.impala.thrift.TTypeNodeType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcRecordIterator {
    private static final Logger LOGGER = LoggerFactory.getLogger(JdbcRecordIterator.class);
    private final Connection conn;
    private final PreparedStatement ps;
    private final ResultSet rs;
    private final List<String> jdbcColumnNames;

    public JdbcRecordIterator(Connection conn, PreparedStatement ps, ResultSet rs, Configuration conf) throws JdbcDatabaseAccessException {
        this.conn = conn;
        this.ps = ps;
        this.rs = rs;
        try {
            ResultSetMetaData metadata = rs.getMetaData();
            int numColumns = metadata.getColumnCount();
            ArrayList<String> columnNames = new ArrayList<String>(numColumns);
            ArrayList<Integer> jdbcColumnTypes = new ArrayList<Integer>(numColumns);
            for (int i = 0; i < numColumns; ++i) {
                columnNames.add(metadata.getColumnName(i + 1));
                jdbcColumnTypes.add(metadata.getColumnType(i + 1));
            }
            this.jdbcColumnNames = columnNames;
        }
        catch (Exception e) {
            LOGGER.error("Error while trying to get column names.", (Throwable)e);
            throw new JdbcDatabaseAccessException("Error while trying to get column names: " + e.getMessage(), e);
        }
        LOGGER.debug("Iterator ColumnNames = {}", this.jdbcColumnNames);
    }

    public Connection getConnection() {
        return this.conn;
    }

    public boolean hasNext() throws JdbcDatabaseAccessException {
        try {
            return this.rs.next();
        }
        catch (Exception e) {
            LOGGER.warn("hasNext() threw exception", (Throwable)e);
            throw new JdbcDatabaseAccessException("Error while retrieving next batch of rows: " + e.getMessage(), e);
        }
    }

    public void next(List<TColumnDesc> colDescs, List<TColumnData> colDatas) throws UnsupportedOperationException {
        Preconditions.checkState((colDescs.size() == colDatas.size() ? 1 : 0) != 0);
        for (int i = 0; i < colDescs.size(); ++i) {
            TColumnType type = colDescs.get(i).getType();
            TColumnData colData = colDatas.get(i);
            if (type.types.get(0).getType() != TTypeNodeType.SCALAR) {
                throw new UnsupportedOperationException("Unsupported column type: " + (Object)((Object)type.types.get(0).getType()));
            }
            Preconditions.checkState((type.getTypesSize() == 1 ? 1 : 0) != 0);
            TScalarType scalarType = type.types.get((int)0).scalar_type;
            try {
                Object value = this.rs.getObject(i + 1);
                if (value == null) {
                    colData.addToIs_null(true);
                    continue;
                }
                switch (scalarType.type) {
                    case TINYINT: {
                        colData.addToByte_vals(this.rs.getByte(i + 1));
                        break;
                    }
                    case SMALLINT: {
                        colData.addToShort_vals(this.rs.getShort(i + 1));
                        break;
                    }
                    case INT: {
                        colData.addToInt_vals(this.rs.getInt(i + 1));
                        break;
                    }
                    case DATE: {
                        LocalDate localDate = Instant.ofEpochMilli(this.rs.getDate(i + 1).getTime()).atZone(ZoneId.systemDefault()).toLocalDate();
                        colData.addToInt_vals((int)localDate.toEpochDay());
                        break;
                    }
                    case BIGINT: {
                        colData.addToLong_vals(this.rs.getLong(i + 1));
                        break;
                    }
                    case DOUBLE: {
                        colData.addToDouble_vals(this.rs.getDouble(i + 1));
                        break;
                    }
                    case FLOAT: {
                        colData.addToDouble_vals(this.rs.getFloat(i + 1));
                        break;
                    }
                    case STRING: {
                        colData.addToString_vals(this.rs.getString(i + 1));
                        break;
                    }
                    case BOOLEAN: {
                        colData.addToBool_vals(this.rs.getBoolean(i + 1));
                        break;
                    }
                    case TIMESTAMP: {
                        colData.addToBinary_vals(SerializationUtils.encodeTimestamp((Timestamp)this.rs.getTimestamp(i + 1, Calendar.getInstance(TimeZone.getTimeZone(ZoneOffset.UTC)))));
                        break;
                    }
                    case DECIMAL: {
                        BigDecimal val = this.rs.getBigDecimal(i + 1);
                        int valPrecision = val.precision();
                        int valScale = val.scale();
                        if (scalarType.scale < valScale || scalarType.precision < valPrecision + scalarType.scale - valScale) {
                            throw new UnsupportedOperationException(String.format("Invalid DECIMAL(%d, %d) for column %s since there is possible loss of precision when casting from DECIMAL(%d, %d)", scalarType.precision, scalarType.scale, colDescs.get(i).getName(), valPrecision, valScale));
                        }
                        if (scalarType.scale > valScale) {
                            val = val.setScale(scalarType.scale);
                        }
                        colData.addToBinary_vals(SerializationUtils.encodeDecimal((BigDecimal)val));
                        break;
                    }
                    default: {
                        throw new UnsupportedOperationException("Unsupported column type: " + (Object)((Object)scalarType.getType()));
                    }
                }
                colData.addToIs_null(false);
                continue;
            }
            catch (SQLException throwables) {
                colData.addToIs_null(true);
            }
        }
    }

    public void close() throws JdbcDatabaseAccessException {
        try {
            this.rs.close();
            this.ps.close();
        }
        catch (Exception e) {
            LOGGER.warn("Caught exception while trying to close database objects", (Throwable)e);
            throw new JdbcDatabaseAccessException("Error while releasing database resources: " + e.getMessage(), e);
        }
    }
}

