/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.common;

import com.google.common.base.Preconditions;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.IMetaStoreClient;
import org.apache.impala.catalog.MetaStoreClientPool;
import org.apache.impala.common.TransactionException;
import org.apache.impala.compat.MetastoreShim;
import org.apache.impala.thrift.TQueryCtx;
import org.apache.log4j.Logger;

public class TransactionKeepalive {
    public static final Logger LOG = Logger.getLogger(TransactionKeepalive.class);
    private static final long MAX_SLEEP_INTERVAL_MILLISECONDS = 100000L;
    private static final long MILLION = 1000000L;
    private final long sleepIntervalMs_;
    private final Thread daemonThread_;
    private final MetaStoreClientPool metaStoreClientPool_;
    private Map<Long, HeartbeatContext> transactions_ = new HashMap<Long, HeartbeatContext>();
    private Map<Long, HeartbeatContext> locks_ = new HashMap<Long, HeartbeatContext>();

    public TransactionKeepalive(MetaStoreClientPool metaStoreClientPool) {
        HiveConf hiveConf = new HiveConf(TransactionKeepalive.class);
        this.sleepIntervalMs_ = Math.min(100000L, hiveConf.getTimeVar(HiveConf.ConfVars.HIVE_TXN_TIMEOUT, TimeUnit.MILLISECONDS) / 3L);
        Preconditions.checkState((this.sleepIntervalMs_ > 0L ? 1 : 0) != 0);
        Preconditions.checkNotNull((Object)metaStoreClientPool);
        this.metaStoreClientPool_ = metaStoreClientPool;
        this.daemonThread_ = new Thread(new DaemonThread());
        this.daemonThread_.setDaemon(true);
        this.daemonThread_.setName("Transaction keepalive thread");
        this.daemonThread_.start();
    }

    public synchronized void addTransaction(Long transactionId, HeartbeatContext ctx) {
        Preconditions.checkNotNull((Object)transactionId);
        Preconditions.checkNotNull((Object)ctx);
        Preconditions.checkState((!this.transactions_.containsKey(transactionId) ? 1 : 0) != 0);
        this.transactions_.put(transactionId, ctx);
    }

    public synchronized void addLock(Long lockId, HeartbeatContext ctx) {
        Preconditions.checkNotNull((Object)lockId);
        Preconditions.checkNotNull((Object)ctx);
        Preconditions.checkState((!this.locks_.containsKey(lockId) ? 1 : 0) != 0);
        this.locks_.put(lockId, ctx);
    }

    public synchronized void deleteTransaction(Long transactionId) {
        Preconditions.checkNotNull((Object)transactionId);
        if (this.transactions_.remove(transactionId) == null) {
            LOG.info((Object)("Transaction id " + transactionId + " was already removed from TransactionKeepalive object or never existed."));
        }
    }

    public synchronized void deleteLock(Long lockId) {
        Preconditions.checkNotNull((Object)lockId);
        if (this.locks_.remove(lockId) == null) {
            LOG.info((Object)("Lock id " + lockId + " was already removed from TransactionKeepalive object or never existed."));
        }
    }

    private class DaemonThread
    implements Runnable {
        private DaemonThread() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            Random rand = new Random();
            try {
                Thread.sleep(rand.nextInt((int)TransactionKeepalive.this.sleepIntervalMs_));
            }
            catch (Throwable e2) {
                LOG.error((Object)"Unexpected exception thrown", e2);
            }
            while (true) {
                try {
                    while (true) {
                        long sleepMillis;
                        Map<Long, HeartbeatContext> copyOfLocks;
                        Map<Long, HeartbeatContext> copyOfTransactions;
                        TransactionKeepalive transactionKeepalive = TransactionKeepalive.this;
                        synchronized (transactionKeepalive) {
                            copyOfTransactions = TransactionKeepalive.this.transactions_.entrySet().stream().collect(Collectors.toMap(e -> (Long)e.getKey(), e -> (HeartbeatContext)e.getValue()));
                            copyOfLocks = TransactionKeepalive.this.locks_.entrySet().stream().collect(Collectors.toMap(e -> (Long)e.getKey(), e -> (HeartbeatContext)e.getValue()));
                        }
                        long durationOfHeartbeatingMillis = 0L;
                        if (!copyOfTransactions.isEmpty() || !copyOfLocks.isEmpty()) {
                            LOG.info((Object)("There are " + String.valueOf(copyOfTransactions.size()) + " open transactions and " + String.valueOf(copyOfLocks.size()) + " independent locks in TransactionKeepalive. Start heartbeating them."));
                            long startHeartbeating = System.nanoTime();
                            this.sendHeartbeatsFor(copyOfTransactions, copyOfLocks);
                            durationOfHeartbeatingMillis = (System.nanoTime() - startHeartbeating) / 1000000L;
                            LOG.info((Object)("Heartbeating the transactions and locks took " + durationOfHeartbeatingMillis + " milliseconds."));
                        }
                        if ((sleepMillis = TransactionKeepalive.this.sleepIntervalMs_ - durationOfHeartbeatingMillis) <= 0L) continue;
                        long randomness = rand.nextInt((int)(sleepMillis / 10L));
                        Thread.sleep(sleepMillis + randomness);
                    }
                }
                catch (Throwable e3) {
                    LOG.error((Object)"Unexpected exception thrown", e3);
                    continue;
                }
                break;
            }
        }

        private void sendHeartbeatsFor(Map<Long, HeartbeatContext> transactions, Map<Long, HeartbeatContext> locks) {
            try (MetaStoreClientPool.MetaStoreClient client = TransactionKeepalive.this.metaStoreClientPool_.getClient();){
                HeartbeatContext ctx;
                IMetaStoreClient hmsClient = client.getHiveClient();
                for (Map.Entry<Long, HeartbeatContext> entry : transactions.entrySet()) {
                    ctx = entry.getValue();
                    if (!this.oldEnough(ctx)) continue;
                    Long transactionId = entry.getKey();
                    this.sendHeartbeat(hmsClient, transactionId, 0L, ctx);
                }
                for (Map.Entry<Long, HeartbeatContext> entry : locks.entrySet()) {
                    ctx = entry.getValue();
                    if (!this.oldEnough(ctx)) continue;
                    Long lockId = entry.getKey();
                    this.sendHeartbeat(hmsClient, 0L, lockId, ctx);
                }
            }
        }

        private boolean oldEnough(HeartbeatContext heartbeatContext) {
            Long ageInMillis = (System.nanoTime() - heartbeatContext.creationTime) / 1000000L;
            return ageInMillis > TransactionKeepalive.this.sleepIntervalMs_;
        }

        private void sendHeartbeat(IMetaStoreClient hmsClient, long transactionId, long lockId, HeartbeatContext context) {
            Preconditions.checkState((transactionId == 0L || lockId == 0L ? 1 : 0) != 0);
            Preconditions.checkState((transactionId != 0L || lockId != 0L ? 1 : 0) != 0);
            try {
                if (!MetastoreShim.heartbeat(hmsClient, transactionId, lockId)) {
                    if (transactionId != 0L) {
                        LOG.warn((Object)("Transaction " + String.valueOf(transactionId) + " of query " + context.toString() + " doesn't exist anymore. Stop heartbeating it."));
                        TransactionKeepalive.this.deleteTransaction(transactionId);
                    }
                    if (lockId != 0L) {
                        LOG.warn((Object)("Lock " + String.valueOf(lockId) + " of query " + context.toString() + " doesn't exist anymore. Stop heartbeating it."));
                        TransactionKeepalive.this.deleteLock(lockId);
                    }
                }
            }
            catch (TransactionException e) {
                LOG.warn((Object)("Caught exception during heartbeating transaction " + String.valueOf(transactionId) + " lock " + String.valueOf(lockId) + " for query " + context.toString()), (Throwable)e);
            }
        }
    }

    public static class HeartbeatContext {
        private TQueryCtx queryCtx;
        private String cause;
        private long creationTime;

        public HeartbeatContext(TQueryCtx queryCtx, long creationTime) {
            this.queryCtx = queryCtx;
            this.creationTime = creationTime;
        }

        public HeartbeatContext(String cause, long creationTime) {
            this.queryCtx = null;
            this.cause = "'" + cause + "'";
            this.creationTime = creationTime;
        }

        public String toString() {
            if (this.queryCtx != null) {
                return this.queryCtx.query_id.toString();
            }
            return this.cause;
        }
    }
}

