/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.catalog;

import com.codahale.metrics.Snapshot;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheStats;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.hadoop.hive.common.FileUtils;
import org.apache.hadoop.hive.metastore.api.ColumnStatisticsData;
import org.apache.hadoop.hive.metastore.api.ColumnStatisticsObj;
import org.apache.hadoop.hive.metastore.api.FieldSchema;
import org.apache.impala.analysis.Expr;
import org.apache.impala.analysis.LiteralExpr;
import org.apache.impala.analysis.NullLiteral;
import org.apache.impala.analysis.PartitionKeyValue;
import org.apache.impala.analysis.ToSqlUtils;
import org.apache.impala.catalog.Catalog;
import org.apache.impala.catalog.CatalogException;
import org.apache.impala.catalog.CatalogObject;
import org.apache.impala.catalog.Column;
import org.apache.impala.catalog.ColumnStats;
import org.apache.impala.catalog.FeCatalog;
import org.apache.impala.catalog.FeFsPartition;
import org.apache.impala.catalog.FeFsTable;
import org.apache.impala.catalog.FeHBaseTable;
import org.apache.impala.catalog.FeIcebergTable;
import org.apache.impala.catalog.FeTable;
import org.apache.impala.catalog.HdfsFileFormat;
import org.apache.impala.catalog.HdfsPartition;
import org.apache.impala.catalog.HdfsStorageDescriptor;
import org.apache.impala.catalog.HdfsTable;
import org.apache.impala.catalog.SideloadTableStats;
import org.apache.impala.catalog.Table;
import org.apache.impala.catalog.TableLoadingException;
import org.apache.impala.catalog.Type;
import org.apache.impala.catalog.VirtualColumn;
import org.apache.impala.catalog.local.CatalogdMetaProvider;
import org.apache.impala.catalog.local.LocalCatalog;
import org.apache.impala.catalog.local.LocalFsTable;
import org.apache.impala.catalog.local.LocalHbaseTable;
import org.apache.impala.catalog.local.LocalIcebergTable;
import org.apache.impala.catalog.local.LocalKuduTable;
import org.apache.impala.catalog.local.LocalView;
import org.apache.impala.catalog.local.MetaProvider;
import org.apache.impala.common.ImpalaException;
import org.apache.impala.common.NotImplementedException;
import org.apache.impala.service.BackendConfig;
import org.apache.impala.thrift.TCatalogObject;
import org.apache.impala.thrift.TColumn;
import org.apache.impala.thrift.TColumnDescriptor;
import org.apache.impala.thrift.TGetCatalogMetricsResult;
import org.apache.impala.thrift.THdfsPartition;
import org.apache.impala.thrift.TTable;
import org.apache.impala.thrift.TTableStats;
import org.apache.impala.thrift.TTableType;
import org.apache.impala.util.AcidUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class FeCatalogUtils {
    private static final Logger LOG = LoggerFactory.getLogger(FeCatalogUtils.class);

    public static Type parseColumnType(FieldSchema fs, String tableName) throws TableLoadingException {
        Type type = Type.parseColumnType(fs.getType());
        if (type == null) {
            throw new TableLoadingException(String.format("Unsupported type '%s' in column '%s' of table '%s'", fs.getType(), fs.getName(), tableName));
        }
        if (type.exceedsMaxNestingDepth()) {
            throw new TableLoadingException(String.format("Type exceeds the maximum nesting depth of %s:\n%s", Type.MAX_NESTING_DEPTH, type.toSql()));
        }
        return type;
    }

    public static ImmutableList<Column> fieldSchemasToColumns(org.apache.hadoop.hive.metastore.api.Table msTbl) throws TableLoadingException {
        boolean isFullAcidTable = AcidUtils.isFullAcidTable(msTbl.getParameters());
        int pos = 0;
        ImmutableList.Builder ret = ImmutableList.builder();
        for (FieldSchema s : Iterables.concat((Iterable)msTbl.getPartitionKeys(), (Iterable)msTbl.getSd().getCols())) {
            if (isFullAcidTable && pos == msTbl.getPartitionKeys().size()) {
                ret.add((Object)AcidUtils.getRowIdColumnType(pos++));
            }
            Type type = FeCatalogUtils.parseColumnType(s, msTbl.getTableName());
            ret.add((Object)new Column(s.getName(), type, s.getComment(), pos++));
        }
        return ret.build();
    }

    public static void validateClusteringColumns(Iterable<Column> cols, String tableName) throws TableLoadingException {
        for (Column c : cols) {
            Type type = c.getType();
            if (type.supportsTablePartitioning()) continue;
            throw new TableLoadingException(String.format("Failed to load metadata for table '%s' because of unsupported partition-column type '%s' in partition column '%s'", tableName, type.toString(), c.getName()));
        }
    }

    public static List<TColumnDescriptor> getTColumnDescriptors(FeTable table) {
        ArrayList<TColumnDescriptor> colDescs = new ArrayList<TColumnDescriptor>();
        for (Column col : table.getColumns()) {
            colDescs.add(col.toDescriptor());
        }
        return colDescs;
    }

    public static void injectColumnStats(List<ColumnStatisticsObj> colStats, FeTable table, SideloadTableStats testStats) {
        for (ColumnStatisticsObj stats : colStats) {
            Column col = table.getColumn(stats.getColName());
            Preconditions.checkNotNull((Object)col, (String)"Unable to find column %s in table %s", (Object)stats.getColName(), (Object)table.getFullName());
            if (!ColumnStats.isSupportedColType(col.getType())) {
                LOG.warn(String.format("Statistics for %s, column %s are not supported as column has type %s", table.getFullName(), col.getName(), col.getType()));
                continue;
            }
            ColumnStatisticsData colStatsData = stats.getStatsData();
            if (testStats != null && testStats.hasColumn(stats.getColName())) {
                colStatsData = testStats.getColumnStats(stats.getColName());
                Preconditions.checkNotNull((Object)colStatsData);
                LOG.info("Sideload stats for " + table.getFullName() + "." + stats.getColName() + ". " + colStatsData);
            }
            if (col.updateStats(colStatsData)) continue;
            LOG.warn(String.format("Failed to load column stats for %s, column %s. Stats may be incompatible with column type %s. Consider regenerating statistics for %s.", table.getFullName(), col.getName(), col.getType(), table.getFullName()));
        }
    }

    public static long getRowCount(Map<String, String> parameters) {
        return FeCatalogUtils.getLongParam("numRows", parameters);
    }

    public static long getTotalSize(Map<String, String> parameters) {
        return FeCatalogUtils.getLongParam("totalSize", parameters);
    }

    private static long getLongParam(String key, Map<String, String> parameters) {
        if (parameters == null) {
            return -1L;
        }
        String value = parameters.get(key);
        if (value == null) {
            return -1L;
        }
        try {
            return Long.valueOf(value);
        }
        catch (NumberFormatException numberFormatException) {
            return -1L;
        }
    }

    public static FeFsPartition loadPartition(FeFsTable table, long partitionId) {
        List<? extends FeFsPartition> partCol = table.loadPartitions(Collections.singleton(partitionId));
        if (partCol.size() != 1) {
            throw new AssertionError((Object)String.format("expected exactly one result fetching partition ID %s from table %s (got %s)", partitionId, table.getFullName(), partCol.size()));
        }
        return (FeFsPartition)Iterables.getOnlyElement(partCol);
    }

    public static Collection<? extends FeFsPartition> loadAllPartitions(FeFsTable table) {
        return table.loadPartitions(table.getPartitionIds());
    }

    public static List<LiteralExpr> parsePartitionKeyValues(FeFsTable table, List<String> hmsPartitionValues) throws CatalogException {
        Preconditions.checkArgument((hmsPartitionValues.size() == table.getNumClusteringCols() ? 1 : 0) != 0, (String)"Cannot parse partition values '%s' for table %s: expected %d values but got %d", hmsPartitionValues, (Object)table.getFullName(), (Object)table.getNumClusteringCols(), (Object)hmsPartitionValues.size());
        ArrayList<LiteralExpr> keyValues = new ArrayList<LiteralExpr>();
        for (String string : hmsPartitionValues) {
            Type type = table.getColumns().get(keyValues.size()).getType();
            if (string.equals(table.getNullPartitionKeyValue())) {
                keyValues.add(NullLiteral.create(type));
                continue;
            }
            try {
                keyValues.add(LiteralExpr.createFromUnescapedStr(string, type));
            }
            catch (Exception ex) {
                LOG.warn(String.format("Failed to create literal expression: type: %s, value: '%s'", type.toSql(), string), (Throwable)ex);
                throw new CatalogException("Invalid partition key value of type: " + type, ex);
            }
        }
        for (Expr expr : keyValues) {
            expr.analyzeNoThrow(null);
        }
        return keyValues;
    }

    public static String getPartitionName(FeFsPartition partition) {
        return FeCatalogUtils.getPartitionName(partition.getTable(), FeCatalogUtils.getPartitionValuesAsStrings(partition, true));
    }

    public static List<String> getPartitionValuesAsStrings(FeFsPartition partition, boolean mapNullsToHiveKey) {
        ArrayList<String> ret = new ArrayList<String>();
        for (LiteralExpr partValue : partition.getPartitionValues()) {
            if (mapNullsToHiveKey) {
                ret.add(PartitionKeyValue.getPartitionKeyValueString(partValue, partition.getTable().getNullPartitionKeyValue()));
                continue;
            }
            ret.add(partValue.getStringValue());
        }
        return ret;
    }

    public static String getPartitionName(HdfsPartition.Builder partBuilder) {
        HdfsTable table = partBuilder.getTable();
        ArrayList<String> partitionValues = new ArrayList<String>();
        for (LiteralExpr partValue : partBuilder.getPartitionValues()) {
            partitionValues.add(PartitionKeyValue.getPartitionKeyValueString(partValue, table.getNullPartitionKeyValue()));
        }
        return FeCatalogUtils.getPartitionName(table, partitionValues);
    }

    public static String getPartitionName(FeFsTable table, List<String> partitionValues) {
        List partitionKeys = table.getClusteringColumns().stream().map(Column::getName).collect(Collectors.toList());
        return FileUtils.makePartName(partitionKeys, partitionValues);
    }

    public static String getPartitionName(List<PartitionKeyValue> partitionKeyValues) {
        List partitionKeys = partitionKeyValues.stream().map(PartitionKeyValue::getColName).collect(Collectors.toList());
        List partitionValues = partitionKeyValues.stream().map(PartitionKeyValue::getLiteralValue).map(l -> PartitionKeyValue.getPartitionKeyValueString(l, "__HIVE_DEFAULT_PARTITION__")).collect(Collectors.toList());
        return FileUtils.makePartName(partitionKeys, partitionValues);
    }

    public static String getConjunctSqlForPartition(FeFsPartition part) {
        ArrayList<String> partColSql = new ArrayList<String>();
        for (Column partCol : part.getTable().getClusteringColumns()) {
            partColSql.add(ToSqlUtils.getIdentSql(partCol.getName()));
        }
        ArrayList<String> conjuncts = new ArrayList<String>();
        for (int i = 0; i < partColSql.size(); ++i) {
            LiteralExpr partVal = part.getPartitionValues().get(i);
            String partValSql = partVal.toSql();
            if (Expr.IS_NULL_LITERAL.apply((Object)partVal) || partValSql.isEmpty()) {
                conjuncts.add((String)partColSql.get(i) + " IS NULL");
                continue;
            }
            conjuncts.add((String)partColSql.get(i) + "=" + partValSql);
        }
        return "(" + Joiner.on((String)" AND ").join(conjuncts) + ")";
    }

    public static Set<HdfsFileFormat> getFileFormats(Iterable<? extends FeFsPartition> partitions) {
        HashSet<HdfsFileFormat> fileFormats = new HashSet<HdfsFileFormat>();
        for (FeFsPartition feFsPartition : partitions) {
            fileFormats.add(feFsPartition.getFileFormat());
        }
        return fileFormats;
    }

    public static THdfsPartition fsPartitionToThrift(FeFsPartition part, CatalogObject.ThriftObjectType type) {
        HdfsStorageDescriptor sd = part.getInputFormatDescriptor();
        THdfsPartition thriftHdfsPart = new THdfsPartition();
        thriftHdfsPart.setHdfs_storage_descriptor(sd.toThrift());
        thriftHdfsPart.setPartitionKeyExprs(Expr.treesToThrift(part.getPartitionValues()));
        thriftHdfsPart.setId(part.getId());
        thriftHdfsPart.setLocation(part.getLocationAsThrift());
        if (part.getWriteId() >= 0L) {
            thriftHdfsPart.setWrite_id(part.getWriteId());
        }
        if (type == CatalogObject.ThriftObjectType.FULL) {
            thriftHdfsPart.setPartition_name(part.getPartitionName());
            thriftHdfsPart.setStats(new TTableStats(part.getNumRows()));
            thriftHdfsPart.setAccess_level(part.getAccessLevel());
            thriftHdfsPart.setIs_marked_cached(part.isMarkedCached());
            thriftHdfsPart.setHms_parameters(Maps.newHashMap(part.getParameters()));
            thriftHdfsPart.setHas_incremental_stats(part.hasIncrementalStats());
            long numBlocks = 0L;
            long totalFileBytes = 0L;
            for (HdfsPartition.FileDescriptor fd : part.getFileDescriptors()) {
                numBlocks += (long)fd.getNumFileBlocks();
                totalFileBytes += fd.getFileLength();
            }
            if (!part.getInsertFileDescriptors().isEmpty()) {
                for (HdfsPartition.FileDescriptor fd : part.getInsertFileDescriptors()) {
                    thriftHdfsPart.addToInsert_file_desc(fd.toThrift());
                }
                for (HdfsPartition.FileDescriptor fd : part.getDeleteFileDescriptors()) {
                    thriftHdfsPart.addToDelete_file_desc(fd.toThrift());
                }
            } else {
                for (HdfsPartition.FileDescriptor fd : part.getFileDescriptors()) {
                    thriftHdfsPart.addToFile_desc(fd.toThrift());
                }
            }
            thriftHdfsPart.setNum_blocks(numBlocks);
            thriftHdfsPart.setTotal_file_size_bytes(totalFileBytes);
        }
        return thriftHdfsPart;
    }

    public static TTable feTableToThrift(FeTable table) throws ImpalaException {
        if (table instanceof Table) {
            return ((Table)table).toThrift();
        }
        TTable res = new TTable(table.getDb().getName(), table.getName());
        res.setTable_stats(table.getTTableStats());
        res.setMetastore_table(table.getMetaStoreTable());
        res.setClustering_columns(new ArrayList<TColumn>());
        for (Column column : table.getClusteringColumns()) {
            res.addToClustering_columns(column.toThrift());
        }
        res.setColumns(new ArrayList<TColumn>());
        for (Column column : table.getNonClusteringColumns()) {
            res.addToColumns(column.toThrift());
        }
        res.setVirtual_columns(new ArrayList<TColumn>());
        for (VirtualColumn virtualColumn : table.getVirtualColumns()) {
            res.addToVirtual_columns(virtualColumn.toThrift());
        }
        if (table instanceof LocalFsTable) {
            res.setTable_type(TTableType.HDFS_TABLE);
            res.setHdfs_table(((LocalFsTable)table).toTHdfsTable(CatalogObject.ThriftObjectType.FULL));
        } else if (table instanceof LocalKuduTable) {
            res.setTable_type(TTableType.KUDU_TABLE);
            res.setKudu_table(((LocalKuduTable)table).toTKuduTable());
        } else if (table instanceof LocalHbaseTable) {
            res.setTable_type(TTableType.HBASE_TABLE);
            res.setHbase_table(FeHBaseTable.Util.getTHBaseTable((FeHBaseTable)table));
        } else if (table instanceof LocalIcebergTable) {
            res.setTable_type(TTableType.ICEBERG_TABLE);
            LocalIcebergTable iceTable = (LocalIcebergTable)table;
            res.setIceberg_table(FeIcebergTable.Utils.getTIcebergTable(iceTable));
            res.setHdfs_table(iceTable.transformToTHdfsTable(true, CatalogObject.ThriftObjectType.FULL));
        } else if (table instanceof LocalView) {
            res.setTable_type(TTableType.VIEW);
        } else {
            throw new NotImplementedException("Unsupported type to export: " + table.getClass());
        }
        return res;
    }

    public static void populateCacheMetrics(FeCatalog catalog, TGetCatalogMetricsResult metrics) {
        Preconditions.checkNotNull((Object)catalog);
        Preconditions.checkNotNull((Object)metrics);
        if (!BackendConfig.INSTANCE.getBackendCfg().use_local_catalog) {
            return;
        }
        Preconditions.checkState((boolean)(catalog instanceof LocalCatalog));
        MetaProvider provider = ((LocalCatalog)catalog).getMetaProvider();
        if (!(provider instanceof CatalogdMetaProvider)) {
            return;
        }
        CacheStats stats = ((CatalogdMetaProvider)provider).getCacheStats();
        metrics.setCache_eviction_count(stats.evictionCount());
        metrics.setCache_hit_count(stats.hitCount());
        metrics.setCache_load_count(stats.loadCount());
        metrics.setCache_load_exception_count(stats.loadExceptionCount());
        metrics.setCache_load_success_count(stats.loadSuccessCount());
        metrics.setCache_miss_count(stats.missCount());
        metrics.setCache_request_count(stats.requestCount());
        metrics.setCache_total_load_time(stats.totalLoadTime());
        metrics.setCache_avg_load_time(stats.averageLoadPenalty());
        metrics.setCache_hit_rate(stats.hitRate());
        metrics.setCache_load_exception_rate(stats.loadExceptionRate());
        metrics.setCache_miss_rate(stats.missRate());
        Snapshot cacheEntrySize = ((CatalogdMetaProvider)provider).getCacheEntrySize();
        metrics.setCache_entry_median_size(cacheEntrySize.getMedian());
        metrics.setCache_entry_99th_size(cacheEntrySize.get99thPercentile());
    }

    public static String debugString(List<TCatalogObject> objects) {
        if (objects == null || objects.size() == 0) {
            return "[]";
        }
        ArrayList<String> catalogObjs = new ArrayList<String>();
        for (TCatalogObject object : objects) {
            catalogObjs.add(String.format("%s version: %d", Catalog.toCatalogObjectKey(object), object.catalog_version));
        }
        return "[" + Joiner.on((String)",").join(catalogObjs) + "]";
    }
}

