/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.catalog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Ticker;
import com.google.common.util.concurrent.Uninterruptibles;
import com.sun.management.GarbageCollectorMXBean;
import com.sun.management.GcInfo;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryUsage;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.NotificationListener;
import org.apache.impala.catalog.CatalogServiceCatalog;
import org.apache.impala.catalog.Db;
import org.apache.impala.catalog.IncompleteTable;
import org.apache.impala.catalog.Table;
import org.apache.impala.common.Reference;
import org.apache.impala.service.BackendConfig;
import org.apache.impala.thrift.TTableName;
import org.apache.impala.util.NoOpEventSequence;
import org.apache.log4j.Logger;

public class CatalogdTableInvalidator {
    public static final Logger LOG = Logger.getLogger(CatalogdTableInvalidator.class);
    @VisibleForTesting
    static Ticker TIME_SOURCE = Ticker.systemTicker();
    private static long DAEMON_MAXIMUM_SLEEP_NANO = TimeUnit.MINUTES.toNanos(5L);
    private final long unusedTableTtlNano_;
    private final boolean invalidateTableOnMemoryPressure_;
    private final CatalogServiceCatalog catalog_;
    private final Thread daemonThread_;
    private final double oldGenFullThreshold_;
    private final double gcInvalidationFraction_;
    @VisibleForTesting
    AtomicLong scanCount_ = new AtomicLong();
    private GarbageCollectorMXBean oldGenGcBean_;
    private String oldGcGenName_;
    private long lastObservedGcCount_;
    private boolean stopped_ = false;
    private long lastInvalidationTime_;

    CatalogdTableInvalidator(CatalogServiceCatalog catalog, long unusedTableTtlSec, boolean invalidateTableOnMemoryPressure, double oldGenFullThreshold, double gcInvalidationFraction) {
        this.catalog_ = catalog;
        this.unusedTableTtlNano_ = TimeUnit.SECONDS.toNanos(unusedTableTtlSec);
        this.oldGenFullThreshold_ = oldGenFullThreshold;
        this.gcInvalidationFraction_ = gcInvalidationFraction;
        this.lastInvalidationTime_ = TIME_SOURCE.read();
        this.invalidateTableOnMemoryPressure_ = invalidateTableOnMemoryPressure && this.tryInstallGcListener();
        this.daemonThread_ = new Thread(new DaemonThread());
        this.daemonThread_.setDaemon(true);
        this.daemonThread_.setName("CatalogTableInvalidator timer");
        this.daemonThread_.start();
    }

    public static CatalogdTableInvalidator create(CatalogServiceCatalog catalog, BackendConfig config) {
        boolean invalidateTableOnMemoryPressure = config.invalidateTablesOnMemoryPressure();
        int timeoutSec = config.getInvalidateTablesTimeoutS();
        double gcOldGenFullThreshold = config.getInvalidateTablesGcOldGenFullThreshold();
        double fractionOnMemoryPressure = config.getInvalidateTablesFractionOnMemoryPressure();
        Preconditions.checkArgument((timeoutSec >= 0 ? 1 : 0) != 0, (Object)"invalidate_tables_timeout_s must be a non-negative integer.");
        Preconditions.checkArgument((gcOldGenFullThreshold >= 0.0 && gcOldGenFullThreshold <= 1.0 ? 1 : 0) != 0, (Object)"invalidate_tables_gc_old_gen_full_threshold must be in [0, 1].");
        Preconditions.checkArgument((fractionOnMemoryPressure >= 0.0 && fractionOnMemoryPressure <= 1.0 ? 1 : 0) != 0, (Object)"invalidate_tables_fraction_on_memory_pressure must be in [0, 1].");
        if (timeoutSec > 0 || invalidateTableOnMemoryPressure) {
            return new CatalogdTableInvalidator(catalog, timeoutSec, invalidateTableOnMemoryPressure, gcOldGenFullThreshold, fractionOnMemoryPressure);
        }
        return null;
    }

    static long nanoTime() {
        return TIME_SOURCE.read();
    }

    private boolean tryInstallGcListener() {
        String commonErrMsg = "Continuing without GC-triggered invalidation of tables.";
        List<GarbageCollectorMXBean> gcbeans = ManagementFactory.getPlatformMXBeans(GarbageCollectorMXBean.class);
        GcNotificationListener gcNotificationListener = new GcNotificationListener();
        boolean foundOldPool = false;
        block0: for (GarbageCollectorMXBean gcbean : gcbeans) {
            for (String poolName : gcbean.getMemoryPoolNames()) {
                if (!poolName.contains("Old")) continue;
                if (!(gcbean instanceof NotificationEmitter)) {
                    LOG.warn((Object)("GCBean " + gcbean.getClass().getName() + " is not supported because it does not implement NotificationEmitter. " + commonErrMsg));
                    return false;
                }
                this.oldGenGcBean_ = gcbean;
                this.oldGcGenName_ = poolName;
                this.lastObservedGcCount_ = gcbean.getCollectionCount();
                foundOldPool = true;
                ((NotificationEmitter)((Object)gcbean)).addNotificationListener(gcNotificationListener, null, null);
                continue block0;
            }
        }
        if (!foundOldPool) {
            LOG.warn((Object)("Cannot find old generation memory pool in the GC beans. " + commonErrMsg));
        }
        return foundOldPool;
    }

    private boolean shouldEvictFromFullHeapAfterGc() {
        if (!this.invalidateTableOnMemoryPressure_) {
            return false;
        }
        long gcCount = this.oldGenGcBean_.getCollectionCount();
        if (gcCount > this.lastObservedGcCount_) {
            this.lastObservedGcCount_ = gcCount;
            GcInfo lastGcInfo = this.oldGenGcBean_.getLastGcInfo();
            if (lastGcInfo == null) {
                LOG.warn((Object)"gcBean.getLastGcInfo() returned null. Table invalidation based on memory pressure was skipped.");
                return false;
            }
            MemoryUsage tenuredGenUsage = lastGcInfo.getMemoryUsageAfterGc().get(this.oldGcGenName_);
            Preconditions.checkState((tenuredGenUsage != null ? 1 : 0) != 0);
            return (double)tenuredGenUsage.getMax() * this.oldGenFullThreshold_ < (double)tenuredGenUsage.getUsed();
        }
        return false;
    }

    private void invalidateSome(double invalidationFraction) {
        ArrayList<Table> tables = new ArrayList<Table>();
        for (Db db : this.catalog_.getAllDbs()) {
            for (Table table : db.getTables()) {
                if (table instanceof IncompleteTable) continue;
                tables.add(table);
            }
        }
        Collections.sort(tables, new Comparator<Table>(){

            @Override
            public int compare(Table o1, Table o2) {
                return Long.compare(o1.getLastUsedTime(), o2.getLastUsedTime());
            }
        });
        int i = 0;
        while ((double)i < (double)tables.size() * invalidationFraction) {
            TTableName tTableName = ((Table)tables.get(i)).getTableName().toThrift();
            Reference<Boolean> tblWasRemoved = new Reference<Boolean>();
            Reference<Boolean> dbWasAdded = new Reference<Boolean>();
            this.catalog_.invalidateTable(tTableName, tblWasRemoved, dbWasAdded, NoOpEventSequence.INSTANCE);
            LOG.info((Object)("Table " + ((Table)tables.get(i)).getFullName() + " invalidated due to memory pressure."));
            ++i;
        }
    }

    private void invalidateOlderThan(long retireAgeNano) {
        long now = TIME_SOURCE.read();
        for (Db db : this.catalog_.getAllDbs()) {
            for (Table table : this.catalog_.getAllTables(db)) {
                long inactivityTime;
                if (table instanceof IncompleteTable || (inactivityTime = now - table.getLastUsedTime()) <= retireAgeNano) continue;
                Reference<Boolean> tblWasRemoved = new Reference<Boolean>();
                Reference<Boolean> dbWasAdded = new Reference<Boolean>();
                TTableName tTableName = table.getTableName().toThrift();
                this.catalog_.invalidateTable(tTableName, tblWasRemoved, dbWasAdded, NoOpEventSequence.INSTANCE);
                LOG.info((Object)("Invalidated " + table.getFullName() + " due to inactivity for " + TimeUnit.NANOSECONDS.toSeconds(inactivityTime) + " seconds."));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void stop() {
        CatalogdTableInvalidator catalogdTableInvalidator = this;
        synchronized (catalogdTableInvalidator) {
            this.stopped_ = true;
            this.notify();
        }
        try {
            this.daemonThread_.join();
        }
        catch (InterruptedException e) {
            LOG.warn((Object)"stop() is interrupted", (Throwable)e);
        }
    }

    @VisibleForTesting
    synchronized void wakeUpForTests() {
        this.notify();
    }

    private class GcNotificationListener
    implements NotificationListener {
        private GcNotificationListener() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void handleNotification(Notification notification, Object handback) {
            CatalogdTableInvalidator catalogdTableInvalidator = CatalogdTableInvalidator.this;
            synchronized (catalogdTableInvalidator) {
                CatalogdTableInvalidator.this.notify();
            }
        }
    }

    private class DaemonThread
    implements Runnable {
        private DaemonThread() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public void run() {
            long sleepTimeNano = DAEMON_MAXIMUM_SLEEP_NANO;
            if (CatalogdTableInvalidator.this.unusedTableTtlNano_ > 0L) {
                sleepTimeNano = Math.min(CatalogdTableInvalidator.this.unusedTableTtlNano_ / 10L, sleepTimeNano);
            }
            while (true) {
                try {
                    while (true) {
                        CatalogdTableInvalidator catalogdTableInvalidator = CatalogdTableInvalidator.this;
                        synchronized (catalogdTableInvalidator) {
                            if (CatalogdTableInvalidator.this.stopped_) {
                                return;
                            }
                            if (CatalogdTableInvalidator.this.shouldEvictFromFullHeapAfterGc()) {
                                CatalogdTableInvalidator.this.invalidateSome(CatalogdTableInvalidator.this.gcInvalidationFraction_);
                                CatalogdTableInvalidator.this.scanCount_.incrementAndGet();
                            }
                            long now = CatalogdTableInvalidator.nanoTime();
                            if (CatalogdTableInvalidator.this.unusedTableTtlNano_ > 0L && now >= CatalogdTableInvalidator.this.lastInvalidationTime_ + sleepTimeNano) {
                                CatalogdTableInvalidator.this.invalidateOlderThan(CatalogdTableInvalidator.this.unusedTableTtlNano_);
                                CatalogdTableInvalidator.this.lastInvalidationTime_ = now;
                                CatalogdTableInvalidator.this.scanCount_.incrementAndGet();
                            }
                            TimeUnit.NANOSECONDS.timedWait(CatalogdTableInvalidator.this, sleepTimeNano);
                        }
                    }
                }
                catch (Exception e) {
                    LOG.warn((Object)"Unexpected exception thrown while attempting to automatically invalidate tables. Will retry in 5 seconds.", (Throwable)e);
                    Uninterruptibles.sleepUninterruptibly((long)5L, (TimeUnit)TimeUnit.SECONDS);
                    continue;
                }
                break;
            }
        }
    }
}

