/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.catalog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.hadoop.hive.metastore.IMetaStoreClient;
import org.apache.hadoop.hive.metastore.api.DataOperationType;
import org.apache.hadoop.hive.metastore.api.LockComponent;
import org.apache.hadoop.hive.metastore.api.LockLevel;
import org.apache.hadoop.hive.metastore.api.LockType;
import org.apache.hadoop.hive.metastore.api.Partition;
import org.apache.impala.analysis.FunctionName;
import org.apache.impala.authorization.AuthorizationPolicy;
import org.apache.impala.catalog.AuthzCacheInvalidation;
import org.apache.impala.catalog.CatalogException;
import org.apache.impala.catalog.CatalogObjectCache;
import org.apache.impala.catalog.DataSource;
import org.apache.impala.catalog.DatabaseNotFoundException;
import org.apache.impala.catalog.Db;
import org.apache.impala.catalog.Function;
import org.apache.impala.catalog.HasName;
import org.apache.impala.catalog.HdfsCachePool;
import org.apache.impala.catalog.HdfsPartition;
import org.apache.impala.catalog.HdfsTable;
import org.apache.impala.catalog.MetaStoreClientPool;
import org.apache.impala.catalog.PartitionNotFoundException;
import org.apache.impala.catalog.Principal;
import org.apache.impala.catalog.PrincipalPrivilege;
import org.apache.impala.catalog.Table;
import org.apache.impala.catalog.TableWriteId;
import org.apache.impala.catalog.Transaction;
import org.apache.impala.catalog.monitor.CatalogMonitor;
import org.apache.impala.common.TransactionException;
import org.apache.impala.common.TransactionKeepalive;
import org.apache.impala.compat.MetastoreShim;
import org.apache.impala.thrift.TCatalogObject;
import org.apache.impala.thrift.TFunction;
import org.apache.impala.thrift.THdfsPartition;
import org.apache.impala.thrift.TImpalaTableType;
import org.apache.impala.thrift.TPartitionKeyValue;
import org.apache.impala.thrift.TPrincipalType;
import org.apache.impala.thrift.TTable;
import org.apache.impala.thrift.TTableName;
import org.apache.impala.thrift.TUniqueId;
import org.apache.impala.util.EventSequence;
import org.apache.impala.util.PatternMatcher;

public abstract class Catalog
implements AutoCloseable {
    public static final long INITIAL_CATALOG_VERSION = 0L;
    public static final TUniqueId INITIAL_CATALOG_SERVICE_ID = new TUniqueId(0L, 0L);
    public static final String DEFAULT_DB = "default";
    private MetaStoreClientPool metaStoreClientPool_;
    protected AuthorizationPolicy authPolicy_ = new AuthorizationPolicy();
    protected AtomicReference<Map<String, Db>> dbCache_ = new AtomicReference(new ConcurrentHashMap());
    protected final CatalogObjectCache<DataSource> dataSources_;
    protected final ConcurrentHashMap<Long, Set<TableWriteId>> txnToWriteIds_ = new ConcurrentHashMap();
    protected final CatalogObjectCache<HdfsCachePool> hdfsCachePools_ = new CatalogObjectCache(false);
    protected final CatalogObjectCache<AuthzCacheInvalidation> authzCacheInvalidation_ = new CatalogObjectCache();
    private TransactionKeepalive transactionKeepalive_;
    private static final CatalogObjectOrder CATALOG_OBJECT_ORDER = new CatalogObjectOrder();

    public Catalog(MetaStoreClientPool metaStoreClientPool) {
        this.dataSources_ = new CatalogObjectCache();
        this.metaStoreClientPool_ = (MetaStoreClientPool)Preconditions.checkNotNull((Object)metaStoreClientPool);
        this.transactionKeepalive_ = MetastoreShim.getMajorVersion() > 2L ? new TransactionKeepalive(this.metaStoreClientPool_) : null;
    }

    public Catalog() {
        this(new MetaStoreClientPool(0, 0));
    }

    public abstract String getAcidUserId();

    public void addDb(Db db) {
        this.dbCache_.get().put(db.getName().toLowerCase(), db);
    }

    public Db getDb(String dbName) {
        Preconditions.checkArgument((dbName != null && !dbName.isEmpty() ? 1 : 0) != 0, (Object)"Null or empty database name given as argument to Catalog.getDb");
        return this.dbCache_.get().get(dbName.toLowerCase());
    }

    public Db removeDb(String dbName) {
        return this.dbCache_.get().remove(dbName.toLowerCase());
    }

    public List<Db> getDbs(PatternMatcher matcher) {
        return Catalog.filterCatalogObjectsByPattern(this.dbCache_.get().values(), matcher);
    }

    public Table getTableNoThrow(String dbName, String tableName) {
        Db db = this.getDb(dbName);
        if (db == null) {
            return null;
        }
        return db.getTable(tableName);
    }

    public Table getTable(String dbName, String tableName) throws DatabaseNotFoundException {
        Db db = this.getDb(dbName);
        if (db == null) {
            throw new DatabaseNotFoundException("Database '" + dbName + "' not found");
        }
        return db.getTable(tableName);
    }

    public Table getTableIfCachedNoThrow(String dbName, String tableName) {
        return this.getTableNoThrow(dbName, tableName);
    }

    public Table getTableIfCached(String dbName, String tableName) throws DatabaseNotFoundException {
        return this.getTable(dbName, tableName);
    }

    public Table removeTable(TTableName tableName) {
        Db db = this.getDb(tableName.getDb_name());
        if (db == null) {
            return null;
        }
        Table tbl = db.removeTable(tableName.getTable_name());
        if (tbl != null && !tbl.isStoredInImpaladCatalogCache()) {
            CatalogMonitor.INSTANCE.getCatalogTableMetrics().removeTable(tbl);
        }
        return tbl;
    }

    public List<String> getTableNames(String dbName, PatternMatcher matcher) throws DatabaseNotFoundException {
        return this.getTableNames(dbName, matcher, Collections.emptySet());
    }

    public List<String> getTableNames(String dbName, PatternMatcher matcher, Set<TImpalaTableType> tableTypes) throws DatabaseNotFoundException {
        Preconditions.checkNotNull((Object)dbName);
        Db db = this.getDb(dbName);
        if (db == null) {
            throw new DatabaseNotFoundException("Database '" + dbName + "' not found");
        }
        return Catalog.filterStringsByPattern(db.getAllTableNames(tableTypes), matcher);
    }

    public boolean containsTable(String dbName, String tableName) {
        Db db = this.getDb(dbName);
        return db == null ? false : db.containsTable(tableName);
    }

    public boolean addDataSource(DataSource dataSource) {
        return this.dataSources_.add(dataSource);
    }

    public DataSource removeDataSource(String dataSourceName) {
        Preconditions.checkNotNull((Object)dataSourceName);
        return this.dataSources_.remove(dataSourceName.toLowerCase());
    }

    public DataSource getDataSource(String dataSourceName) {
        Preconditions.checkNotNull((Object)dataSourceName);
        return this.dataSources_.get(dataSourceName.toLowerCase());
    }

    public List<DataSource> getDataSources() {
        return this.dataSources_.getValues();
    }

    public List<String> getDataSourceNames(String pattern) {
        return Catalog.filterStringsByPattern(this.dataSources_.keySet(), PatternMatcher.createHivePatternMatcher(pattern));
    }

    public List<DataSource> getDataSources(PatternMatcher matcher) {
        return Catalog.filterCatalogObjectsByPattern(this.dataSources_.getValues(), matcher);
    }

    public boolean addFunction(Function fn) {
        Db db = this.getDb(fn.dbName());
        if (db == null) {
            return false;
        }
        return db.addFunction(fn);
    }

    public Function getFunction(Function desc, Function.CompareMode mode) {
        Db db = this.getDb(desc.dbName());
        if (db == null) {
            return null;
        }
        return db.getFunction(desc, mode);
    }

    public Function removeFunction(Function desc) {
        Db db = this.getDb(desc.dbName());
        if (db == null) {
            return null;
        }
        return db.removeFunction(desc);
    }

    public boolean containsFunction(FunctionName name) {
        Db db = this.getDb(name.getDb());
        if (db == null) {
            return false;
        }
        return db.containsFunction(name.getFunction());
    }

    public boolean addHdfsCachePool(HdfsCachePool cachePool) {
        return this.hdfsCachePools_.add(cachePool);
    }

    public HdfsCachePool getHdfsCachePool(String poolName) {
        return this.hdfsCachePools_.get(poolName);
    }

    public AuthzCacheInvalidation getAuthzCacheInvalidation(String markerName) {
        return this.authzCacheInvalidation_.get((String)Preconditions.checkNotNull((Object)markerName));
    }

    @Override
    public void close() {
        this.metaStoreClientPool_.close();
    }

    @VisibleForTesting
    public MetaStoreClientPool getMetaStoreClientPool() {
        return this.metaStoreClientPool_;
    }

    @VisibleForTesting
    public void setMetaStoreClientPool(MetaStoreClientPool pool) {
        this.metaStoreClientPool_ = pool;
    }

    public MetaStoreClientPool.MetaStoreClient getMetaStoreClient() {
        return this.metaStoreClientPool_.getClient();
    }

    public MetaStoreClientPool.MetaStoreClient getMetaStoreClient(EventSequence timeline) {
        MetaStoreClientPool.MetaStoreClient client = this.getMetaStoreClient();
        timeline.markEvent("Got Metastore client");
        return client;
    }

    public int getNumHmsClientsIdle() {
        return this.metaStoreClientPool_.getNumHmsClientsIdle();
    }

    public int getNumHmsClientsInUse() {
        return this.metaStoreClientPool_.getNumHmsClientsInUse();
    }

    public static List<String> filterStringsByPattern(Iterable<String> candidates, PatternMatcher matcher) {
        Preconditions.checkNotNull((Object)matcher);
        ArrayList<String> filtered = new ArrayList<String>();
        for (String candidate : candidates) {
            if (!matcher.matches(candidate)) continue;
            filtered.add(candidate);
        }
        Collections.sort(filtered, String.CASE_INSENSITIVE_ORDER);
        return filtered;
    }

    public static <T extends HasName> List<T> filterCatalogObjectsByPattern(Iterable<? extends T> candidates, PatternMatcher matcher) {
        Preconditions.checkNotNull((Object)matcher);
        ArrayList<HasName> filtered = new ArrayList<HasName>();
        for (HasName candidate : candidates) {
            if (!matcher.matches(candidate.getName())) continue;
            filtered.add(candidate);
        }
        Collections.sort(filtered, CATALOG_OBJECT_ORDER);
        return filtered;
    }

    public HdfsPartition getHdfsPartition(String dbName, String tableName, Partition msPart) throws CatalogException {
        ArrayList<TPartitionKeyValue> partitionSpec = new ArrayList<TPartitionKeyValue>();
        Table table = this.getTable(dbName, tableName);
        if (!(table instanceof HdfsTable)) {
            throw new PartitionNotFoundException("Not an HdfsTable: " + dbName + "." + tableName);
        }
        for (int i = 0; i < msPart.getValues().size(); ++i) {
            partitionSpec.add(new TPartitionKeyValue(((HdfsTable)table).getColumns().get(i).getName(), (String)msPart.getValues().get(i)));
        }
        return this.getHdfsPartition(table.getDb().getName(), table.getName(), partitionSpec);
    }

    public HdfsPartition getHdfsPartition(String dbName, String tableName, List<TPartitionKeyValue> partitionSpec) throws CatalogException {
        String partitionNotFoundMsg = "Partition not found: " + Joiner.on((String)", ").join(partitionSpec);
        Table table = this.getTable(dbName, tableName);
        if (!(table instanceof HdfsTable)) {
            throw new PartitionNotFoundException(partitionNotFoundMsg);
        }
        HdfsPartition partition = ((HdfsTable)table).getPartitionFromThriftPartitionSpec(partitionSpec);
        if (partition == null) {
            throw new PartitionNotFoundException(partitionNotFoundMsg);
        }
        return partition;
    }

    public boolean containsHdfsPartition(String dbName, String tableName, List<TPartitionKeyValue> partitionSpec) throws CatalogException {
        try {
            return this.getHdfsPartition(dbName, tableName, partitionSpec) != null;
        }
        catch (PartitionNotFoundException e) {
            return false;
        }
    }

    public TCatalogObject getTCatalogObject(TCatalogObject objectDesc) throws CatalogException {
        return this.getTCatalogObject(objectDesc, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public TCatalogObject getTCatalogObject(TCatalogObject objectDesc, boolean isHumanReadable) throws CatalogException {
        TCatalogObject result = new TCatalogObject();
        switch (objectDesc.getType()) {
            case DATABASE: {
                Db db = this.getDb(objectDesc.getDb().getDb_name());
                if (db == null) {
                    throw new CatalogException("Database not found: " + objectDesc.getDb().getDb_name());
                }
                result.setType(db.getCatalogObjectType());
                result.setCatalog_version(db.getCatalogVersion());
                result.setDb(db.toThrift());
                break;
            }
            case TABLE: 
            case VIEW: {
                Table table = this.getTable(objectDesc.getTable().getDb_name(), objectDesc.getTable().getTbl_name());
                if (table == null) {
                    throw new CatalogException("Table not found: " + objectDesc.getTable().getTbl_name());
                }
                table.takeReadLock();
                try {
                    result.setType(table.getCatalogObjectType());
                    result.setCatalog_version(table.getCatalogVersion());
                    result.setTable(isHumanReadable ? table.toHumanReadableThrift() : table.toThrift());
                    break;
                }
                finally {
                    table.releaseReadLock();
                }
            }
            case FUNCTION: {
                TFunction tfn = objectDesc.getFn();
                Function desc = Function.fromThrift(tfn);
                Function fn = this.getFunction(desc, Function.CompareMode.IS_INDISTINGUISHABLE);
                if (fn == null) {
                    throw new CatalogException("Function not found: " + tfn);
                }
                result.setType(fn.getCatalogObjectType());
                result.setCatalog_version(fn.getCatalogVersion());
                result.setFn(fn.toThrift());
                break;
            }
            case DATA_SOURCE: {
                String dataSrcName = objectDesc.getData_source().getName();
                DataSource dataSrc = this.getDataSource(dataSrcName);
                if (dataSrc == null) {
                    throw new CatalogException("Data source not found: " + dataSrcName);
                }
                result.setType(dataSrc.getCatalogObjectType());
                result.setCatalog_version(dataSrc.getCatalogVersion());
                result.setData_source(dataSrc.toThrift());
                break;
            }
            case HDFS_CACHE_POOL: {
                HdfsCachePool pool = this.getHdfsCachePool(objectDesc.getCache_pool().getPool_name());
                if (pool == null) {
                    throw new CatalogException("Hdfs cache pool not found: " + objectDesc.getCache_pool().getPool_name());
                }
                result.setType(pool.getCatalogObjectType());
                result.setCatalog_version(pool.getCatalogVersion());
                result.setCache_pool(pool.toThrift());
                break;
            }
            case PRINCIPAL: {
                Principal principal = this.authPolicy_.getPrincipal(objectDesc.getPrincipal().getPrincipal_name(), objectDesc.getPrincipal().getPrincipal_type());
                if (principal == null) {
                    throw new CatalogException("Principal not found: " + objectDesc.getPrincipal().getPrincipal_name());
                }
                result.setType(principal.getCatalogObjectType());
                result.setCatalog_version(principal.getCatalogVersion());
                result.setPrincipal(principal.toThrift());
                break;
            }
            case PRIVILEGE: {
                Principal tmpPrincipal = this.authPolicy_.getPrincipal(objectDesc.getPrivilege().getPrincipal_id(), objectDesc.getPrivilege().getPrincipal_type());
                if (tmpPrincipal == null) {
                    throw new CatalogException(String.format("No %s associated with ID: %d", Principal.toString(objectDesc.getPrivilege().getPrincipal_type()).toLowerCase(), objectDesc.getPrivilege().getPrincipal_id()));
                }
                String privilegeName = PrincipalPrivilege.buildPrivilegeName(objectDesc.getPrivilege());
                PrincipalPrivilege privilege = tmpPrincipal.getPrivilege(privilegeName);
                if (privilege != null) {
                    result.setType(privilege.getCatalogObjectType());
                    result.setCatalog_version(privilege.getCatalogVersion());
                    result.setPrivilege(privilege.toThrift());
                    return result;
                }
                throw new CatalogException(String.format("%s '%s' does not contain privilege: '%s'", Principal.toString(tmpPrincipal.getPrincipalType()), tmpPrincipal.getName(), privilegeName));
            }
            case AUTHZ_CACHE_INVALIDATION: {
                AuthzCacheInvalidation authzCacheInvalidation = this.getAuthzCacheInvalidation(objectDesc.getAuthz_cache_invalidation().getMarker_name());
                if (authzCacheInvalidation == null) {
                    throw new CatalogException("Authz cache invalidation not found: " + objectDesc.getAuthz_cache_invalidation().getMarker_name());
                }
                result.setType(authzCacheInvalidation.getCatalogObjectType());
                result.setCatalog_version(authzCacheInvalidation.getCatalogVersion());
                result.setAuthz_cache_invalidation(authzCacheInvalidation.toThrift());
                break;
            }
            default: {
                throw new IllegalStateException("Unexpected TCatalogObject type: " + (Object)((Object)objectDesc.getType()));
            }
        }
        return result;
    }

    public static boolean isDefaultDb(String dbName) {
        return DEFAULT_DB.equals(dbName.toLowerCase());
    }

    public static String toCatalogObjectKey(TCatalogObject catalogObject) {
        Preconditions.checkNotNull((Object)catalogObject);
        switch (catalogObject.getType()) {
            case DATABASE: {
                return "DATABASE:" + catalogObject.getDb().getDb_name().toLowerCase();
            }
            case TABLE: {
                TTable tbl = catalogObject.getTable();
                return "TABLE:" + tbl.getDb_name().toLowerCase() + "." + tbl.getTbl_name().toLowerCase();
            }
            case VIEW: {
                TTable view = catalogObject.getTable();
                return "VIEW:" + view.getDb_name().toLowerCase() + "." + view.getTbl_name().toLowerCase();
            }
            case HDFS_PARTITION: {
                THdfsPartition part = catalogObject.getHdfs_partition();
                return "HDFS_PARTITION:" + part.getDb_name().toLowerCase() + "." + part.getTbl_name().toLowerCase() + ":" + (String)Preconditions.checkNotNull((Object)part.getPartition_name());
            }
            case FUNCTION: {
                return "FUNCTION:" + catalogObject.getFn().getName() + "(" + catalogObject.getFn().getSignature() + ")";
            }
            case PRINCIPAL: {
                String principalName = catalogObject.getPrincipal().getPrincipal_name();
                if (catalogObject.getPrincipal().getPrincipal_type() == TPrincipalType.ROLE) {
                    principalName = principalName.toLowerCase();
                }
                return "PRINCIPAL:" + principalName + "." + catalogObject.getPrincipal().getPrincipal_type().name();
            }
            case PRIVILEGE: {
                return "PRIVILEGE:" + PrincipalPrivilege.buildPrivilegeName(catalogObject.getPrivilege()) + "." + catalogObject.getPrivilege().getPrincipal_id() + "." + (Object)((Object)catalogObject.getPrivilege().getPrincipal_type());
            }
            case HDFS_CACHE_POOL: {
                return "HDFS_CACHE_POOL:" + catalogObject.getCache_pool().getPool_name().toLowerCase();
            }
            case DATA_SOURCE: {
                return "DATA_SOURCE:" + catalogObject.getData_source().getName().toLowerCase();
            }
            case AUTHZ_CACHE_INVALIDATION: {
                return "AUTHZ_CACHE_INVALIDATION:" + catalogObject.getAuthz_cache_invalidation().getMarker_name().toLowerCase();
            }
            case CATALOG: {
                return "CATALOG_SERVICE_ID";
            }
        }
        throw new IllegalStateException("Unsupported catalog object type: " + (Object)((Object)catalogObject.getType()));
    }

    public static boolean keyEquals(TCatalogObject first, TCatalogObject second) {
        return Catalog.toCatalogObjectKey(first).equals(Catalog.toCatalogObjectKey(second));
    }

    public Transaction openTransaction(IMetaStoreClient hmsClient, TransactionKeepalive.HeartbeatContext ctx) throws TransactionException {
        return new Transaction(hmsClient, this.transactionKeepalive_, this.getAcidUserId(), ctx);
    }

    public long lockTableStandalone(String dbName, String tableName, TransactionKeepalive.HeartbeatContext ctx, int lockMaxWaitTime) throws TransactionException {
        return this.lockTableInternal(dbName, tableName, 0L, DataOperationType.NO_TXN, ctx, lockMaxWaitTime);
    }

    public void lockTableInTransaction(String dbName, String tableName, Transaction transaction, DataOperationType opType, TransactionKeepalive.HeartbeatContext ctx, int lockMaxWaitTime) throws TransactionException {
        Preconditions.checkState((transaction.getId() > 0L ? 1 : 0) != 0);
        this.lockTableInternal(dbName, tableName, transaction.getId(), opType, ctx, lockMaxWaitTime);
    }

    private long lockTableInternal(String dbName, String tableName, long txnId, DataOperationType opType, TransactionKeepalive.HeartbeatContext ctx, int lockMaxWaitTime) throws TransactionException {
        Preconditions.checkState((txnId >= 0L ? 1 : 0) != 0);
        LockComponent lockComponent = new LockComponent();
        lockComponent.setDbname(dbName);
        lockComponent.setTablename(tableName);
        lockComponent.setLevel(LockLevel.TABLE);
        lockComponent.setType(LockType.EXCLUSIVE);
        lockComponent.setOperationType(opType);
        List<LockComponent> lockComponents = Arrays.asList(lockComponent);
        long lockId = -1L;
        try (MetaStoreClientPool.MetaStoreClient client = this.metaStoreClientPool_.getClient();){
            lockId = MetastoreShim.acquireLock(client.getHiveClient(), txnId, lockComponents, lockMaxWaitTime);
            if (txnId == 0L) {
                this.transactionKeepalive_.addLock(lockId, ctx);
            }
        }
        return lockId;
    }

    public void releaseTableLock(long lockId) throws TransactionException {
        try (MetaStoreClientPool.MetaStoreClient client = this.metaStoreClientPool_.getClient();){
            this.transactionKeepalive_.deleteLock(lockId);
            MetastoreShim.releaseLock(client.getHiveClient(), lockId);
        }
    }

    public Set<TableWriteId> getWriteIds(Long txnId) {
        Preconditions.checkNotNull((Object)txnId);
        return Collections.unmodifiableSet(this.txnToWriteIds_.getOrDefault(txnId, Collections.emptySet()));
    }

    public void addWriteId(Long txnId, TableWriteId tableWriteId) {
        Preconditions.checkNotNull((Object)txnId);
        Preconditions.checkNotNull((Object)tableWriteId);
        this.txnToWriteIds_.computeIfAbsent(txnId, k -> new HashSet()).add(tableWriteId);
    }

    public Set<TableWriteId> removeWriteIds(Long txnId) {
        Preconditions.checkNotNull((Object)txnId);
        Set<TableWriteId> resultSet = this.txnToWriteIds_.remove(txnId);
        return resultSet != null ? resultSet : Collections.emptySet();
    }

    public void clearWriteIds() {
        this.txnToWriteIds_.clear();
    }

    private static class CatalogObjectOrder
    implements Comparator<HasName> {
        private CatalogObjectOrder() {
        }

        @Override
        public int compare(HasName o1, HasName o2) {
            return String.CASE_INSENSITIVE_ORDER.compare(o1.getName(), o2.getName());
        }
    }
}

