/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.analysis;

import com.google.common.base.Preconditions;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.impala.analysis.ColumnDef;
import org.apache.impala.analysis.HdfsUri;
import org.apache.impala.analysis.TypeDef;
import org.apache.impala.catalog.ArrayType;
import org.apache.impala.catalog.MapType;
import org.apache.impala.catalog.ScalarType;
import org.apache.impala.catalog.StructField;
import org.apache.impala.catalog.StructType;
import org.apache.impala.catalog.Type;
import org.apache.impala.common.AnalysisException;
import org.apache.impala.common.FileSystemUtil;
import org.apache.impala.util.FileAnalysisUtil;
import org.apache.parquet.format.converter.ParquetMetadataConverter;
import org.apache.parquet.hadoop.ParquetFileReader;
import org.apache.parquet.hadoop.metadata.ParquetMetadata;
import org.apache.parquet.schema.GroupType;
import org.apache.parquet.schema.LogicalTypeAnnotation;
import org.apache.parquet.schema.MessageType;
import org.apache.parquet.schema.PrimitiveType;
import org.apache.parquet.schema.Type;

class ParquetSchemaExtractor {
    private static final String ERROR_MSG = "Failed to convert Parquet type\n%s\nto an Impala %s type:\n%s\n";

    ParquetSchemaExtractor() {
    }

    private static MessageType loadParquetSchema(Path pathToFile) throws AnalysisException {
        FileAnalysisUtil.CheckIfFile(pathToFile);
        ParquetMetadata readFooter = null;
        try {
            readFooter = ParquetFileReader.readFooter((Configuration)FileSystemUtil.getConfiguration(), (Path)pathToFile, (ParquetMetadataConverter.MetadataFilter)ParquetMetadataConverter.NO_FILTER);
        }
        catch (FileNotFoundException e) {
            throw new AnalysisException("File not found: " + e);
        }
        catch (IOException e) {
            throw new AnalysisException("Failed to open file as a parquet file: " + e);
        }
        catch (RuntimeException e) {
            if (e.toString().contains("is not a Parquet file")) {
                throw new AnalysisException("File is not a parquet file: " + pathToFile);
            }
            throw e;
        }
        return readFooter.getFileMetaData().getSchema();
    }

    private static Type convertPrimitiveParquetType(org.apache.parquet.schema.Type parquetType) throws AnalysisException {
        Preconditions.checkState((boolean)parquetType.isPrimitive());
        PrimitiveType prim = parquetType.asPrimitiveType();
        switch (prim.getPrimitiveTypeName()) {
            case BINARY: {
                return Type.STRING;
            }
            case BOOLEAN: {
                return Type.BOOLEAN;
            }
            case DOUBLE: {
                return Type.DOUBLE;
            }
            case FIXED_LEN_BYTE_ARRAY: {
                throw new AnalysisException("Unsupported parquet type FIXED_LEN_BYTE_ARRAY for field " + parquetType.getName());
            }
            case FLOAT: {
                return Type.FLOAT;
            }
            case INT32: {
                return Type.INT;
            }
            case INT64: {
                return Type.BIGINT;
            }
            case INT96: {
                return Type.TIMESTAMP;
            }
        }
        Preconditions.checkState((boolean)false, (Object)("Unexpected parquet primitive type: " + prim.getPrimitiveTypeName()));
        return null;
    }

    private static MapType convertMap(GroupType outerGroup) throws AnalysisException {
        if (outerGroup.getFieldCount() != 1) {
            throw new AnalysisException(String.format(ERROR_MSG, outerGroup.toString(), "MAP", "The logical MAP type must have exactly 1 inner field."));
        }
        org.apache.parquet.schema.Type innerField = outerGroup.getType(0);
        if (!innerField.isRepetition(Type.Repetition.REPEATED)) {
            throw new AnalysisException(String.format(ERROR_MSG, outerGroup.toString(), "MAP", "The logical MAP type must have a repeated inner field."));
        }
        if (innerField.isPrimitive()) {
            throw new AnalysisException(String.format(ERROR_MSG, outerGroup.toString(), "MAP", "The inner field of the logical MAP type must be a group."));
        }
        GroupType innerGroup = innerField.asGroupType();
        if (innerGroup.getFieldCount() != 2) {
            throw new AnalysisException(String.format(ERROR_MSG, outerGroup.toString(), "MAP", "The inner field of the logical MAP type must have exactly 2 fields."));
        }
        org.apache.parquet.schema.Type key = innerGroup.getType(0);
        if (!key.getName().equals("key")) {
            throw new AnalysisException(String.format(ERROR_MSG, outerGroup.toString(), "MAP", "The name of the first field of the inner field of the logical MAP type must be 'key'"));
        }
        if (!key.isPrimitive()) {
            throw new AnalysisException(String.format(ERROR_MSG, outerGroup.toString(), "MAP", "The key type of the logical MAP type must be primitive."));
        }
        org.apache.parquet.schema.Type value = innerGroup.getType(1);
        if (!value.getName().equals("value")) {
            throw new AnalysisException(String.format(ERROR_MSG, outerGroup.toString(), "MAP", "The name of the second field of the inner field of the logical MAP type must be 'value'"));
        }
        return new MapType(ParquetSchemaExtractor.convertParquetType(key), ParquetSchemaExtractor.convertParquetType(value));
    }

    private static StructType convertStruct(GroupType outerGroup) throws AnalysisException {
        ArrayList<StructField> structFields = new ArrayList<StructField>();
        for (org.apache.parquet.schema.Type field : outerGroup.getFields()) {
            StructField f = new StructField(field.getName(), ParquetSchemaExtractor.convertParquetType(field));
            structFields.add(f);
        }
        return new StructType(structFields);
    }

    private static ArrayType convertArray(GroupType outerGroup) throws AnalysisException {
        if (outerGroup.getFieldCount() != 1) {
            throw new AnalysisException(String.format(ERROR_MSG, outerGroup.toString(), "LIST", "The logical LIST type must have exactly 1 inner field."));
        }
        org.apache.parquet.schema.Type innerField = outerGroup.getType(0);
        if (!innerField.isRepetition(Type.Repetition.REPEATED)) {
            throw new AnalysisException(String.format(ERROR_MSG, outerGroup.toString(), "LIST", "The inner field of the logical LIST type must be repeated."));
        }
        if (innerField.isPrimitive() || innerField.getOriginalType() != null) {
            return new ArrayType(ParquetSchemaExtractor.convertParquetType(innerField));
        }
        GroupType innerGroup = innerField.asGroupType();
        if (innerGroup.getFieldCount() != 1) {
            return new ArrayType(ParquetSchemaExtractor.convertStruct(innerGroup));
        }
        return new ArrayType(ParquetSchemaExtractor.convertParquetType(innerGroup.getType(0)));
    }

    private static Type convertLogicalParquetType(org.apache.parquet.schema.Type parquetType) throws AnalysisException {
        LogicalTypeAnnotation logicalType = parquetType.getLogicalTypeAnnotation();
        if (logicalType instanceof LogicalTypeAnnotation.ListLogicalTypeAnnotation) {
            return ParquetSchemaExtractor.convertArray(parquetType.asGroupType());
        }
        if (logicalType instanceof LogicalTypeAnnotation.MapLogicalTypeAnnotation || logicalType instanceof LogicalTypeAnnotation.MapKeyValueTypeAnnotation) {
            return ParquetSchemaExtractor.convertMap(parquetType.asGroupType());
        }
        PrimitiveType prim = parquetType.asPrimitiveType();
        if (prim.getPrimitiveTypeName() == PrimitiveType.PrimitiveTypeName.BINARY && (logicalType instanceof LogicalTypeAnnotation.StringLogicalTypeAnnotation || logicalType instanceof LogicalTypeAnnotation.EnumLogicalTypeAnnotation)) {
            return Type.STRING;
        }
        if (prim.getPrimitiveTypeName() == PrimitiveType.PrimitiveTypeName.INT64 && logicalType instanceof LogicalTypeAnnotation.TimestampLogicalTypeAnnotation) {
            return Type.TIMESTAMP;
        }
        if (prim.getPrimitiveTypeName() == PrimitiveType.PrimitiveTypeName.INT32 && logicalType instanceof LogicalTypeAnnotation.DateLogicalTypeAnnotation) {
            return Type.DATE;
        }
        if ((prim.getPrimitiveTypeName() == PrimitiveType.PrimitiveTypeName.INT32 || prim.getPrimitiveTypeName() == PrimitiveType.PrimitiveTypeName.INT64) && logicalType instanceof LogicalTypeAnnotation.IntLogicalTypeAnnotation && ((LogicalTypeAnnotation.IntLogicalTypeAnnotation)logicalType).isSigned()) {
            switch (((LogicalTypeAnnotation.IntLogicalTypeAnnotation)logicalType).getBitWidth()) {
                case 8: {
                    return Type.TINYINT;
                }
                case 16: {
                    return Type.SMALLINT;
                }
                case 32: {
                    return Type.INT;
                }
                case 64: {
                    return Type.BIGINT;
                }
            }
        }
        if (logicalType instanceof LogicalTypeAnnotation.DecimalLogicalTypeAnnotation) {
            LogicalTypeAnnotation.DecimalLogicalTypeAnnotation decimal = (LogicalTypeAnnotation.DecimalLogicalTypeAnnotation)logicalType;
            return ScalarType.createDecimalType(decimal.getPrecision(), decimal.getScale());
        }
        throw new AnalysisException("Unsupported logical parquet type " + logicalType + " (primitive type is " + prim.getPrimitiveTypeName().name() + ") for field " + parquetType.getName());
    }

    private static Type convertParquetType(org.apache.parquet.schema.Type field) throws AnalysisException {
        Type type = null;
        type = field.getLogicalTypeAnnotation() != null ? ParquetSchemaExtractor.convertLogicalParquetType(field) : (field.isPrimitive() ? ParquetSchemaExtractor.convertPrimitiveParquetType(field) : ParquetSchemaExtractor.convertStruct(field.asGroupType()));
        return type;
    }

    static List<ColumnDef> extract(HdfsUri location) throws AnalysisException {
        MessageType parquetSchema = ParquetSchemaExtractor.loadParquetSchema(location.getPath());
        List fields = parquetSchema.getFields();
        ArrayList<ColumnDef> schema = new ArrayList<ColumnDef>();
        for (org.apache.parquet.schema.Type field : fields) {
            Type type = ParquetSchemaExtractor.convertParquetType(field);
            Preconditions.checkNotNull((Object)type);
            String colName = field.getName();
            HashMap<ColumnDef.Option, Object> option = new HashMap<ColumnDef.Option, Object>();
            option.put(ColumnDef.Option.COMMENT, "Inferred from Parquet file.");
            schema.add(new ColumnDef(colName, new TypeDef(type), option));
        }
        return schema;
    }
}

