/*
 * Decompiled with CFR 0.152.
 */
package org.apache.impala.analysis;

import com.google.common.base.Preconditions;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.List;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.impala.analysis.Analyzer;
import org.apache.impala.analysis.HdfsUri;
import org.apache.impala.analysis.PartitionSpec;
import org.apache.impala.analysis.QueryStringBuilder;
import org.apache.impala.analysis.StatementBase;
import org.apache.impala.analysis.TableName;
import org.apache.impala.analysis.TableRef;
import org.apache.impala.analysis.ToSqlOptions;
import org.apache.impala.authorization.Privilege;
import org.apache.impala.catalog.FeFsTable;
import org.apache.impala.catalog.FeIcebergTable;
import org.apache.impala.catalog.FeTable;
import org.apache.impala.common.AnalysisException;
import org.apache.impala.common.FileSystemUtil;
import org.apache.impala.thrift.TLoadDataReq;
import org.apache.impala.thrift.TTableName;
import org.apache.impala.util.FsPermissionChecker;

public class LoadDataStmt
extends StatementBase {
    private final TableName tableName_;
    private final HdfsUri sourceDataPath_;
    private final PartitionSpec partitionSpec_;
    private final boolean overwrite_;
    private String dbName_;
    private FeTable table_;
    private String createTmpTblQuery_;
    private String insertTblQuery_;
    private String dropTmpTblQuery_;

    public LoadDataStmt(TableName tableName, HdfsUri sourceDataPath, boolean overwrite, PartitionSpec partitionSpec) {
        Preconditions.checkNotNull((Object)tableName);
        Preconditions.checkNotNull((Object)sourceDataPath);
        this.tableName_ = tableName;
        this.sourceDataPath_ = sourceDataPath;
        this.overwrite_ = overwrite;
        this.partitionSpec_ = partitionSpec;
    }

    public String getTbl() {
        return this.tableName_.getTbl();
    }

    public String getDb() {
        Preconditions.checkNotNull((Object)this.dbName_);
        return this.dbName_;
    }

    @Override
    public String toSql(ToSqlOptions options) {
        StringBuilder sb = new StringBuilder("LOAD DATA INPATH '");
        sb.append(this.sourceDataPath_ + "' ");
        if (this.overwrite_) {
            sb.append("OVERWRITE ");
        }
        sb.append("INTO TABLE " + this.tableName_.toString());
        if (this.partitionSpec_ != null) {
            sb.append(" " + this.partitionSpec_.toSql(options));
        }
        return sb.toString();
    }

    @Override
    public void collectTableRefs(List<TableRef> tblRefs) {
        tblRefs.add(new TableRef(this.tableName_.toPath(), null));
    }

    @Override
    public void analyze(Analyzer analyzer) throws AnalysisException {
        this.dbName_ = analyzer.getTargetDbName(this.tableName_);
        this.table_ = analyzer.getTable(this.tableName_, Privilege.INSERT);
        if (!(this.table_ instanceof FeFsTable)) {
            throw new AnalysisException("LOAD DATA only supported for HDFS tables: " + this.dbName_ + "." + this.getTbl());
        }
        Analyzer.checkTableCapability(this.table_, Analyzer.OperationType.WRITE);
        Analyzer.ensureTableNotTransactional(this.table_, "LOAD DATA");
        if (this.partitionSpec_ != null) {
            if (this.table_ instanceof FeIcebergTable) {
                throw new AnalysisException("PARTITION clause is not supported for Iceberg tables.");
            }
            this.partitionSpec_.setTableName(this.tableName_);
            this.partitionSpec_.setPartitionShouldExist();
            this.partitionSpec_.setPrivilegeRequirement(Privilege.INSERT);
            this.partitionSpec_.analyze(analyzer);
        } else if (this.table_.getMetaStoreTable().getPartitionKeysSize() > 0) {
            throw new AnalysisException("Table is partitioned but no partition spec was specified: " + this.dbName_ + "." + this.getTbl());
        }
        this.analyzePaths(analyzer, (FeFsTable)this.table_);
        if (this.table_ instanceof FeIcebergTable) {
            this.analyzeLoadIntoIcebergTable();
        }
    }

    private void analyzePaths(Analyzer analyzer, FeFsTable table) throws AnalysisException {
        this.sourceDataPath_.analyze(analyzer, Privilege.ALL);
        try {
            Path source = this.sourceDataPath_.getPath();
            FileSystem fs = source.getFileSystem(FileSystemUtil.getConfiguration());
            if (!FileSystemUtil.isValidLoadDataInpath(fs)) {
                throw new AnalysisException(String.format("INPATH location '%s' must point to one of the supported filesystem URI scheme (%s).", this.sourceDataPath_, FileSystemUtil.getValidLoadDataInpathSchemes()));
            }
            if (!fs.exists(source)) {
                throw new AnalysisException(String.format("INPATH location '%s' does not exist.", this.sourceDataPath_));
            }
            FsPermissionChecker checker = FsPermissionChecker.getInstance();
            boolean shouldCheckPerms = FileSystemUtil.FsType.getFsType(fs.getScheme()) != FileSystemUtil.FsType.ADLS;
            boolean authzEnabled = analyzer.isAuthzEnabled();
            if (fs.isDirectory(source)) {
                if (FileSystemUtil.getTotalNumVisibleFiles(source) == 0) {
                    throw new AnalysisException(String.format("INPATH location '%s' contains no visible files.", this.sourceDataPath_));
                }
                if (FileSystemUtil.containsVisibleSubdirectory(source)) {
                    throw new AnalysisException(String.format("INPATH location '%s' cannot contain non-hidden subdirectories.", this.sourceDataPath_));
                }
                if (shouldCheckPerms && !checker.checkAccess(source, fs, FsAction.READ_WRITE, authzEnabled)) {
                    throw new AnalysisException(String.format("Unable to LOAD DATA from %s because Impala does not have READ and WRITE permissions on this directory", source));
                }
            } else {
                if (FileSystemUtil.isHiddenFile(source.getName())) {
                    throw new AnalysisException(String.format("INPATH location '%s' points to a hidden file.", source));
                }
                if (shouldCheckPerms && !checker.checkAccess(source.getParent(), fs, FsAction.WRITE, authzEnabled)) {
                    throw new AnalysisException(String.format("Unable to LOAD DATA from %s because Impala does not have WRITE permissions on its parent directory %s", source, source.getParent()));
                }
                if (shouldCheckPerms && !checker.checkAccess(source, fs, FsAction.READ, authzEnabled)) {
                    throw new AnalysisException(String.format("Unable to LOAD DATA from %s because Impala does not have READ permissions on this file", source));
                }
            }
            FeFsTable.Utils.checkWriteAccess(table, this.partitionSpec_ != null ? this.partitionSpec_.getPartitionSpecKeyValues() : null, "LOAD DATA");
        }
        catch (FileNotFoundException e) {
            throw new AnalysisException("File not found: " + e.getMessage(), e);
        }
        catch (IOException e) {
            throw new AnalysisException("Error accessing filesystem: " + e.getMessage(), e);
        }
    }

    private void analyzeLoadIntoIcebergTable() throws AnalysisException {
        Path sourcePath = this.sourceDataPath_.getPath();
        String tmpTableName = QueryStringBuilder.createTmpTableName(this.dbName_, this.tableName_.getTbl());
        QueryStringBuilder.Create createTableQueryBuilder = QueryStringBuilder.Create.builder().table(tmpTableName, true);
        try {
            String magicString;
            FileSystem fs = sourcePath.getFileSystem(FileSystemUtil.getConfiguration());
            Path filePathForLike = sourcePath;
            if (fs.isDirectory(sourcePath)) {
                RemoteIterator<? extends FileStatus> fileStatuses = FileSystemUtil.listFiles(fs, sourcePath, true, "");
                while (fileStatuses.hasNext()) {
                    FileStatus fileStatus = (FileStatus)fileStatuses.next();
                    String fileName = fileStatus.getPath().getName();
                    if (!fileStatus.isFile() || FileSystemUtil.isHiddenFile(fileName)) continue;
                    filePathForLike = fileStatus.getPath();
                    break;
                }
            }
            if ((magicString = FileSystemUtil.readMagicString(filePathForLike)).substring(0, 4).equals("PAR1")) {
                createTableQueryBuilder.like("PARQUET", "%s").storedAs("PARQUET");
            } else if (magicString.substring(0, 3).equals("ORC")) {
                createTableQueryBuilder.like("ORC", "%s").storedAs("ORC");
            } else {
                throw new AnalysisException(String.format("INPATH contains unsupported LOAD format, file '%s' has '%s' magic string.", filePathForLike, magicString));
            }
            createTableQueryBuilder.tableLocation("%s");
            createTableQueryBuilder.property("TEMPORARY", "true");
        }
        catch (IOException e) {
            throw new AnalysisException("Failed to generate CREATE TABLE subquery statement. ", e);
        }
        this.createTmpTblQuery_ = createTableQueryBuilder.build();
        QueryStringBuilder.Insert insertTblQueryBuilder = QueryStringBuilder.Insert.builder().overwrite(this.overwrite_).table(this.tableName_.toString());
        QueryStringBuilder.Select insertSelectTblQueryBuilder = QueryStringBuilder.Select.builder().selectList("*").from(tmpTableName);
        insertTblQueryBuilder.select(insertSelectTblQueryBuilder);
        this.insertTblQuery_ = insertTblQueryBuilder.build();
        this.dropTmpTblQuery_ = QueryStringBuilder.Drop.builder().table(tmpTableName).build();
    }

    public TLoadDataReq toThrift() {
        TLoadDataReq loadDataReq = new TLoadDataReq();
        loadDataReq.setTable_name(new TTableName(this.getDb(), this.getTbl()));
        loadDataReq.setSource_path(this.sourceDataPath_.toString());
        loadDataReq.setOverwrite(this.overwrite_);
        if (this.partitionSpec_ != null) {
            loadDataReq.setPartition_spec(this.partitionSpec_.toThrift());
        }
        if (this.table_ instanceof FeIcebergTable) {
            loadDataReq.setIceberg_tbl(true);
            loadDataReq.setCreate_tmp_tbl_query_template(this.createTmpTblQuery_);
            loadDataReq.setInsert_into_dst_tbl_query(this.insertTblQuery_);
            loadDataReq.setDrop_tmp_tbl_query(this.dropTmpTblQuery_);
        }
        return loadDataReq;
    }
}

