/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.util;

import java.nio.ByteBuffer;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Random;
import org.apache.iceberg.relocated.com.google.common.primitives.UnsignedBytes;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.RandomUtil;
import org.apache.iceberg.util.ZOrderByteUtils;
import org.junit.Assert;
import org.junit.Test;

public class TestZOrderByteUtil {
    private static final byte IIIIIIII = -1;
    private static final byte IOIOIOIO = -86;
    private static final byte OIOIOIOI = 85;
    private static final byte OOOOIIII = 15;
    private static final byte OOOOOOOI = 1;
    private static final byte OOOOOOOO = 0;
    private static final int NUM_TESTS = 100000;
    private static final int NUM_INTERLEAVE_TESTS = 1000;
    private final Random random = new Random(42L);

    private String bytesToString(byte[] bytes) {
        StringBuilder result = new StringBuilder();
        for (byte b : bytes) {
            result.append(String.format("%8s", Integer.toBinaryString(b & 0xFF)).replace(' ', '0'));
        }
        return result.toString();
    }

    private byte[] generateRandomBytes() {
        int length = Math.abs(this.random.nextInt(100) + 1);
        return this.generateRandomBytes(length);
    }

    private byte[] generateRandomBytes(int length) {
        byte[] result = new byte[length];
        this.random.nextBytes(result);
        return result;
    }

    private String interleaveStrings(String[] strings) {
        StringBuilder result = new StringBuilder();
        int totalLength = Arrays.stream(strings).mapToInt(String::length).sum();
        int substringIndex = 0;
        int characterIndex = 0;
        while (characterIndex < totalLength) {
            for (String str : strings) {
                if (substringIndex >= str.length()) continue;
                result.append(str.charAt(substringIndex));
                ++characterIndex;
            }
            ++substringIndex;
        }
        return result.toString();
    }

    @Test
    public void testInterleaveRandomExamples() {
        for (int test = 0; test < 1000; ++test) {
            int numByteArrays = Math.abs(this.random.nextInt(6)) + 1;
            byte[][] testBytes = new byte[numByteArrays][];
            String[] testStrings = new String[numByteArrays];
            for (int byteIndex = 0; byteIndex < numByteArrays; ++byteIndex) {
                testBytes[byteIndex] = this.generateRandomBytes();
                testStrings[byteIndex] = this.bytesToString(testBytes[byteIndex]);
            }
            int zOrderSize = Arrays.stream(testBytes).mapToInt(column -> ((byte[])column).length).sum();
            byte[] byteResult = ZOrderByteUtils.interleaveBits((byte[][])testBytes, (int)zOrderSize);
            String byteResultAsString = this.bytesToString(byteResult);
            String stringResult = this.interleaveStrings(testStrings);
            Assert.assertEquals((String)"String interleave didn't match byte interleave", (Object)stringResult, (Object)byteResultAsString);
        }
    }

    @Test
    public void testReuseInterleaveBuffer() {
        int numByteArrays = 2;
        int colLength = 16;
        ByteBuffer interleaveBuffer = ByteBuffer.allocate(numByteArrays * colLength);
        for (int test = 0; test < 1000; ++test) {
            byte[][] testBytes = new byte[numByteArrays][];
            String[] testStrings = new String[numByteArrays];
            for (int byteIndex = 0; byteIndex < numByteArrays; ++byteIndex) {
                testBytes[byteIndex] = this.generateRandomBytes(colLength);
                testStrings[byteIndex] = this.bytesToString(testBytes[byteIndex]);
            }
            byte[] byteResult = ZOrderByteUtils.interleaveBits((byte[][])testBytes, (int)(numByteArrays * colLength), (ByteBuffer)interleaveBuffer);
            String byteResultAsString = this.bytesToString(byteResult);
            String stringResult = this.interleaveStrings(testStrings);
            Assert.assertEquals((String)"String interleave didn't match byte interleave", (Object)stringResult, (Object)byteResultAsString);
        }
    }

    @Test
    public void testInterleaveEmptyBits() {
        byte[][] test = new byte[4][10];
        byte[] expected = new byte[40];
        Assert.assertArrayEquals((String)"Should combine empty arrays", (byte[])expected, (byte[])ZOrderByteUtils.interleaveBits((byte[][])test, (int)40));
    }

    @Test
    public void testInterleaveFullBits() {
        byte[][] test = new byte[][]{{-1, -1}, {-1}, new byte[0], {-1, -1, -1}};
        byte[] expected = new byte[]{-1, -1, -1, -1, -1, -1};
        Assert.assertArrayEquals((String)"Should combine full arrays", (byte[])expected, (byte[])ZOrderByteUtils.interleaveBits((byte[][])test, (int)6));
    }

    @Test
    public void testInterleaveMixedBits() {
        byte[][] test = new byte[][]{{1, -1, 0, 15}, {1, 0, -1}, {1}, {1}};
        byte[] expected = new byte[]{0, 0, 0, 15, -86, -86, 85, 85, 15};
        Assert.assertArrayEquals((String)"Should combine mixed byte arrays", (byte[])expected, (byte[])ZOrderByteUtils.interleaveBits((byte[][])test, (int)9));
    }

    @Test
    public void testIntOrdering() {
        ByteBuffer aBuffer = ZOrderByteUtils.allocatePrimitiveBuffer();
        ByteBuffer bBuffer = ZOrderByteUtils.allocatePrimitiveBuffer();
        for (int i = 0; i < 100000; ++i) {
            int aInt = this.random.nextInt();
            int bInt = this.random.nextInt();
            int intCompare = Integer.signum(Integer.compare(aInt, bInt));
            byte[] aBytes = ZOrderByteUtils.intToOrderedBytes((int)aInt, (ByteBuffer)aBuffer).array();
            byte[] bBytes = ZOrderByteUtils.intToOrderedBytes((int)bInt, (ByteBuffer)bBuffer).array();
            int byteCompare = Integer.signum(UnsignedBytes.lexicographicalComparator().compare(aBytes, bBytes));
            Assert.assertEquals((String)String.format("Ordering of ints should match ordering of bytes, %s ~ %s -> %s != %s ~ %s -> %s ", aInt, bInt, intCompare, Arrays.toString(aBytes), Arrays.toString(bBytes), byteCompare), (long)intCompare, (long)byteCompare);
        }
    }

    @Test
    public void testLongOrdering() {
        ByteBuffer aBuffer = ZOrderByteUtils.allocatePrimitiveBuffer();
        ByteBuffer bBuffer = ZOrderByteUtils.allocatePrimitiveBuffer();
        for (int i = 0; i < 100000; ++i) {
            long aLong = this.random.nextInt();
            long bLong = this.random.nextInt();
            int longCompare = Integer.signum(Long.compare(aLong, bLong));
            byte[] aBytes = ZOrderByteUtils.longToOrderedBytes((long)aLong, (ByteBuffer)aBuffer).array();
            byte[] bBytes = ZOrderByteUtils.longToOrderedBytes((long)bLong, (ByteBuffer)bBuffer).array();
            int byteCompare = Integer.signum(UnsignedBytes.lexicographicalComparator().compare(aBytes, bBytes));
            Assert.assertEquals((String)String.format("Ordering of longs should match ordering of bytes, %s ~ %s -> %s != %s ~ %s -> %s ", aLong, bLong, longCompare, Arrays.toString(aBytes), Arrays.toString(bBytes), byteCompare), (long)longCompare, (long)byteCompare);
        }
    }

    @Test
    public void testShortOrdering() {
        ByteBuffer aBuffer = ZOrderByteUtils.allocatePrimitiveBuffer();
        ByteBuffer bBuffer = ZOrderByteUtils.allocatePrimitiveBuffer();
        for (int i = 0; i < 100000; ++i) {
            short aShort = (short)(this.random.nextInt() % 32768);
            short bShort = (short)(this.random.nextInt() % 32768);
            int longCompare = Integer.signum(Long.compare(aShort, bShort));
            byte[] aBytes = ZOrderByteUtils.shortToOrderedBytes((short)aShort, (ByteBuffer)aBuffer).array();
            byte[] bBytes = ZOrderByteUtils.shortToOrderedBytes((short)bShort, (ByteBuffer)bBuffer).array();
            int byteCompare = Integer.signum(UnsignedBytes.lexicographicalComparator().compare(aBytes, bBytes));
            Assert.assertEquals((String)String.format("Ordering of longs should match ordering of bytes, %s ~ %s -> %s != %s ~ %s -> %s ", aShort, bShort, longCompare, Arrays.toString(aBytes), Arrays.toString(bBytes), byteCompare), (long)longCompare, (long)byteCompare);
        }
    }

    @Test
    public void testTinyOrdering() {
        ByteBuffer aBuffer = ZOrderByteUtils.allocatePrimitiveBuffer();
        ByteBuffer bBuffer = ZOrderByteUtils.allocatePrimitiveBuffer();
        for (int i = 0; i < 100000; ++i) {
            byte aByte = (byte)(this.random.nextInt() % 128);
            byte bByte = (byte)(this.random.nextInt() % 128);
            int longCompare = Integer.signum(Long.compare(aByte, bByte));
            byte[] aBytes = ZOrderByteUtils.tinyintToOrderedBytes((byte)aByte, (ByteBuffer)aBuffer).array();
            byte[] bBytes = ZOrderByteUtils.tinyintToOrderedBytes((byte)bByte, (ByteBuffer)bBuffer).array();
            int byteCompare = Integer.signum(UnsignedBytes.lexicographicalComparator().compare(aBytes, bBytes));
            Assert.assertEquals((String)String.format("Ordering of longs should match ordering of bytes, %s ~ %s -> %s != %s ~ %s -> %s ", aByte, bByte, longCompare, Arrays.toString(aBytes), Arrays.toString(bBytes), byteCompare), (long)longCompare, (long)byteCompare);
        }
    }

    @Test
    public void testFloatOrdering() {
        ByteBuffer aBuffer = ZOrderByteUtils.allocatePrimitiveBuffer();
        ByteBuffer bBuffer = ZOrderByteUtils.allocatePrimitiveBuffer();
        for (int i = 0; i < 100000; ++i) {
            float aFloat = this.random.nextFloat();
            float bFloat = this.random.nextFloat();
            int floatCompare = Integer.signum(Float.compare(aFloat, bFloat));
            byte[] aBytes = ZOrderByteUtils.floatToOrderedBytes((float)aFloat, (ByteBuffer)aBuffer).array();
            byte[] bBytes = ZOrderByteUtils.floatToOrderedBytes((float)bFloat, (ByteBuffer)bBuffer).array();
            int byteCompare = Integer.signum(UnsignedBytes.lexicographicalComparator().compare(aBytes, bBytes));
            Assert.assertEquals((String)String.format("Ordering of floats should match ordering of bytes, %s ~ %s -> %s != %s ~ %s -> %s ", Float.valueOf(aFloat), Float.valueOf(bFloat), floatCompare, Arrays.toString(aBytes), Arrays.toString(bBytes), byteCompare), (long)floatCompare, (long)byteCompare);
        }
    }

    @Test
    public void testDoubleOrdering() {
        ByteBuffer aBuffer = ZOrderByteUtils.allocatePrimitiveBuffer();
        ByteBuffer bBuffer = ZOrderByteUtils.allocatePrimitiveBuffer();
        for (int i = 0; i < 100000; ++i) {
            double aDouble = this.random.nextDouble();
            double bDouble = this.random.nextDouble();
            int doubleCompare = Integer.signum(Double.compare(aDouble, bDouble));
            byte[] aBytes = ZOrderByteUtils.doubleToOrderedBytes((double)aDouble, (ByteBuffer)aBuffer).array();
            byte[] bBytes = ZOrderByteUtils.doubleToOrderedBytes((double)bDouble, (ByteBuffer)bBuffer).array();
            int byteCompare = Integer.signum(UnsignedBytes.lexicographicalComparator().compare(aBytes, bBytes));
            Assert.assertEquals((String)String.format("Ordering of doubles should match ordering of bytes, %s ~ %s -> %s != %s ~ %s -> %s ", aDouble, bDouble, doubleCompare, Arrays.toString(aBytes), Arrays.toString(bBytes), byteCompare), (long)doubleCompare, (long)byteCompare);
        }
    }

    @Test
    public void testStringOrdering() {
        CharsetEncoder encoder = StandardCharsets.UTF_8.newEncoder();
        ByteBuffer aBuffer = ByteBuffer.allocate(128);
        ByteBuffer bBuffer = ByteBuffer.allocate(128);
        for (int i = 0; i < 100000; ++i) {
            String aString = (String)RandomUtil.generatePrimitive((Type.PrimitiveType)Types.StringType.get(), (Random)this.random);
            String bString = (String)RandomUtil.generatePrimitive((Type.PrimitiveType)Types.StringType.get(), (Random)this.random);
            int stringCompare = Integer.signum(aString.compareTo(bString));
            byte[] aBytes = ZOrderByteUtils.stringToOrderedBytes((String)aString, (int)128, (ByteBuffer)aBuffer, (CharsetEncoder)encoder).array();
            byte[] bBytes = ZOrderByteUtils.stringToOrderedBytes((String)bString, (int)128, (ByteBuffer)bBuffer, (CharsetEncoder)encoder).array();
            int byteCompare = Integer.signum(UnsignedBytes.lexicographicalComparator().compare(aBytes, bBytes));
            Assert.assertEquals((String)String.format("Ordering of strings should match ordering of bytes, %s ~ %s -> %s != %s ~ %s -> %s ", aString, bString, stringCompare, Arrays.toString(aBytes), Arrays.toString(bBytes), byteCompare), (long)stringCompare, (long)byteCompare);
        }
    }

    @Test
    public void testByteTruncateOrFill() {
        ByteBuffer aBuffer = ByteBuffer.allocate(128);
        ByteBuffer bBuffer = ByteBuffer.allocate(128);
        for (int i = 0; i < 100000; ++i) {
            byte[] aBytesRaw = (byte[])RandomUtil.generatePrimitive((Type.PrimitiveType)Types.BinaryType.get(), (Random)this.random);
            byte[] bBytesRaw = (byte[])RandomUtil.generatePrimitive((Type.PrimitiveType)Types.BinaryType.get(), (Random)this.random);
            int stringCompare = Integer.signum(UnsignedBytes.lexicographicalComparator().compare(aBytesRaw, bBytesRaw));
            byte[] aBytes = ZOrderByteUtils.byteTruncateOrFill((byte[])aBytesRaw, (int)128, (ByteBuffer)aBuffer).array();
            byte[] bBytes = ZOrderByteUtils.byteTruncateOrFill((byte[])bBytesRaw, (int)128, (ByteBuffer)bBuffer).array();
            int byteCompare = Integer.signum(UnsignedBytes.lexicographicalComparator().compare(aBytes, bBytes));
            Assert.assertEquals((String)String.format("Ordering of strings should match ordering of bytes, %s ~ %s -> %s != %s ~ %s -> %s ", aBytesRaw, bBytesRaw, stringCompare, Arrays.toString(aBytes), Arrays.toString(bBytes), byteCompare), (long)stringCompare, (long)byteCompare);
        }
    }
}

