/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.io;

import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.apache.iceberg.io.ByteBufferInputStream;
import org.apache.iceberg.io.MultiBufferInputStream;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.Assert;
import org.junit.Test;

public abstract class TestByteBufferInputStreams {
    protected abstract ByteBufferInputStream newStream();

    protected abstract void checkOriginalData();

    @Test
    public void testRead0() throws Exception {
        byte[] bytes = new byte[]{};
        ByteBufferInputStream stream = this.newStream();
        Assert.assertEquals((String)"Should read 0 bytes", (long)0L, (long)stream.read(bytes));
        int bytesRead = stream.read(new byte[100]);
        Assert.assertTrue((String)"Should read to end of stream", (bytesRead < 100 ? 1 : 0) != 0);
        Assert.assertEquals((String)"Should read 0 bytes at end of stream", (long)0L, (long)stream.read(bytes));
    }

    @Test
    public void testReadAll() throws Exception {
        byte[] bytes = new byte[35];
        ByteBufferInputStream stream = this.newStream();
        int bytesRead = stream.read(bytes);
        Assert.assertEquals((String)"Should read the entire buffer", (long)bytes.length, (long)bytesRead);
        for (int i = 0; i < bytes.length; ++i) {
            Assert.assertEquals((String)"Byte i should be i", (long)i, (long)bytes[i]);
            Assert.assertEquals((String)"Should advance position", (long)35L, (long)stream.getPos());
        }
        Assert.assertEquals((String)"Should have no more remaining content", (long)0L, (long)stream.available());
        Assert.assertEquals((String)"Should return -1 at end of stream", (long)-1L, (long)stream.read(bytes));
        Assert.assertEquals((String)"Should have no more remaining content", (long)0L, (long)stream.available());
        this.checkOriginalData();
    }

    @Test
    public void testSmallReads() throws Exception {
        for (int size = 1; size < 36; ++size) {
            byte[] bytes = new byte[size];
            ByteBufferInputStream stream = this.newStream();
            long length = stream.available();
            int lastBytesRead = bytes.length;
            int offset = 0;
            while ((long)offset < length) {
                Assert.assertEquals((String)"Should read requested len", (long)bytes.length, (long)lastBytesRead);
                lastBytesRead = stream.read(bytes, 0, bytes.length);
                Assert.assertEquals((String)"Should advance position", (long)(offset + lastBytesRead), (long)stream.getPos());
                for (int i = 0; i < lastBytesRead; ++i) {
                    Assert.assertEquals((String)"Byte i should be i", (long)(offset + i), (long)bytes[i]);
                }
                offset += bytes.length;
            }
            Assert.assertEquals((String)"Should read fewer bytes at end of buffer", (long)(length % (long)bytes.length), (long)(lastBytesRead % bytes.length));
            Assert.assertEquals((String)"Should have no more remaining content", (long)0L, (long)stream.available());
            Assert.assertEquals((String)"Should return -1 at end of stream", (long)-1L, (long)stream.read(bytes));
            Assert.assertEquals((String)"Should have no more remaining content", (long)0L, (long)stream.available());
        }
        this.checkOriginalData();
    }

    @Test
    public void testPartialBufferReads() throws Exception {
        for (int size = 1; size < 35; ++size) {
            byte[] bytes = new byte[33];
            ByteBufferInputStream stream = this.newStream();
            int lastBytesRead = size;
            for (int offset = 0; offset < bytes.length; offset += size) {
                Assert.assertEquals((String)"Should read requested len", (long)size, (long)lastBytesRead);
                lastBytesRead = stream.read(bytes, offset, Math.min(size, bytes.length - offset));
                Assert.assertEquals((String)"Should advance position", (long)(lastBytesRead > 0 ? (long)(offset + lastBytesRead) : (long)offset), (long)stream.getPos());
            }
            Assert.assertEquals((String)"Should read fewer bytes at end of buffer", (long)(bytes.length % size), (long)(lastBytesRead % size));
            for (int i = 0; i < bytes.length; ++i) {
                Assert.assertEquals((String)"Byte i should be i", (long)i, (long)bytes[i]);
            }
            Assert.assertEquals((String)"Should have no more remaining content", (long)2L, (long)stream.available());
            Assert.assertEquals((String)"Should return 2 more bytes", (long)2L, (long)stream.read(bytes));
            Assert.assertEquals((String)"Should have no more remaining content", (long)0L, (long)stream.available());
            Assert.assertEquals((String)"Should return -1 at end of stream", (long)-1L, (long)stream.read(bytes));
            Assert.assertEquals((String)"Should have no more remaining content", (long)0L, (long)stream.available());
        }
        this.checkOriginalData();
    }

    @Test
    public void testReadByte() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        int length = stream.available();
        for (int i = 0; i < length; ++i) {
            Assert.assertEquals((String)"Position should increment", (long)i, (long)stream.getPos());
            Assert.assertEquals((long)i, (long)stream.read());
        }
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> stream.read()).isInstanceOf(EOFException.class)).hasMessage(null);
        this.checkOriginalData();
    }

    @Test
    public void testSlice() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        int length = stream.available();
        ByteBuffer empty = stream.slice(0);
        Assert.assertNotNull((String)"slice(0) should produce a non-null buffer", (Object)empty);
        Assert.assertEquals((String)"slice(0) should produce an empty buffer", (long)0L, (long)empty.remaining());
        Assert.assertEquals((String)"Position should be at start", (long)0L, (long)stream.getPos());
        int i = 0;
        while (stream.available() > 0) {
            int bytesToSlice = Math.min(stream.available(), 10);
            ByteBuffer buffer = stream.slice(bytesToSlice);
            for (int j = 0; j < bytesToSlice; ++j) {
                Assert.assertEquals((String)"Data should be correct", (long)(i + j), (long)buffer.get());
            }
            i += bytesToSlice;
        }
        Assert.assertEquals((String)"Position should be at end", (long)length, (long)stream.getPos());
        this.checkOriginalData();
    }

    @Test
    public void testSliceBuffers0() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        Assert.assertEquals((String)"Should return an empty list", Collections.emptyList(), (Object)stream.sliceBuffers(0L));
    }

    @Test
    public void testWholeSliceBuffers() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        int length = stream.available();
        List buffers = stream.sliceBuffers((long)stream.available());
        Assert.assertEquals((String)"Should consume all buffers", (long)length, (long)stream.getPos());
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> stream.sliceBuffers((long)length)).isInstanceOf(EOFException.class)).hasMessage(null);
        ByteBufferInputStream copy = ByteBufferInputStream.wrap((List)buffers);
        for (int i = 0; i < length; ++i) {
            Assert.assertEquals((String)"Slice should have identical data", (long)i, (long)copy.read());
        }
        this.checkOriginalData();
    }

    @Test
    public void testSliceBuffersCoverage() throws Exception {
        for (int size = 1; size < 36; ++size) {
            ByteBufferInputStream stream = this.newStream();
            int length = stream.available();
            ArrayList buffers = Lists.newArrayList();
            while (stream.available() > 0) {
                buffers.addAll(stream.sliceBuffers((long)Math.min(size, stream.available())));
            }
            Assert.assertEquals((String)"Should consume all content", (long)length, (long)stream.getPos());
            MultiBufferInputStream newStream = new MultiBufferInputStream((List)buffers);
            for (int i = 0; i < length; ++i) {
                Assert.assertEquals((String)"Data should be correct", (long)i, (long)newStream.read());
            }
        }
        this.checkOriginalData();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSliceBuffersModification() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        int length = stream.available();
        int sliceLength = 5;
        List buffers = stream.sliceBuffers((long)sliceLength);
        Assert.assertEquals((String)"Should advance the original stream", (long)(length - sliceLength), (long)stream.available());
        Assert.assertEquals((String)"Should advance the original stream position", (long)sliceLength, (long)stream.getPos());
        Assert.assertEquals((String)"Should return a slice of the first buffer", (long)1L, (long)buffers.size());
        ByteBuffer buffer = (ByteBuffer)buffers.get(0);
        Assert.assertEquals((String)"Should have requested bytes", (long)sliceLength, (long)buffer.remaining());
        buffer.limit(sliceLength + 1);
        for (int i = 0; i < sliceLength + 1; ++i) {
            Assert.assertEquals((String)"Should have correct data", (long)i, (long)buffer.get());
        }
        Assert.assertEquals((String)"Reading a slice shouldn't advance the original stream", (long)sliceLength, (long)stream.getPos());
        Assert.assertEquals((String)"Reading a slice shouldn't change the underlying data", (long)sliceLength, (long)stream.read());
        buffer.limit(sliceLength + 2);
        byte originalValue = buffer.duplicate().get();
        ByteBuffer undoBuffer = buffer.duplicate();
        try {
            buffer.put((byte)-1);
            Assert.assertEquals((String)"Writing to a slice shouldn't advance the original stream", (long)(sliceLength + 1), (long)stream.getPos());
            Assert.assertEquals((String)"Writing to a slice should change the underlying data", (long)255L, (long)stream.read());
        }
        finally {
            undoBuffer.put(originalValue);
        }
    }

    @Test
    public void testSkip() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        while (stream.available() > 0) {
            int bytesToSkip = Math.min(stream.available(), 10);
            Assert.assertEquals((String)"Should skip all, regardless of backing buffers", (long)bytesToSkip, (long)stream.skip((long)bytesToSkip));
        }
        stream = this.newStream();
        Assert.assertEquals((long)0L, (long)stream.skip(0L));
        int length = stream.available();
        Assert.assertEquals((String)"Should stop at end when out of bytes", (long)length, (long)stream.skip((long)(length + 10)));
        Assert.assertEquals((String)"Should return -1 when at end", (long)-1L, (long)stream.skip(10L));
    }

    @Test
    public void testSkipFully() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        long lastPosition = 0L;
        while (stream.available() > 0) {
            int bytesToSkip = Math.min(stream.available(), 10);
            stream.skipFully((long)bytesToSkip);
            Assert.assertEquals((String)"Should skip all, regardless of backing buffers", (long)bytesToSkip, (long)(stream.getPos() - lastPosition));
            lastPosition = stream.getPos();
        }
        ByteBufferInputStream stream2 = this.newStream();
        stream2.skipFully(0L);
        Assert.assertEquals((long)0L, (long)stream2.getPos());
        int length = stream2.available();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> stream2.skipFully((long)(length + 10))).isInstanceOf(EOFException.class)).hasMessageStartingWith("Not enough bytes to skip");
    }

    @Test
    public void testMark() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        stream.read(new byte[7]);
        stream.mark(100);
        long mark = stream.getPos();
        byte[] expected = new byte[100];
        int expectedBytesRead = stream.read(expected);
        long end = stream.getPos();
        stream.reset();
        Assert.assertEquals((String)"Position should return to the mark", (long)mark, (long)stream.getPos());
        byte[] afterReset = new byte[100];
        int bytesReadAfterReset = stream.read(afterReset);
        Assert.assertEquals((String)"Should read the same number of bytes", (long)expectedBytesRead, (long)bytesReadAfterReset);
        Assert.assertEquals((String)"Read should end at the same position", (long)end, (long)stream.getPos());
        Assert.assertArrayEquals((String)"Content should be equal", (byte[])expected, (byte[])afterReset);
    }

    @Test
    public void testMarkTwice() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        stream.read(new byte[7]);
        stream.mark(1);
        stream.mark(100);
        long mark = stream.getPos();
        byte[] expected = new byte[100];
        int expectedBytesRead = stream.read(expected);
        long end = stream.getPos();
        stream.reset();
        Assert.assertEquals((String)"Position should return to the mark", (long)mark, (long)stream.getPos());
        byte[] afterReset = new byte[100];
        int bytesReadAfterReset = stream.read(afterReset);
        Assert.assertEquals((String)"Should read the same number of bytes", (long)expectedBytesRead, (long)bytesReadAfterReset);
        Assert.assertEquals((String)"Read should end at the same position", (long)end, (long)stream.getPos());
        Assert.assertArrayEquals((String)"Content should be equal", (byte[])expected, (byte[])afterReset);
    }

    @Test
    public void testMarkAtStart() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        stream.mark(100);
        long mark = stream.getPos();
        byte[] expected = new byte[10];
        Assert.assertEquals((String)"Should read 10 bytes", (long)10L, (long)stream.read(expected));
        long end = stream.getPos();
        stream.reset();
        Assert.assertEquals((String)"Position should return to the mark", (long)mark, (long)stream.getPos());
        byte[] afterReset = new byte[10];
        Assert.assertEquals((String)"Should read 10 bytes", (long)10L, (long)stream.read(afterReset));
        Assert.assertEquals((String)"Read should end at the same position", (long)end, (long)stream.getPos());
        Assert.assertArrayEquals((String)"Content should be equal", (byte[])expected, (byte[])afterReset);
    }

    @Test
    public void testMarkAtEnd() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        int bytesRead = stream.read(new byte[100]);
        Assert.assertTrue((String)"Should read to end of stream", (bytesRead < 100 ? 1 : 0) != 0);
        stream.mark(100);
        long mark = stream.getPos();
        byte[] expected = new byte[10];
        Assert.assertEquals((String)"Should read 0 bytes", (long)-1L, (long)stream.read(expected));
        long end = stream.getPos();
        stream.reset();
        Assert.assertEquals((String)"Position should return to the mark", (long)mark, (long)stream.getPos());
        byte[] afterReset = new byte[10];
        Assert.assertEquals((String)"Should read 0 bytes", (long)-1L, (long)stream.read(afterReset));
        Assert.assertEquals((String)"Read should end at the same position", (long)end, (long)stream.getPos());
        Assert.assertArrayEquals((String)"Content should be equal", (byte[])expected, (byte[])afterReset);
    }

    @Test
    public void testMarkUnset() {
        ByteBufferInputStream stream = this.newStream();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> stream.reset()).isInstanceOf(IOException.class)).hasMessageStartingWith("No mark defined");
    }

    @Test
    public void testMarkAndResetTwiceOverSameRange() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        byte[] expected = new byte[6];
        stream.mark(10);
        Assert.assertEquals((String)"Should read expected bytes", (long)expected.length, (long)stream.read(expected));
        stream.reset();
        stream.mark(10);
        byte[] firstRead = new byte[6];
        Assert.assertEquals((String)"Should read firstRead bytes", (long)firstRead.length, (long)stream.read(firstRead));
        stream.reset();
        byte[] secondRead = new byte[6];
        Assert.assertEquals((String)"Should read secondRead bytes", (long)secondRead.length, (long)stream.read(secondRead));
        Assert.assertArrayEquals((String)"First read should be correct", (byte[])expected, (byte[])firstRead);
        Assert.assertArrayEquals((String)"Second read should be correct", (byte[])expected, (byte[])secondRead);
    }

    @Test
    public void testMarkLimit() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        stream.mark(5);
        Assert.assertEquals((String)"Should read 5 bytes", (long)5L, (long)stream.read(new byte[5]));
        stream.reset();
        Assert.assertEquals((String)"Should read 6 bytes", (long)6L, (long)stream.read(new byte[6]));
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> stream.reset()).isInstanceOf(IOException.class)).hasMessageStartingWith("No mark defined");
    }

    @Test
    public void testMarkDoubleReset() throws Exception {
        ByteBufferInputStream stream = this.newStream();
        stream.mark(5);
        Assert.assertEquals((String)"Should read 5 bytes", (long)5L, (long)stream.read(new byte[5]));
        stream.reset();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> stream.reset()).isInstanceOf(IOException.class)).hasMessageStartingWith("No mark defined");
    }
}

