/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.hadoop;

import com.github.benmanes.caffeine.cache.Cache;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.iceberg.AssertHelpers;
import org.apache.iceberg.CachingCatalog;
import org.apache.iceberg.MetadataTableType;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.Table;
import org.apache.iceberg.TestableCachingCatalog;
import org.apache.iceberg.catalog.Catalog;
import org.apache.iceberg.catalog.Namespace;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.hadoop.HadoopCatalog;
import org.apache.iceberg.hadoop.HadoopTableTestBase;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableMap;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.util.FakeTicker;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ListAssert;
import org.assertj.core.api.OptionalAssert;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class TestCachingCatalog
extends HadoopTableTestBase {
    private static final Duration EXPIRATION_TTL = Duration.ofMinutes(5L);
    private static final Duration HALF_OF_EXPIRATION = EXPIRATION_TTL.dividedBy(2L);
    private FakeTicker ticker;

    @Before
    public void beforeEach() {
        this.ticker = new FakeTicker();
    }

    @After
    public void afterEach() {
        this.ticker = null;
    }

    @Test
    public void testInvalidateMetadataTablesIfBaseTableIsModified() throws Exception {
        Catalog catalog = CachingCatalog.wrap((Catalog)this.hadoopCatalog());
        TableIdentifier tableIdent = TableIdentifier.of((String[])new String[]{"db", "ns1", "ns2", "tbl"});
        Table table = catalog.createTable(tableIdent, SCHEMA, SPEC, (Map)ImmutableMap.of((Object)"key2", (Object)"value2"));
        table.newAppend().appendFile(FILE_A).commit();
        Snapshot oldSnapshot = table.currentSnapshot();
        TableIdentifier filesMetaTableIdent = TableIdentifier.of((String[])new String[]{"db", "ns1", "ns2", "tbl", "files"});
        Table filesMetaTable = catalog.loadTable(filesMetaTableIdent);
        TableIdentifier manifestsMetaTableIdent = TableIdentifier.of((String[])new String[]{"db", "ns1", "ns2", "tbl", "manifests"});
        Table manifestsMetaTable = catalog.loadTable(manifestsMetaTableIdent);
        table.newAppend().appendFile(FILE_B).commit();
        Table filesMetaTable2 = catalog.loadTable(filesMetaTableIdent);
        Table manifestsMetaTable2 = catalog.loadTable(manifestsMetaTableIdent);
        Assert.assertEquals((Object)filesMetaTable2, (Object)filesMetaTable);
        Assert.assertEquals((Object)manifestsMetaTable2, (Object)manifestsMetaTable);
        Assert.assertNotEquals((Object)table.currentSnapshot(), (Object)oldSnapshot);
        Assert.assertEquals((Object)filesMetaTable2.currentSnapshot(), (Object)table.currentSnapshot());
        Assert.assertEquals((Object)manifestsMetaTable2.currentSnapshot(), (Object)table.currentSnapshot());
    }

    @Test
    public void testInvalidateMetadataTablesIfBaseTableIsDropped() throws IOException {
        Catalog catalog = CachingCatalog.wrap((Catalog)this.hadoopCatalog());
        TableIdentifier tableIdent = TableIdentifier.of((String[])new String[]{"db", "ns1", "ns2", "tbl"});
        Table table = catalog.createTable(tableIdent, SCHEMA, SPEC, (Map)ImmutableMap.of((Object)"key2", (Object)"value2"));
        table.newAppend().appendFile(FILE_A).commit();
        Snapshot oldSnapshot = table.currentSnapshot();
        for (MetadataTableType type : MetadataTableType.values()) {
            catalog.loadTable(TableIdentifier.parse((String)(tableIdent + "." + type.name())));
            catalog.loadTable(TableIdentifier.parse((String)(tableIdent + "." + type.name().toLowerCase(Locale.ROOT))));
        }
        catalog.dropTable(tableIdent);
        table = catalog.createTable(tableIdent, SCHEMA, SPEC, (Map)ImmutableMap.of((Object)"key2", (Object)"value2"));
        table.newAppend().appendFile(FILE_B).commit();
        Snapshot newSnapshot = table.currentSnapshot();
        Assert.assertNotEquals((String)"Snapshots must be different", (Object)oldSnapshot, (Object)newSnapshot);
        for (MetadataTableType type : MetadataTableType.values()) {
            TableIdentifier metadataIdent1 = TableIdentifier.parse((String)(tableIdent + "." + type.name()));
            Table metadataTable1 = catalog.loadTable(metadataIdent1);
            Assert.assertEquals((String)"Snapshot must be new", (Object)newSnapshot, (Object)metadataTable1.currentSnapshot());
            TableIdentifier metadataIdent2 = TableIdentifier.parse((String)(tableIdent + "." + type.name().toLowerCase(Locale.ROOT)));
            Table metadataTable2 = catalog.loadTable(metadataIdent2);
            Assert.assertEquals((String)"Snapshot must be new", (Object)newSnapshot, (Object)metadataTable2.currentSnapshot());
        }
    }

    @Test
    public void testTableName() throws Exception {
        Catalog catalog = CachingCatalog.wrap((Catalog)this.hadoopCatalog());
        TableIdentifier tableIdent = TableIdentifier.of((String[])new String[]{"db", "ns1", "ns2", "tbl"});
        catalog.createTable(tableIdent, SCHEMA, SPEC, (Map)ImmutableMap.of((Object)"key2", (Object)"value2"));
        Table table = catalog.loadTable(tableIdent);
        Assert.assertEquals((String)"Name must match", (Object)"hadoop.db.ns1.ns2.tbl", (Object)table.name());
        TableIdentifier snapshotsTableIdent = TableIdentifier.of((String[])new String[]{"db", "ns1", "ns2", "tbl", "snapshots"});
        Table snapshotsTable = catalog.loadTable(snapshotsTableIdent);
        Assert.assertEquals((String)"Name must match", (Object)"hadoop.db.ns1.ns2.tbl.snapshots", (Object)snapshotsTable.name());
    }

    @Test
    public void testTableExpiresAfterInterval() throws IOException {
        TestableCachingCatalog catalog = TestableCachingCatalog.wrap((Catalog)this.hadoopCatalog(), EXPIRATION_TTL, this.ticker);
        Namespace namespace = Namespace.of((String[])new String[]{"db", "ns1", "ns2"});
        TableIdentifier tableIdent = TableIdentifier.of((Namespace)namespace, (String)"tbl");
        catalog.createTable(tableIdent, SCHEMA, SPEC, (Map)ImmutableMap.of((Object)"key", (Object)"value"));
        Assertions.assertThat((Map)catalog.cache().asMap()).containsKey((Object)tableIdent);
        ((OptionalAssert)Assertions.assertThat(catalog.remainingAgeFor(tableIdent)).isPresent()).get().isEqualTo((Object)EXPIRATION_TTL);
        this.ticker.advance(HALF_OF_EXPIRATION);
        Assertions.assertThat((Map)catalog.cache().asMap()).containsKey((Object)tableIdent);
        ((OptionalAssert)Assertions.assertThat(catalog.ageOf(tableIdent)).isPresent()).get().isEqualTo((Object)HALF_OF_EXPIRATION);
        this.ticker.advance(HALF_OF_EXPIRATION.plus(Duration.ofSeconds(10L)));
        Assertions.assertThat((Map)catalog.cache().asMap()).doesNotContainKey((Object)tableIdent);
        Assert.assertNotSame((String)"CachingCatalog should return a new instance after expiration", (Object)this.table, (Object)catalog.loadTable(tableIdent));
    }

    @Test
    public void testCatalogExpirationTtlRefreshesAfterAccessViaCatalog() throws IOException {
        TestableCachingCatalog catalog = TestableCachingCatalog.wrap((Catalog)this.hadoopCatalog(), EXPIRATION_TTL, this.ticker);
        Namespace namespace = Namespace.of((String[])new String[]{"db", "ns1", "ns2"});
        TableIdentifier tableIdent = TableIdentifier.of((Namespace)namespace, (String)"tbl");
        catalog.createTable(tableIdent, SCHEMA, SPEC, (Map)ImmutableMap.of((Object)"key", (Object)"value"));
        Assertions.assertThat((Map)catalog.cache().asMap()).containsKey((Object)tableIdent);
        ((OptionalAssert)Assertions.assertThat(catalog.ageOf(tableIdent)).isPresent()).get().isEqualTo((Object)Duration.ZERO);
        this.ticker.advance(HALF_OF_EXPIRATION);
        Assertions.assertThat((Map)catalog.cache().asMap()).containsKey((Object)tableIdent);
        ((OptionalAssert)Assertions.assertThat(catalog.ageOf(tableIdent)).isPresent()).get().isEqualTo((Object)HALF_OF_EXPIRATION);
        ((OptionalAssert)Assertions.assertThat(catalog.remainingAgeFor(tableIdent)).isPresent()).get().isEqualTo((Object)HALF_OF_EXPIRATION);
        Duration oneMinute = Duration.ofMinutes(1L);
        this.ticker.advance(oneMinute);
        Assertions.assertThat((Map)catalog.cache().asMap()).containsKey((Object)tableIdent);
        ((OptionalAssert)Assertions.assertThat(catalog.ageOf(tableIdent)).isPresent()).get().isEqualTo((Object)HALF_OF_EXPIRATION.plus(oneMinute));
        Assertions.assertThat(catalog.remainingAgeFor(tableIdent)).get().isEqualTo((Object)HALF_OF_EXPIRATION.minus(oneMinute));
        Table table = catalog.loadTable(tableIdent);
        Assertions.assertThat(catalog.ageOf(tableIdent)).get().isEqualTo((Object)Duration.ZERO);
        Assertions.assertThat(catalog.remainingAgeFor(tableIdent)).get().isEqualTo((Object)EXPIRATION_TTL);
        this.ticker.advance(HALF_OF_EXPIRATION);
        Assertions.assertThat(catalog.ageOf(tableIdent)).get().isEqualTo((Object)HALF_OF_EXPIRATION);
        Assertions.assertThat(catalog.remainingAgeFor(tableIdent)).get().isEqualTo((Object)HALF_OF_EXPIRATION);
        table.refresh();
        Assertions.assertThat(catalog.ageOf(tableIdent)).get().isEqualTo((Object)HALF_OF_EXPIRATION);
        Assertions.assertThat(catalog.remainingAgeFor(tableIdent)).get().isEqualTo((Object)HALF_OF_EXPIRATION);
        table.newAppend().appendFile(FILE_A).commit();
        Assertions.assertThat(catalog.ageOf(tableIdent)).get().isEqualTo((Object)HALF_OF_EXPIRATION);
        Assertions.assertThat(catalog.remainingAgeFor(tableIdent)).get().isEqualTo((Object)HALF_OF_EXPIRATION);
    }

    @Test
    public void testCacheExpirationEagerlyRemovesMetadataTables() throws IOException {
        TestableCachingCatalog catalog = TestableCachingCatalog.wrap((Catalog)this.hadoopCatalog(), EXPIRATION_TTL, this.ticker);
        Namespace namespace = Namespace.of((String[])new String[]{"db", "ns1", "ns2"});
        TableIdentifier tableIdent = TableIdentifier.of((Namespace)namespace, (String)"tbl");
        Table table = catalog.createTable(tableIdent, SCHEMA, SPEC, (Map)ImmutableMap.of((Object)"key2", (Object)"value2"));
        Assertions.assertThat((Map)catalog.cache().asMap()).containsKey((Object)tableIdent);
        table.newAppend().appendFile(FILE_A).commit();
        Assertions.assertThat((Map)catalog.cache().asMap()).containsKey((Object)tableIdent);
        Assertions.assertThat(catalog.ageOf(tableIdent)).get().isEqualTo((Object)Duration.ZERO);
        this.ticker.advance(HALF_OF_EXPIRATION);
        Assertions.assertThat((Map)catalog.cache().asMap()).containsKey((Object)tableIdent);
        Assertions.assertThat(catalog.ageOf(tableIdent)).get().isEqualTo((Object)HALF_OF_EXPIRATION);
        Arrays.stream(TestCachingCatalog.metadataTables(tableIdent)).forEach(arg_0 -> ((TestableCachingCatalog)catalog).loadTable(arg_0));
        Assertions.assertThat((Map)catalog.cache().asMap()).containsKeys((Object[])TestCachingCatalog.metadataTables(tableIdent));
        ((ListAssert)Assertions.assertThat(Arrays.stream(TestCachingCatalog.metadataTables(tableIdent)).map(catalog::ageOf)).isNotEmpty()).allMatch(age -> age.isPresent() && ((Duration)age.get()).equals(Duration.ZERO));
        Assert.assertEquals((String)"Loading a non-cached metadata table should refresh the main table's age", Optional.of(EXPIRATION_TTL), catalog.remainingAgeFor(tableIdent));
        this.ticker.advance(HALF_OF_EXPIRATION);
        Arrays.stream(TestCachingCatalog.metadataTables(tableIdent)).forEach(arg_0 -> ((TestableCachingCatalog)catalog).loadTable(arg_0));
        ((ListAssert)Assertions.assertThat(Arrays.stream(TestCachingCatalog.metadataTables(tableIdent)).map(catalog::ageOf)).isNotEmpty()).allMatch(age -> age.isPresent() && ((Duration)age.get()).equals(Duration.ZERO));
        Assert.assertEquals((String)"Accessing a cached metadata table should not affect the main table's age", Optional.of(HALF_OF_EXPIRATION), catalog.remainingAgeFor(tableIdent));
        this.ticker.advance(HALF_OF_EXPIRATION);
        Assertions.assertThat((Map)catalog.cache().asMap()).doesNotContainKey((Object)tableIdent);
        Arrays.stream(TestCachingCatalog.metadataTables(tableIdent)).forEach(metadataTable -> Assert.assertFalse((String)"When a data table expires, its metadata tables should expire regardless of age", (boolean)catalog.cache().asMap().containsKey(metadataTable)));
    }

    @Test
    public void testDeadlock() throws IOException, InterruptedException {
        HadoopCatalog underlyingCatalog = this.hadoopCatalog();
        TestableCachingCatalog catalog = TestableCachingCatalog.wrap((Catalog)underlyingCatalog, Duration.ofSeconds(1L), this.ticker);
        Namespace namespace = Namespace.of((String[])new String[]{"db", "ns1", "ns2"});
        int numThreads = 20;
        ArrayList createdTables = Lists.newArrayList();
        for (int i = 0; i < numThreads; ++i) {
            TableIdentifier tableIdent = TableIdentifier.of((Namespace)namespace, (String)("tbl" + i));
            catalog.createTable(tableIdent, SCHEMA, SPEC, (Map)ImmutableMap.of((Object)"key", (Object)"value"));
            createdTables.add(tableIdent);
        }
        Cache<TableIdentifier, Table> cache = catalog.cache();
        AtomicInteger cacheGetCount = new AtomicInteger(0);
        AtomicInteger cacheCleanupCount = new AtomicInteger(0);
        ExecutorService executor = Executors.newFixedThreadPool(numThreads);
        for (int i = 0; i < numThreads; ++i) {
            if (i % 2 == 0) {
                String table2 = "tbl" + i;
                executor.submit(() -> {
                    this.ticker.advance(Duration.ofSeconds(2L));
                    cache.get((Object)TableIdentifier.of((Namespace)namespace, (String)table2), arg_0 -> ((HadoopCatalog)underlyingCatalog).loadTable(arg_0));
                    cacheGetCount.incrementAndGet();
                });
                continue;
            }
            executor.submit(() -> {
                this.ticker.advance(Duration.ofSeconds(2L));
                cache.cleanUp();
                cacheCleanupCount.incrementAndGet();
            });
        }
        executor.awaitTermination(2L, TimeUnit.SECONDS);
        Assertions.assertThat((AtomicInteger)cacheGetCount).hasValue(numThreads / 2);
        Assertions.assertThat((AtomicInteger)cacheCleanupCount).hasValue(numThreads / 2);
        executor.shutdown();
        createdTables.forEach(table -> catalog.dropTable((TableIdentifier)table, true));
    }

    @Test
    public void testCachingCatalogRejectsExpirationIntervalOfZero() {
        AssertHelpers.assertThrows((String)"Caching catalog should disallow an expiration interval of zero, as zero signifies not to cache at all", IllegalArgumentException.class, () -> TestableCachingCatalog.wrap((Catalog)this.hadoopCatalog(), Duration.ZERO, this.ticker));
    }

    @Test
    public void testCacheExpirationIsDisabledByANegativeValue() throws IOException {
        TestableCachingCatalog catalog = TestableCachingCatalog.wrap((Catalog)this.hadoopCatalog(), Duration.ofMillis(-1L), this.ticker);
        Assert.assertFalse((String)"When a negative value is used as the expiration interval, the cache should not expire entries based on a TTL", (boolean)catalog.isCacheExpirationEnabled());
    }

    @Test
    public void testInvalidateTableForChainedCachingCatalogs() throws Exception {
        TestableCachingCatalog wrappedCatalog = TestableCachingCatalog.wrap((Catalog)this.hadoopCatalog(), EXPIRATION_TTL, this.ticker);
        TestableCachingCatalog catalog = TestableCachingCatalog.wrap((Catalog)wrappedCatalog, EXPIRATION_TTL, this.ticker);
        Namespace namespace = Namespace.of((String[])new String[]{"db", "ns1", "ns2"});
        TableIdentifier tableIdent = TableIdentifier.of((Namespace)namespace, (String)"tbl");
        catalog.createTable(tableIdent, SCHEMA, SPEC, (Map)ImmutableMap.of((Object)"key2", (Object)"value2"));
        Assertions.assertThat((Map)catalog.cache().asMap()).containsKey((Object)tableIdent);
        catalog.invalidateTable(tableIdent);
        Assertions.assertThat((Map)catalog.cache().asMap()).doesNotContainKey((Object)tableIdent);
        Assertions.assertThat((Map)wrappedCatalog.cache().asMap()).doesNotContainKey((Object)tableIdent);
    }

    public static TableIdentifier[] metadataTables(TableIdentifier tableIdent) {
        return (TableIdentifier[])Arrays.stream(MetadataTableType.values()).map(type -> TableIdentifier.parse((String)(tableIdent + "." + type.name().toLowerCase(Locale.ROOT)))).toArray(TableIdentifier[]::new);
    }
}

