/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.IntFunction;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.fs.ByteBufferPositionedReadable;
import org.apache.hadoop.fs.FileRange;
import org.apache.hadoop.fs.PositionedReadable;
import org.apache.hadoop.fs.impl.CombinedFileRange;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.functional.Function4RaisingIOE;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.LimitedPrivate(value={"Filesystems"})
@InterfaceStability.Unstable
public final class VectoredReadUtils {
    private static final int TMP_BUFFER_MAX_SIZE = 65536;
    private static final Logger LOG = LoggerFactory.getLogger(VectoredReadUtils.class);
    public static final Consumer<ByteBuffer> LOG_BYTE_BUFFER_RELEASED = buffer -> LOG.debug("Release buffer of length {}: {}", (Object)buffer.limit(), buffer);

    public static <T extends FileRange> T validateRangeRequest(T range) throws EOFException {
        Objects.requireNonNull(range, "range is null");
        Preconditions.checkArgument(range.getLength() >= 0, "length is negative in %s", range);
        if (range.getOffset() < 0L) {
            throw new EOFException("position is negative in range " + range);
        }
        return range;
    }

    public static void validateVectoredReadRanges(List<? extends FileRange> ranges) throws EOFException {
        VectoredReadUtils.validateAndSortRanges(ranges, Optional.empty());
    }

    public static void readVectored(PositionedReadable stream, List<? extends FileRange> ranges, IntFunction<ByteBuffer> allocate) throws EOFException {
        VectoredReadUtils.readVectored(stream, ranges, allocate, LOG_BYTE_BUFFER_RELEASED);
    }

    public static void readVectored(PositionedReadable stream, List<? extends FileRange> ranges, IntFunction<ByteBuffer> allocate, Consumer<ByteBuffer> release) throws EOFException {
        for (FileRange fileRange : VectoredReadUtils.validateAndSortRanges(ranges, Optional.empty())) {
            fileRange.setData(VectoredReadUtils.readRangeFrom(stream, fileRange, allocate, release));
        }
    }

    public static CompletableFuture<ByteBuffer> readRangeFrom(PositionedReadable stream, FileRange range, IntFunction<ByteBuffer> allocate) throws EOFException {
        return VectoredReadUtils.readRangeFrom(stream, range, allocate, LOG_BYTE_BUFFER_RELEASED);
    }

    public static CompletableFuture<ByteBuffer> readRangeFrom(PositionedReadable stream, FileRange range, IntFunction<ByteBuffer> allocate, Consumer<ByteBuffer> release) throws EOFException {
        VectoredReadUtils.validateRangeRequest(range);
        CompletableFuture<ByteBuffer> result = new CompletableFuture<ByteBuffer>();
        ByteBuffer buffer = allocate.apply(range.getLength());
        try {
            if (stream instanceof ByteBufferPositionedReadable) {
                LOG.debug("ByteBufferPositionedReadable.readFully of {}", (Object)range);
                ((ByteBufferPositionedReadable)((Object)stream)).readFully(range.getOffset(), buffer);
                buffer.flip();
            } else {
                VectoredReadUtils.readNonByteBufferPositionedReadable(stream, range, buffer);
            }
            result.complete(buffer);
        }
        catch (IOException ioe) {
            LOG.debug("Failed to read {}", (Object)range, (Object)ioe);
            release.accept(buffer);
            result.completeExceptionally(ioe);
        }
        return result;
    }

    private static void readNonByteBufferPositionedReadable(PositionedReadable stream, FileRange range, ByteBuffer buffer) throws IOException {
        if (buffer.isDirect()) {
            LOG.debug("Reading {} into a direct byte buffer from {}", (Object)range, (Object)stream);
            VectoredReadUtils.readInDirectBuffer(range, buffer, (Function4RaisingIOE<Long, byte[], Integer, Integer, Void>)((Function4RaisingIOE)(position, buffer1, offset, length) -> {
                stream.readFully((long)position, (byte[])buffer1, (int)offset, (int)length);
                return null;
            }));
            buffer.flip();
        } else {
            LOG.debug("Reading {} into a byte buffer from {}", (Object)range, (Object)stream);
            stream.readFully(range.getOffset(), buffer.array(), buffer.arrayOffset(), range.getLength());
        }
    }

    public static void readInDirectBuffer(FileRange range, ByteBuffer buffer, Function4RaisingIOE<Long, byte[], Integer, Integer, Void> operation) throws IOException {
        int currentLength;
        LOG.debug("Reading {} into a direct buffer", (Object)range);
        VectoredReadUtils.validateRangeRequest(range);
        int length = range.getLength();
        if (length == 0) {
            return;
        }
        long position = range.getOffset();
        int tmpBufferMaxSize = Math.min(65536, length);
        byte[] tmp = new byte[tmpBufferMaxSize];
        for (int readBytes = 0; readBytes < length; readBytes += currentLength) {
            currentLength = readBytes + tmpBufferMaxSize < length ? tmpBufferMaxSize : length - readBytes;
            LOG.debug("Reading {} bytes from position {} (bytes read={}", currentLength, position, readBytes);
            operation.apply((Object)position, (Object)tmp, (Object)0, (Object)currentLength);
            buffer.put(tmp, 0, currentLength);
            position += (long)currentLength;
        }
    }

    public static boolean isOrderedDisjoint(List<? extends FileRange> input, int chunkSize, int minimumSeek) {
        long previous = -minimumSeek;
        for (FileRange fileRange : input) {
            long offset = fileRange.getOffset();
            long end = fileRange.getOffset() + (long)fileRange.getLength();
            if (offset % (long)chunkSize != 0L || end % (long)chunkSize != 0L || offset - previous < (long)minimumSeek) {
                return false;
            }
            previous = end;
        }
        return true;
    }

    public static long roundDown(long offset, int chunkSize) {
        if (chunkSize > 1) {
            return offset - offset % (long)chunkSize;
        }
        return offset;
    }

    public static long roundUp(long offset, int chunkSize) {
        if (chunkSize > 1) {
            long next = offset + (long)chunkSize - 1L;
            return next - next % (long)chunkSize;
        }
        return offset;
    }

    public static List<? extends FileRange> validateAndSortRanges(List<? extends FileRange> input, Optional<Long> fileLength) throws EOFException {
        List<? extends FileRange> sortedRanges;
        Objects.requireNonNull(input, "Null input list");
        if (input.isEmpty()) {
            LOG.debug("Empty input list");
            return input;
        }
        if (input.size() == 1) {
            VectoredReadUtils.validateRangeRequest(input.get(0));
            sortedRanges = input;
        } else {
            sortedRanges = VectoredReadUtils.sortRangeList(input);
            FileRange prev = null;
            for (FileRange fileRange : sortedRanges) {
                VectoredReadUtils.validateRangeRequest(fileRange);
                if (prev != null) {
                    Preconditions.checkArgument(fileRange.getOffset() >= prev.getOffset() + (long)prev.getLength(), "Overlapping ranges %s and %s", prev, fileRange);
                }
                prev = fileRange;
            }
        }
        if (fileLength.isPresent()) {
            FileRange last = sortedRanges.get(sortedRanges.size() - 1);
            Long l = fileLength.get();
            if (last.getOffset() >= l) {
                throw new EOFException("Range starts beyond the file length (" + l + "): " + last);
            }
            if (last.getOffset() + (long)last.getLength() > l) {
                throw new EOFException("Range extends beyond the file length (" + l + "): " + last);
            }
        }
        return sortedRanges;
    }

    public static List<? extends FileRange> sortRangeList(List<? extends FileRange> input) {
        ArrayList<? extends FileRange> l = new ArrayList<FileRange>(input);
        l.sort(Comparator.comparingLong(FileRange::getOffset));
        return l;
    }

    @InterfaceStability.Stable
    public static FileRange[] sortRanges(List<? extends FileRange> input) {
        return VectoredReadUtils.sortRangeList(input).toArray(new FileRange[0]);
    }

    public static List<CombinedFileRange> mergeSortedRanges(List<? extends FileRange> sortedRanges, int chunkSize, int minimumSeek, int maxSize) {
        CombinedFileRange current = null;
        ArrayList<CombinedFileRange> result = new ArrayList<CombinedFileRange>(sortedRanges.size());
        for (FileRange fileRange : sortedRanges) {
            long start = VectoredReadUtils.roundDown(fileRange.getOffset(), chunkSize);
            long end = VectoredReadUtils.roundUp(fileRange.getOffset() + (long)fileRange.getLength(), chunkSize);
            if (current != null && current.merge(start, end, fileRange, minimumSeek, maxSize)) continue;
            current = new CombinedFileRange(start, end, fileRange);
            result.add(current);
        }
        return result;
    }

    public static ByteBuffer sliceTo(ByteBuffer readData, long readOffset, FileRange request) {
        int offsetChange = (int)(request.getOffset() - readOffset);
        int requestLength = request.getLength();
        readData = readData.slice();
        readData.position(offsetChange);
        readData.limit(offsetChange + requestLength);
        readData = readData.slice();
        return readData;
    }

    public static boolean hasVectorIOCapability(String capability) {
        switch (capability.toLowerCase(Locale.ENGLISH)) {
            case "in:readvectored": {
                return true;
            }
        }
        return false;
    }

    private VectoredReadUtils() {
        throw new UnsupportedOperationException();
    }
}

