/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hive.testutils;

import com.google.common.annotations.VisibleForTesting;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.net.BindException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import org.apache.hadoop.conf.Configuration;
import org.apache.zookeeper.common.ClientX509Util;
import org.apache.zookeeper.server.ServerCnxnFactory;
import org.apache.zookeeper.server.ZooKeeperServer;
import org.apache.zookeeper.server.persistence.FileTxnLog;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MiniZooKeeperCluster {
    private static final Logger LOG = LoggerFactory.getLogger(MiniZooKeeperCluster.class);
    private static final int TICK_TIME = 2000;
    private static final int DEFAULT_CONNECTION_TIMEOUT = 30000;
    private static final String LOCALHOST_KEY_STORE_NAME = "keystore.jks";
    private static final String TRUST_STORE_NAME = "truststore.jks";
    private static final String KEY_STORE_TRUST_STORE_PASSWORD = "HiveJdbc";
    private int connectionTimeout;
    private boolean started;
    private int defaultClientPort = 0;
    private List<ServerCnxnFactory> standaloneServerFactoryList;
    private List<ZooKeeperServer> zooKeeperServers;
    private List<Integer> clientPortList;
    private int activeZKServerIndex;
    private int tickTime = 0;
    private Configuration configuration;
    private boolean sslEnabled = false;

    public MiniZooKeeperCluster() {
        this(new Configuration());
    }

    public MiniZooKeeperCluster(boolean sslEnabled) {
        this(new Configuration(), sslEnabled);
    }

    public MiniZooKeeperCluster(Configuration configuration) {
        this(configuration, false);
    }

    public MiniZooKeeperCluster(Configuration configuration, boolean sslEnabled) {
        this.sslEnabled = sslEnabled;
        this.started = false;
        this.configuration = configuration;
        this.activeZKServerIndex = -1;
        this.zooKeeperServers = new ArrayList<ZooKeeperServer>();
        this.clientPortList = new ArrayList<Integer>();
        this.standaloneServerFactoryList = new ArrayList<ServerCnxnFactory>();
        this.connectionTimeout = configuration.getInt("zookeeper.session.timeout.localHBaseCluster", 30000);
    }

    public void addClientPort(int clientPort) {
        this.clientPortList.add(clientPort);
    }

    @VisibleForTesting
    public List<Integer> getClientPortList() {
        return this.clientPortList;
    }

    private boolean hasValidClientPortInList(int index) {
        return this.clientPortList.size() > index && this.clientPortList.get(index) > 0;
    }

    public void setDefaultClientPort(int clientPort) {
        if (clientPort <= 0) {
            throw new IllegalArgumentException("Invalid default ZK client port: " + clientPort);
        }
        this.defaultClientPort = clientPort;
    }

    private int selectClientPort(int seedPort) {
        int i;
        int returnClientPort = seedPort + 1;
        if (returnClientPort == 0) {
            returnClientPort = this.defaultClientPort > 0 ? this.defaultClientPort : 49152 + new Random().nextInt(16128);
        }
        block0: do {
            for (i = 0; i < this.clientPortList.size(); ++i) {
                if (returnClientPort != this.clientPortList.get(i)) continue;
                ++returnClientPort;
                continue block0;
            }
        } while (i != this.clientPortList.size());
        return returnClientPort;
    }

    public void setTickTime(int tickTime) {
        this.tickTime = tickTime;
    }

    public int getBackupZooKeeperServerNum() {
        return this.zooKeeperServers.size() - 1;
    }

    public int getZooKeeperServerNum() {
        return this.zooKeeperServers.size();
    }

    private static void setupTestEnv(boolean sslEnabled) {
        System.setProperty("zookeeper.4lw.commands.whitelist", "*");
        System.setProperty("zookeeper.preAllocSize", "100");
        if (sslEnabled) {
            System.setProperty("zookeeper.authProvider.x509", "org.apache.zookeeper.server.auth.X509AuthenticationProvider");
        }
        FileTxnLog.setPreallocSize((long)102400L);
    }

    public int startup(File baseDir) throws IOException, InterruptedException {
        int numZooKeeperServers = this.clientPortList.size();
        if (numZooKeeperServers == 0) {
            numZooKeeperServers = 1;
        }
        return this.startup(baseDir, numZooKeeperServers);
    }

    public int startup(File baseDir, int numZooKeeperServers) throws IOException, InterruptedException {
        if (numZooKeeperServers <= 0) {
            return -1;
        }
        MiniZooKeeperCluster.setupTestEnv(this.sslEnabled);
        this.shutdown();
        int tentativePort = -1;
        for (int i = 0; i < numZooKeeperServers; ++i) {
            ServerCnxnFactory standaloneServerFactory;
            File dir = new File(baseDir, "zookeeper_" + i).getAbsoluteFile();
            this.createDir(dir);
            int tickTimeToUse = this.tickTime > 0 ? this.tickTime : 2000;
            int currentClientPort = this.hasValidClientPortInList(i) ? this.clientPortList.get(i) : (tentativePort = this.selectClientPort(tentativePort));
            ZooKeeperServer server = new ZooKeeperServer(dir, dir, tickTimeToUse);
            server.setMinSessionTimeout(this.configuration.getInt("hbase.zookeeper.property.minSessionTimeout", -1));
            server.setMaxSessionTimeout(this.configuration.getInt("hbase.zookeeper.property.maxSessionTimeout", -1));
            while (true) {
                try {
                    standaloneServerFactory = this.createServerCnxnFactory(currentClientPort);
                }
                catch (BindException e) {
                    LOG.debug("Failed binding ZK Server to client port: " + currentClientPort, (Throwable)e);
                    if (this.hasValidClientPortInList(i)) {
                        return -1;
                    }
                    currentClientPort = tentativePort = this.selectClientPort(tentativePort);
                    continue;
                }
                break;
            }
            standaloneServerFactory.startup(server);
            if (!this.sslEnabled && !MiniZooKeeperCluster.waitForServerUp(currentClientPort, this.connectionTimeout)) {
                throw new IOException("Waiting for startup of standalone server");
            }
            if (this.clientPortList.size() <= i) {
                this.clientPortList.add(currentClientPort);
            } else if (this.clientPortList.get(i) <= 0) {
                this.clientPortList.remove(i);
                this.clientPortList.add(i, currentClientPort);
            }
            this.standaloneServerFactoryList.add(standaloneServerFactory);
            this.zooKeeperServers.add(server);
        }
        this.activeZKServerIndex = 0;
        this.started = true;
        int clientPort = this.clientPortList.get(this.activeZKServerIndex);
        LOG.info("Started MiniZooKeeperCluster and ran successful 'stat' on client port=" + clientPort);
        return clientPort;
    }

    private ServerCnxnFactory createServerCnxnFactory(int currentClientPort) throws IOException {
        ServerCnxnFactory serverCnxnFactory = null;
        if (this.sslEnabled) {
            System.setProperty("zookeeper.serverCnxnFactory", "org.apache.zookeeper.server.NettyServerCnxnFactory");
            String dataFileDir = !System.getProperty("test.data.files", "").isEmpty() ? System.getProperty("test.data.files") : this.configuration.get("test.data.files").replace('\\', '/').replace("c:", "");
            ClientX509Util x509Util = new ClientX509Util();
            System.setProperty(x509Util.getSslKeystoreLocationProperty(), dataFileDir + File.separator + LOCALHOST_KEY_STORE_NAME);
            System.setProperty(x509Util.getSslKeystorePasswdProperty(), KEY_STORE_TRUST_STORE_PASSWORD);
            System.setProperty(x509Util.getSslTruststoreLocationProperty(), dataFileDir + File.separator + TRUST_STORE_NAME);
            System.setProperty(x509Util.getSslTruststorePasswdProperty(), KEY_STORE_TRUST_STORE_PASSWORD);
            serverCnxnFactory = ServerCnxnFactory.createFactory();
            serverCnxnFactory.configure(new InetSocketAddress(currentClientPort), this.configuration.getInt("hbase.zookeeper.property.maxClientCnxns", 300), -1, true);
        } else {
            serverCnxnFactory = ServerCnxnFactory.createFactory();
            serverCnxnFactory.configure(new InetSocketAddress(currentClientPort), this.configuration.getInt("hbase.zookeeper.property.maxClientCnxns", 300));
        }
        return serverCnxnFactory;
    }

    private void createDir(File dir) throws IOException {
        try {
            if (!dir.exists()) {
                dir.mkdirs();
            }
        }
        catch (SecurityException e) {
            throw new IOException("creating dir: " + dir, e);
        }
    }

    public void shutdown() throws IOException {
        for (int i = 0; i < this.standaloneServerFactoryList.size(); ++i) {
            ServerCnxnFactory standaloneServerFactory = this.standaloneServerFactoryList.get(i);
            int clientPort = this.clientPortList.get(i);
            standaloneServerFactory.shutdown();
            if (MiniZooKeeperCluster.waitForServerDown(clientPort, this.connectionTimeout)) continue;
            throw new IOException("Waiting for shutdown of standalone server");
        }
        this.standaloneServerFactoryList.clear();
        for (ZooKeeperServer zkServer : this.zooKeeperServers) {
            zkServer.getZKDatabase().close();
            zkServer.shutdown(true);
        }
        this.zooKeeperServers.clear();
        if (this.started) {
            this.started = false;
            this.activeZKServerIndex = 0;
            this.clientPortList.clear();
            LOG.info("Shutdown MiniZK cluster with all ZK servers");
        }
    }

    public int killCurrentActiveZooKeeperServer() throws IOException, InterruptedException {
        if (!this.started || this.activeZKServerIndex < 0) {
            return -1;
        }
        ServerCnxnFactory standaloneServerFactory = this.standaloneServerFactoryList.get(this.activeZKServerIndex);
        int clientPort = this.clientPortList.get(this.activeZKServerIndex);
        standaloneServerFactory.shutdown();
        if (!MiniZooKeeperCluster.waitForServerDown(clientPort, this.connectionTimeout)) {
            throw new IOException("Waiting for shutdown of standalone server");
        }
        this.zooKeeperServers.get(this.activeZKServerIndex).getZKDatabase().close();
        this.standaloneServerFactoryList.remove(this.activeZKServerIndex);
        this.clientPortList.remove(this.activeZKServerIndex);
        this.zooKeeperServers.remove(this.activeZKServerIndex);
        LOG.info("Kill the current active ZK servers in the cluster on client port: " + clientPort);
        if (this.standaloneServerFactoryList.isEmpty()) {
            return -1;
        }
        clientPort = this.clientPortList.get(this.activeZKServerIndex);
        LOG.info("Activate a backup zk server in the cluster on client port: " + clientPort);
        return clientPort;
    }

    public void killOneBackupZooKeeperServer() throws IOException, InterruptedException {
        if (!this.started || this.activeZKServerIndex < 0 || this.standaloneServerFactoryList.size() <= 1) {
            return;
        }
        int backupZKServerIndex = this.activeZKServerIndex + 1;
        ServerCnxnFactory standaloneServerFactory = this.standaloneServerFactoryList.get(backupZKServerIndex);
        int clientPort = this.clientPortList.get(backupZKServerIndex);
        standaloneServerFactory.shutdown();
        if (!MiniZooKeeperCluster.waitForServerDown(clientPort, this.connectionTimeout)) {
            throw new IOException("Waiting for shutdown of standalone server");
        }
        this.zooKeeperServers.get(backupZKServerIndex).getZKDatabase().close();
        this.standaloneServerFactoryList.remove(backupZKServerIndex);
        this.clientPortList.remove(backupZKServerIndex);
        this.zooKeeperServers.remove(backupZKServerIndex);
        LOG.info("Kill one backup ZK servers in the cluster on client port: " + clientPort);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean waitForServerDown(int port, long timeout) throws IOException {
        long start = System.currentTimeMillis();
        while (true) {
            try (Socket sock = new Socket("localhost", port);){
                OutputStream outstream = sock.getOutputStream();
                outstream.write("stat".getBytes());
                outstream.flush();
            }
            catch (IOException e) {
                return true;
            }
            if (System.currentTimeMillis() > start + timeout) break;
            try {
                Thread.sleep(250L);
            }
            catch (InterruptedException e) {
                throw (InterruptedIOException)new InterruptedIOException().initCause(e);
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean waitForServerUp(int port, long timeout) throws IOException {
        long start = System.currentTimeMillis();
        while (true) {
            try {
                Socket sock = new Socket("localhost", port);
                BufferedReader reader = null;
                try {
                    OutputStream outstream = sock.getOutputStream();
                    outstream.write("stat".getBytes());
                    outstream.flush();
                    InputStreamReader isr = new InputStreamReader(sock.getInputStream());
                    reader = new BufferedReader(isr);
                    String line = reader.readLine();
                    if (line != null && line.startsWith("Zookeeper version:")) {
                        boolean bl = true;
                        return bl;
                    }
                }
                finally {
                    sock.close();
                    if (reader != null) {
                        reader.close();
                    }
                }
            }
            catch (IOException e) {
                LOG.info("server localhost:" + port + " not up " + e);
            }
            if (System.currentTimeMillis() > start + timeout) {
                return false;
            }
            try {
                Thread.sleep(250L);
            }
            catch (InterruptedException e) {
                throw (InterruptedIOException)new InterruptedIOException().initCause(e);
            }
        }
    }

    public int getClientPort() {
        return this.activeZKServerIndex < 0 || this.activeZKServerIndex >= this.clientPortList.size() ? -1 : this.clientPortList.get(this.activeZKServerIndex);
    }
}

