/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.compress.compressors;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.zip.GZIPInputStream;
import org.apache.commons.compress.AbstractTest;
import org.apache.commons.compress.compressors.CompressorInputStream;
import org.apache.commons.compress.compressors.CompressorOutputStream;
import org.apache.commons.compress.compressors.CompressorStreamFactory;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream;
import org.apache.commons.compress.compressors.gzip.GzipParameters;
import org.apache.commons.io.IOUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;

public final class GZipTest
extends AbstractTest {
    @Test
    public void testConcatenatedStreamsReadFirstOnly() throws Exception {
        File input = GZipTest.getFile("multiple.gz");
        try (InputStream is = Files.newInputStream(input.toPath(), new OpenOption[0]);
             CompressorInputStream in = new CompressorStreamFactory().createCompressorInputStream("gz", is);){
            Assertions.assertEquals((int)97, (int)in.read());
            Assertions.assertEquals((int)-1, (int)in.read());
        }
    }

    @Test
    public void testConcatenatedStreamsReadFully() throws Exception {
        File input = GZipTest.getFile("multiple.gz");
        try (InputStream is = Files.newInputStream(input.toPath(), new OpenOption[0]);
             GzipCompressorInputStream in = new GzipCompressorInputStream(is, true);){
            Assertions.assertEquals((int)97, (int)in.read());
            Assertions.assertEquals((int)98, (int)in.read());
            Assertions.assertEquals((int)0, (int)in.available());
            Assertions.assertEquals((int)-1, (int)in.read());
        }
    }

    @Test
    public void testCorruptedInput() throws Exception {
        byte[] data = GZipTest.readAllBytes("bla.tgz");
        try (ByteArrayInputStream in = new ByteArrayInputStream(data, 0, data.length - 1);
             CompressorInputStream cin = new CompressorStreamFactory().createCompressorInputStream("gz", (InputStream)in);
             ByteArrayOutputStream out = new ByteArrayOutputStream();){
            Assertions.assertThrows(IOException.class, () -> IOUtils.copy((InputStream)cin, (OutputStream)out), (String)"Expected an exception");
        }
    }

    private void testExtraFlags(int compressionLevel, int flag, int bufferSize) throws Exception {
        byte[] content = GZipTest.readAllBytes("test3.xml");
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        GzipParameters parameters = new GzipParameters();
        parameters.setCompressionLevel(compressionLevel);
        parameters.setBufferSize(bufferSize);
        try (GzipCompressorOutputStream out = new GzipCompressorOutputStream((OutputStream)bout, parameters);){
            IOUtils.copy((InputStream)new ByteArrayInputStream(content), (OutputStream)out);
            out.flush();
        }
        Assertions.assertEquals((int)flag, (int)bout.toByteArray()[8], (String)"extra flags (XFL)");
    }

    @Test
    public void testExtraFlagsBestCompression() throws Exception {
        this.testExtraFlags(9, 2, 1024);
    }

    @Test
    public void testExtraFlagsDefaultCompression() throws Exception {
        this.testExtraFlags(-1, 0, 4096);
    }

    @Test
    public void testExtraFlagsFastestCompression() throws Exception {
        this.testExtraFlags(1, 4, 128);
    }

    @Test
    public void testGzipCreation() throws Exception {
        File input = GZipTest.getFile("test1.xml");
        File output = this.newTempFile("test1.xml.gz");
        try (OutputStream out = Files.newOutputStream(output.toPath(), new OpenOption[0]);
             CompressorOutputStream cos = new CompressorStreamFactory().createCompressorOutputStream("gz", out);){
            Files.copy(input.toPath(), (OutputStream)cos);
        }
    }

    @Test
    public void testGzipUnarchive() throws Exception {
        File input = GZipTest.getFile("bla.tgz");
        File output = this.newTempFile("bla.tar");
        try (InputStream is = Files.newInputStream(input.toPath(), new OpenOption[0]);
             CompressorInputStream in = new CompressorStreamFactory().createCompressorInputStream("gz", is);){
            Files.copy((InputStream)in, output.toPath(), new CopyOption[0]);
        }
    }

    @Test
    public void testInteroperabilityWithGzipCompressorInputStream() throws Exception {
        byte[] content;
        try (InputStream fis = GZipTest.newInputStream("test3.xml");){
            content = IOUtils.toByteArray((InputStream)fis);
        }
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        GzipParameters parameters = new GzipParameters();
        parameters.setCompressionLevel(9);
        parameters.setOperatingSystem(3);
        parameters.setFilename("test3.xml");
        Assertions.assertEquals((Object)parameters.getFilename(), (Object)parameters.getFileName());
        parameters.setFileName("test3.xml");
        Assertions.assertEquals((Object)parameters.getFilename(), (Object)parameters.getFileName());
        parameters.setComment("Test file");
        parameters.setModificationTime(System.currentTimeMillis());
        try (GzipCompressorOutputStream out = new GzipCompressorOutputStream((OutputStream)bout, parameters);){
            out.write(content);
            out.flush();
        }
        try (GzipCompressorInputStream in = new GzipCompressorInputStream((InputStream)new ByteArrayInputStream(bout.toByteArray()));){
            byte[] content2 = IOUtils.toByteArray((InputStream)in);
            Assertions.assertArrayEquals((byte[])content, (byte[])content2, (String)"uncompressed content");
        }
    }

    @Test
    public void testInteroperabilityWithGZIPInputStream() throws Exception {
        byte[] content;
        try (InputStream fis = GZipTest.newInputStream("test3.xml");){
            content = IOUtils.toByteArray((InputStream)fis);
        }
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        GzipParameters parameters = new GzipParameters();
        parameters.setCompressionLevel(9);
        parameters.setOperatingSystem(3);
        parameters.setFilename("test3.xml");
        Assertions.assertEquals((Object)parameters.getFilename(), (Object)parameters.getFileName());
        parameters.setFileName("test3.xml");
        Assertions.assertEquals((Object)parameters.getFilename(), (Object)parameters.getFileName());
        parameters.setComment("Test file");
        parameters.setModificationTime(System.currentTimeMillis());
        try (GzipCompressorOutputStream out = new GzipCompressorOutputStream((OutputStream)bout, parameters);){
            out.write(content);
            out.flush();
        }
        GZIPInputStream in = new GZIPInputStream(new ByteArrayInputStream(bout.toByteArray()));
        byte[] content2 = IOUtils.toByteArray((InputStream)in);
        Assertions.assertArrayEquals((byte[])content, (byte[])content2, (String)"uncompressed content");
    }

    @ParameterizedTest
    @ValueSource(ints={0, -1})
    public void testInvalidBufferSize(int bufferSize) {
        GzipParameters parameters = new GzipParameters();
        Assertions.assertThrows(IllegalArgumentException.class, () -> parameters.setBufferSize(bufferSize), (String)"IllegalArgumentException not thrown");
    }

    @ParameterizedTest
    @ValueSource(ints={10, -5})
    public void testInvalidCompressionLevel(int compressionLevel) {
        GzipParameters parameters = new GzipParameters();
        Assertions.assertThrows(IllegalArgumentException.class, () -> parameters.setCompressionLevel(compressionLevel), (String)"IllegalArgumentException not thrown");
    }

    @Test
    public void testMetadataRoundTrip() throws Exception {
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        GzipParameters parameters = new GzipParameters();
        parameters.setCompressionLevel(9);
        parameters.setModificationTime(123456000L);
        parameters.setOperatingSystem(13);
        parameters.setFilename("test3.xml");
        Assertions.assertEquals((Object)parameters.getFilename(), (Object)parameters.getFileName());
        parameters.setFileName("test3.xml");
        Assertions.assertEquals((Object)parameters.getFilename(), (Object)parameters.getFileName());
        parameters.setComment("Umlaute m\u00c3\u00b6glich?");
        try (GzipCompressorOutputStream out = new GzipCompressorOutputStream((OutputStream)bout, parameters);){
            Files.copy(GZipTest.getFile("test3.xml").toPath(), (OutputStream)out);
        }
        GzipCompressorInputStream input = new GzipCompressorInputStream((InputStream)new ByteArrayInputStream(bout.toByteArray()));
        input.close();
        GzipParameters readParams = input.getMetaData();
        Assertions.assertEquals((int)9, (int)readParams.getCompressionLevel());
        Assertions.assertEquals((long)123456000L, (long)readParams.getModificationTime());
        Assertions.assertEquals((int)13, (int)readParams.getOperatingSystem());
        Assertions.assertEquals((Object)"test3.xml", (Object)readParams.getFileName());
        Assertions.assertEquals((Object)"test3.xml", (Object)readParams.getFilename());
        Assertions.assertEquals((Object)"Umlaute m\u00c3\u00b6glich?", (Object)readParams.getComment());
    }

    @Test
    public void testMultiByteReadConsistentlyReturnsMinusOneAtEof() throws IOException {
        File input = GZipTest.getFile("bla.tgz");
        byte[] buf = new byte[2];
        try (InputStream is = Files.newInputStream(input.toPath(), new OpenOption[0]);
             GzipCompressorInputStream in = new GzipCompressorInputStream(is);){
            IOUtils.toByteArray((InputStream)in);
            Assertions.assertEquals((int)-1, (int)in.read(buf));
            Assertions.assertEquals((int)-1, (int)in.read(buf));
        }
    }

    @Test
    public void testOverWrite() throws Exception {
        GzipCompressorOutputStream out = new GzipCompressorOutputStream((OutputStream)new ByteArrayOutputStream());
        out.close();
        Assertions.assertThrows(IOException.class, () -> out.write(0), (String)"IOException expected");
    }

    @Test
    public void testSingleByteReadConsistentlyReturnsMinusOneAtEof() throws IOException {
        File input = GZipTest.getFile("bla.tgz");
        try (InputStream is = Files.newInputStream(input.toPath(), new OpenOption[0]);
             GzipCompressorInputStream in = new GzipCompressorInputStream(is);){
            IOUtils.toByteArray((InputStream)in);
            Assertions.assertEquals((int)-1, (int)in.read());
            Assertions.assertEquals((int)-1, (int)in.read());
        }
    }
}

