/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import org.apache.hadoop.hbase.regionserver.Chunk;
import org.apache.hadoop.hbase.regionserver.CompactingMemStore;
import org.apache.hadoop.hbase.regionserver.HeapMemoryManager;
import org.apache.hadoop.hbase.regionserver.OffheapChunk;
import org.apache.hadoop.hbase.regionserver.OnheapChunk;
import org.apache.hadoop.util.StringUtils;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class ChunkCreator {
    private static final Logger LOG = LoggerFactory.getLogger(ChunkCreator.class);
    private AtomicInteger chunkID = new AtomicInteger(1);
    public static final int SIZEOF_CHUNK_HEADER = 4;
    private Map<Integer, Chunk> chunkIdMap = new ConcurrentHashMap<Integer, Chunk>();
    private final int chunkSize;
    private final boolean offheap;
    @VisibleForTesting
    static ChunkCreator INSTANCE;
    @VisibleForTesting
    static boolean chunkPoolDisabled;
    private MemStoreChunkPool pool;

    @VisibleForTesting
    ChunkCreator(int chunkSize, boolean offheap, long globalMemStoreSize, float poolSizePercentage, float initialCountPercentage, HeapMemoryManager heapMemoryManager) {
        this.chunkSize = chunkSize;
        this.offheap = offheap;
        this.pool = this.initializePool(globalMemStoreSize, poolSizePercentage, initialCountPercentage);
        if (heapMemoryManager != null && this.pool != null) {
            heapMemoryManager.registerTuneObserver(this.pool);
        }
    }

    @SuppressWarnings(value={"LI_LAZY_INIT_STATIC"}, justification="Method is called by single thread at the starting of RS")
    @VisibleForTesting
    public static ChunkCreator initialize(int chunkSize, boolean offheap, long globalMemStoreSize, float poolSizePercentage, float initialCountPercentage, HeapMemoryManager heapMemoryManager) {
        if (INSTANCE != null) {
            return INSTANCE;
        }
        INSTANCE = new ChunkCreator(chunkSize, offheap, globalMemStoreSize, poolSizePercentage, initialCountPercentage, heapMemoryManager);
        return INSTANCE;
    }

    static ChunkCreator getInstance() {
        return INSTANCE;
    }

    Chunk getChunk() {
        return this.getChunk(CompactingMemStore.IndexType.ARRAY_MAP, this.chunkSize);
    }

    Chunk getChunk(CompactingMemStore.IndexType chunkIndexType) {
        return this.getChunk(chunkIndexType, this.chunkSize);
    }

    Chunk getChunk(CompactingMemStore.IndexType chunkIndexType, int size) {
        Chunk chunk = null;
        if (this.pool != null && size == this.chunkSize && (chunk = this.pool.getChunk()) == null && LOG.isTraceEnabled()) {
            LOG.trace("The chunk pool is full. Reached maxCount= " + this.pool.getMaxCount() + ". Creating chunk onheap.");
        }
        if (chunk == null) {
            chunk = this.createChunk(false, chunkIndexType, size);
        }
        chunk.init();
        return chunk;
    }

    Chunk getJumboChunk(CompactingMemStore.IndexType chunkIndexType, int jumboSize) {
        if (jumboSize <= this.chunkSize) {
            LOG.warn("Jumbo chunk size " + jumboSize + " must be more than regular chunk size " + this.chunkSize + ". Converting to regular chunk.");
            return this.getChunk(chunkIndexType, this.chunkSize);
        }
        return this.getChunk(chunkIndexType, jumboSize + 4);
    }

    private Chunk createChunk(boolean pool, CompactingMemStore.IndexType chunkIndexType, int size) {
        Chunk chunk = null;
        int id = this.chunkID.getAndIncrement();
        assert (id > 0);
        chunk = pool && this.offheap ? new OffheapChunk(size, id, pool) : new OnheapChunk(size, id, pool);
        if (pool || chunkIndexType == CompactingMemStore.IndexType.CHUNK_MAP) {
            this.chunkIdMap.put(chunk.getId(), chunk);
        }
        return chunk;
    }

    private Chunk createChunkForPool() {
        return this.createChunk(true, CompactingMemStore.IndexType.ARRAY_MAP, this.chunkSize);
    }

    @VisibleForTesting
    Chunk getChunk(int id) {
        return this.chunkIdMap.get(id);
    }

    int getChunkSize() {
        return this.chunkSize;
    }

    boolean isOffheap() {
        return this.offheap;
    }

    private void removeChunks(Set<Integer> chunkIDs) {
        this.chunkIdMap.keySet().removeAll(chunkIDs);
    }

    Chunk removeChunk(int chunkId) {
        return this.chunkIdMap.remove(chunkId);
    }

    @VisibleForTesting
    int numberOfMappedChunks() {
        return this.chunkIdMap.size();
    }

    @VisibleForTesting
    void clearChunkIds() {
        this.chunkIdMap.clear();
    }

    @VisibleForTesting
    static void clearDisableFlag() {
        chunkPoolDisabled = false;
    }

    private MemStoreChunkPool initializePool(long globalMemStoreSize, float poolSizePercentage, float initialCountPercentage) {
        if (poolSizePercentage <= 0.0f) {
            LOG.info("PoolSizePercentage is less than 0. So not using pool");
            return null;
        }
        if (chunkPoolDisabled) {
            return null;
        }
        if ((double)poolSizePercentage > 1.0) {
            throw new IllegalArgumentException("hbase.hregion.memstore.chunkpool.maxsize must be between 0.0 and 1.0");
        }
        int maxCount = (int)((float)globalMemStoreSize * poolSizePercentage / (float)this.getChunkSize());
        if ((double)initialCountPercentage > 1.0 || initialCountPercentage < 0.0f) {
            throw new IllegalArgumentException("hbase.hregion.memstore.chunkpool.initialsize must be between 0.0 and 1.0");
        }
        int initialCount = (int)(initialCountPercentage * (float)maxCount);
        LOG.info("Allocating MemStoreChunkPool with chunk size " + StringUtils.byteDesc((long)this.getChunkSize()) + ", max count " + maxCount + ", initial count " + initialCount);
        return new MemStoreChunkPool(maxCount, initialCount, poolSizePercentage);
    }

    @VisibleForTesting
    int getMaxCount() {
        if (this.pool != null) {
            return this.pool.getMaxCount();
        }
        return 0;
    }

    @VisibleForTesting
    int getPoolSize() {
        if (this.pool != null) {
            return this.pool.reclaimedChunks.size();
        }
        return 0;
    }

    @VisibleForTesting
    boolean isChunkInPool(int chunkId) {
        if (this.pool != null) {
            Chunk c = this.getChunk(chunkId);
            if (c == null) {
                return false;
            }
            return this.pool.reclaimedChunks.contains(c);
        }
        return false;
    }

    @VisibleForTesting
    void clearChunksInPool() {
        if (this.pool != null) {
            this.pool.reclaimedChunks.clear();
        }
    }

    synchronized void putbackChunks(Set<Integer> chunks) {
        if (this.pool == null) {
            this.removeChunks(chunks);
            return;
        }
        for (int chunkID : chunks) {
            Chunk chunk = this.getChunk(chunkID);
            if (chunk == null) continue;
            if (chunk.isJumbo()) {
                this.removeChunk(chunkID);
                continue;
            }
            this.pool.putbackChunks(chunk);
        }
    }

    static {
        chunkPoolDisabled = false;
    }

    private class MemStoreChunkPool
    implements HeapMemoryManager.HeapMemoryTuneObserver {
        private int maxCount;
        private final BlockingQueue<Chunk> reclaimedChunks;
        private final float poolSizePercentage;
        private final ScheduledExecutorService scheduleThreadPool;
        private static final int statThreadPeriod = 300;
        private final AtomicLong chunkCount = new AtomicLong();
        private final LongAdder reusedChunkCount = new LongAdder();

        MemStoreChunkPool(int maxCount, int initialCount, float poolSizePercentage) {
            this.maxCount = maxCount;
            this.poolSizePercentage = poolSizePercentage;
            this.reclaimedChunks = new LinkedBlockingQueue<Chunk>();
            for (int i = 0; i < initialCount; ++i) {
                Chunk chunk = ChunkCreator.this.createChunkForPool();
                chunk.init();
                this.reclaimedChunks.add(chunk);
            }
            this.chunkCount.set(initialCount);
            String n = Thread.currentThread().getName();
            this.scheduleThreadPool = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat(n + "-MemStoreChunkPool Statistics").setDaemon(true).build());
            this.scheduleThreadPool.scheduleAtFixedRate(new StatisticsThread(), 300L, 300L, TimeUnit.SECONDS);
        }

        Chunk getChunk() {
            Chunk chunk = (Chunk)this.reclaimedChunks.poll();
            if (chunk != null) {
                chunk.reset();
                this.reusedChunkCount.increment();
            } else {
                long created;
                while ((created = this.chunkCount.get()) < (long)this.maxCount) {
                    if (!this.chunkCount.compareAndSet(created, created + 1L)) continue;
                    chunk = ChunkCreator.this.createChunkForPool();
                    break;
                }
            }
            return chunk;
        }

        private void putbackChunks(Chunk c) {
            int toAdd = this.maxCount - this.reclaimedChunks.size();
            if (c.isFromPool() && toAdd > 0) {
                this.reclaimedChunks.add(c);
            } else {
                ChunkCreator.this.removeChunk(c.getId());
            }
        }

        private int getMaxCount() {
            return this.maxCount;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void onHeapMemoryTune(long newMemstoreSize, long newBlockCacheSize) {
            if (ChunkCreator.this.isOffheap()) {
                LOG.warn("Not tuning the chunk pool as it is offheap");
                return;
            }
            int newMaxCount = (int)((float)newMemstoreSize * this.poolSizePercentage / (float)ChunkCreator.this.getChunkSize());
            if (newMaxCount != this.maxCount) {
                if (newMaxCount > this.maxCount) {
                    LOG.info("Max count for chunks increased from " + this.maxCount + " to " + newMaxCount);
                    this.maxCount = newMaxCount;
                } else {
                    LOG.info("Max count for chunks decreased from " + this.maxCount + " to " + newMaxCount);
                    this.maxCount = newMaxCount;
                    if (this.reclaimedChunks.size() > newMaxCount) {
                        MemStoreChunkPool memStoreChunkPool = this;
                        synchronized (memStoreChunkPool) {
                            while (this.reclaimedChunks.size() > newMaxCount) {
                                this.reclaimedChunks.poll();
                            }
                        }
                    }
                }
            }
        }

        private class StatisticsThread
        extends Thread {
            StatisticsThread() {
                super("MemStoreChunkPool.StatisticsThread");
                this.setDaemon(true);
            }

            @Override
            public void run() {
                this.logStats();
            }

            private void logStats() {
                if (!LOG.isDebugEnabled()) {
                    return;
                }
                long created = MemStoreChunkPool.this.chunkCount.get();
                long reused = MemStoreChunkPool.this.reusedChunkCount.sum();
                long total = created + reused;
                LOG.debug("Stats: current pool size=" + MemStoreChunkPool.this.reclaimedChunks.size() + ",created chunk count=" + created + ",reused chunk count=" + reused + ",reuseRatio=" + (total == 0L ? "0" : StringUtils.formatPercent((double)((float)reused / (float)total), (int)2)));
            }
        }
    }
}

