/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.nodemanager;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileContext;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.thirdparty.com.google.common.collect.ImmutableList;
import org.apache.hadoop.util.DiskChecker;
import org.apache.hadoop.util.DiskValidator;
import org.apache.hadoop.util.DiskValidatorFactory;
import org.apache.hadoop.yarn.conf.YarnConfiguration;
import org.apache.hadoop.yarn.exceptions.YarnRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DirectoryCollection {
    private static final Logger LOG = LoggerFactory.getLogger(DirectoryCollection.class);
    private final Configuration conf = new YarnConfiguration();
    private final DiskValidator diskValidator;
    private boolean diskUtilizationThresholdEnabled;
    private boolean diskFreeSpaceThresholdEnabled;
    private boolean subAccessibilityValidationEnabled;
    private List<String> localDirs;
    private List<String> errorDirs;
    private List<String> fullDirs;
    private Map<String, DiskErrorInformation> directoryErrorInfo;
    private final ReentrantReadWriteLock.ReadLock readLock;
    private final ReentrantReadWriteLock.WriteLock writeLock;
    private int numFailures;
    private float diskUtilizationPercentageCutoffHigh;
    private float diskUtilizationPercentageCutoffLow;
    private long diskFreeSpaceCutoffLow;
    private long diskFreeSpaceCutoffHigh;
    private int goodDirsDiskUtilizationPercentage;
    private Set<DirsChangeListener> dirsChangeListeners;

    static List<String> concat(List<String> l1, List<String> l2) {
        ArrayList<String> ret = new ArrayList<String>(l1.size() + l2.size());
        ret.addAll(l1);
        ret.addAll(l2);
        return ret;
    }

    public DirectoryCollection(String[] dirs) {
        this(dirs, 100.0f, 100.0f, 0L, 0L);
    }

    public DirectoryCollection(String[] dirs, float utilizationPercentageCutOff) {
        this(dirs, utilizationPercentageCutOff, utilizationPercentageCutOff, 0L, 0L);
    }

    public DirectoryCollection(String[] dirs, long utilizationSpaceCutOff) {
        this(dirs, 100.0f, 100.0f, utilizationSpaceCutOff, utilizationSpaceCutOff);
    }

    public DirectoryCollection(String[] dirs, long utilizationSpaceCutOffLow, long utilizationSpaceCutOffHigh) {
        this(dirs, 100.0f, 100.0f, utilizationSpaceCutOffLow, utilizationSpaceCutOffHigh);
    }

    public DirectoryCollection(String[] dirs, float utilizationPercentageCutOffHigh, float utilizationPercentageCutOffLow, long utilizationSpaceCutOff) {
        this(dirs, utilizationPercentageCutOffHigh, utilizationPercentageCutOffLow, utilizationSpaceCutOff, utilizationSpaceCutOff);
    }

    public DirectoryCollection(String[] dirs, float utilizationPercentageCutOffHigh, float utilizationPercentageCutOffLow, long utilizationSpaceCutOffLow, long utilizationSpaceCutOffHigh) {
        try {
            String diskValidatorName = this.conf.get("yarn.nodemanager.disk-validator", "basic");
            this.diskValidator = DiskValidatorFactory.getInstance((String)diskValidatorName);
            LOG.info("Disk Validator '" + diskValidatorName + "' is loaded.");
        }
        catch (Exception e) {
            throw new YarnRuntimeException((Throwable)e);
        }
        this.diskUtilizationThresholdEnabled = this.conf.getBoolean("yarn.nodemanager.disk-health-checker.disk-utilization-threshold.enabled", true);
        this.diskFreeSpaceThresholdEnabled = this.conf.getBoolean("yarn.nodemanager.disk-health-checker.disk-free-space-threshold.enabled", true);
        this.subAccessibilityValidationEnabled = this.conf.getBoolean("yarn.nodemanager.disk-health-checker.working-dir-content-accessibility-validation.enabled", false);
        this.localDirs = new ArrayList<String>(Arrays.asList(dirs));
        this.errorDirs = new ArrayList<String>();
        this.fullDirs = new ArrayList<String>();
        this.directoryErrorInfo = new ConcurrentHashMap<String, DiskErrorInformation>();
        ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
        this.readLock = lock.readLock();
        this.writeLock = lock.writeLock();
        this.setDiskUtilizationPercentageCutoff(utilizationPercentageCutOffHigh, utilizationPercentageCutOffLow);
        this.setDiskUtilizationSpaceCutoff(utilizationSpaceCutOffLow, utilizationSpaceCutOffHigh);
        this.dirsChangeListeners = Collections.newSetFromMap(new ConcurrentHashMap());
    }

    void registerDirsChangeListener(DirsChangeListener listener) {
        if (this.dirsChangeListeners.add(listener)) {
            listener.onDirsChanged();
        }
    }

    void deregisterDirsChangeListener(DirsChangeListener listener) {
        this.dirsChangeListeners.remove(listener);
    }

    List<String> getGoodDirs() {
        this.readLock.lock();
        try {
            ImmutableList immutableList = ImmutableList.copyOf(this.localDirs);
            return immutableList;
        }
        finally {
            this.readLock.unlock();
        }
    }

    List<String> getFailedDirs() {
        this.readLock.lock();
        try {
            List<String> list = Collections.unmodifiableList(DirectoryCollection.concat(this.errorDirs, this.fullDirs));
            return list;
        }
        finally {
            this.readLock.unlock();
        }
    }

    List<String> getFullDirs() {
        this.readLock.lock();
        try {
            ImmutableList immutableList = ImmutableList.copyOf(this.fullDirs);
            return immutableList;
        }
        finally {
            this.readLock.unlock();
        }
    }

    @InterfaceStability.Evolving
    List<String> getErroredDirs() {
        this.readLock.lock();
        try {
            ImmutableList immutableList = ImmutableList.copyOf(this.errorDirs);
            return immutableList;
        }
        finally {
            this.readLock.unlock();
        }
    }

    int getNumFailures() {
        this.readLock.lock();
        try {
            int n = this.numFailures;
            return n;
        }
        finally {
            this.readLock.unlock();
        }
    }

    @InterfaceStability.Evolving
    DiskErrorInformation getDirectoryErrorInfo(String dirName) {
        this.readLock.lock();
        try {
            DiskErrorInformation diskErrorInformation = this.directoryErrorInfo.get(dirName);
            return diskErrorInformation;
        }
        finally {
            this.readLock.unlock();
        }
    }

    @InterfaceStability.Evolving
    boolean isDiskUnHealthy(String dirName) {
        this.readLock.lock();
        try {
            boolean bl = this.directoryErrorInfo.containsKey(dirName);
            return bl;
        }
        finally {
            this.readLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean createNonExistentDirs(FileContext localFs, FsPermission perm) {
        boolean failed = false;
        ArrayList<String> localDirectories = null;
        this.readLock.lock();
        try {
            localDirectories = new ArrayList<String>(this.localDirs);
        }
        finally {
            this.readLock.unlock();
        }
        for (String dir : localDirectories) {
            try {
                this.createDir(localFs, new Path(dir), perm);
            }
            catch (IOException e) {
                LOG.warn("Unable to create directory " + dir + " error " + e.getMessage() + ", removing from the list of valid directories.");
                this.writeLock.lock();
                try {
                    this.localDirs.remove(dir);
                    this.errorDirs.add(dir);
                    this.directoryErrorInfo.put(dir, new DiskErrorInformation(DiskErrorCause.OTHER, "Cannot create directory : " + dir + ", error " + e.getMessage()));
                    ++this.numFailures;
                }
                finally {
                    this.writeLock.unlock();
                }
                failed = true;
            }
        }
        return !failed;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean checkDirs() {
        boolean setChanged = false;
        HashSet<String> preCheckGoodDirs = null;
        HashSet<String> preCheckFullDirs = null;
        HashSet<String> preCheckOtherErrorDirs = null;
        List<String> failedDirs = null;
        List<String> allLocalDirs = null;
        this.readLock.lock();
        try {
            preCheckGoodDirs = new HashSet<String>(this.localDirs);
            preCheckFullDirs = new HashSet<String>(this.fullDirs);
            preCheckOtherErrorDirs = new HashSet<String>(this.errorDirs);
            failedDirs = DirectoryCollection.concat(this.errorDirs, this.fullDirs);
            allLocalDirs = DirectoryCollection.concat(this.localDirs, failedDirs);
        }
        finally {
            this.readLock.unlock();
        }
        Map<String, DiskErrorInformation> dirsFailedCheck = this.testDirs(allLocalDirs, preCheckGoodDirs);
        this.writeLock.lock();
        try {
            this.localDirs.clear();
            this.errorDirs.clear();
            this.fullDirs.clear();
            this.directoryErrorInfo.clear();
            for (Map.Entry<String, DiskErrorInformation> entry : dirsFailedCheck.entrySet()) {
                String dir = entry.getKey();
                DiskErrorInformation errorInformation = entry.getValue();
                switch (entry.getValue().cause) {
                    case DISK_FULL: {
                        this.fullDirs.add(entry.getKey());
                        break;
                    }
                    case OTHER: {
                        this.errorDirs.add(entry.getKey());
                        break;
                    }
                    default: {
                        LOG.warn((Object)((Object)entry.getValue().cause) + " is unknown for disk error.");
                    }
                }
                this.directoryErrorInfo.put(entry.getKey(), errorInformation);
                if (!preCheckGoodDirs.contains(dir)) continue;
                LOG.warn("Directory " + dir + " error, " + errorInformation.message + ", removing from list of valid directories");
                setChanged = true;
                ++this.numFailures;
            }
            for (String dir : allLocalDirs) {
                if (dirsFailedCheck.containsKey(dir)) continue;
                this.localDirs.add(dir);
                if (!preCheckFullDirs.contains(dir) && !preCheckOtherErrorDirs.contains(dir)) continue;
                setChanged = true;
                LOG.info("Directory " + dir + " passed disk check, adding to list of valid directories.");
            }
            HashSet<String> postCheckFullDirs = new HashSet<String>(this.fullDirs);
            HashSet<String> postCheckOtherDirs = new HashSet<String>(this.errorDirs);
            for (String dir : preCheckFullDirs) {
                if (!postCheckOtherDirs.contains(dir)) continue;
                LOG.warn("Directory " + dir + " error " + dirsFailedCheck.get((Object)dir).message);
            }
            for (String dir : preCheckOtherErrorDirs) {
                if (!postCheckFullDirs.contains(dir)) continue;
                LOG.warn("Directory " + dir + " error " + dirsFailedCheck.get((Object)dir).message);
            }
            this.setGoodDirsDiskUtilizationPercentage();
            if (setChanged) {
                for (DirsChangeListener listener : this.dirsChangeListeners) {
                    listener.onDirsChanged();
                }
            }
            boolean bl = setChanged;
            return bl;
        }
        finally {
            this.writeLock.unlock();
        }
    }

    Map<String, DiskErrorInformation> testDirs(List<String> dirs, Set<String> goodDirs) {
        HashMap<String, DiskErrorInformation> ret = new HashMap<String, DiskErrorInformation>(0);
        for (String dir : dirs) {
            LOG.debug("Start testing dir accessibility: {}", (Object)dir);
            File testDir = new File(dir);
            boolean goodDir = goodDirs.contains(dir);
            Stream.of(this.validateDisk(testDir), this.validateUsageOverPercentageLimit(testDir, goodDir), this.validateDiskFreeSpaceUnderLimit(testDir, goodDir), this.validateSubsAccessibility(testDir)).filter(Objects::nonNull).findFirst().ifPresent(diskErrorInformation -> ret.put(dir, (DiskErrorInformation)diskErrorInformation));
        }
        return ret;
    }

    private DiskErrorInformation validateDisk(File dir) {
        try {
            this.diskValidator.checkStatus(dir);
            LOG.debug("Dir {} pass throw the disk validation", (Object)dir);
            return null;
        }
        catch (IOException | UncheckedIOException | SecurityException e) {
            return new DiskErrorInformation(DiskErrorCause.OTHER, e.getMessage());
        }
    }

    private DiskErrorInformation validateUsageOverPercentageLimit(File dir, boolean isGoodDir) {
        if (!this.diskUtilizationThresholdEnabled) {
            return null;
        }
        float diskUtilizationPercentageCutoff = isGoodDir ? this.diskUtilizationPercentageCutoffHigh : this.diskUtilizationPercentageCutoffLow;
        float freePercentage = 100.0f * ((float)dir.getUsableSpace() / (float)dir.getTotalSpace());
        float usedPercentage = 100.0f - freePercentage;
        if (usedPercentage > diskUtilizationPercentageCutoff || usedPercentage >= 100.0f) {
            return new DiskErrorInformation(DiskErrorCause.DISK_FULL, "used space above threshold of " + diskUtilizationPercentageCutoff + "%");
        }
        LOG.debug("Dir {} pass throw the usage over percentage validation", (Object)dir);
        return null;
    }

    private DiskErrorInformation validateDiskFreeSpaceUnderLimit(File dir, boolean isGoodDir) {
        if (!this.diskFreeSpaceThresholdEnabled) {
            return null;
        }
        long freeSpaceCutoff = isGoodDir ? this.diskFreeSpaceCutoffLow : this.diskFreeSpaceCutoffHigh;
        long freeSpace = dir.getUsableSpace() / 0x100000L;
        if (freeSpace < freeSpaceCutoff) {
            return new DiskErrorInformation(DiskErrorCause.DISK_FULL, "free space below limit of " + freeSpaceCutoff + "MB");
        }
        LOG.debug("Dir {} pass throw the free space validation", (Object)dir);
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private DiskErrorInformation validateSubsAccessibility(File dir) {
        if (!this.subAccessibilityValidationEnabled) {
            return null;
        }
        try (Stream<java.nio.file.Path> walk = Files.walk(dir.toPath(), new FileVisitOption[0]);){
            List subs = walk.map(java.nio.file.Path::toFile).collect(Collectors.toList());
            Iterator iterator = subs.iterator();
            while (iterator.hasNext()) {
                File sub = (File)iterator.next();
                if (sub.isDirectory()) {
                    DiskChecker.checkDir((File)sub);
                    continue;
                }
                if (!Files.isReadable(sub.toPath())) {
                    DiskErrorInformation diskErrorInformation = new DiskErrorInformation(DiskErrorCause.OTHER, "Can not read " + sub);
                    return diskErrorInformation;
                }
                LOG.debug("{} under {} is accessible", (Object)sub, (Object)dir);
            }
            return null;
        }
        catch (IOException | UncheckedIOException | SecurityException e) {
            return new DiskErrorInformation(DiskErrorCause.OTHER, e.getMessage());
        }
    }

    private void createDir(FileContext localFs, Path dir, FsPermission perm) throws IOException {
        block5: {
            if (dir == null) {
                return;
            }
            try {
                localFs.getFileStatus(dir);
            }
            catch (FileNotFoundException e) {
                this.createDir(localFs, dir.getParent(), perm);
                try {
                    localFs.mkdir(dir, perm, false);
                }
                catch (FileAlreadyExistsException fileAlreadyExistsException) {
                    // empty catch block
                }
                if (perm.equals((Object)perm.applyUMask(localFs.getUMask()))) break block5;
                localFs.setPermission(dir, perm);
            }
        }
    }

    @VisibleForTesting
    float getDiskUtilizationPercentageCutoffHigh() {
        return this.diskUtilizationPercentageCutoffHigh;
    }

    @VisibleForTesting
    float getDiskUtilizationPercentageCutoffLow() {
        return this.diskUtilizationPercentageCutoffLow;
    }

    public void setDiskUtilizationPercentageCutoff(float utilizationPercentageCutOffHigh, float utilizationPercentageCutOffLow) {
        this.diskUtilizationPercentageCutoffHigh = Math.max(0.0f, Math.min(100.0f, utilizationPercentageCutOffHigh));
        this.diskUtilizationPercentageCutoffLow = Math.max(0.0f, Math.min(this.diskUtilizationPercentageCutoffHigh, utilizationPercentageCutOffLow));
    }

    public long getDiskUtilizationSpaceCutoff() {
        return this.getDiskUtilizationSpaceCutoffLow();
    }

    @VisibleForTesting
    long getDiskUtilizationSpaceCutoffLow() {
        return this.diskFreeSpaceCutoffLow;
    }

    @VisibleForTesting
    long getDiskUtilizationSpaceCutoffHigh() {
        return this.diskFreeSpaceCutoffHigh;
    }

    @VisibleForTesting
    boolean getDiskUtilizationThresholdEnabled() {
        return this.diskUtilizationThresholdEnabled;
    }

    @VisibleForTesting
    boolean getDiskFreeSpaceThresholdEnabled() {
        return this.diskFreeSpaceThresholdEnabled;
    }

    @VisibleForTesting
    void setDiskUtilizationThresholdEnabled(boolean utilizationEnabled) {
        this.diskUtilizationThresholdEnabled = utilizationEnabled;
    }

    @VisibleForTesting
    void setDiskFreeSpaceThresholdEnabled(boolean freeSpaceEnabled) {
        this.diskFreeSpaceThresholdEnabled = freeSpaceEnabled;
    }

    public void setDiskUtilizationSpaceCutoff(long freeSpaceCutoff) {
        this.setDiskUtilizationSpaceCutoff(freeSpaceCutoff, freeSpaceCutoff);
    }

    public void setDiskUtilizationSpaceCutoff(long freeSpaceCutoffLow, long freeSpaceCutoffHigh) {
        this.diskFreeSpaceCutoffLow = Math.max(0L, freeSpaceCutoffLow);
        this.diskFreeSpaceCutoffHigh = Math.max(this.diskFreeSpaceCutoffLow, Math.max(0L, freeSpaceCutoffHigh));
    }

    private void setGoodDirsDiskUtilizationPercentage() {
        long totalSpace = 0L;
        long usableSpace = 0L;
        for (String dir : this.localDirs) {
            File f = new File(dir);
            if (!f.isDirectory()) continue;
            totalSpace += f.getTotalSpace();
            usableSpace += f.getUsableSpace();
        }
        if (totalSpace != 0L) {
            long tmp = (totalSpace - usableSpace) * 100L / totalSpace;
            if (Integer.MIN_VALUE < tmp && Integer.MAX_VALUE > tmp) {
                this.goodDirsDiskUtilizationPercentage = Math.toIntExact(tmp);
            }
        } else {
            this.goodDirsDiskUtilizationPercentage = 0;
        }
    }

    public int getGoodDirsDiskUtilizationPercentage() {
        return this.goodDirsDiskUtilizationPercentage;
    }

    @VisibleForTesting
    public void setSubAccessibilityValidationEnabled(boolean subAccessibilityValidationEnabled) {
        this.subAccessibilityValidationEnabled = subAccessibilityValidationEnabled;
    }

    public static interface DirsChangeListener {
        public void onDirsChanged();
    }

    static class DiskErrorInformation {
        DiskErrorCause cause;
        String message;

        DiskErrorInformation(DiskErrorCause cause, String message) {
            this.cause = cause;
            this.message = message;
        }
    }

    public static enum DiskErrorCause {
        DISK_FULL,
        OTHER;

    }
}

