/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.timeline;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.TreeMap;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.time.FastDateFormat;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.yarn.server.timeline.RollingLevelDBTimelineStore;
import org.fusesource.leveldbjni.JniDBFactory;
import org.iq80.leveldb.DB;
import org.iq80.leveldb.Options;
import org.iq80.leveldb.WriteBatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RollingLevelDB {
    private static final Logger LOG = LoggerFactory.getLogger(RollingLevelDB.class);
    private static JniDBFactory factory = new JniDBFactory();
    private FastDateFormat fdf;
    private SimpleDateFormat sdf;
    private GregorianCalendar cal = new GregorianCalendar(TimeZone.getTimeZone("GMT"));
    private final TreeMap<Long, DB> rollingdbs;
    private final TreeMap<Long, DB> rollingdbsToEvict;
    private final String name;
    private volatile long nextRollingCheckMillis = 0L;
    private FileSystem lfs = null;
    private Path rollingDBPath;
    private Configuration conf;
    private RollingPeriod rollingPeriod;
    private long ttl;
    private boolean ttlEnabled;

    RollingLevelDB(String name) {
        this.name = name;
        this.rollingdbs = new TreeMap();
        this.rollingdbsToEvict = new TreeMap();
    }

    protected String getName() {
        return this.name;
    }

    protected long currentTimeMillis() {
        return System.currentTimeMillis();
    }

    public long getNextRollingTimeMillis() {
        return this.nextRollingCheckMillis;
    }

    public long getTimeToLive() {
        return this.ttl;
    }

    public boolean getTimeToLiveEnabled() {
        return this.ttlEnabled;
    }

    protected void setNextRollingTimeMillis(long timestamp) {
        this.nextRollingCheckMillis = timestamp;
        LOG.info("Next rolling time for " + this.getName() + " is " + this.fdf.format(this.nextRollingCheckMillis));
    }

    public void init(Configuration config) throws Exception {
        LOG.info("Initializing RollingLevelDB for " + this.getName());
        this.conf = config;
        this.ttl = this.conf.getLong("yarn.timeline-service.ttl-ms", 604800000L);
        this.ttlEnabled = this.conf.getBoolean("yarn.timeline-service.ttl-enable", true);
        this.rollingDBPath = new Path(this.conf.get("yarn.timeline-service.leveldb-timeline-store.path"), "leveldb-timeline-store");
        this.initFileSystem();
        this.initRollingPeriod();
        this.initHistoricalDBs();
    }

    protected void initFileSystem() throws IOException {
        this.lfs = FileSystem.getLocal((Configuration)this.conf);
        boolean success = this.lfs.mkdirs(this.rollingDBPath, RollingLevelDBTimelineStore.LEVELDB_DIR_UMASK);
        if (!success) {
            throw new IOException("Failed to create leveldb root directory " + this.rollingDBPath);
        }
    }

    protected synchronized void initRollingPeriod() {
        String lcRollingPeriod = this.conf.get("yarn.timeline-service.rolling-period", "hourly");
        this.rollingPeriod = RollingPeriod.valueOf(lcRollingPeriod.toUpperCase(Locale.ENGLISH));
        this.fdf = FastDateFormat.getInstance((String)this.rollingPeriod.dateFormat(), (TimeZone)TimeZone.getTimeZone("GMT"));
        this.sdf = new SimpleDateFormat(this.rollingPeriod.dateFormat());
        this.sdf.setTimeZone(this.fdf.getTimeZone());
    }

    protected synchronized void initHistoricalDBs() throws IOException {
        FileStatus[] statuses;
        Path rollingDBGlobPath = new Path(this.rollingDBPath, this.getName() + ".*");
        for (FileStatus status : statuses = this.lfs.globStatus(rollingDBGlobPath)) {
            String dbName = FilenameUtils.getExtension((String)status.getPath().toString());
            try {
                Long dbStartTime = this.sdf.parse(dbName).getTime();
                this.initRollingLevelDB(dbStartTime, status.getPath());
            }
            catch (ParseException pe) {
                LOG.warn("Failed to initialize rolling leveldb " + dbName + " for " + this.getName());
            }
        }
    }

    private void initRollingLevelDB(Long dbStartTime, Path rollingInstanceDBPath) {
        if (this.rollingdbs.containsKey(dbStartTime)) {
            return;
        }
        Options options = new Options();
        options.createIfMissing(true);
        options.cacheSize(this.conf.getLong("yarn.timeline-service.leveldb-timeline-store.read-cache-size", 0x6400000L));
        options.maxOpenFiles(this.conf.getInt("yarn.timeline-service.leveldb-timeline-store.max-open-files", 1000));
        options.writeBufferSize(this.conf.getInt("yarn.timeline-service.leveldb-timeline-store.write-buffer-size", 0x1000000));
        LOG.info("Initializing rolling leveldb instance :" + rollingInstanceDBPath + " for start time: " + dbStartTime);
        DB db = null;
        try {
            db = factory.open(new File(rollingInstanceDBPath.toUri().getPath()), options);
            this.rollingdbs.put(dbStartTime, db);
            String dbName = this.fdf.format((Object)dbStartTime);
            LOG.info("Added rolling leveldb instance " + dbName + " to " + this.getName());
        }
        catch (IOException ioe) {
            LOG.warn("Failed to open rolling leveldb instance :" + new File(rollingInstanceDBPath.toUri().getPath()), (Throwable)ioe);
        }
    }

    synchronized DB getPreviousDB(DB db) {
        DB cur;
        Iterator<DB> iterator = this.rollingdbs.values().iterator();
        DB prev = null;
        while (iterator.hasNext() && (cur = iterator.next()) != db) {
            prev = cur;
        }
        return prev;
    }

    synchronized long getStartTimeFor(DB db) {
        long startTime = -1L;
        for (Map.Entry<Long, DB> entry : this.rollingdbs.entrySet()) {
            if (entry.getValue() != db) continue;
            startTime = entry.getKey();
        }
        return startTime;
    }

    public synchronized DB getDBForStartTime(long startTime) {
        Map.Entry<Long, DB> entry;
        if ((startTime = Math.min(startTime, this.currentTimeMillis())) >= this.getNextRollingTimeMillis()) {
            this.roll(startTime);
        }
        if ((entry = this.rollingdbs.floorEntry(startTime)) == null) {
            return null;
        }
        return entry.getValue();
    }

    private void roll(long startTime) {
        LOG.info("Rolling new DB instance for " + this.getName());
        long currentStartTime = this.computeCurrentCheckMillis(startTime);
        this.setNextRollingTimeMillis(this.computeNextCheckMillis(currentStartTime));
        String currentRollingDBInstance = this.fdf.format(currentStartTime);
        String currentRollingDBName = this.getName() + "." + currentRollingDBInstance;
        Path currentRollingDBPath = new Path(this.rollingDBPath, currentRollingDBName);
        if (this.getTimeToLiveEnabled()) {
            this.scheduleOldDBsForEviction();
        }
        this.initRollingLevelDB(currentStartTime, currentRollingDBPath);
    }

    private synchronized void scheduleOldDBsForEviction() {
        long evictionThreshold = this.computeCurrentCheckMillis(this.currentTimeMillis() - this.getTimeToLive());
        LOG.info("Scheduling " + this.getName() + " DBs older than " + this.fdf.format(evictionThreshold) + " for eviction");
        Iterator<Map.Entry<Long, DB>> iterator = this.rollingdbs.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<Long, DB> entry = iterator.next();
            if (entry.getKey() >= evictionThreshold) continue;
            LOG.info("Scheduling " + this.getName() + " eviction for " + this.fdf.format((Object)entry.getKey()));
            iterator.remove();
            this.rollingdbsToEvict.put(entry.getKey(), entry.getValue());
        }
    }

    public synchronized void evictOldDBs() {
        LOG.info("Evicting " + this.getName() + " DBs scheduled for eviction");
        Iterator<Map.Entry<Long, DB>> iterator = this.rollingdbsToEvict.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<Long, DB> entry = iterator.next();
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{(Closeable)entry.getValue()});
            String dbName = this.fdf.format((Object)entry.getKey());
            Path path = new Path(this.rollingDBPath, this.getName() + "." + dbName);
            try {
                LOG.info("Removing old db directory contents in " + path);
                this.lfs.delete(path, true);
            }
            catch (IOException ioe) {
                LOG.warn("Failed to evict old db " + path, (Throwable)ioe);
            }
            iterator.remove();
        }
    }

    public void stop() throws Exception {
        for (DB db : this.rollingdbs.values()) {
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{db});
        }
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{this.lfs});
    }

    private long computeNextCheckMillis(long now) {
        return this.computeCheckMillis(now, true);
    }

    public long computeCurrentCheckMillis(long now) {
        return this.computeCheckMillis(now, false);
    }

    private synchronized long computeCheckMillis(long now, boolean next) {
        this.cal.setTimeInMillis(now);
        this.cal.set(13, 0);
        this.cal.set(14, 0);
        if (this.rollingPeriod == RollingPeriod.DAILY) {
            this.cal.set(11, 0);
            this.cal.set(12, 0);
            if (next) {
                this.cal.add(5, 1);
            }
        } else if (this.rollingPeriod == RollingPeriod.HALF_DAILY) {
            int hour = this.cal.get(10) / 12 * 12;
            this.cal.set(10, hour);
            this.cal.set(12, 0);
            if (next) {
                this.cal.add(11, 12);
            }
        } else if (this.rollingPeriod == RollingPeriod.QUARTER_DAILY) {
            int hour = this.cal.get(10) / 6 * 6;
            this.cal.set(10, hour);
            this.cal.set(12, 0);
            if (next) {
                this.cal.add(11, 6);
            }
        } else if (this.rollingPeriod == RollingPeriod.HOURLY) {
            this.cal.set(12, 0);
            if (next) {
                this.cal.add(11, 1);
            }
        } else if (this.rollingPeriod == RollingPeriod.MINUTELY) {
            int minute = this.cal.get(12) / 5 * 5;
            this.cal.set(12, minute);
            if (next) {
                this.cal.add(12, 5);
            }
        }
        return this.cal.getTimeInMillis();
    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    static enum RollingPeriod {
        DAILY{

            @Override
            public String dateFormat() {
                return "yyyy-MM-dd";
            }
        }
        ,
        HALF_DAILY{

            @Override
            public String dateFormat() {
                return "yyyy-MM-dd-HH";
            }
        }
        ,
        QUARTER_DAILY{

            @Override
            public String dateFormat() {
                return "yyyy-MM-dd-HH";
            }
        }
        ,
        HOURLY{

            @Override
            public String dateFormat() {
                return "yyyy-MM-dd-HH";
            }
        }
        ,
        MINUTELY{

            @Override
            public String dateFormat() {
                return "yyyy-MM-dd-HH-mm";
            }
        };


        public abstract String dateFormat();
    }

    public static class RollingWriteBatch {
        private final DB db;
        private final WriteBatch writeBatch;

        public RollingWriteBatch(DB db, WriteBatch writeBatch) {
            this.db = db;
            this.writeBatch = writeBatch;
        }

        public DB getDB() {
            return this.db;
        }

        public WriteBatch getWriteBatch() {
            return this.writeBatch;
        }

        public void write() {
            this.db.write(this.writeBatch);
        }

        public void close() {
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{this.writeBatch});
        }
    }
}

