/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.tosfs.ops;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.tosfs.common.Tasks;
import org.apache.hadoop.fs.tosfs.conf.ConfKeys;
import org.apache.hadoop.fs.tosfs.object.MultipartUpload;
import org.apache.hadoop.fs.tosfs.object.ObjectInfo;
import org.apache.hadoop.fs.tosfs.object.ObjectStorage;
import org.apache.hadoop.fs.tosfs.object.ObjectUtils;
import org.apache.hadoop.fs.tosfs.object.Part;
import org.apache.hadoop.fs.tosfs.util.CommonUtils;
import org.apache.hadoop.util.Lists;
import org.apache.hadoop.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RenameOp {
    private static final Logger LOG = LoggerFactory.getLogger(RenameOp.class);
    private static final int RENAME_RETRY_TIMES = 3;
    private final Configuration conf;
    private final ObjectStorage storage;
    private final ExecutorService renamePool;
    private final boolean renameObjectEnabled;

    public RenameOp(Configuration conf, ObjectStorage storage, ExecutorService taskThreadPool) {
        this.conf = conf;
        this.storage = storage;
        this.renamePool = taskThreadPool;
        this.renameObjectEnabled = conf.getBoolean(ConfKeys.FS_OBJECT_RENAME_ENABLED.key(storage.scheme()), false);
    }

    public void renameDir(Path src, Path dst) {
        String srcKey = ObjectUtils.pathToKey(src, true);
        String dstKey = ObjectUtils.pathToKey(dst, true);
        this.renameDir(srcKey, dstKey);
    }

    public void renameFile(Path src, Path dst, long length) {
        String srcKey = ObjectUtils.pathToKey(src, false);
        String dstKey = ObjectUtils.pathToKey(dst, false);
        this.renameFile(srcKey, dstKey, length);
    }

    private void renameDir(String srcKey, String dstKey) {
        Iterable<ObjectInfo> objs = this.storage.listAll(srcKey, "");
        if (this.renameObjectEnabled) {
            Tasks.foreach(objs).executeWith(this.renamePool).throwFailureWhenFinished().retry(3).revertWith(sourceInfo -> {
                String newDstKey = dstKey + sourceInfo.key().substring(srcKey.length());
                String newSrcKey = sourceInfo.key();
                LOG.debug("Try to rollback dest key {} to source key {}", (Object)newDstKey, (Object)newSrcKey);
                this.storage.rename(newDstKey, newSrcKey);
            }).run(sourceInfo -> {
                String newDstKey = dstKey + sourceInfo.key().substring(srcKey.length());
                String newSrcKey = sourceInfo.key();
                LOG.debug("Try to rename src key {} to dest key {}", (Object)newSrcKey, (Object)newDstKey);
                this.storage.rename(newSrcKey, newDstKey);
            });
        } else {
            Tasks.foreach(objs).executeWith(this.renamePool).throwFailureWhenFinished().retry(3).revertWith(sourceInfo -> {
                String newDstKey = dstKey + sourceInfo.key().substring(srcKey.length());
                this.storage.delete(newDstKey);
            }).run(sourceInfo -> {
                String newDstKey = dstKey + sourceInfo.key().substring(srcKey.length());
                LOG.debug("Try to rename src key {} to dest key {}", (Object)sourceInfo.key(), (Object)newDstKey);
                try {
                    if (ObjectInfo.isDir(newDstKey)) {
                        this.mkdir(newDstKey);
                    } else {
                        this.copyFile(sourceInfo.key(), newDstKey, sourceInfo.size());
                    }
                }
                catch (IOException e) {
                    throw new UncheckedIOException(String.format("Failed to copy source file %s to dest file %s", sourceInfo.key(), newDstKey), e);
                }
            });
            this.storage.deleteAll(srcKey);
        }
    }

    private void renameFile(String srcKey, String dstKey, long fileSize) {
        if (this.renameObjectEnabled) {
            this.storage.rename(srcKey, dstKey);
        } else {
            Tasks.foreach(0).throwFailureWhenFinished().retry(3).revertWith(obj -> this.storage.delete(dstKey)).run(obj -> {
                try {
                    this.copyFile(srcKey, dstKey, fileSize);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(String.format("Failed to copy source file %s to dest file %s", srcKey, dstKey), e);
                }
            });
            Tasks.foreach(0).throwFailureWhenFinished().retry(3).run(obj -> this.storage.delete(srcKey));
        }
    }

    private void copyFile(String srcKey, String dstKey, long srcSize) throws IOException {
        long byteSizePerPart = this.conf.getLong(ConfKeys.FS_MULTIPART_SIZE.key(this.storage.scheme()), 0x800000L);
        long multiPartCopyThreshold = this.conf.getLong(ConfKeys.FS_MULTIPART_COPY_THRESHOLD.key(this.storage.scheme()), 0x500000L);
        if (srcSize > multiPartCopyThreshold) {
            this.uploadPartCopy(srcKey, srcSize, dstKey, byteSizePerPart);
        } else {
            this.storage.copy(srcKey, dstKey);
        }
    }

    private void uploadPartCopy(String srcKey, long srcSize, String dstKey, long byteSizePerPart) {
        MultipartUpload multipartUpload2 = this.storage.createMultipartUpload(dstKey);
        try {
            Preconditions.checkState((byteSizePerPart >= (long)multipartUpload2.minPartSize() ? 1 : 0) != 0, (String)"Configured upload part size %s must be greater than or equals to the minimal part size %s, please check configure key %s.", (Object[])new Object[]{byteSizePerPart, multipartUpload2.minPartSize(), ConfKeys.FS_MULTIPART_SIZE.key(this.storage.scheme())});
            AtomicInteger partNumGetter = new AtomicInteger(0);
            ArrayList results = Lists.newArrayList();
            for (long start = 0L; start < srcSize; start += byteSizePerPart) {
                long end = Math.min(start + byteSizePerPart, srcSize) - 1L;
                Preconditions.checkArgument((end >= 0L ? 1 : 0) != 0, (String)"Invalid copy range start: %s, end: %s", (Object[])new Object[]{start, end});
                CompletableFuture<Part> result = this.asyncUploadPartCopy(srcKey, multipartUpload2, partNumGetter.incrementAndGet(), start, end);
                results.add(result);
            }
            List<Part> parts = results.stream().map(CompletableFuture::join).sorted(Comparator.comparing(Part::num)).collect(Collectors.toList());
            this.finishUpload(multipartUpload2.key(), multipartUpload2.uploadId(), parts);
        }
        catch (Exception e) {
            LOG.error("Encountering error when upload part copy", (Throwable)e);
            CommonUtils.runQuietly(() -> this.storage.abortMultipartUpload(multipartUpload2.key(), multipartUpload2.uploadId()));
            throw e;
        }
    }

    protected void finishUpload(String key, String uploadId, List<Part> uploadParts) {
        this.storage.completeUpload(key, uploadId, uploadParts);
    }

    private CompletableFuture<Part> asyncUploadPartCopy(String srcKey, MultipartUpload multipartUpload2, int partNum, long copyRangeStart, long copyRangeEnd) {
        return CompletableFuture.supplyAsync(() -> this.storage.uploadPartCopy(srcKey, multipartUpload2.key(), multipartUpload2.uploadId(), partNum, copyRangeStart, copyRangeEnd), this.renamePool).whenComplete((part, err) -> {
            if (err != null) {
                LOG.error("Failed to upload part copy, src key: {}, multipartUpload: {}, partNum: {}, copy range start: {}, copy range end: {}", new Object[]{srcKey, multipartUpload2, partNum, copyRangeStart, copyRangeEnd, err});
            }
        });
    }

    private void mkdir(String key) {
        this.storage.put(key, new byte[0]);
    }
}

