/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.tosfs;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FsServerDefaults;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIOException;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.XAttrSetFlag;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.tosfs.RawFSUtils;
import org.apache.hadoop.fs.tosfs.RawFileStatus;
import org.apache.hadoop.fs.tosfs.RawLocatedFileStatus;
import org.apache.hadoop.fs.tosfs.TosChecksum;
import org.apache.hadoop.fs.tosfs.commit.MagicOutputStream;
import org.apache.hadoop.fs.tosfs.common.Bytes;
import org.apache.hadoop.fs.tosfs.common.ThreadPools;
import org.apache.hadoop.fs.tosfs.conf.ConfKeys;
import org.apache.hadoop.fs.tosfs.object.ChecksumInfo;
import org.apache.hadoop.fs.tosfs.object.Constants;
import org.apache.hadoop.fs.tosfs.object.DirectoryStorage;
import org.apache.hadoop.fs.tosfs.object.ObjectInfo;
import org.apache.hadoop.fs.tosfs.object.ObjectMultiRangeInputStream;
import org.apache.hadoop.fs.tosfs.object.ObjectOutputStream;
import org.apache.hadoop.fs.tosfs.object.ObjectRangeInputStream;
import org.apache.hadoop.fs.tosfs.object.ObjectStorage;
import org.apache.hadoop.fs.tosfs.object.ObjectStorageFactory;
import org.apache.hadoop.fs.tosfs.object.ObjectUtils;
import org.apache.hadoop.fs.tosfs.object.exceptions.InvalidObjectKeyException;
import org.apache.hadoop.fs.tosfs.ops.DefaultFsOps;
import org.apache.hadoop.fs.tosfs.ops.DirectoryFsOps;
import org.apache.hadoop.fs.tosfs.ops.FsOps;
import org.apache.hadoop.fs.tosfs.util.FSUtils;
import org.apache.hadoop.fs.tosfs.util.FuseUtils;
import org.apache.hadoop.fs.tosfs.util.Range;
import org.apache.hadoop.fs.tosfs.util.RemoteIterators;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.thirdparty.com.google.common.collect.Iterators;
import org.apache.hadoop.util.DataChecksum;
import org.apache.hadoop.util.Lists;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RawFileSystem
extends FileSystem {
    private static final Logger LOG = LoggerFactory.getLogger(RawFileSystem.class);
    private static final String MULTIPART_THREAD_POOL_PREFIX = "rawfs-multipart-thread-pool";
    private static final String TASK_THREAD_POOL_PREFIX = "rawfs-task-thread-pool";
    private static final String DFS_BLOCK_SIZE_KEY = "dfs.blocksize";
    private static final long DFS_BLOCK_SIZE_DEFAULT = 0x8000000L;
    private String scheme;
    private String username;
    private Path workingDir;
    private URI uri;
    private String bucket;
    private ObjectStorage storage;
    private ExecutorService taskThreadPool;
    private ExecutorService uploadThreadPool;
    private FsOps fsOps;

    public URI getUri() {
        return this.uri;
    }

    public String getScheme() {
        return this.scheme;
    }

    @VisibleForTesting
    String bucket() {
        return this.bucket;
    }

    public void setConf(Configuration conf) {
        super.setConf(conf);
    }

    public FSDataInputStream open(Path path, int bufferSize) throws IOException {
        LOG.debug("Opening '{}' for reading.", (Object)path);
        RawFileStatus status = this.innerFileStatus(path);
        if (status.isDirectory()) {
            throw new FileNotFoundException(String.format("Can't open %s because it is a directory", path));
        }
        long rangeSize = this.getConf().getLong("fs.objectstorage.stream.range-size", Long.MAX_VALUE);
        Preconditions.checkArgument((rangeSize > 0L ? 1 : 0) != 0, (Object)"Object storage range size must be positive.");
        ObjectMultiRangeInputStream fsIn = new ObjectMultiRangeInputStream(this.taskThreadPool, this.storage, path, status.getLen(), rangeSize, status.checksum());
        return new FSDataInputStream((InputStream)((Object)fsIn));
    }

    public FSDataInputStream open(Path path, byte[] expectedChecksum, Range range) {
        return new FSDataInputStream((InputStream)((Object)new ObjectRangeInputStream(this.storage, path, range, expectedChecksum)));
    }

    public FSDataOutputStream create(Path path, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        RawFileStatus fileStatus = this.getFileStatusOrNull(path);
        if (fileStatus != null) {
            if (fileStatus.isDirectory()) {
                throw new FileAlreadyExistsException(path + " is a directory");
            }
            if (!overwrite) {
                throw new FileAlreadyExistsException(path + " already exists");
            }
            LOG.debug("Overwriting file {}", (Object)path);
        }
        if (MagicOutputStream.isMagic(path)) {
            return new FSDataOutputStream((OutputStream)new MagicOutputStream(this, this.storage, this.uploadThreadPool, this.getConf(), this.makeQualified(path)), null);
        }
        ObjectOutputStream out = new ObjectOutputStream(this.storage, this.uploadThreadPool, this.getConf(), this.makeQualified(path), true);
        if (fileStatus == null && FuseUtils.fuseEnabled()) {
            out.close();
            out = new ObjectOutputStream(this.storage, this.uploadThreadPool, this.getConf(), this.makeQualified(path), true);
        }
        return new FSDataOutputStream((OutputStream)out, null);
    }

    public RemoteIterator<LocatedFileStatus> listFiles(Path f, boolean recursive) throws IOException {
        RawFileStatus fileStatus;
        Path path = this.makeQualified(f);
        LOG.debug("listFiles({}, {})", (Object)path, (Object)recursive);
        RemoteIterator<LocatedFileStatus> subFiles = RemoteIterators.fromIterable(this.fsOps.listDir(path, recursive, key -> !ObjectInfo.isDir(key)), this::toLocatedFileStatus);
        if (!subFiles.hasNext() && (fileStatus = this.innerFileStatus(path)).isFile()) {
            return RemoteIterators.fromSingleton(this.toLocatedFileStatus(fileStatus));
        }
        return subFiles;
    }

    private RawLocatedFileStatus toLocatedFileStatus(RawFileStatus status) throws IOException {
        return new RawLocatedFileStatus(status, status.isFile() ? this.getFileBlockLocations(status, 0L, status.getLen()) : null);
    }

    public FSDataOutputStream createNonRecursive(Path path, FsPermission permission, EnumSet<CreateFlag> flag, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        Path qualified = this.makeQualified(path);
        return this.create(qualified, permission, flag.contains(CreateFlag.OVERWRITE), bufferSize, replication, blockSize, progress);
    }

    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        throw new IOException("Not supported");
    }

    public boolean rename(Path src, Path dst) throws IOException {
        Path dstPath;
        FileStatus srcStatus;
        LOG.debug("Rename source path {} to dest path {}", (Object)src, (Object)dst);
        Future<FileStatus> srcStatusFuture = this.taskThreadPool.submit(() -> this.checkAndGetSrcStatus(src));
        Future<Path> destPathFuture = this.taskThreadPool.submit(() -> this.checkAndGetDstPath(src, dst));
        try {
            srcStatus = srcStatusFuture.get();
            dstPath = destPathFuture.get();
            if (src.equals((Object)dstPath)) {
                return true;
            }
        }
        catch (InterruptedException | ExecutionException e) {
            LOG.error("Failed to rename path, src: {}, dst: {}", new Object[]{src, dst, e});
            return false;
        }
        if (srcStatus.isDirectory()) {
            this.fsOps.renameDir(srcStatus.getPath(), dstPath);
        } else {
            this.fsOps.renameFile(srcStatus.getPath(), dstPath, srcStatus.getLen());
        }
        return true;
    }

    private Path checkAndGetDstPath(Path src, Path dest) throws IOException {
        RawFileStatus destStatus = this.getFileStatusOrNull(dest);
        Path finalDstPath = dest;
        if (destStatus != null && destStatus.isDirectory()) {
            finalDstPath = new Path(dest, src.getName());
        }
        if (src.equals((Object)finalDstPath)) {
            return finalDstPath;
        }
        if (RawFSUtils.inSubtree(src, finalDstPath)) {
            throw new IOException(String.format("Failed to rename since it is prohibited to rename dest path %s under src path %s", finalDstPath, src));
        }
        RawFileStatus finalDstStatus = destStatus;
        if (destStatus != null && destStatus.isDirectory()) {
            finalDstStatus = this.getFileStatusOrNull(finalDstPath);
        }
        if (finalDstStatus != null) {
            throw new FileAlreadyExistsException(String.format("Failed to rename since the dest path %s already exists.", finalDstPath));
        }
        return finalDstPath;
    }

    private FileStatus checkAndGetSrcStatus(Path src) throws IOException {
        RawFileStatus srcStatus = this.innerFileStatus(src);
        if (src.isRoot()) {
            throw new IOException(String.format("Cannot rename the root directory %s to another name", src));
        }
        return srcStatus;
    }

    public boolean delete(Path f, boolean recursive) throws IOException {
        LOG.debug("Delete path {} - recursive {}", (Object)f, (Object)recursive);
        try {
            FileStatus fileStatus = this.getFileStatus(f);
            Path path = fileStatus.getPath();
            if (path.isRoot()) {
                return this.deleteRoot(path, recursive);
            }
            if (fileStatus.isDirectory()) {
                this.fsOps.deleteDir(path, recursive);
            } else {
                this.fsOps.deleteFile(path);
            }
            return true;
        }
        catch (FileNotFoundException e) {
            LOG.debug("Couldn't delete {} - does not exist", (Object)f);
            return false;
        }
    }

    private boolean deleteRoot(Path root, boolean recursive) throws IOException {
        LOG.info("Delete the {} root directory of {}", (Object)this.bucket, (Object)recursive);
        boolean isEmptyDir = this.fsOps.isEmptyDirectory(root);
        if (isEmptyDir) {
            return true;
        }
        if (recursive) {
            return false;
        }
        throw new PathIOException(this.bucket, "Cannot delete root path");
    }

    public RawFileStatus[] listStatus(Path f) throws IOException {
        LOG.debug("List status for path: {}", (Object)f);
        return (RawFileStatus[])Iterators.toArray(this.listStatus(f, false), RawFileStatus.class);
    }

    public Iterator<RawFileStatus> listStatus(Path f, boolean recursive) throws IOException {
        Path path = this.makeQualified(f);
        Iterator<RawFileStatus> iterator = this.fsOps.listDir(path, recursive, key -> true).iterator();
        if (iterator.hasNext()) {
            return iterator;
        }
        RawFileStatus fileStatus = this.innerFileStatus(path);
        if (fileStatus.isFile()) {
            return Collections.singletonList(fileStatus).iterator();
        }
        return Collections.emptyIterator();
    }

    public RemoteIterator<FileStatus> listStatusIterator(final Path p) throws IOException {
        return new RemoteIterator<FileStatus>(){
            private FileSystem.DirectoryEntries entries;
            private int index;
            {
                this.entries = RawFileSystem.this.listStatusBatch(p, null);
                this.index = 0;
            }

            public boolean hasNext() {
                return this.index < this.entries.getEntries().length || this.entries.hasMore();
            }

            private void fetchMore() throws IOException {
                byte[] token = this.entries.getToken();
                this.entries = RawFileSystem.this.listStatusBatch(p, token);
                this.index = 0;
            }

            public FileStatus next() throws IOException {
                if (!this.hasNext()) {
                    throw new NoSuchElementException("No more items in iterator");
                }
                if (this.index == this.entries.getEntries().length) {
                    this.fetchMore();
                    if (!this.hasNext()) {
                        throw new NoSuchElementException("No more items in iterator");
                    }
                }
                return this.entries.getEntries()[this.index++];
            }
        };
    }

    public static long dateToLong(Date date) {
        return date == null ? 0L : date.getTime();
    }

    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    public void setWorkingDirectory(Path newDir) {
        this.workingDir = newDir;
    }

    public boolean mkdirs(Path path, FsPermission permission) throws IOException {
        try {
            RawFileStatus fileStatus = this.innerFileStatus(path);
            if (fileStatus.isDirectory()) {
                return true;
            }
            throw new FileAlreadyExistsException("Path is a file: " + path);
        }
        catch (FileNotFoundException e) {
            Path dir = this.makeQualified(path);
            this.validatePath(dir);
            this.fsOps.mkdirs(dir);
            return true;
        }
    }

    private void validatePath(Path path) throws IOException {
        Path parent = path.getParent();
        while (true) {
            try {
                RawFileStatus fileStatus = this.innerFileStatus(parent);
                if (!fileStatus.isDirectory()) {
                    throw new FileAlreadyExistsException(String.format("Can't make directory for path '%s', it is a file.", parent));
                }
            }
            catch (FileNotFoundException fileNotFoundException) {
                if ((parent = parent.getParent()) != null) continue;
            }
            break;
        }
    }

    public FileStatus getFileStatus(Path path) throws IOException {
        try {
            return this.innerFileStatus(path);
        }
        catch (ParentNotDirectoryException e) {
            throw new FileNotFoundException(e.getMessage());
        }
    }

    RawFileStatus innerFileStatus(Path f) throws ParentNotDirectoryException, FileNotFoundException {
        Path qualifiedPath = f.makeQualified(this.uri, this.workingDir);
        RawFileStatus fileStatus = this.getFileStatusOrNull(qualifiedPath);
        if (fileStatus == null) {
            throw new FileNotFoundException(String.format("No such file or directory: %s", qualifiedPath));
        }
        return fileStatus;
    }

    public RawFileStatus getFileStatusOrNull(Path path) throws ParentNotDirectoryException {
        Path qualifiedPath = path.makeQualified(this.uri, this.workingDir);
        String key = ObjectUtils.pathToKey(qualifiedPath);
        if (key.isEmpty()) {
            return new RawFileStatus(0L, true, 0L, 0L, qualifiedPath, this.username, Constants.MAGIC_CHECKSUM);
        }
        try {
            ObjectInfo obj = this.storage.objectStatus(key);
            if (obj == null) {
                return null;
            }
            return this.objectToFileStatus(obj);
        }
        catch (InvalidObjectKeyException e) {
            String msg = String.format("The object key %s is a invalid key, detail: %s", key, e.getMessage());
            throw new ParentNotDirectoryException(msg);
        }
    }

    private RawFileStatus objectToFileStatus(ObjectInfo obj) {
        Path keyPath = this.makeQualified(ObjectUtils.keyToPath(obj.key()));
        long blockSize = obj.isDir() ? 0L : this.getDefaultBlockSize(keyPath);
        long modificationTime = RawFileSystem.dateToLong(obj.mtime());
        return new RawFileStatus(obj.size(), obj.isDir(), blockSize, modificationTime, keyPath, this.username, obj.checksum());
    }

    @Deprecated
    public long getDefaultBlockSize() {
        return this.getConf().getLongBytes(DFS_BLOCK_SIZE_KEY, 0x8000000L);
    }

    public FsServerDefaults getServerDefaults(Path p) {
        Configuration config = this.getConf();
        return new FsServerDefaults(this.getDefaultBlockSize(), config.getInt("dfs.bytes-per-checksum", 512), 65536, this.getDefaultReplication(), config.getInt("io.file.buffer.size", 4096), false, 0L, DataChecksum.Type.CRC32, "");
    }

    private void stopAllServices() {
        ThreadPools.shutdown(this.uploadThreadPool, 30L, TimeUnit.SECONDS);
        ThreadPools.shutdown(this.taskThreadPool, 30L, TimeUnit.SECONDS);
    }

    public void initialize(URI name, Configuration conf) throws IOException {
        super.initialize(name, conf);
        this.setConf(conf);
        this.scheme = FSUtils.scheme(conf, name);
        this.username = UserGroupInformation.getCurrentUser().getShortUserName();
        this.workingDir = new Path("/user", this.username).makeQualified(name, null);
        this.uri = URI.create(this.scheme + "://" + name.getAuthority());
        this.bucket = this.uri.getAuthority();
        this.storage = ObjectStorageFactory.create(this.scheme, this.bucket, this.getConf());
        if (this.storage.bucket() == null) {
            throw new FileNotFoundException(String.format("Bucket: %s not found.", name.getAuthority()));
        }
        int taskThreadPoolSize = this.getConf().getInt(ConfKeys.FS_TASK_THREAD_POOL_SIZE.key(this.storage.scheme()), ConfKeys.FS_TASK_THREAD_POOL_SIZE_DEFAULT);
        this.taskThreadPool = ThreadPools.newWorkerPool(TASK_THREAD_POOL_PREFIX, taskThreadPoolSize);
        int uploadThreadPoolSize = this.getConf().getInt(ConfKeys.FS_MULTIPART_THREAD_POOL_SIZE.key(this.storage.scheme()), ConfKeys.FS_MULTIPART_THREAD_POOL_SIZE_DEFAULT);
        this.uploadThreadPool = ThreadPools.newWorkerPool(MULTIPART_THREAD_POOL_PREFIX, uploadThreadPoolSize);
        this.fsOps = this.storage.bucket().isDirectory() ? new DirectoryFsOps((DirectoryStorage)this.storage, this::objectToFileStatus) : new DefaultFsOps(this.storage, this.getConf(), this.taskThreadPool, this::objectToFileStatus);
    }

    public void close() throws IOException {
        try {
            super.close();
            this.storage.close();
        }
        finally {
            this.stopAllServices();
        }
    }

    public ObjectStorage storage() {
        return this.storage;
    }

    public ExecutorService uploadThreadPool() {
        return this.uploadThreadPool;
    }

    public FileChecksum getFileChecksum(Path f, long length) throws IOException {
        Preconditions.checkArgument((length >= 0L ? 1 : 0) != 0);
        RawFileStatus fileStatus = this.innerFileStatus(f);
        if (fileStatus.isDirectory()) {
            throw new FileNotFoundException(String.format("Path is not a file, %s", f));
        }
        if (!this.getConf().getBoolean(ConfKeys.FS_CHECKSUM_ENABLED.key(this.storage.scheme()), true)) {
            return null;
        }
        ChecksumInfo csInfo = this.storage.checksumInfo();
        return new TosChecksum(csInfo.algorithm(), fileStatus.checksum());
    }

    public String getCanonicalServiceName() {
        return null;
    }

    public void setXAttr(Path path, String name, byte[] value, EnumSet<XAttrSetFlag> flag) throws IOException {
        Preconditions.checkNotNull((Object)name, (Object)"xAttr name must not be null.");
        Preconditions.checkArgument((!name.isEmpty() ? 1 : 0) != 0, (Object)"xAttr name must not be empty.");
        Preconditions.checkNotNull((Object)value, (Object)"xAttr value must not be null.");
        if (this.getFileStatus(path).isFile()) {
            Path qualifiedPath = path.makeQualified(this.uri, this.workingDir);
            String key = ObjectUtils.pathToKey(qualifiedPath);
            Map<String, String> existedTags = this.storage.getTags(key);
            this.validateXAttrFlag(name, existedTags.containsKey(name), flag);
            String newValue = Bytes.toString(value);
            String previousValue = existedTags.put(name, newValue);
            if (!newValue.equals(previousValue)) {
                this.storage.putTags(key, existedTags);
            }
        }
    }

    public Map<String, byte[]> getXAttrs(Path path) throws IOException {
        if (this.getFileStatus(path).isDirectory()) {
            return new HashMap<String, byte[]>();
        }
        Path qualifiedPath = path.makeQualified(this.uri, this.workingDir);
        String key = ObjectUtils.pathToKey(qualifiedPath);
        Map<String, String> tags = this.storage.getTags(key);
        return tags.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, t -> Bytes.toBytes((String)t.getValue())));
    }

    public byte[] getXAttr(Path path, String name) throws IOException {
        Map<String, byte[]> xAttrs = this.getXAttrs(path);
        if (xAttrs.containsKey(name)) {
            return xAttrs.get(name);
        }
        throw new IOException("Attribute with name " + name + " is not found.");
    }

    public Map<String, byte[]> getXAttrs(Path path, List<String> names) throws IOException {
        Map<String, byte[]> xAttrs = this.getXAttrs(path);
        xAttrs.keySet().retainAll(names);
        if (xAttrs.size() == names.size()) {
            return xAttrs;
        }
        List badNames = names.stream().filter(n -> !xAttrs.containsKey(n)).collect(Collectors.toList());
        throw new IOException("Attributes with name " + badNames + " are not found.");
    }

    public List<String> listXAttrs(Path path) throws IOException {
        return Lists.newArrayList(this.getXAttrs(path).keySet());
    }

    public void removeXAttr(Path path, String name) throws IOException {
        Path qualifiedPath;
        String key;
        Map<String, String> existedTags;
        if (this.getFileStatus(path).isFile() && (existedTags = this.storage.getTags(key = ObjectUtils.pathToKey(qualifiedPath = path.makeQualified(this.uri, this.workingDir)))).remove(name) != null) {
            this.storage.putTags(key, existedTags);
        }
    }

    private void validateXAttrFlag(String xAttrName, boolean xAttrExists, EnumSet<XAttrSetFlag> flag) throws IOException {
        if (xAttrExists) {
            if (!flag.contains(XAttrSetFlag.REPLACE)) {
                throw new IOException("XAttr: " + xAttrName + " already exists. The REPLACE flag must be specified.");
            }
        } else if (!flag.contains(XAttrSetFlag.CREATE)) {
            throw new IOException("XAttr: " + xAttrName + " does not exist. The CREATE flag must be specified.");
        }
    }
}

