/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.mapreduce.lib.input;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.HdfsBlockLocation;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.RawLocalFileSystem;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.mapred.SplitLocationInfo;
import org.apache.hadoop.mapreduce.InputSplit;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.JobContext;
import org.apache.hadoop.mapreduce.lib.input.FileInputFormat;
import org.apache.hadoop.mapreduce.lib.input.FileSplit;
import org.apache.hadoop.mapreduce.lib.input.InvalidInputException;
import org.apache.hadoop.mapreduce.lib.input.TextInputFormat;
import org.apache.hadoop.util.Lists;
import org.apache.hadoop.util.Sets;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestFileInputFormat {
    private static final Logger LOG = LoggerFactory.getLogger(TestFileInputFormat.class);
    private static String testTmpDir = System.getProperty("test.build.data", "/tmp");
    private static final Path TEST_ROOT_DIR = new Path(testTmpDir, "TestFIF");
    private static FileSystem localFs;
    private int numThreads;

    public void initTestFileInputFormat(int pNumThreads) {
        this.numThreads = pNumThreads;
        LOG.info("Running with numThreads: " + pNumThreads);
    }

    public static Collection<Object[]> data() {
        Object[][] data = new Object[][]{{1}, {5}};
        return Arrays.asList(data);
    }

    @BeforeEach
    public void setup() throws IOException {
        LOG.info("Using Test Dir: " + TEST_ROOT_DIR);
        localFs = FileSystem.getLocal((Configuration)new Configuration());
        localFs.delete(TEST_ROOT_DIR, true);
        localFs.mkdirs(TEST_ROOT_DIR);
    }

    @AfterEach
    public void cleanup() throws IOException {
        localFs.delete(TEST_ROOT_DIR, true);
    }

    @MethodSource(value={"data"})
    @ParameterizedTest
    public void testNumInputFilesRecursively(int pNumThreads) throws Exception {
        this.initTestFileInputFormat(pNumThreads);
        Configuration conf = this.getConfiguration();
        conf.set("mapreduce.input.fileinputformat.input.dir.recursive", "true");
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fileInputFormat = new TextInputFormat();
        List splits = fileInputFormat.getSplits((JobContext)job);
        Assertions.assertEquals((int)3, (int)splits.size(), (String)"Input splits are not correct");
        this.verifySplits(Lists.newArrayList((Object[])new String[]{"test:/a1/a2/file2", "test:/a1/a2/file3", "test:/a1/file1"}), splits);
        conf = this.getConfiguration();
        conf.set("mapred.input.dir.recursive", "true");
        job = Job.getInstance((Configuration)conf);
        splits = fileInputFormat.getSplits((JobContext)job);
        this.verifySplits(Lists.newArrayList((Object[])new String[]{"test:/a1/a2/file2", "test:/a1/a2/file3", "test:/a1/file1"}), splits);
    }

    @MethodSource(value={"data"})
    @ParameterizedTest
    public void testNumInputFilesWithoutRecursively(int pNumThreads) throws Exception {
        this.initTestFileInputFormat(pNumThreads);
        Configuration conf = this.getConfiguration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fileInputFormat = new TextInputFormat();
        List splits = fileInputFormat.getSplits((JobContext)job);
        Assertions.assertEquals((int)2, (int)splits.size(), (String)"Input splits are not correct");
        this.verifySplits(Lists.newArrayList((Object[])new String[]{"test:/a1/a2", "test:/a1/file1"}), splits);
    }

    @MethodSource(value={"data"})
    @ParameterizedTest
    public void testNumInputFilesIgnoreDirs(int pNumThreads) throws Exception {
        this.initTestFileInputFormat(pNumThreads);
        Configuration conf = this.getConfiguration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        conf.setBoolean("mapreduce.input.fileinputformat.input.dir.nonrecursive.ignore.subdirs", true);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fileInputFormat = new TextInputFormat();
        List splits = fileInputFormat.getSplits((JobContext)job);
        Assertions.assertEquals((int)1, (int)splits.size(), (String)"Input splits are not correct");
        this.verifySplits(Lists.newArrayList((Object[])new String[]{"test:/a1/file1"}), splits);
    }

    @MethodSource(value={"data"})
    @ParameterizedTest
    public void testListLocatedStatus(int pNumThreads) throws Exception {
        this.initTestFileInputFormat(pNumThreads);
        Configuration conf = this.getConfiguration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        conf.setBoolean("fs.test.impl.disable.cache", false);
        conf.set("mapreduce.input.fileinputformat.inputdir", "test:///a1/a2");
        MockFileSystem mockFs = (MockFileSystem)new Path("test:///").getFileSystem(conf);
        Assertions.assertEquals((int)0, (int)mockFs.numListLocatedStatusCalls, (String)"listLocatedStatus already called");
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fileInputFormat = new TextInputFormat();
        List splits = fileInputFormat.getSplits((JobContext)job);
        Assertions.assertEquals((int)2, (int)splits.size(), (String)"Input splits are not correct");
        Assertions.assertEquals((int)1, (int)mockFs.numListLocatedStatusCalls, (String)"listLocatedStatus calls");
        FileSystem.closeAll();
    }

    @MethodSource(value={"data"})
    @ParameterizedTest
    public void testSplitLocationInfo(int pNumThreads) throws Exception {
        this.initTestFileInputFormat(pNumThreads);
        Configuration conf = this.getConfiguration();
        conf.set("mapreduce.input.fileinputformat.inputdir", "test:///a1/a2");
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fileInputFormat = new TextInputFormat();
        List splits = fileInputFormat.getSplits((JobContext)job);
        String[] locations = ((InputSplit)splits.get(0)).getLocations();
        Assertions.assertEquals((int)2, (int)locations.length);
        SplitLocationInfo[] locationInfo = ((InputSplit)splits.get(0)).getLocationInfo();
        Assertions.assertEquals((int)2, (int)locationInfo.length);
        SplitLocationInfo localhostInfo = locations[0].equals("localhost") ? locationInfo[0] : locationInfo[1];
        SplitLocationInfo otherhostInfo = locations[0].equals("otherhost") ? locationInfo[0] : locationInfo[1];
        Assertions.assertTrue((boolean)localhostInfo.isOnDisk());
        Assertions.assertTrue((boolean)localhostInfo.isInMemory());
        Assertions.assertTrue((boolean)otherhostInfo.isOnDisk());
        Assertions.assertFalse((boolean)otherhostInfo.isInMemory());
    }

    @MethodSource(value={"data"})
    @ParameterizedTest
    public void testListStatusSimple(int pNumThreads) throws IOException {
        this.initTestFileInputFormat(pNumThreads);
        Configuration conf = new Configuration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        List<Path> expectedPaths = TestFileInputFormat.configureTestSimple(conf, localFs);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fif = new TextInputFormat();
        List statuses = fif.listStatus((JobContext)job);
        TestFileInputFormat.verifyFileStatuses(expectedPaths, statuses, localFs);
    }

    @MethodSource(value={"data"})
    @ParameterizedTest
    public void testListStatusNestedRecursive(int pNumThreads) throws IOException {
        this.initTestFileInputFormat(pNumThreads);
        Configuration conf = new Configuration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        List<Path> expectedPaths = TestFileInputFormat.configureTestNestedRecursive(conf, localFs);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fif = new TextInputFormat();
        List statuses = fif.listStatus((JobContext)job);
        TestFileInputFormat.verifyFileStatuses(expectedPaths, statuses, localFs);
    }

    @MethodSource(value={"data"})
    @ParameterizedTest
    public void testListStatusNestedNonRecursive(int pNumThreads) throws IOException {
        this.initTestFileInputFormat(pNumThreads);
        Configuration conf = new Configuration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        List<Path> expectedPaths = TestFileInputFormat.configureTestNestedNonRecursive(conf, localFs);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fif = new TextInputFormat();
        List statuses = fif.listStatus((JobContext)job);
        TestFileInputFormat.verifyFileStatuses(expectedPaths, statuses, localFs);
    }

    @MethodSource(value={"data"})
    @ParameterizedTest
    public void testListStatusErrorOnNonExistantDir(int pNumThreads) throws IOException {
        this.initTestFileInputFormat(pNumThreads);
        Configuration conf = new Configuration();
        conf.setInt("mapreduce.input.fileinputformat.list-status.num-threads", this.numThreads);
        TestFileInputFormat.configureTestErrorOnNonExistantDir(conf, localFs);
        Job job = Job.getInstance((Configuration)conf);
        TextInputFormat fif = new TextInputFormat();
        try {
            fif.listStatus((JobContext)job);
            Assertions.fail((String)"Expecting an IOException for a missing Input path");
        }
        catch (IOException e) {
            Path expectedExceptionPath = new Path(TEST_ROOT_DIR, "input2");
            expectedExceptionPath = localFs.makeQualified(expectedExceptionPath);
            Assertions.assertInstanceOf(InvalidInputException.class, (Object)e);
            Assertions.assertEquals((Object)("Input path does not exist: " + expectedExceptionPath.toString()), (Object)e.getMessage());
        }
    }

    @MethodSource(value={"data"})
    @ParameterizedTest
    public void testShrinkStatus(int pNumThreads) throws IOException {
        this.initTestFileInputFormat(pNumThreads);
        Configuration conf = this.getConfiguration();
        MockFileSystem mockFs = (MockFileSystem)new Path("test:///").getFileSystem(conf);
        Path dir1 = new Path("test:/a1");
        RemoteIterator statuses = mockFs.listLocatedStatus(dir1);
        boolean verified = false;
        while (statuses.hasNext()) {
            LocatedFileStatus orig = (LocatedFileStatus)statuses.next();
            LocatedFileStatus shrink = (LocatedFileStatus)FileInputFormat.shrinkStatus((FileStatus)orig);
            Assertions.assertTrue((boolean)orig.equals((Object)shrink));
            if (shrink.getBlockLocations() != null) {
                Assertions.assertEquals((int)orig.getBlockLocations().length, (int)shrink.getBlockLocations().length);
                for (int i = 0; i < shrink.getBlockLocations().length; ++i) {
                    verified = true;
                    BlockLocation location = shrink.getBlockLocations()[i];
                    BlockLocation actual = orig.getBlockLocations()[i];
                    Assertions.assertNotNull((Object)((HdfsBlockLocation)actual).getLocatedBlock());
                    Assertions.assertEquals((Object)BlockLocation.class.getName(), (Object)location.getClass().getName());
                    Assertions.assertArrayEquals((Object[])actual.getHosts(), (Object[])location.getHosts());
                    Assertions.assertArrayEquals((Object[])actual.getCachedHosts(), (Object[])location.getCachedHosts());
                    Assertions.assertArrayEquals((Object[])actual.getStorageIds(), (Object[])location.getStorageIds());
                    Assertions.assertArrayEquals((Object[])actual.getStorageTypes(), (Object[])location.getStorageTypes());
                    Assertions.assertArrayEquals((Object[])actual.getTopologyPaths(), (Object[])location.getTopologyPaths());
                    Assertions.assertArrayEquals((Object[])actual.getNames(), (Object[])location.getNames());
                    Assertions.assertEquals((long)actual.getLength(), (long)location.getLength());
                    Assertions.assertEquals((long)actual.getOffset(), (long)location.getOffset());
                    Assertions.assertEquals((Object)actual.isCorrupt(), (Object)location.isCorrupt());
                }
                continue;
            }
            Assertions.assertTrue((orig.getBlockLocations() == null ? 1 : 0) != 0);
        }
        Assertions.assertTrue((boolean)verified);
    }

    public static List<Path> configureTestSimple(Configuration conf, FileSystem localFs) throws IOException {
        Path base1 = new Path(TEST_ROOT_DIR, "input1");
        Path base2 = new Path(TEST_ROOT_DIR, "input2");
        conf.set("mapreduce.input.fileinputformat.inputdir", localFs.makeQualified(base1) + "," + localFs.makeQualified(base2));
        localFs.mkdirs(base1);
        localFs.mkdirs(base2);
        Path in1File1 = new Path(base1, "file1");
        Path in1File2 = new Path(base1, "file2");
        localFs.createNewFile(in1File1);
        localFs.createNewFile(in1File2);
        Path in2File1 = new Path(base2, "file1");
        Path in2File2 = new Path(base2, "file2");
        localFs.createNewFile(in2File1);
        localFs.createNewFile(in2File2);
        ArrayList expectedPaths = Lists.newArrayList((Object[])new Path[]{in1File1, in1File2, in2File1, in2File2});
        return expectedPaths;
    }

    public static List<Path> configureTestNestedRecursive(Configuration conf, FileSystem localFs) throws IOException {
        Path base1 = new Path(TEST_ROOT_DIR, "input1");
        conf.set("mapreduce.input.fileinputformat.inputdir", localFs.makeQualified(base1).toString());
        conf.setBoolean("mapreduce.input.fileinputformat.input.dir.recursive", true);
        localFs.mkdirs(base1);
        Path inDir1 = new Path(base1, "dir1");
        Path inDir2 = new Path(base1, "dir2");
        Path inFile1 = new Path(base1, "file1");
        Path dir1File1 = new Path(inDir1, "file1");
        Path dir1File2 = new Path(inDir1, "file2");
        Path dir2File1 = new Path(inDir2, "file1");
        Path dir2File2 = new Path(inDir2, "file2");
        localFs.mkdirs(inDir1);
        localFs.mkdirs(inDir2);
        localFs.createNewFile(inFile1);
        localFs.createNewFile(dir1File1);
        localFs.createNewFile(dir1File2);
        localFs.createNewFile(dir2File1);
        localFs.createNewFile(dir2File2);
        ArrayList expectedPaths = Lists.newArrayList((Object[])new Path[]{inFile1, dir1File1, dir1File2, dir2File1, dir2File2});
        return expectedPaths;
    }

    public static List<Path> configureTestNestedNonRecursive(Configuration conf, FileSystem localFs) throws IOException {
        Path base1 = new Path(TEST_ROOT_DIR, "input1");
        conf.set("mapreduce.input.fileinputformat.inputdir", localFs.makeQualified(base1).toString());
        conf.setBoolean("mapreduce.input.fileinputformat.input.dir.recursive", false);
        localFs.mkdirs(base1);
        Path inDir1 = new Path(base1, "dir1");
        Path inDir2 = new Path(base1, "dir2");
        Path inFile1 = new Path(base1, "file1");
        Path dir1File1 = new Path(inDir1, "file1");
        Path dir1File2 = new Path(inDir1, "file2");
        Path dir2File1 = new Path(inDir2, "file1");
        Path dir2File2 = new Path(inDir2, "file2");
        localFs.mkdirs(inDir1);
        localFs.mkdirs(inDir2);
        localFs.createNewFile(inFile1);
        localFs.createNewFile(dir1File1);
        localFs.createNewFile(dir1File2);
        localFs.createNewFile(dir2File1);
        localFs.createNewFile(dir2File2);
        ArrayList expectedPaths = Lists.newArrayList((Object[])new Path[]{inFile1, inDir1, inDir2});
        return expectedPaths;
    }

    public static List<Path> configureTestErrorOnNonExistantDir(Configuration conf, FileSystem localFs) throws IOException {
        Path base1 = new Path(TEST_ROOT_DIR, "input1");
        Path base2 = new Path(TEST_ROOT_DIR, "input2");
        conf.set("mapreduce.input.fileinputformat.inputdir", localFs.makeQualified(base1) + "," + localFs.makeQualified(base2));
        conf.setBoolean("mapreduce.input.fileinputformat.input.dir.recursive", true);
        localFs.mkdirs(base1);
        Path inFile1 = new Path(base1, "file1");
        Path inFile2 = new Path(base1, "file2");
        localFs.createNewFile(inFile1);
        localFs.createNewFile(inFile2);
        ArrayList expectedPaths = Lists.newArrayList();
        return expectedPaths;
    }

    public static void verifyFileStatuses(List<Path> expectedPaths, List<FileStatus> fetchedStatuses, FileSystem localFs) {
        Assertions.assertEquals((int)expectedPaths.size(), (int)fetchedStatuses.size());
        Iterable fqExpectedPaths = expectedPaths.stream().map(input -> localFs.makeQualified(input)).collect(Collectors.toList());
        HashSet expectedPathSet = Sets.newHashSet((Iterable)fqExpectedPaths);
        for (FileStatus fileStatus : fetchedStatuses) {
            if (expectedPathSet.remove(localFs.makeQualified(fileStatus.getPath()))) continue;
            Assertions.fail((String)("Found extra fetched status: " + fileStatus.getPath()));
        }
        Assertions.assertEquals((int)0, (int)expectedPathSet.size(), (String)("Not all expectedPaths matched: " + expectedPathSet));
    }

    private void verifySplits(List<String> expected, List<InputSplit> splits) {
        Iterable pathsFromSplits = splits.stream().map(input -> ((FileSplit)input).getPath().toString()).collect(Collectors.toList());
        HashSet expectedSet = Sets.newHashSet(expected);
        for (String splitPathString : pathsFromSplits) {
            if (expectedSet.remove(splitPathString)) continue;
            Assertions.fail((String)("Found extra split: " + splitPathString));
        }
        Assertions.assertEquals((int)0, (int)expectedSet.size(), (String)("Not all expectedPaths matched: " + expectedSet));
    }

    private Configuration getConfiguration() {
        Configuration conf = new Configuration();
        conf.set("fs.test.impl.disable.cache", "true");
        conf.setClass("fs.test.impl", MockFileSystem.class, FileSystem.class);
        conf.set("mapreduce.input.fileinputformat.inputdir", "test:///a1");
        return conf;
    }

    static class MockFileSystem
    extends RawLocalFileSystem {
        int numListLocatedStatusCalls = 0;

        MockFileSystem() {
        }

        public FileStatus[] listStatus(Path f) throws FileNotFoundException, IOException {
            if (f.toString().equals("test:/a1")) {
                return new FileStatus[]{new FileStatus(0L, true, 1, 150L, 150L, new Path("test:/a1/a2")), new FileStatus(10L, false, 1, 150L, 150L, new Path("test:/a1/file1"))};
            }
            if (f.toString().equals("test:/a1/a2")) {
                return new FileStatus[]{new FileStatus(10L, false, 1, 150L, 150L, new Path("test:/a1/a2/file2")), new FileStatus(10L, false, 1, 151L, 150L, new Path("test:/a1/a2/file3"))};
            }
            return new FileStatus[0];
        }

        public FileStatus[] globStatus(Path pathPattern, PathFilter filter) throws IOException {
            return new FileStatus[]{new FileStatus(10L, true, 1, 150L, 150L, pathPattern)};
        }

        public FileStatus[] listStatus(Path f, PathFilter filter) throws FileNotFoundException, IOException {
            return this.listStatus(f);
        }

        public BlockLocation[] getFileBlockLocations(FileStatus file, long start, long len) throws IOException {
            DatanodeInfo[] ds = new DatanodeInfo[]{new DatanodeDescriptor(new DatanodeID("127.0.0.1", "localhost", "abcd", 9866, 9867, 9868, 9869)), new DatanodeDescriptor(new DatanodeID("1.0.0.1", "otherhost", "efgh", 9866, 9867, 9868, 9869))};
            long blockLen = len / 3L;
            ExtendedBlock b1 = new ExtendedBlock("bpid", 0L, blockLen, 0L);
            ExtendedBlock b2 = new ExtendedBlock("bpid", 1L, blockLen, 1L);
            ExtendedBlock b3 = new ExtendedBlock("bpid", 2L, len - 2L * blockLen, 2L);
            String[] names = new String[]{"localhost:9866", "otherhost:9866"};
            String[] hosts = new String[]{"localhost", "otherhost"};
            String[] cachedHosts = new String[]{"localhost"};
            BlockLocation loc1 = new BlockLocation(names, hosts, cachedHosts, new String[0], 0L, blockLen, false);
            BlockLocation loc2 = new BlockLocation(names, hosts, cachedHosts, new String[0], blockLen, blockLen, false);
            BlockLocation loc3 = new BlockLocation(names, hosts, cachedHosts, new String[0], 2L * blockLen, len - 2L * blockLen, false);
            return new BlockLocation[]{new HdfsBlockLocation(loc1, new LocatedBlock(b1, ds)), new HdfsBlockLocation(loc2, new LocatedBlock(b2, ds)), new HdfsBlockLocation(loc3, new LocatedBlock(b3, ds))};
        }

        protected RemoteIterator<LocatedFileStatus> listLocatedStatus(Path f, PathFilter filter) throws FileNotFoundException, IOException {
            ++this.numListLocatedStatusCalls;
            return super.listLocatedStatus(f, filter);
        }
    }
}

