/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.mapreduce.lib.input;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.StringJoiner;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.compress.bzip2.BZip2TextFileWriter;
import org.apache.hadoop.io.compress.bzip2.BZip2Utils;
import org.apache.hadoop.mapreduce.lib.input.BaseLineRecordReaderHelper;
import org.apache.hadoop.util.Preconditions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public abstract class BaseTestLineRecordReaderBZip2 {
    private static final byte[] LF = new byte[]{10};
    private static final byte[] CR = new byte[]{13};
    private static final byte[] CR_LF = new byte[]{13, 10};
    private Configuration conf;
    private FileSystem fs;
    private Path tempFile;

    public Configuration getConf() {
        return this.conf;
    }

    public FileSystem getFs() {
        return this.fs;
    }

    public Path getTempFile() {
        return this.tempFile;
    }

    @BeforeEach
    public void setUp() throws Exception {
        this.conf = new Configuration();
        Path workDir = new Path(System.getProperty("test.build.data", "target"), "data/" + this.getClass().getSimpleName());
        this.fs = workDir.getFileSystem(this.conf);
        Path inputDir = new Path(workDir, "input");
        this.tempFile = new Path(inputDir, "test.txt.bz2");
    }

    @AfterEach
    public void tearDown() throws Exception {
        this.fs.delete(this.tempFile, false);
    }

    @Test
    public void firstBlockEndsWithLF() throws Exception {
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE, 1000, LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1001L, 2L});
    }

    @Test
    public void firstBlockEndsWithLFSecondBlockStartsWithLF() throws Exception {
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE, 1000, LF);
            writer.writeManyRecords(254, 254, LF);
            writer.writeRecord(1, LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1255L, 2L});
    }

    @Test
    public void firstBlockEndsWithLFSecondBlockStartsWithCR() throws Exception {
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE, 1000, LF);
            writer.writeRecord(1, CR);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1001L, 2L});
    }

    @Test
    public void firstBlockEndsWithCRLF() throws Exception {
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE, 1000, CR_LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1001L, 2L});
    }

    @Test
    public void lastRecordContentSpanAcrossBlocks() throws Exception {
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE - 50, 999, LF);
            writer.writeRecord(100, LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1000L, 3L});
    }

    @Test
    public void lastRecordOfBlockHasItsLFInNextBlock() throws Exception {
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE - 50, 999, LF);
            writer.writeRecord(51, LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1000L, 3L});
    }

    @Test
    public void lastRecordOfFirstBlockHasItsCRLFInSecondBlock() throws Exception {
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE - 50, 999, LF);
            writer.writeRecord(52, CR_LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1000L, 3L});
    }

    @Test
    public void lastRecordOfFirstBlockHasItsCRLFPartlyInSecondBlock() throws Exception {
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE - 50, 999, LF);
            writer.writeRecord(51, CR_LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1000L, 3L});
    }

    @Test
    public void lastByteInFirstBlockIsCRFirstByteInSecondBlockIsNotLF() throws Exception {
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE, 1000, CR);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
            writer.writeRecord(10, LF);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1001L, 2L});
    }

    @Test
    public void usingCRDelimiterWithSmallestBufferSize() throws Exception {
        this.conf.set("io.file.buffer.size", "1");
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE - 50, 999, CR);
            writer.writeRecord(100, CR);
            writer.writeRecord(10, CR);
            writer.writeRecord(10, CR);
            writer.writeRecord(10, CR);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1000L, 3L});
    }

    @Test
    public void delimitedByCRSpanningThreeBlocks() throws Exception {
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeRecord(3 * BZip2TextFileWriter.BLOCK_SIZE, CR);
            writer.writeRecord(3 * BZip2TextFileWriter.BLOCK_SIZE, CR);
            writer.writeRecord(3 * BZip2TextFileWriter.BLOCK_SIZE, CR);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1L, 0L, 1L, 0L, 0L, 1L, 0L, 0L, 0L});
    }

    @Test
    public void customDelimiterLastThreeBytesInBlockAreDelimiter() throws Exception {
        byte[] delimiter = new byte[]{101, 110, 100};
        this.setDelimiter(delimiter);
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE, 1000, delimiter);
            writer.writeRecord(10, delimiter);
            writer.writeRecord(10, delimiter);
            writer.writeRecord(10, delimiter);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1001L, 2L});
    }

    @Test
    public void customDelimiterDelimiterSpansAcrossBlocks() throws Exception {
        byte[] delimiter = new byte[]{101, 110, 100};
        this.setDelimiter(delimiter);
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE - 50, 999, delimiter);
            writer.writeRecord(52, delimiter);
            writer.writeRecord(10, delimiter);
            writer.writeRecord(10, delimiter);
            writer.writeRecord(10, delimiter);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1001L, 2L});
    }

    @Test
    public void customDelimiterLastRecordDelimiterStartsAtNextBlockStart() throws Exception {
        byte[] delimiter = new byte[]{101, 110, 100};
        this.setDelimiter(delimiter);
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE - 50, 999, delimiter);
            writer.writeRecord(53, delimiter);
            writer.writeRecord(10, delimiter);
            writer.writeRecord(10, delimiter);
            writer.writeRecord(10, delimiter);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1000L, 3L});
    }

    @Test
    public void customDelimiterLastBlockBytesShareCommonPrefixWithDelimiter() throws Exception {
        byte[] delimiter = new byte[]{101, 110, 100};
        this.setDelimiter(delimiter);
        try (BZip2TextFileWriter writer = new BZip2TextFileWriter(this.tempFile, this.conf);){
            writer.writeManyRecords(BZip2TextFileWriter.BLOCK_SIZE - 4, 999, delimiter);
            writer.write("an enchanting tale coming to an end");
            writer.writeRecord(10, delimiter);
            writer.writeRecord(10, delimiter);
            writer.writeRecord(10, delimiter);
        }
        this.assertRecordCountsPerSplit(this.tempFile, new long[]{1000L, 3L});
    }

    protected abstract BaseLineRecordReaderHelper newReader(Path var1);

    private void assertRecordCountsPerSplit(Path path, long[] countsIfSplitAtBlocks) throws IOException {
        RecordCountAssert countAssert = new RecordCountAssert(path, countsIfSplitAtBlocks);
        countAssert.assertSingleSplit();
        countAssert.assertSplittingAtBlocks();
        countAssert.assertSplittingJustAfterSecondBlockStarts();
        countAssert.assertSplittingEachBlockRangeInThreeParts();
        countAssert.assertSplitsAroundBlockStartOffsets();
    }

    private long getFileSize(Path path) throws IOException {
        return this.fs.getFileStatus(path).getLen();
    }

    private void setDelimiter(byte[] delimiter) {
        this.conf.set("textinputformat.record.delimiter", new String(delimiter, StandardCharsets.UTF_8));
    }

    private class RecordCountAssert {
        private final BaseLineRecordReaderHelper reader;
        private final long numBlocks;
        private final long[] countsIfSplitAtBlocks;
        private final long fileSize;
        private final long totalRecords;
        private final List<Long> nextBlockOffsets;

        RecordCountAssert(Path path, long[] countsIfSplitAtBlocks) throws IOException {
            this.reader = BaseTestLineRecordReaderBZip2.this.newReader(path);
            this.countsIfSplitAtBlocks = countsIfSplitAtBlocks;
            this.fileSize = BaseTestLineRecordReaderBZip2.this.getFileSize(path);
            this.totalRecords = Arrays.stream(countsIfSplitAtBlocks).sum();
            this.numBlocks = countsIfSplitAtBlocks.length;
            this.nextBlockOffsets = BZip2Utils.getNextBlockMarkerOffsets((Path)path, (Configuration)BaseTestLineRecordReaderBZip2.this.conf);
            Assertions.assertEquals((long)this.numBlocks, (long)(this.nextBlockOffsets.size() + 1));
        }

        private void assertSingleSplit() throws IOException {
            Assertions.assertEquals((long)this.totalRecords, (long)this.reader.countRecords(0L, this.fileSize));
        }

        private void assertSplittingAtBlocks() throws IOException {
            this.assertSplits(this.getSplitsAtBlocks());
        }

        private void assertSplittingJustAfterSecondBlockStarts() throws IOException {
            if (this.numBlocks <= 1L) {
                return;
            }
            long recordsInFirstTwoBlocks = this.countsIfSplitAtBlocks[0] + this.countsIfSplitAtBlocks[1];
            long remainingRecords = this.totalRecords - recordsInFirstTwoBlocks;
            long firstSplitSize = this.nextBlockOffsets.get(0) + 1L;
            Assertions.assertEquals((long)recordsInFirstTwoBlocks, (long)this.reader.countRecords(0L, firstSplitSize));
            Assertions.assertEquals((long)remainingRecords, (long)this.reader.countRecords(firstSplitSize, this.fileSize - firstSplitSize));
        }

        private void assertSplittingEachBlockRangeInThreeParts() throws IOException {
            for (SplitRange splitRange : this.getSplitsAtBlocks()) {
                long[] expectedNumRecordsPerPart = new long[]{splitRange.expectedNumRecords, 0L, 0L};
                List<SplitRange> parts = splitRange.divide(expectedNumRecordsPerPart);
                this.assertSplits(parts);
            }
        }

        private void assertSplitsAroundBlockStartOffsets() throws IOException {
            for (SplitRange split : this.getSplitsAtBlocks()) {
                this.assertSplit(split.withLength(1L));
                if (split.start > 0L) {
                    this.assertSplit(split.moveBy(-2L).withLength(3L));
                    this.assertSplit(split.moveBy(-2L).withLength(2L).withExpectedNumRecords(0L));
                    this.assertSplit(split.moveBy(-1L).withLength(2L));
                    this.assertSplit(split.moveBy(-1L).withLength(1L).withExpectedNumRecords(0L));
                }
                this.assertSplit(split.moveBy(1L).withLength(1L).withExpectedNumRecords(0L));
                this.assertSplit(split.moveBy(2L).withLength(1L).withExpectedNumRecords(0L));
            }
        }

        private List<SplitRange> getSplitsAtBlocks() {
            ArrayList<SplitRange> splits = new ArrayList<SplitRange>();
            int i = 0;
            while ((long)i < this.numBlocks) {
                String name = "Block" + i;
                long start = i == 0 ? 0L : this.nextBlockOffsets.get(i - 1);
                long end = (long)i == this.numBlocks - 1L ? this.fileSize : this.nextBlockOffsets.get(i);
                long length = end - start;
                long expectedNumRecords = this.countsIfSplitAtBlocks[i];
                splits.add(new SplitRange(name, start, length, expectedNumRecords));
                ++i;
            }
            return splits;
        }

        private void assertSplits(Iterable<SplitRange> splitRanges) throws IOException {
            for (SplitRange splitRange : splitRanges) {
                this.assertSplit(splitRange);
            }
        }

        private void assertSplit(SplitRange splitRange) throws IOException {
            String message = splitRange.toString();
            long actual = this.reader.countRecords(splitRange.start, splitRange.length);
            Assertions.assertEquals((long)splitRange.expectedNumRecords, (long)actual, (String)message);
        }
    }

    private static class SplitRange {
        private final String name;
        private final long start;
        private final long length;
        private final long expectedNumRecords;

        SplitRange(String name, long start, long length, long expectedNumRecords) {
            this.name = name;
            this.start = start;
            this.length = length;
            this.expectedNumRecords = expectedNumRecords;
        }

        public String toString() {
            return new StringJoiner(", ", SplitRange.class.getSimpleName() + "[", "]").add("name='" + this.name + "'").add("start=" + this.start).add("length=" + this.length).add("expectedNumRecords=" + this.expectedNumRecords).toString();
        }

        List<SplitRange> divide(long[] expectedNumRecordsPerPart) {
            int numParts = expectedNumRecordsPerPart.length;
            Preconditions.checkArgument((numParts > 0 ? 1 : 0) != 0);
            long minPartSize = this.length / (long)numParts;
            Preconditions.checkArgument((minPartSize > 0L ? 1 : 0) != 0);
            long lastPartExtraSize = this.length % (long)numParts;
            ArrayList<SplitRange> partRanges = new ArrayList<SplitRange>();
            long partStart = this.start;
            for (int i = 0; i < numParts; ++i) {
                String partName = this.name + "_Part" + i;
                long extraSize = i == numParts - 1 ? lastPartExtraSize : 0L;
                long partSize = minPartSize + extraSize;
                long partExpectedNumRecords = expectedNumRecordsPerPart[i];
                partRanges.add(new SplitRange(partName, partStart, partSize, partExpectedNumRecords));
                partStart += partSize;
            }
            return partRanges;
        }

        SplitRange withLength(long newLength) {
            return new SplitRange(this.name, this.start, newLength, this.expectedNumRecords);
        }

        SplitRange withExpectedNumRecords(long newExpectedNumRecords) {
            return new SplitRange(this.name, this.start, this.length, newExpectedNumRecords);
        }

        SplitRange moveBy(long delta) {
            return new SplitRange(this.name, this.start + delta, this.length, this.expectedNumRecords);
        }
    }
}

