/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Random;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;

public class TestByteBufferPread {
    private static MiniDFSCluster cluster;
    private static FileSystem fs;
    private static byte[] fileContents;
    private static Path testFile;
    private static Random rand;
    private static final long SEED = 3735928559L;
    private static final int BLOCK_SIZE = 4096;
    private static final int FILE_SIZE = 49152;

    @BeforeAll
    public static void setup() throws IOException {
        Configuration conf = new Configuration();
        conf.setLong("dfs.blocksize", 4096L);
        cluster = new MiniDFSCluster.Builder(conf).numDataNodes(3).build();
        fs = cluster.getFileSystem();
        fileContents = new byte[49152];
        rand = new Random(3735928559L);
        rand.nextBytes(fileContents);
        testFile = new Path("/byte-buffer-pread-test.dat");
        try (FSDataOutputStream out = fs.create(testFile, (short)3);){
            out.write(fileContents);
        }
    }

    @Test
    public void testPreadWithHeapByteBuffer() throws IOException {
        this.testPreadWithByteBuffer(ByteBuffer.allocate(49152));
        this.testPreadWithFullByteBuffer(ByteBuffer.allocate(49152));
        this.testPreadWithPositionedByteBuffer(ByteBuffer.allocate(49152));
        this.testPreadWithLimitedByteBuffer(ByteBuffer.allocate(49152));
        this.testPositionedPreadWithByteBuffer(ByteBuffer.allocate(49152));
        this.testPreadFullyWithByteBuffer(ByteBuffer.allocate(49152));
    }

    @Test
    public void testPreadWithDirectByteBuffer() throws IOException {
        this.testPreadWithByteBuffer(ByteBuffer.allocateDirect(49152));
        this.testPreadWithFullByteBuffer(ByteBuffer.allocateDirect(49152));
        this.testPreadWithPositionedByteBuffer(ByteBuffer.allocateDirect(49152));
        this.testPreadWithLimitedByteBuffer(ByteBuffer.allocateDirect(49152));
        this.testPositionedPreadWithByteBuffer(ByteBuffer.allocateDirect(49152));
        this.testPreadFullyWithByteBuffer(ByteBuffer.allocateDirect(49152));
    }

    private void testPreadWithByteBuffer(ByteBuffer buffer) throws IOException {
        int totalBytesRead = 0;
        try (FSDataInputStream in = fs.open(testFile);){
            int bytesRead;
            while ((bytesRead = in.read((long)totalBytesRead, buffer)) > 0) {
                Assertions.assertEquals((int)(totalBytesRead += bytesRead), (int)buffer.position());
            }
            Assertions.assertFalse((boolean)buffer.hasRemaining());
            buffer.position(0);
            byte[] bufferContents = new byte[49152];
            buffer.get(bufferContents);
            Assertions.assertArrayEquals((byte[])bufferContents, (byte[])fileContents);
        }
    }

    private void testPreadWithFullByteBuffer(ByteBuffer buffer) throws IOException {
        byte[] existingBufferBytes = new byte[49152];
        rand.nextBytes(existingBufferBytes);
        buffer.put(existingBufferBytes);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        try (FSDataInputStream in = fs.open(testFile);){
            Assertions.assertEquals((int)0, (int)in.read(buffer));
            Assertions.assertFalse((boolean)buffer.hasRemaining());
            buffer.position(0);
            byte[] bufferContents = new byte[49152];
            buffer.get(bufferContents);
            Assertions.assertArrayEquals((byte[])bufferContents, (byte[])existingBufferBytes);
        }
    }

    private void testPreadWithLimitedByteBuffer(ByteBuffer buffer) throws IOException {
        int totalBytesRead = 0;
        buffer.limit(24576);
        try (FSDataInputStream in = fs.open(testFile);){
            int bytesRead;
            while ((bytesRead = in.read((long)totalBytesRead, buffer)) > 0) {
                Assertions.assertEquals((int)(totalBytesRead += bytesRead), (int)buffer.position());
            }
            Assertions.assertEquals((int)totalBytesRead, (int)24576);
            Assertions.assertFalse((boolean)buffer.hasRemaining());
            buffer.position(0);
            byte[] bufferContents = new byte[24576];
            buffer.get(bufferContents);
            Assertions.assertArrayEquals((byte[])bufferContents, (byte[])Arrays.copyOfRange(fileContents, 0, 24576));
        }
    }

    private void testPreadWithPositionedByteBuffer(ByteBuffer buffer) throws IOException {
        int totalBytesRead = 0;
        buffer.position(24576);
        try (FSDataInputStream in = fs.open(testFile);){
            int bytesRead;
            while ((bytesRead = in.read((long)totalBytesRead, buffer)) > 0) {
                Assertions.assertEquals((int)((totalBytesRead += bytesRead) + 24576), (int)buffer.position());
            }
            Assertions.assertEquals((int)totalBytesRead, (int)24576);
            Assertions.assertFalse((boolean)buffer.hasRemaining());
            buffer.position(24576);
            byte[] bufferContents = new byte[24576];
            buffer.get(bufferContents);
            Assertions.assertArrayEquals((byte[])bufferContents, (byte[])Arrays.copyOfRange(fileContents, 0, 24576));
        }
    }

    private void testPositionedPreadWithByteBuffer(ByteBuffer buffer) throws IOException {
        int totalBytesRead = 0;
        try (FSDataInputStream in = fs.open(testFile);){
            int bytesRead;
            while ((bytesRead = in.read((long)(totalBytesRead + 24576), buffer)) > 0) {
                Assertions.assertEquals((int)(totalBytesRead += bytesRead), (int)buffer.position());
            }
            Assertions.assertEquals((int)totalBytesRead, (int)24576);
            Assertions.assertEquals((int)buffer.position(), (int)24576);
            Assertions.assertTrue((boolean)buffer.hasRemaining());
            buffer.position(0);
            byte[] bufferContents = new byte[24576];
            buffer.get(bufferContents);
            Assertions.assertArrayEquals((byte[])bufferContents, (byte[])Arrays.copyOfRange(fileContents, 24576, 49152));
        }
    }

    private void testPreadFullyWithByteBuffer(ByteBuffer buffer) throws IOException {
        int totalBytesRead = 0;
        try (FSDataInputStream in = fs.open(testFile);){
            in.readFully((long)totalBytesRead, buffer);
            Assertions.assertFalse((boolean)buffer.hasRemaining());
            buffer.position(0);
            byte[] bufferContents = new byte[49152];
            buffer.get(bufferContents);
            Assertions.assertArrayEquals((byte[])bufferContents, (byte[])fileContents);
        }
    }

    @AfterAll
    public static void shutdown() throws IOException {
        try {
            fs.delete(testFile, false);
            fs.close();
        }
        finally {
            cluster.shutdown(true);
        }
    }
}

