/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.IOException;
import java.util.concurrent.ThreadLocalRandom;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FsShell;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.SafeModeAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.AppendTestUtil;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManagerTestUtil;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.datanode.DataNodeFaultInjector;
import org.apache.hadoop.hdfs.server.datanode.FsDatasetTestUtils;
import org.apache.hadoop.hdfs.server.namenode.FSDirTruncateOp;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogLoader;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.NameNodeAdapter;
import org.apache.hadoop.hdfs.tools.DFSAdmin;
import org.apache.hadoop.hdfs.util.RwLockMode;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.test.LambdaTestUtils;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.assertj.core.api.AbstractBooleanAssert;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

public class TestFileTruncate {
    static final Logger LOG;
    static final int BLOCK_SIZE = 4;
    static final short REPLICATION = 3;
    static final int DATANODE_NUM = 3;
    static final int SUCCESS_ATTEMPTS = 300;
    static final int RECOVERY_ATTEMPTS = 600;
    static final long SLEEP = 100L;
    static final long LOW_SOFTLIMIT = 100L;
    static final long LOW_HARDLIMIT = 200L;
    static final int SHORT_HEARTBEAT = 1;
    static Configuration conf;
    static MiniDFSCluster cluster;
    static DistributedFileSystem fs;
    private Path parent;

    @BeforeEach
    public void setUp() throws IOException {
        conf = new HdfsConfiguration();
        conf.setLong("dfs.namenode.fs-limits.min-block-size", 4L);
        conf.setInt("dfs.bytes-per-checksum", 4);
        conf.setInt("dfs.heartbeat.interval", 1);
        conf.setLong("dfs.namenode.reconstruction.pending.timeout-sec", 1L);
        cluster = new MiniDFSCluster.Builder(conf).format(true).numDataNodes(3).waitSafeMode(true).build();
        fs = cluster.getFileSystem();
        this.parent = new Path("/test");
    }

    @AfterEach
    public void tearDown() throws IOException {
        if (fs != null) {
            fs.close();
            fs = null;
        }
        if (cluster != null) {
            cluster.shutdown();
            cluster = null;
        }
    }

    @Test
    public void testBasicTruncate() throws IOException {
        int startingFileSize = 12;
        fs.mkdirs(this.parent);
        fs.setQuota(this.parent, 100L, 1000L);
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        for (int fileLength = startingFileSize; fileLength > 0; fileLength -= 3) {
            for (int toTruncate = 0; toTruncate <= fileLength; ++toTruncate) {
                Path p = new Path(this.parent, "testBasicTruncate" + fileLength);
                TestFileTruncate.writeContents(contents, fileLength, p);
                int newLength = fileLength - toTruncate;
                org.junit.jupiter.api.Assertions.assertTrue((boolean)fs.hasPathCapability(p, "fs.capability.paths.truncate"), (String)"DFS supports truncate");
                boolean isReady = fs.truncate(p, (long)newLength);
                LOG.info("fileLength=" + fileLength + ", newLength=" + newLength + ", toTruncate=" + toTruncate + ", isReady=" + isReady);
                org.junit.jupiter.api.Assertions.assertEquals((Object)isReady, (Object)(toTruncate == 0 || newLength % 4 == 0 ? 1 : 0), (String)"File must be closed for zero truncate or truncating at the block boundary");
                if (!isReady) {
                    TestFileTruncate.checkBlockRecovery(p);
                }
                ContentSummary cs = fs.getContentSummary(this.parent);
                org.junit.jupiter.api.Assertions.assertEquals((long)cs.getSpaceConsumed(), (long)(newLength * 3), (String)"Bad disk space usage");
                TestFileTruncate.checkFullFile(p, newLength, contents);
            }
        }
        fs.delete(this.parent, true);
    }

    @Test
    public void testMultipleTruncate() throws IOException {
        Path dir = new Path("/testMultipleTruncate");
        fs.mkdirs(dir);
        Path p = new Path(dir, "file");
        byte[] data = new byte[400];
        ThreadLocalRandom.current().nextBytes(data);
        TestFileTruncate.writeContents(data, data.length, p);
        int n = data.length;
        while (n > 0) {
            int newLength = ThreadLocalRandom.current().nextInt(n);
            org.junit.jupiter.api.Assertions.assertTrue((boolean)fs.hasPathCapability(p, "fs.capability.paths.truncate"), (String)"DFS supports truncate");
            boolean isReady = fs.truncate(p, (long)newLength);
            LOG.info("newLength=" + newLength + ", isReady=" + isReady);
            org.junit.jupiter.api.Assertions.assertEquals((Object)isReady, (Object)(newLength % 4 == 0 ? 1 : 0), (String)"File must be closed for truncating at the block boundary");
            org.junit.jupiter.api.Assertions.assertEquals((Object)isReady, (Object)fs.truncate(p, (long)newLength), (String)"Truncate is not idempotent");
            if (!isReady) {
                TestFileTruncate.checkBlockRecovery(p);
            }
            TestFileTruncate.checkFullFile(p, newLength, data);
            n = newLength;
        }
        fs.delete(dir, true);
    }

    @Test
    public void testSnapshotTruncateThenDeleteSnapshot() throws IOException {
        Path dir = new Path("/testSnapshotTruncateThenDeleteSnapshot");
        fs.mkdirs(dir);
        fs.allowSnapshot(dir);
        Path p = new Path(dir, "file");
        byte[] data = new byte[4];
        ThreadLocalRandom.current().nextBytes(data);
        TestFileTruncate.writeContents(data, data.length, p);
        String snapshot = "s0";
        fs.createSnapshot(dir, "s0");
        Block lastBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock().getBlock().getLocalBlock();
        int newLength = data.length - 1;
        assert (newLength % 4 != 0) : " newLength must not be multiple of BLOCK_SIZE";
        org.junit.jupiter.api.Assertions.assertTrue((boolean)fs.hasPathCapability(p, "fs.capability.paths.truncate"), (String)"DFS supports truncate");
        boolean isReady = fs.truncate(p, (long)newLength);
        LOG.info("newLength=" + newLength + ", isReady=" + isReady);
        org.junit.jupiter.api.Assertions.assertEquals((Object)isReady, (Object)(newLength % 4 == 0 ? 1 : 0), (String)"File must be closed for truncating at the block boundary");
        fs.deleteSnapshot(dir, "s0");
        if (!isReady) {
            TestFileTruncate.checkBlockRecovery(p);
        }
        TestFileTruncate.checkFullFile(p, newLength, data);
        TestFileTruncate.assertBlockNotPresent(lastBlock);
        fs.delete(dir, true);
    }

    @Test
    @Timeout(value=90L)
    public void testTruncateTwiceTogether() throws Exception {
        Path dir = new Path("/testTruncateTwiceTogether");
        fs.mkdirs(dir);
        Path p = new Path(dir, "file");
        byte[] data = new byte[400];
        ThreadLocalRandom.current().nextBytes(data);
        TestFileTruncate.writeContents(data, data.length, p);
        DataNodeFaultInjector originInjector = DataNodeFaultInjector.get();
        DataNodeFaultInjector injector = new DataNodeFaultInjector(){

            public void delay() {
                try {
                    Thread.sleep(5000L);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
        };
        DataNodeFaultInjector.set((DataNodeFaultInjector)injector);
        Thread t = new Thread(() -> {
            String hdfsCacheDisableKey = "fs.hdfs.impl.disable.cache";
            boolean originCacheDisable = conf.getBoolean(hdfsCacheDisableKey, false);
            try {
                conf.setBoolean(hdfsCacheDisableKey, true);
                FileSystem fs1 = FileSystem.get((Configuration)conf);
                fs1.truncate(p, (long)(data.length - 1));
            }
            catch (IOException iOException) {
            }
            finally {
                conf.setBoolean(hdfsCacheDisableKey, originCacheDisable);
            }
        });
        t.start();
        t.join();
        NameNodeAdapter.getLeaseManager(cluster.getNamesystem()).setLeasePeriod(100L, 200L);
        LambdaTestUtils.intercept(RemoteException.class, (String)"/testTruncateTwiceTogether/file is being truncated", () -> fs.truncate(p, (long)(data.length - 2)));
        TestFileTruncate.checkBlockRecovery(p);
        TestFileTruncate.assertFileLength(p, data.length - 1);
        DataNodeFaultInjector.set((DataNodeFaultInjector)originInjector);
        NameNodeAdapter.getLeaseManager(cluster.getNamesystem()).setLeasePeriod(60000L, conf.getLong("dfs.namenode.lease-hard-limit-sec", 1200L) * 1000L);
        fs.delete(dir, true);
    }

    @Test
    public void testTruncateWithOtherOperations() throws IOException {
        Path dir = new Path("/testTruncateOtherOperations");
        fs.mkdirs(dir);
        Path p = new Path(dir, "file");
        byte[] data = new byte[8];
        ThreadLocalRandom.current().nextBytes(data);
        TestFileTruncate.writeContents(data, data.length, p);
        int newLength = data.length - 1;
        boolean isReady = fs.truncate(p, (long)newLength);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)isReady);
        fs.setReplication(p, (short)2);
        fs.setPermission(p, FsPermission.createImmutable((short)292));
        Path q = new Path(dir, "newFile");
        fs.rename(p, q);
        TestFileTruncate.checkBlockRecovery(q);
        TestFileTruncate.checkFullFile(q, newLength, data);
        cluster.restartNameNode(new String[0]);
        TestFileTruncate.checkFullFile(q, newLength, data);
        fs.delete(dir, true);
    }

    @Test
    public void testSnapshotWithAppendTruncate() throws IOException, InterruptedException {
        this.testSnapshotWithAppendTruncate(0, 1, 2);
        this.testSnapshotWithAppendTruncate(0, 2, 1);
        this.testSnapshotWithAppendTruncate(1, 0, 2);
        this.testSnapshotWithAppendTruncate(1, 2, 0);
        this.testSnapshotWithAppendTruncate(2, 0, 1);
        this.testSnapshotWithAppendTruncate(2, 1, 0);
    }

    void testSnapshotWithAppendTruncate(int ... deleteOrder) throws IOException, InterruptedException {
        FSDirectory fsDir = cluster.getNamesystem().getFSDirectory();
        fs.mkdirs(this.parent);
        fs.setQuota(this.parent, 100L, 1000L);
        fs.allowSnapshot(this.parent);
        String truncateFile = "testSnapshotWithAppendTruncate";
        Path src = new Path(this.parent, truncateFile);
        int[] length = new int[4];
        length[0] = 10;
        DFSTestUtil.createFile((FileSystem)fs, src, 64, length[0], 4L, (short)3, 0L);
        Block firstBlk = TestFileTruncate.getLocatedBlocks(src).get(0).getBlock().getLocalBlock();
        Path[] snapshotFiles = new Path[4];
        ContentSummary contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(30L);
        String[] ss = new String[]{"ss0", "ss1", "ss2", "ss3"};
        Path snapshotDir = fs.createSnapshot(this.parent, ss[0]);
        snapshotFiles[0] = new Path(snapshotDir, truncateFile);
        length[1] = length[2] = length[0] + 4 + 1;
        DFSTestUtil.appendFile((FileSystem)fs, src, 5);
        Block lastBlk = TestFileTruncate.getLocatedBlocks(src).getLastLocatedBlock().getBlock().getLocalBlock();
        contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(45L);
        snapshotDir = fs.createSnapshot(this.parent, ss[1]);
        snapshotFiles[1] = new Path(snapshotDir, truncateFile);
        snapshotDir = fs.createSnapshot(this.parent, ss[2]);
        snapshotFiles[2] = new Path(snapshotDir, truncateFile);
        DFSTestUtil.appendFile((FileSystem)fs, src, 5);
        Block appendedBlk = TestFileTruncate.getLocatedBlocks(src).getLastLocatedBlock().getBlock().getLocalBlock();
        contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(60L);
        int newLength = length[0] + 2;
        boolean isReady = fs.truncate(src, (long)newLength);
        BlockManagerTestUtil.waitForMarkedDeleteQueueIsEmpty(cluster.getNamesystem(0).getBlockManager());
        org.junit.jupiter.api.Assertions.assertTrue((boolean)isReady, (String)"Recovery is not expected.");
        TestFileTruncate.assertFileLength(snapshotFiles[2], length[2]);
        TestFileTruncate.assertFileLength(snapshotFiles[1], length[1]);
        TestFileTruncate.assertFileLength(snapshotFiles[0], length[0]);
        TestFileTruncate.assertBlockNotPresent(appendedBlk);
        contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(48L);
        newLength = length[0] - 2;
        isReady = fs.truncate(src, (long)newLength);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)isReady, (String)"Recovery is not expected.");
        TestFileTruncate.assertFileLength(snapshotFiles[2], length[2]);
        TestFileTruncate.assertFileLength(snapshotFiles[1], length[1]);
        TestFileTruncate.assertFileLength(snapshotFiles[0], length[0]);
        contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(48L);
        isReady = fs.truncate(src, (long)(newLength -= 2));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)isReady, (String)"Recovery is expected.");
        TestFileTruncate.checkBlockRecovery(src);
        TestFileTruncate.assertFileLength(snapshotFiles[2], length[2]);
        TestFileTruncate.assertFileLength(snapshotFiles[1], length[1]);
        TestFileTruncate.assertFileLength(snapshotFiles[0], length[0]);
        Block replacedBlk = TestFileTruncate.getLocatedBlocks(src).getLastLocatedBlock().getBlock().getLocalBlock();
        contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(54L);
        snapshotDir = fs.createSnapshot(this.parent, ss[3]);
        snapshotFiles[3] = new Path(snapshotDir, truncateFile);
        length[3] = newLength;
        int numINodes = fsDir.getInodeMapSize();
        isReady = fs.delete(src, false);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)isReady, (String)"Delete failed.");
        TestFileTruncate.assertFileLength(snapshotFiles[3], length[3]);
        TestFileTruncate.assertFileLength(snapshotFiles[2], length[2]);
        TestFileTruncate.assertFileLength(snapshotFiles[1], length[1]);
        TestFileTruncate.assertFileLength(snapshotFiles[0], length[0]);
        org.junit.jupiter.api.Assertions.assertEquals((int)numINodes, (int)fsDir.getInodeMapSize(), (String)"Number of INodes should not change");
        fs.deleteSnapshot(this.parent, ss[3]);
        BlockManagerTestUtil.waitForMarkedDeleteQueueIsEmpty(cluster.getNamesystem(0).getBlockManager());
        TestFileTruncate.assertBlockExists(firstBlk);
        TestFileTruncate.assertBlockExists(lastBlk);
        TestFileTruncate.assertBlockNotPresent(replacedBlk);
        contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(48L);
        fs.deleteSnapshot(this.parent, ss[deleteOrder[0]]);
        TestFileTruncate.assertFileLength(snapshotFiles[deleteOrder[1]], length[deleteOrder[1]]);
        TestFileTruncate.assertFileLength(snapshotFiles[deleteOrder[2]], length[deleteOrder[2]]);
        TestFileTruncate.assertBlockExists(firstBlk);
        TestFileTruncate.assertBlockExists(lastBlk);
        org.junit.jupiter.api.Assertions.assertEquals((int)numINodes, (int)fsDir.getInodeMapSize(), (String)"Number of INodes should not change");
        contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(48L);
        fs.deleteSnapshot(this.parent, ss[deleteOrder[1]]);
        BlockManagerTestUtil.waitForMarkedDeleteQueueIsEmpty(cluster.getNamesystem(0).getBlockManager());
        TestFileTruncate.assertFileLength(snapshotFiles[deleteOrder[2]], length[deleteOrder[2]]);
        TestFileTruncate.assertBlockExists(firstBlk);
        contentSummary = fs.getContentSummary(this.parent);
        if (fs.exists(snapshotFiles[0])) {
            TestFileTruncate.assertBlockNotPresent(lastBlk);
            Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(36L);
        } else {
            Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(48L);
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)numINodes, (int)fsDir.getInodeMapSize(), (String)"Number of INodes should not change");
        fs.deleteSnapshot(this.parent, ss[deleteOrder[2]]);
        BlockManagerTestUtil.waitForMarkedDeleteQueueIsEmpty(cluster.getNamesystem(0).getBlockManager());
        TestFileTruncate.assertBlockNotPresent(firstBlk);
        TestFileTruncate.assertBlockNotPresent(lastBlk);
        contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(0L);
        org.junit.jupiter.api.Assertions.assertNotEquals((int)numINodes, (int)fsDir.getInodeMapSize(), (String)"Number of INodes should change");
    }

    @Test
    public void testSnapshotWithTruncates() throws IOException, InterruptedException {
        this.testSnapshotWithTruncates(0, 1, 2);
        this.testSnapshotWithTruncates(0, 2, 1);
        this.testSnapshotWithTruncates(1, 0, 2);
        this.testSnapshotWithTruncates(1, 2, 0);
        this.testSnapshotWithTruncates(2, 0, 1);
        this.testSnapshotWithTruncates(2, 1, 0);
    }

    void testSnapshotWithTruncates(int ... deleteOrder) throws IOException, InterruptedException {
        fs.mkdirs(this.parent);
        fs.setQuota(this.parent, 100L, 1000L);
        fs.allowSnapshot(this.parent);
        String truncateFile = "testSnapshotWithTruncates";
        Path src = new Path(this.parent, truncateFile);
        int[] length = new int[3];
        length[0] = 12;
        DFSTestUtil.createFile((FileSystem)fs, src, 64, length[0], 4L, (short)3, 0L);
        Block firstBlk = TestFileTruncate.getLocatedBlocks(src).get(0).getBlock().getLocalBlock();
        Block lastBlk = TestFileTruncate.getLocatedBlocks(src).getLastLocatedBlock().getBlock().getLocalBlock();
        Path[] snapshotFiles = new Path[3];
        ContentSummary contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(36L);
        String[] ss = new String[]{"ss0", "ss1", "ss2"};
        Path snapshotDir = fs.createSnapshot(this.parent, ss[0]);
        snapshotFiles[0] = new Path(snapshotDir, truncateFile);
        length[1] = 8;
        boolean isReady = fs.truncate(src, 8L);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)isReady, (String)"Recovery is not expected.");
        contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(36L);
        snapshotDir = fs.createSnapshot(this.parent, ss[1]);
        snapshotFiles[1] = new Path(snapshotDir, truncateFile);
        length[2] = 6;
        isReady = fs.truncate(src, 6L);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)isReady, (String)"Recovery is expected.");
        TestFileTruncate.checkBlockRecovery(src);
        snapshotDir = fs.createSnapshot(this.parent, ss[2]);
        snapshotFiles[2] = new Path(snapshotDir, truncateFile);
        TestFileTruncate.assertFileLength(snapshotFiles[0], length[0]);
        TestFileTruncate.assertBlockExists(lastBlk);
        contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(42L);
        fs.deleteSnapshot(this.parent, ss[deleteOrder[0]]);
        BlockManagerTestUtil.waitForMarkedDeleteQueueIsEmpty(cluster.getNamesystem(0).getBlockManager());
        TestFileTruncate.assertFileLength(snapshotFiles[deleteOrder[1]], length[deleteOrder[1]]);
        TestFileTruncate.assertFileLength(snapshotFiles[deleteOrder[2]], length[deleteOrder[2]]);
        TestFileTruncate.assertFileLength(src, length[2]);
        TestFileTruncate.assertBlockExists(firstBlk);
        contentSummary = fs.getContentSummary(this.parent);
        if (fs.exists(snapshotFiles[0])) {
            Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(42L);
            TestFileTruncate.assertBlockExists(lastBlk);
        } else {
            Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(30L);
            TestFileTruncate.assertBlockNotPresent(lastBlk);
        }
        fs.deleteSnapshot(this.parent, ss[deleteOrder[1]]);
        BlockManagerTestUtil.waitForMarkedDeleteQueueIsEmpty(cluster.getNamesystem(0).getBlockManager());
        TestFileTruncate.assertFileLength(snapshotFiles[deleteOrder[2]], length[deleteOrder[2]]);
        TestFileTruncate.assertFileLength(src, length[2]);
        TestFileTruncate.assertBlockExists(firstBlk);
        contentSummary = fs.getContentSummary(this.parent);
        if (fs.exists(snapshotFiles[0])) {
            Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(42L);
            TestFileTruncate.assertBlockExists(lastBlk);
        } else if (fs.exists(snapshotFiles[1])) {
            Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(30L);
            TestFileTruncate.assertBlockNotPresent(lastBlk);
        } else {
            Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(18L);
            TestFileTruncate.assertBlockNotPresent(lastBlk);
        }
        fs.deleteSnapshot(this.parent, ss[deleteOrder[2]]);
        BlockManagerTestUtil.waitForMarkedDeleteQueueIsEmpty(cluster.getNamesystem(0).getBlockManager());
        TestFileTruncate.assertFileLength(src, length[2]);
        TestFileTruncate.assertBlockExists(firstBlk);
        contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(18L);
        Assertions.assertThat((long)contentSummary.getLength()).isEqualTo(6L);
        fs.delete(src, false);
        BlockManagerTestUtil.waitForMarkedDeleteQueueIsEmpty(cluster.getNamesystem().getBlockManager());
        TestFileTruncate.assertBlockNotPresent(firstBlk);
        contentSummary = fs.getContentSummary(this.parent);
        Assertions.assertThat((long)contentSummary.getSpaceConsumed()).isEqualTo(0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testTruncateFailure() throws IOException {
        int startingFileSize = 10;
        int toTruncate = 1;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path dir = new Path("/dir");
        Path p = new Path(dir, "testTruncateFailure");
        out.write(contents, 0, startingFileSize);
        try (FSDataOutputStream out = fs.create(p, false, 4, (short)3, 4L);){
            fs.truncate(p, 0L);
            org.junit.jupiter.api.Assertions.fail((String)"Truncate must fail on open file.");
        }
        out = fs.append(p);
        try {
            fs.truncate(p, 0L);
            org.junit.jupiter.api.Assertions.fail((String)"Truncate must fail for append.");
        }
        catch (IOException expected) {
            GenericTestUtils.assertExceptionContains((String)"Failed to TRUNCATE_FILE", (Throwable)expected);
        }
        finally {
            out.close();
        }
        try {
            fs.truncate(p, -1L);
            org.junit.jupiter.api.Assertions.fail((String)"Truncate must fail for a negative new length.");
        }
        catch (HadoopIllegalArgumentException expected) {
            GenericTestUtils.assertExceptionContains((String)"Cannot truncate to a negative file size", (Throwable)expected);
        }
        try {
            fs.truncate(p, (long)(startingFileSize + 1));
            org.junit.jupiter.api.Assertions.fail((String)"Truncate must fail for a larger new length.");
        }
        catch (Exception expected) {
            GenericTestUtils.assertExceptionContains((String)"Cannot truncate to a larger file size", (Throwable)expected);
        }
        try {
            fs.truncate(dir, 0L);
            org.junit.jupiter.api.Assertions.fail((String)"Truncate must fail for a directory.");
        }
        catch (Exception expected) {
            GenericTestUtils.assertExceptionContains((String)"Path is not a file", (Throwable)expected);
        }
        try {
            fs.truncate(new Path(dir, "non-existing"), 0L);
            org.junit.jupiter.api.Assertions.fail((String)"Truncate must fail for a non-existing file.");
        }
        catch (Exception expected) {
            GenericTestUtils.assertExceptionContains((String)"File does not exist", (Throwable)expected);
        }
        fs.setPermission(p, FsPermission.createImmutable((short)436));
        UserGroupInformation fooUgi = UserGroupInformation.createUserForTesting((String)"foo", (String[])new String[]{"foo"});
        try {
            FileSystem foofs = DFSTestUtil.getFileSystemAs(fooUgi, conf);
            foofs.truncate(p, 0L);
            org.junit.jupiter.api.Assertions.fail((String)"Truncate must fail for no WRITE permission.");
        }
        catch (Exception expected) {
            GenericTestUtils.assertExceptionContains((String)"Permission denied", (Throwable)expected);
        }
        cluster.shutdownDataNodes();
        NameNodeAdapter.getLeaseManager(cluster.getNamesystem()).setLeasePeriod(100L, 200L);
        int newLength = startingFileSize - toTruncate;
        boolean isReady = fs.truncate(p, (long)newLength);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)isReady).as("truncate should have triggered block recovery.", new Object[0])).isFalse();
        try {
            fs.truncate(p, 0L);
            org.junit.jupiter.api.Assertions.fail((String)"Truncate must fail since a truncate is already in progress.");
        }
        catch (IOException expected) {
            GenericTestUtils.assertExceptionContains((String)"Failed to TRUNCATE_FILE", (Throwable)expected);
        }
        boolean recoveryTriggered = false;
        for (int i = 0; i < 600; ++i) {
            String leaseHolder = NameNodeAdapter.getLeaseHolderForPath(cluster.getNameNode(), p.toUri().getPath());
            if (leaseHolder.startsWith("HDFS_NameNode")) {
                recoveryTriggered = true;
                break;
            }
            try {
                Thread.sleep(100L);
                continue;
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)recoveryTriggered).as("lease recovery should have occurred in ~60000 ms.", new Object[0])).isTrue();
        cluster.startDataNodes(conf, 3, true, HdfsServerConstants.StartupOption.REGULAR, null);
        cluster.waitActive();
        TestFileTruncate.checkBlockRecovery(p);
        NameNodeAdapter.getLeaseManager(cluster.getNamesystem()).setLeasePeriod(60000L, conf.getLong("dfs.namenode.lease-hard-limit-sec", 1200L) * 1000L);
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.delete(p, false);
    }

    @Test
    @Timeout(value=60L)
    public void testTruncateWithDataNodesRestart() throws Exception {
        int startingFileSize = 12;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path p = new Path(this.parent, "testTruncateWithDataNodesRestart");
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        LocatedBlock oldBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        int dn = 0;
        int toTruncateLength = 1;
        int newLength = startingFileSize - toTruncateLength;
        cluster.getDataNodes().get(dn).shutdown();
        this.truncateAndRestartDN(p, dn, newLength);
        TestFileTruncate.checkBlockRecovery(p);
        LocatedBlock newBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        org.junit.jupiter.api.Assertions.assertEquals((long)newBlock.getBlock().getBlockId(), (long)oldBlock.getBlock().getBlockId());
        org.junit.jupiter.api.Assertions.assertEquals((long)newBlock.getBlock().getGenerationStamp(), (long)(oldBlock.getBlock().getGenerationStamp() + 1L));
        Thread.sleep(2000L);
        cluster.triggerBlockReports();
        DFSTestUtil.waitReplication((FileSystem)fs, p, (short)3);
        FsDatasetTestUtils utils = cluster.getFsDatasetTestUtils(dn);
        org.junit.jupiter.api.Assertions.assertEquals((long)utils.getStoredDataLength(newBlock.getBlock()), (long)newBlock.getBlockSize());
        org.junit.jupiter.api.Assertions.assertEquals((long)utils.getStoredGenerationStamp(newBlock.getBlock()), (long)newBlock.getBlock().getGenerationStamp());
        FileStatus fileStatus = fs.getFileStatus(p);
        Assertions.assertThat((long)fileStatus.getLen()).isEqualTo((long)newLength);
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.delete(this.parent, true);
    }

    @Test
    @Timeout(value=60L)
    public void testCopyOnTruncateWithDataNodesRestart() throws Exception {
        int startingFileSize = 12;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path p = new Path(this.parent, "testCopyOnTruncateWithDataNodesRestart");
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        LocatedBlock oldBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        fs.allowSnapshot(this.parent);
        fs.createSnapshot(this.parent, "ss0");
        int dn = 1;
        int toTruncateLength = 1;
        int newLength = startingFileSize - toTruncateLength;
        cluster.getDataNodes().get(dn).shutdown();
        this.truncateAndRestartDN(p, dn, newLength);
        TestFileTruncate.checkBlockRecovery(p);
        LocatedBlock newBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        org.junit.jupiter.api.Assertions.assertNotEquals((long)newBlock.getBlock().getBlockId(), (long)oldBlock.getBlock().getBlockId());
        org.junit.jupiter.api.Assertions.assertEquals((long)newBlock.getBlock().getGenerationStamp(), (long)(oldBlock.getBlock().getGenerationStamp() + 1L));
        DFSTestUtil.waitReplication((FileSystem)fs, p, (short)3);
        FsDatasetTestUtils utils = cluster.getFsDatasetTestUtils(dn);
        org.junit.jupiter.api.Assertions.assertEquals((long)utils.getStoredDataLength(newBlock.getBlock()), (long)newBlock.getBlockSize());
        org.junit.jupiter.api.Assertions.assertEquals((long)utils.getStoredDataLength(oldBlock.getBlock()), (long)oldBlock.getBlockSize());
        org.junit.jupiter.api.Assertions.assertEquals((long)utils.getStoredGenerationStamp(oldBlock.getBlock()), (long)oldBlock.getBlock().getGenerationStamp());
        FileStatus fileStatus = fs.getFileStatus(p);
        Assertions.assertThat((long)fileStatus.getLen()).isEqualTo((long)newLength);
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.deleteSnapshot(this.parent, "ss0");
        fs.delete(this.parent, true);
    }

    @Test
    @Timeout(value=60L)
    public void testTruncateWithDataNodesRestartImmediately() throws Exception {
        int startingFileSize = 12;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path p = new Path(this.parent, "testTruncateWithDataNodesRestartImmediately");
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        LocatedBlock oldBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        int dn0 = 0;
        int dn1 = 1;
        int toTruncateLength = 1;
        int newLength = startingFileSize - toTruncateLength;
        boolean isReady = fs.truncate(p, (long)newLength);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)isReady);
        cluster.restartDataNode(dn0, false, true);
        cluster.restartDataNode(dn1, false, true);
        cluster.waitActive();
        TestFileTruncate.checkBlockRecovery(p);
        LocatedBlock newBlock = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock();
        org.junit.jupiter.api.Assertions.assertEquals((long)newBlock.getBlock().getBlockId(), (long)oldBlock.getBlock().getBlockId());
        org.junit.jupiter.api.Assertions.assertEquals((long)newBlock.getBlock().getGenerationStamp(), (long)(oldBlock.getBlock().getGenerationStamp() + 1L));
        Thread.sleep(2000L);
        cluster.triggerBlockReports();
        DFSTestUtil.waitReplication((FileSystem)fs, p, (short)3);
        FsDatasetTestUtils utils = cluster.getFsDatasetTestUtils(dn0);
        org.junit.jupiter.api.Assertions.assertEquals((long)utils.getStoredDataLength(newBlock.getBlock()), (long)newBlock.getBlockSize());
        org.junit.jupiter.api.Assertions.assertEquals((long)utils.getStoredGenerationStamp(newBlock.getBlock()), (long)newBlock.getBlock().getGenerationStamp());
        utils = cluster.getFsDatasetTestUtils(dn1);
        org.junit.jupiter.api.Assertions.assertEquals((long)utils.getStoredDataLength(newBlock.getBlock()), (long)newBlock.getBlockSize());
        org.junit.jupiter.api.Assertions.assertEquals((long)utils.getStoredGenerationStamp(newBlock.getBlock()), (long)newBlock.getBlock().getGenerationStamp());
        FileStatus fileStatus = fs.getFileStatus(p);
        Assertions.assertThat((long)fileStatus.getLen()).isEqualTo((long)newLength);
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.delete(this.parent, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    @Timeout(value=60L)
    public void testTruncateWithDataNodesShutdownImmediately() throws Exception {
        int startingFileSize = 12;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        Path p = new Path(this.parent, "testTruncateWithDataNodesShutdownImmediately");
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        int toTruncateLength = 1;
        int newLength = startingFileSize - toTruncateLength;
        boolean isReady = fs.truncate(p, (long)newLength);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)isReady);
        cluster.shutdownDataNodes();
        cluster.setDataNodesDead();
        try {
            for (int i = 0; i < 300 && cluster.isDataNodeUp(); ++i) {
                Thread.sleep(100L);
            }
            org.junit.jupiter.api.Assertions.assertFalse((boolean)cluster.isDataNodeUp(), (String)"All DataNodes should be down.");
            LocatedBlocks blocks = TestFileTruncate.getLocatedBlocks(p);
            org.junit.jupiter.api.Assertions.assertTrue((boolean)blocks.isUnderConstruction());
        }
        finally {
            cluster.startDataNodes(conf, 3, true, HdfsServerConstants.StartupOption.REGULAR, null);
            cluster.waitActive();
        }
        TestFileTruncate.checkBlockRecovery(p);
        fs.delete(this.parent, true);
    }

    @Test
    public void testTruncateEditLogLoad() throws IOException {
        fs.setSafeMode(SafeModeAction.ENTER);
        fs.saveNamespace();
        fs.setSafeMode(SafeModeAction.LEAVE);
        int startingFileSize = 10;
        int toTruncate = 1;
        String s = "/testTruncateEditLogLoad";
        Path p = new Path("/testTruncateEditLogLoad");
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        int newLength = startingFileSize - toTruncate;
        boolean isReady = fs.truncate(p, (long)newLength);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)isReady).as("truncate should have triggered block recovery.", new Object[0])).isFalse();
        cluster.restartNameNode(new String[0]);
        String holder = UserGroupInformation.getCurrentUser().getUserName();
        cluster.getNamesystem().recoverLease("/testTruncateEditLogLoad", holder, "");
        TestFileTruncate.checkBlockRecovery(p);
        TestFileTruncate.checkFullFile(p, newLength, contents);
        fs.delete(p, false);
    }

    @Test
    public void testUpgradeAndRestart() throws IOException {
        fs.mkdirs(this.parent);
        fs.setQuota(this.parent, 100L, 1000L);
        fs.allowSnapshot(this.parent);
        String truncateFile = "testUpgrade";
        Path p = new Path(this.parent, truncateFile);
        int startingFileSize = 8;
        int toTruncate = 1;
        byte[] contents = AppendTestUtil.initBuffer(startingFileSize);
        TestFileTruncate.writeContents(contents, startingFileSize, p);
        Path snapshotDir = fs.createSnapshot(this.parent, "ss0");
        Path snapshotFile = new Path(snapshotDir, truncateFile);
        int newLengthBeforeUpgrade = startingFileSize - toTruncate;
        boolean isReady = fs.truncate(p, (long)newLengthBeforeUpgrade);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)isReady).as("truncate should have triggered block recovery.", new Object[0])).isFalse();
        TestFileTruncate.checkBlockRecovery(p);
        TestFileTruncate.checkFullFile(p, newLengthBeforeUpgrade, contents);
        TestFileTruncate.assertFileLength(snapshotFile, startingFileSize);
        long totalBlockBefore = cluster.getNamesystem().getBlocksTotal();
        TestFileTruncate.restartCluster(HdfsServerConstants.StartupOption.UPGRADE);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)cluster.getNamesystem().isInSafeMode()).as("SafeMode should be OFF", new Object[0])).isFalse();
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)cluster.getNamesystem().isUpgradeFinalized()).as("NameNode should be performing upgrade.", new Object[0])).isFalse();
        FileStatus fileStatus = fs.getFileStatus(p);
        Assertions.assertThat((long)fileStatus.getLen()).isEqualTo((long)newLengthBeforeUpgrade);
        int newLengthAfterUpgrade = newLengthBeforeUpgrade - toTruncate;
        Block oldBlk = TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock().getBlock().getLocalBlock();
        isReady = fs.truncate(p, (long)newLengthAfterUpgrade);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)isReady).as("truncate should have triggered block recovery.", new Object[0])).isFalse();
        fileStatus = fs.getFileStatus(p);
        Assertions.assertThat((long)fileStatus.getLen()).isEqualTo((long)newLengthAfterUpgrade);
        ((AbstractLongAssert)Assertions.assertThat((long)TestFileTruncate.getLocatedBlocks(p).getLastLocatedBlock().getBlock().getLocalBlock().getBlockId()).as("Should copy on truncate during upgrade", new Object[0])).isNotEqualTo(oldBlk.getBlockId());
        TestFileTruncate.checkBlockRecovery(p);
        TestFileTruncate.checkFullFile(p, newLengthAfterUpgrade, contents);
        ((AbstractLongAssert)Assertions.assertThat((long)cluster.getNamesystem().getBlocksTotal()).as("Total block count should be unchanged from copy-on-truncate", new Object[0])).isEqualTo(totalBlockBefore);
        TestFileTruncate.restartCluster(HdfsServerConstants.StartupOption.ROLLBACK);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)fs.exists(p)).as("File does not exist " + p, new Object[0])).isTrue();
        fileStatus = fs.getFileStatus(p);
        Assertions.assertThat((long)fileStatus.getLen()).isEqualTo((long)newLengthBeforeUpgrade);
        TestFileTruncate.checkFullFile(p, newLengthBeforeUpgrade, contents);
        ((AbstractLongAssert)Assertions.assertThat((long)cluster.getNamesystem().getBlocksTotal()).as("Total block count should be unchanged from rolling back", new Object[0])).isEqualTo(totalBlockBefore);
        TestFileTruncate.restartCluster(HdfsServerConstants.StartupOption.REGULAR);
        ((AbstractLongAssert)Assertions.assertThat((long)cluster.getNamesystem().getBlocksTotal()).as("Total block count should be unchanged from start-up", new Object[0])).isEqualTo(totalBlockBefore);
        TestFileTruncate.checkFullFile(p, newLengthBeforeUpgrade, contents);
        TestFileTruncate.assertFileLength(snapshotFile, startingFileSize);
        fs.setSafeMode(SafeModeAction.ENTER);
        fs.saveNamespace();
        cluster.restartNameNode(true);
        ((AbstractLongAssert)Assertions.assertThat((long)cluster.getNamesystem().getBlocksTotal()).as("Total block count should be unchanged from start-up", new Object[0])).isEqualTo(totalBlockBefore);
        TestFileTruncate.checkFullFile(p, newLengthBeforeUpgrade, contents);
        TestFileTruncate.assertFileLength(snapshotFile, startingFileSize);
        fs.deleteSnapshot(this.parent, "ss0");
        fs.delete(this.parent, true);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)fs.exists(p)).as("File " + p + " shouldn't exist", new Object[0])).isFalse();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testTruncateRecovery() throws IOException {
        long blockRecoveryId;
        Block truncateBlock;
        BlockInfo oldBlock;
        FSNamesystem fsn = cluster.getNamesystem();
        String client = "client";
        String clientMachine = "clientMachine";
        String src = "/test/testTruncateRecovery";
        Path srcPath = new Path(src);
        byte[] contents = AppendTestUtil.initBuffer(4);
        TestFileTruncate.writeContents(contents, 4, srcPath);
        INodesInPath iip = fsn.getFSDirectory().getINodesInPath(src, FSDirectory.DirOp.WRITE);
        INodeFile file = iip.getLastINode().asFile();
        long initialGenStamp = file.getLastBlock().getGenerationStamp();
        fsn.writeLock(RwLockMode.GLOBAL);
        try {
            oldBlock = file.getLastBlock();
            truncateBlock = FSDirTruncateOp.prepareFileForTruncate((FSNamesystem)fsn, (INodesInPath)iip, (String)client, (String)clientMachine, (long)1L, null);
            Assertions.assertThat((long)truncateBlock.getBlockId()).isEqualTo(oldBlock.getBlockId());
            Assertions.assertThat((long)truncateBlock.getNumBytes()).isEqualTo(oldBlock.getNumBytes());
            Assertions.assertThat((long)truncateBlock.getGenerationStamp()).isEqualTo(fsn.getBlockManager().getBlockIdManager().getGenerationStamp());
            Assertions.assertThat((Comparable)file.getLastBlock().getBlockUCState()).isEqualTo((Object)HdfsServerConstants.BlockUCState.UNDER_RECOVERY);
            blockRecoveryId = file.getLastBlock().getUnderConstructionFeature().getBlockRecoveryId();
            Assertions.assertThat((long)blockRecoveryId).isEqualTo(initialGenStamp + 1L);
            fsn.getEditLog().logTruncate(src, client, clientMachine, 3L, Time.now(), truncateBlock);
        }
        finally {
            fsn.writeUnlock(RwLockMode.GLOBAL, "testTruncateRecovery");
        }
        TestFileTruncate.writeContents(contents, 4, srcPath);
        fs.allowSnapshot(this.parent);
        fs.createSnapshot(this.parent, "ss0");
        iip = fsn.getFSDirectory().getINodesInPath(src, FSDirectory.DirOp.WRITE);
        file = iip.getLastINode().asFile();
        file.recordModification(iip.getLatestSnapshotId(), true);
        Assertions.assertThat((boolean)file.isBlockInLatestSnapshot((BlockInfo)((BlockInfoContiguous)file.getLastBlock()))).isEqualTo(true);
        initialGenStamp = file.getLastBlock().getGenerationStamp();
        fsn.writeLock(RwLockMode.GLOBAL);
        try {
            oldBlock = file.getLastBlock();
            truncateBlock = FSDirTruncateOp.prepareFileForTruncate((FSNamesystem)fsn, (INodesInPath)iip, (String)client, (String)clientMachine, (long)1L, null);
            Assertions.assertThat((long)truncateBlock.getBlockId()).isNotEqualTo(oldBlock.getBlockId());
            Assertions.assertThat((truncateBlock.getNumBytes() < oldBlock.getNumBytes() ? 1 : 0) != 0).isEqualTo(true);
            Assertions.assertThat((long)truncateBlock.getGenerationStamp()).isEqualTo(fsn.getBlockManager().getBlockIdManager().getGenerationStamp());
            Assertions.assertThat((Comparable)file.getLastBlock().getBlockUCState()).isEqualTo((Object)HdfsServerConstants.BlockUCState.UNDER_RECOVERY);
            blockRecoveryId = file.getLastBlock().getUnderConstructionFeature().getBlockRecoveryId();
            Assertions.assertThat((long)blockRecoveryId).isEqualTo(initialGenStamp + 1L);
            fsn.getEditLog().logTruncate(src, client, clientMachine, 3L, Time.now(), truncateBlock);
        }
        finally {
            fsn.writeUnlock(RwLockMode.GLOBAL, "testTruncateRecovery");
        }
        TestFileTruncate.checkBlockRecovery(srcPath);
        fs.deleteSnapshot(this.parent, "ss0");
        fs.delete(this.parent, true);
    }

    @Test
    public void testTruncateShellCommand() throws Exception {
        Path src = new Path("/test/testTruncateShellCommand");
        int oldLength = 9;
        int newLength = 5;
        String[] argv = new String[]{"-truncate", String.valueOf(5), src.toString()};
        this.runTruncateShellCommand(src, 9, argv);
        TestFileTruncate.checkBlockRecovery(src);
        Assertions.assertThat((long)fs.getFileStatus(src).getLen()).isEqualTo(5L);
        fs.delete(this.parent, true);
    }

    @Test
    public void testTruncateShellCommandOnBlockBoundary() throws Exception {
        Path src = new Path("/test/testTruncateShellCommandOnBoundary");
        int oldLength = 8;
        int newLength = 4;
        String[] argv = new String[]{"-truncate", String.valueOf(4), src.toString()};
        this.runTruncateShellCommand(src, 8, argv);
        Assertions.assertThat((long)fs.getFileStatus(src).getLen()).isEqualTo(4L);
        fs.delete(this.parent, true);
    }

    @Test
    public void testTruncateShellCommandWithWaitOption() throws Exception {
        Path src = new Path("/test/testTruncateShellCommandWithWaitOption");
        int oldLength = 9;
        int newLength = 5;
        String[] argv = new String[]{"-truncate", "-w", String.valueOf(5), src.toString()};
        this.runTruncateShellCommand(src, 9, argv);
        Assertions.assertThat((long)fs.getFileStatus(src).getLen()).isEqualTo(5L);
        fs.delete(this.parent, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void runTruncateShellCommand(Path src, int oldLength, String[] shellOpts) throws Exception {
        TestFileTruncate.writeContents(AppendTestUtil.initBuffer(oldLength), oldLength, src);
        Assertions.assertThat((long)fs.getFileStatus(src).getLen()).isEqualTo((long)oldLength);
        try (FsShell shell = null;){
            shell = new FsShell(conf);
            Assertions.assertThat((int)ToolRunner.run((Tool)shell, (String[])shellOpts)).isEqualTo(0);
        }
    }

    @Test
    public void testTruncate4Symlink() throws IOException {
        int fileLength = 12;
        fs.mkdirs(this.parent);
        byte[] contents = AppendTestUtil.initBuffer(12);
        Path file = new Path(this.parent, "testTruncate4Symlink");
        TestFileTruncate.writeContents(contents, 12, file);
        Path link = new Path(this.parent, "link");
        fs.createSymlink(file, link, false);
        int newLength = 4;
        boolean isReady = fs.truncate(link, 4L);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)isReady, (String)"Recovery is not expected.");
        FileStatus fileStatus = fs.getFileStatus(file);
        Assertions.assertThat((long)fileStatus.getLen()).isEqualTo(4L);
        ContentSummary cs = fs.getContentSummary(this.parent);
        org.junit.jupiter.api.Assertions.assertEquals((long)cs.getSpaceConsumed(), (long)12L, (String)"Bad disk space usage");
        TestFileTruncate.checkFullFile(file, 4, contents);
        fs.delete(this.parent, true);
    }

    @Test
    public void testTruncateWithRollingUpgrade() throws Exception {
        DFSAdmin dfsadmin = new DFSAdmin(cluster.getConfiguration(0));
        DistributedFileSystem dfs = cluster.getFileSystem();
        dfs.setSafeMode(SafeModeAction.ENTER);
        int status = dfsadmin.run(new String[]{"-rollingUpgrade", "prepare"});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)status, (String)"could not prepare for rolling upgrade");
        dfs.setSafeMode(SafeModeAction.LEAVE);
        Path dir = new Path("/testTruncateWithRollingUpgrade");
        fs.mkdirs(dir);
        Path p = new Path(dir, "file");
        byte[] data = new byte[3];
        ThreadLocalRandom.current().nextBytes(data);
        TestFileTruncate.writeContents(data, data.length, p);
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)cluster.getNamesystem().getFSDirectory().getBlockManager().getTotalBlocks(), (String)"block num should 1");
        boolean isReady = fs.truncate(p, 2L);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)isReady, (String)"should be copy-on-truncate");
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)cluster.getNamesystem().getFSDirectory().getBlockManager().getTotalBlocks(), (String)"block num should 2");
        fs.delete(p, true);
        BlockManagerTestUtil.waitForMarkedDeleteQueueIsEmpty(cluster.getNamesystem().getBlockManager());
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)cluster.getNamesystem().getFSDirectory().getBlockManager().getTotalBlocks(), (String)"block num should 0");
        status = dfsadmin.run(new String[]{"-rollingUpgrade", "finalize"});
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)status, (String)"could not finalize rolling upgrade");
    }

    static void writeContents(byte[] contents, int fileLength, Path p) throws IOException {
        FSDataOutputStream out = fs.create(p, true, 4, (short)3, 4L);
        out.write(contents, 0, fileLength);
        out.close();
    }

    static void checkBlockRecovery(Path p) throws IOException {
        TestFileTruncate.checkBlockRecovery(p, fs);
    }

    public static void checkBlockRecovery(Path p, DistributedFileSystem dfs) throws IOException {
        TestFileTruncate.checkBlockRecovery(p, dfs, 300, 100L);
    }

    public static void checkBlockRecovery(Path p, DistributedFileSystem dfs, int attempts, long sleepMs) throws IOException {
        boolean success = false;
        for (int i = 0; i < attempts; ++i) {
            boolean noLastBlock;
            LocatedBlocks blocks = TestFileTruncate.getLocatedBlocks(p, dfs);
            boolean bl = noLastBlock = blocks.getLastLocatedBlock() == null;
            if (!blocks.isUnderConstruction() && (noLastBlock || blocks.isLastBlockComplete())) {
                success = true;
                break;
            }
            try {
                Thread.sleep(sleepMs);
                continue;
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)success).as("inode should complete in ~" + sleepMs * (long)attempts + " ms.", new Object[0])).isTrue();
    }

    static LocatedBlocks getLocatedBlocks(Path src) throws IOException {
        return TestFileTruncate.getLocatedBlocks(src, fs);
    }

    static LocatedBlocks getLocatedBlocks(Path src, DistributedFileSystem dfs) throws IOException {
        return dfs.getClient().getLocatedBlocks(src.toString(), 0L, Long.MAX_VALUE);
    }

    static void assertBlockExists(Block blk) {
        org.junit.jupiter.api.Assertions.assertNotNull((Object)cluster.getNamesystem().getStoredBlock(blk), (String)("BlocksMap does not contain block: " + blk));
    }

    static void assertBlockNotPresent(Block blk) {
        org.junit.jupiter.api.Assertions.assertNull((Object)cluster.getNamesystem().getStoredBlock(blk), (String)("BlocksMap should not contain block: " + blk));
    }

    static void assertFileLength(Path file, long length) throws IOException {
        byte[] data = DFSTestUtil.readFileBuffer((FileSystem)fs, file);
        org.junit.jupiter.api.Assertions.assertEquals((long)length, (long)data.length, (String)"Wrong data size in snapshot.");
    }

    static void checkFullFile(Path p, int newLength, byte[] contents) throws IOException {
        AppendTestUtil.checkFullFile((FileSystem)fs, p, newLength, contents, p.toString());
    }

    static void restartCluster(HdfsServerConstants.StartupOption o) throws IOException {
        cluster.shutdown();
        if (HdfsServerConstants.StartupOption.ROLLBACK == o) {
            NameNode.doRollback((Configuration)conf, (boolean)false);
        }
        cluster = new MiniDFSCluster.Builder(conf).numDataNodes(3).format(false).startupOption(o == HdfsServerConstants.StartupOption.ROLLBACK ? HdfsServerConstants.StartupOption.REGULAR : o).dnStartupOption(o != HdfsServerConstants.StartupOption.ROLLBACK ? HdfsServerConstants.StartupOption.REGULAR : o).build();
        fs = cluster.getFileSystem();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void truncateAndRestartDN(Path p, int dn, int newLength) throws IOException {
        try {
            boolean isReady = fs.truncate(p, (long)newLength);
            org.junit.jupiter.api.Assertions.assertFalse((boolean)isReady);
        }
        finally {
            cluster.restartDataNode(dn, false, true);
            cluster.waitActive();
        }
    }

    @Test
    public void testQuotaOnTruncateWithSnapshot() throws Exception {
        Path root = new Path("/");
        Path dirPath = new Path(root, "dir");
        org.junit.jupiter.api.Assertions.assertTrue((boolean)fs.mkdirs(dirPath));
        Path filePath = new Path(dirPath, "file");
        DFSTestUtil.createFile((FileSystem)fs, filePath, 10L, (short)3, 0L);
        fs.allowSnapshot(dirPath);
        fs.createSnapshot(dirPath, "s1");
        org.junit.jupiter.api.Assertions.assertEquals((long)fs.getContentSummary(root).getSpaceConsumed(), (long)fs.getQuotaUsage(root).getSpaceConsumed());
        boolean blockrecovery = fs.truncate(filePath, 5L);
        if (!blockrecovery) {
            TestFileTruncate.checkBlockRecovery(filePath, fs, 300, 100L);
        }
        org.junit.jupiter.api.Assertions.assertEquals((long)fs.getContentSummary(root).getSpaceConsumed(), (long)fs.getQuotaUsage(root).getSpaceConsumed());
        fs.deleteSnapshot(dirPath, "s1");
        org.junit.jupiter.api.Assertions.assertEquals((long)fs.getContentSummary(root).getSpaceConsumed(), (long)fs.getQuotaUsage(root).getSpaceConsumed());
    }

    @Test
    public void testConcatOnInodeRefernce() throws IOException {
        String dir = "/testConcat";
        Path trgDir = new Path(dir);
        fs.mkdirs(new Path(dir), FsPermission.getDirDefault());
        Path trg = new Path(dir, "file");
        DFSTestUtil.createFile((FileSystem)fs, trg, 512L, (short)2, 0L);
        String dir2 = "/dir2";
        Path srcDir = new Path(dir2);
        fs.mkdirs(srcDir);
        fs.allowSnapshot(srcDir);
        Path src = new Path(srcDir, "file1");
        DFSTestUtil.createFile((FileSystem)fs, src, 512L, (short)2, 0L);
        fs.createSnapshot(srcDir, "s1");
        fs.rename(src, trgDir);
        fs.deleteSnapshot(srcDir, "s1");
        Path[] srcs = new Path[]{new Path(dir, "file1")};
        org.junit.jupiter.api.Assertions.assertEquals((long)2L, (long)fs.getContentSummary(new Path(dir)).getFileCount());
        fs.concat(trg, srcs);
        org.junit.jupiter.api.Assertions.assertEquals((long)1L, (long)fs.getContentSummary(new Path(dir)).getFileCount());
    }

    @Test
    public void testQuotaSpaceConsumedWithSnapshots() throws IOException {
        Path root = new Path("/");
        Path dir = new Path(root, "dir");
        fs.mkdirs(dir);
        fs.allowSnapshot(dir);
        Path file2 = new Path(dir, "file2");
        DFSTestUtil.createFile((FileSystem)fs, file2, 30L, (short)1, 0L);
        fs.createSnapshot(dir, "s1");
        boolean isReady = fs.truncate(file2, 20L);
        if (!isReady) {
            TestFileTruncate.checkBlockRecovery(file2);
        }
        fs.createSnapshot(dir, "s2");
        isReady = fs.truncate(file2, 10L);
        if (!isReady) {
            TestFileTruncate.checkBlockRecovery(file2);
        }
        fs.deleteSnapshot(dir, "s1");
        fs.deleteSnapshot(dir, "s2");
        org.junit.jupiter.api.Assertions.assertEquals((long)fs.getContentSummary(root).getSpaceConsumed(), (long)fs.getQuotaUsage(root).getSpaceConsumed());
        fs.delete(dir, true);
        org.junit.jupiter.api.Assertions.assertEquals((long)fs.getContentSummary(root).getSpaceConsumed(), (long)fs.getQuotaUsage(root).getSpaceConsumed());
    }

    @Test
    public void testTruncatewithRenameandSnapshot() throws Exception {
        Path dir = new Path("/dir");
        fs.mkdirs(dir, new FsPermission(511));
        Path file = new Path(dir, "file");
        Path movedFile = new Path("/file");
        DFSTestUtil.createFile((FileSystem)fs, file, 10L, (short)3, 0L);
        fs.allowSnapshot(dir);
        Path snapshotPath = fs.createSnapshot(dir, "s0");
        org.junit.jupiter.api.Assertions.assertTrue((boolean)fs.exists(snapshotPath));
        fs.rename(file, new Path("/"));
        boolean isReady = fs.truncate(movedFile, 5L);
        if (!isReady) {
            TestFileTruncate.checkBlockRecovery(movedFile);
        }
        FileStatus fileStatus = fs.getFileStatus(movedFile);
        org.junit.jupiter.api.Assertions.assertEquals((long)5L, (long)fileStatus.getLen());
        LocatedBlocks locations = fs.getClient().getNamenode().getBlockLocations("/dir/.snapshot/s0/file", 0L, 10L);
        org.junit.jupiter.api.Assertions.assertEquals((long)10L, (long)locations.get(0).getBlockSize());
    }

    static {
        GenericTestUtils.setLogLevel((Logger)NameNode.stateChangeLog, (Level)Level.TRACE);
        GenericTestUtils.setLogLevel((Logger)FSEditLogLoader.LOG, (Level)Level.TRACE);
        LOG = LoggerFactory.getLogger(TestFileTruncate.class);
    }
}

