/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.IOException;
import java.util.EnumSet;
import java.util.function.Supplier;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.CryptoProtocolVersion;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSOutputStream;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.ViewDistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.BlockType;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.DatanodeInfoWithStorage;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.HdfsFileStatus;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockUnderConstructionFeature;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.DataNodeTestUtils;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.TestInterDatanodeProtocol;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.LeaseManager;
import org.apache.hadoop.hdfs.server.namenode.NameNodeAdapter;
import org.apache.hadoop.io.EnumSetWritable;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.util.DataChecksum;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class TestLeaseRecovery {
    static final int BLOCK_SIZE = 1024;
    static final short REPLICATION_NUM = 3;
    private static final long LEASE_PERIOD = 300L;
    private MiniDFSCluster cluster;

    @AfterEach
    public void shutdown() throws IOException {
        if (this.cluster != null) {
            this.cluster.shutdown();
            this.cluster = null;
        }
    }

    static void checkMetaInfo(ExtendedBlock b, DataNode dn) throws IOException {
        TestInterDatanodeProtocol.checkMetaInfo(b, dn);
    }

    static int min(Integer ... x) {
        int m = x[0];
        for (int i = 1; i < x.length; ++i) {
            if (x[i] >= m) continue;
            m = x[i];
        }
        return m;
    }

    void waitLeaseRecovery(MiniDFSCluster cluster) {
        cluster.setLeasePeriod(300L, 300L);
        try {
            Thread.sleep(6000L);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    @Test
    public void testBlockSynchronization() throws Exception {
        int ORG_FILE_SIZE = 3000;
        HdfsConfiguration conf = new HdfsConfiguration();
        conf.setLong("dfs.blocksize", 1024L);
        this.cluster = new MiniDFSCluster.Builder((Configuration)conf).numDataNodes(5).build();
        this.cluster.waitActive();
        DistributedFileSystem dfs = this.cluster.getFileSystem();
        String filestr = "/foo";
        Path filepath = new Path(filestr);
        DFSTestUtil.createFile((FileSystem)dfs, filepath, 3000L, (short)3, 0L);
        Assertions.assertTrue((boolean)dfs.exists(filepath));
        DFSTestUtil.waitReplication((FileSystem)dfs, filepath, (short)3);
        LocatedBlock locatedblock = TestInterDatanodeProtocol.getLastLocatedBlock(dfs.dfs.getNamenode(), filestr);
        DatanodeInfoWithStorage[] datanodeinfos = locatedblock.getLocations();
        Assertions.assertEquals((int)3, (int)datanodeinfos.length);
        DataNode[] datanodes = new DataNode[3];
        for (int i = 0; i < 3; ++i) {
            datanodes[i] = this.cluster.getDataNode(datanodeinfos[i].getIpcPort());
            Assertions.assertTrue((datanodes[i] != null ? 1 : 0) != 0);
        }
        ExtendedBlock lastblock = locatedblock.getBlock();
        DataNode.LOG.info("newblocks=" + lastblock);
        for (int i = 0; i < 3; ++i) {
            TestLeaseRecovery.checkMetaInfo(lastblock, datanodes[i]);
        }
        DataNode.LOG.info("dfs.dfs.clientName=" + dfs.dfs.clientName);
        this.cluster.getNameNodeRpc().append(filestr, dfs.dfs.clientName, new EnumSetWritable(EnumSet.of(CreateFlag.APPEND)));
        this.waitLeaseRecovery(this.cluster);
        Block[] updatedmetainfo = new Block[3];
        long oldSize = lastblock.getNumBytes();
        lastblock = TestInterDatanodeProtocol.getLastLocatedBlock(dfs.dfs.getNamenode(), filestr).getBlock();
        long currentGS = lastblock.getGenerationStamp();
        for (int i = 0; i < 3; ++i) {
            updatedmetainfo[i] = DataNodeTestUtils.getFSDataset(datanodes[i]).getStoredBlock(lastblock.getBlockPoolId(), lastblock.getBlockId());
            Assertions.assertEquals((long)lastblock.getBlockId(), (long)updatedmetainfo[i].getBlockId());
            Assertions.assertEquals((long)oldSize, (long)updatedmetainfo[i].getNumBytes());
            Assertions.assertEquals((long)currentGS, (long)updatedmetainfo[i].getGenerationStamp());
        }
        System.out.println("Testing that lease recovery cannot happen during safemode.");
        filestr = "/foo.safemode";
        filepath = new Path(filestr);
        dfs.create(filepath, (short)1);
        this.cluster.getNameNodeRpc().setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_ENTER, false);
        Assertions.assertTrue((boolean)dfs.dfs.exists(filestr));
        DFSTestUtil.waitReplication((FileSystem)dfs, filepath, (short)1);
        this.waitLeaseRecovery(this.cluster);
        LeaseManager lm = NameNodeAdapter.getLeaseManager(this.cluster.getNamesystem());
        Assertions.assertTrue((lm.countLease() == 1 ? 1 : 0) != 0, (String)("Found " + lm.countLease() + " lease, expected 1"));
        this.cluster.getNameNodeRpc().setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_LEAVE, false);
    }

    @Test
    public void testBlockRecoveryWithLessMetafile() throws Exception {
        int count;
        Configuration conf = new Configuration();
        conf.set("dfs.block.local-path-access.user", UserGroupInformation.getCurrentUser().getShortUserName());
        this.cluster = new MiniDFSCluster.Builder(conf).numDataNodes(1).build();
        Path file = new Path("/testRecoveryFile");
        DistributedFileSystem dfs = this.cluster.getFileSystem();
        FSDataOutputStream out = dfs.create(file);
        int FILE_SIZE = 0x200000;
        for (count = 0; count < 0x200000; count += 4) {
            out.writeBytes("Data");
        }
        out.hsync();
        ((DFSOutputStream)out.getWrappedStream()).abort();
        LocatedBlocks locations = this.cluster.getNameNodeRpc().getBlockLocations(file.toString(), 0L, (long)count);
        ExtendedBlock block = locations.get(0).getBlock();
        int CHECKSUM_SIZE = 4;
        int bytesPerChecksum = conf.getInt("dfs.bytes-per-checksum", 512);
        int metaFileSize = (0x200000 + bytesPerChecksum - 1) / bytesPerChecksum * 4 + 8;
        int newMetaFileSize = metaFileSize - 4;
        this.cluster.truncateMeta(0, block, newMetaFileSize);
        MiniDFSCluster.DataNodeProperties dnProp = this.cluster.stopDataNode(0);
        this.cluster.restartDataNode(dnProp, true);
        DistributedFileSystem newdfs = (DistributedFileSystem)FileSystem.newInstance((Configuration)this.cluster.getConfiguration(0));
        count = 0;
        while (++count < 10 && !newdfs.recoverLease(file)) {
            Thread.sleep(1000L);
        }
        Assertions.assertTrue((boolean)newdfs.recoverLease(file), (String)"File should be closed");
        long expectedNewFileLen = 0x200000 - bytesPerChecksum;
        long newFileLen = newdfs.getFileStatus(file).getLen();
        Assertions.assertEquals((long)newFileLen, (long)expectedNewFileLen);
    }

    @Test
    public void testBlockRecoveryRetryAfterFailedRecovery() throws Exception {
        int count;
        Configuration conf = new Configuration();
        this.cluster = new MiniDFSCluster.Builder(conf).numDataNodes(3).build();
        Path file = new Path("/testBlockRecoveryRetryAfterFailedRecovery");
        DistributedFileSystem dfs = this.cluster.getFileSystem();
        FSDataOutputStream out = dfs.create(file);
        int FILE_SIZE = 131072;
        for (count = 0; count < 131072; count += 8) {
            out.writeBytes("DE K9SUL");
        }
        out.hsync();
        ((DFSOutputStream)out.getWrappedStream()).abort();
        LocatedBlocks locations = this.cluster.getNameNodeRpc().getBlockLocations(file.toString(), 0L, (long)count);
        ExtendedBlock block = locations.get(0).getBlock();
        this.cluster.getDataNodes().get(0).getFSDataset().finalizeBlock(block, false);
        this.cluster.deleteMeta(0, block);
        DistributedFileSystem newDfs = (DistributedFileSystem)FileSystem.newInstance((Configuration)this.cluster.getConfiguration(0));
        count = 0;
        while (count++ < 15 && !newDfs.recoverLease(file)) {
            Thread.sleep(1000L);
        }
        Assertions.assertTrue((boolean)newDfs.recoverLease(file), (String)"File should be closed");
    }

    @Test
    public void testLeaseRecoveryAndAppend() throws Exception {
        this.testLeaseRecoveryAndAppend(new Configuration());
    }

    @Test
    public void testLeaseRecoveryAndAppendWithViewDFS() throws Exception {
        Configuration conf = new Configuration();
        conf.set("fs.hdfs.impl", ViewDistributedFileSystem.class.getName());
        this.testLeaseRecoveryAndAppend(conf);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testLeaseRecoveryAndAppend(Configuration conf) throws Exception {
        try {
            this.cluster = new MiniDFSCluster.Builder(conf).numDataNodes(1).build();
            Path file = new Path("/testLeaseRecovery");
            DistributedFileSystem dfs = this.cluster.getFileSystem();
            FSDataOutputStream out = dfs.create(file);
            out.hflush();
            out.hsync();
            ((DFSOutputStream)out.getWrappedStream()).abort();
            DistributedFileSystem newdfs = (DistributedFileSystem)FileSystem.newInstance((Configuration)this.cluster.getConfiguration(0));
            try {
                newdfs.append(file);
                Assertions.fail((String)"Append to a file(lease is held by another client) should fail");
            }
            catch (RemoteException e) {
                Assertions.assertTrue((boolean)e.getMessage().contains("file lease is currently owned"));
            }
            boolean recoverLease = newdfs.recoverLease(file);
            Assertions.assertTrue((boolean)recoverLease);
            FSDataOutputStream append = newdfs.append(file);
            append.write("test".getBytes());
            append.close();
        }
        finally {
            if (this.cluster != null) {
                this.cluster.shutdown();
                this.cluster = null;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLeaseRecoveryEmptyCommittedLastBlock() throws Exception {
        Configuration conf = new Configuration();
        DFSClient client = null;
        try {
            this.cluster = new MiniDFSCluster.Builder(conf).numDataNodes(1).build();
            DistributedFileSystem dfs = this.cluster.getFileSystem();
            client = new DFSClient(this.cluster.getNameNode().getServiceRpcAddress(), conf);
            String file = "/test/f1";
            Path filePath = new Path(file);
            this.createCommittedNotCompleteFile(client, file, null, 1);
            INodeFile inode = this.cluster.getNamesystem().getFSDirectory().getINode(filePath.toString()).asFile();
            Assertions.assertTrue((boolean)inode.isUnderConstruction());
            Assertions.assertEquals((int)1, (int)inode.numBlocks());
            Assertions.assertNotNull((Object)inode.getLastBlock());
            try {
                dfs.append(filePath);
                Assertions.fail((String)"Append to a file(lease is held by another client) should fail");
            }
            catch (RemoteException e) {
                Assertions.assertTrue((boolean)e.getMessage().contains("file lease is currently owned"));
            }
            Assertions.assertEquals((Object)false, (Object)client.recoverLease(file));
            for (int i = 0; i < 10 && !client.recoverLease(file); ++i) {
                Thread.sleep(1000L);
            }
            Assertions.assertTrue((boolean)client.recoverLease(file));
            inode = this.cluster.getNamesystem().getFSDirectory().getINode(filePath.toString()).asFile();
            Assertions.assertTrue((!inode.isUnderConstruction() ? 1 : 0) != 0);
            Assertions.assertEquals((int)0, (int)inode.numBlocks());
            Assertions.assertNull((Object)inode.getLastBlock());
            FSDataOutputStream append = dfs.append(filePath);
            append.write("test".getBytes());
            append.close();
        }
        finally {
            if (this.cluster != null) {
                this.cluster.shutdown();
                this.cluster = null;
            }
            if (client != null) {
                client.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLeaseManagerRecoversEmptyCommittedLastBlock() throws Exception {
        Configuration conf = new Configuration();
        DFSClient client = null;
        try {
            this.cluster = new MiniDFSCluster.Builder(conf).numDataNodes(1).build();
            client = new DFSClient(this.cluster.getNameNode().getServiceRpcAddress(), conf);
            String file = "/test/f1";
            this.createCommittedNotCompleteFile(client, file, null, 1);
            this.waitLeaseRecovery(this.cluster);
            GenericTestUtils.waitFor(() -> {
                String holder = NameNodeAdapter.getLeaseHolderForPath(this.cluster.getNameNode(), file);
                return holder == null;
            }, (long)100L, (long)10000L);
        }
        finally {
            if (this.cluster != null) {
                this.cluster.shutdown();
                this.cluster = null;
            }
            if (client != null) {
                client.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testAbortedRecovery() throws Exception {
        Configuration conf = new Configuration();
        DFSClient client = null;
        try {
            this.cluster = new MiniDFSCluster.Builder(conf).numDataNodes(1).build();
            client = new DFSClient(this.cluster.getNameNode().getServiceRpcAddress(), conf);
            String file = "/test/f1";
            HdfsFileStatus stat = client.getNamenode().create("/test/f1", new FsPermission("777"), client.clientName, new EnumSetWritable(EnumSet.of(CreateFlag.CREATE)), true, (short)1, 0x8000000L, new CryptoProtocolVersion[0], null, null);
            Assertions.assertNotNull((Object)NameNodeAdapter.getLeaseHolderForPath(this.cluster.getNameNode(), "/test/f1"));
            ExtendedBlock block = client.getNamenode().addBlock("/test/f1", client.clientName, null, DatanodeInfo.EMPTY_ARRAY, stat.getFileId(), new String[0], null).getBlock();
            ExtendedBlock updatedBlock = client.getNamenode().updateBlockForPipeline(block, client.clientName).getBlock();
            updatedBlock.setNumBytes(1234L);
            BlockManager bm = this.cluster.getNamesystem().getBlockManager();
            BlockInfo storedBlock = bm.getStoredBlock(block.getLocalBlock());
            BlockUnderConstructionFeature uc = storedBlock.getUnderConstructionFeature();
            uc.setExpectedLocations(updatedBlock.getLocalBlock(), uc.getExpectedStorageLocations(), BlockType.CONTIGUOUS);
            client.getNamenode().complete("/test/f1", client.clientName, block, stat.getFileId());
            Assertions.assertNotNull((Object)NameNodeAdapter.getLeaseHolderForPath(this.cluster.getNameNode(), "/test/f1"));
            this.cluster.setLeasePeriod(300L, 300L);
            GenericTestUtils.waitFor((Supplier)new Supplier<Boolean>(){

                @Override
                public Boolean get() {
                    String holder = NameNodeAdapter.getLeaseHolderForPath(TestLeaseRecovery.this.cluster.getNameNode(), "/test/f1");
                    return holder == null;
                }
            }, (long)100L, (long)20000L);
            Assertions.assertTrue((boolean)storedBlock.isDeleted());
        }
        finally {
            if (this.cluster != null) {
                this.cluster.shutdown();
                this.cluster = null;
            }
            if (client != null) {
                client.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLeaseManagerRecoversCommittedLastBlockWithContent() throws Exception {
        Configuration conf = new Configuration();
        DFSClient client = null;
        try {
            this.cluster = new MiniDFSCluster.Builder(conf).numDataNodes(3).build();
            client = new DFSClient(this.cluster.getNameNode().getServiceRpcAddress(), conf);
            String file = "/test/f2";
            byte[] bytesToWrite = new byte[]{123};
            this.createCommittedNotCompleteFile(client, file, bytesToWrite, 3);
            this.waitLeaseRecovery(this.cluster);
            DistributedFileSystem hdfs = this.cluster.getFileSystem();
            try (FSDataOutputStream op = null;){
                op = hdfs.append(new Path(file));
                op.write(23);
            }
            try (FSDataInputStream stream = null;){
                stream = this.cluster.getFileSystem().open(new Path(file));
                Assertions.assertEquals((int)123, (int)stream.readByte());
                Assertions.assertEquals((int)23, (int)stream.readByte());
            }
            GenericTestUtils.waitFor(() -> {
                String holder = NameNodeAdapter.getLeaseHolderForPath(this.cluster.getNameNode(), file);
                return holder == null;
            }, (long)100L, (long)10000L);
        }
        finally {
            if (this.cluster != null) {
                this.cluster.shutdown();
                this.cluster = null;
            }
            if (client != null) {
                client.close();
            }
        }
    }

    private void createCommittedNotCompleteFile(DFSClient client, String file, byte[] bytesToWrite, int repFactor) throws IOException {
        HdfsFileStatus stat = client.getNamenode().create(file, new FsPermission("777"), client.clientName, new EnumSetWritable(EnumSet.of(CreateFlag.CREATE)), true, (short)repFactor, 0x8000000L, new CryptoProtocolVersion[0], null, null);
        LocatedBlock blk = client.getNamenode().addBlock(file, client.clientName, null, DatanodeInfo.EMPTY_ARRAY, stat.getFileId(), new String[0], null);
        ExtendedBlock finalBlock = blk.getBlock();
        if (bytesToWrite != null) {
            DFSOutputStream s = new DFSOutputStream(client, file, stat, EnumSet.of(CreateFlag.CREATE), null, DataChecksum.newDataChecksum((DataChecksum.Type)DataChecksum.Type.CRC32C, (int)512), null, true);
            s.start();
            s.write(bytesToWrite);
            s.hflush();
            finalBlock = s.getBlock();
            s.abort();
        }
        boolean closed = client.getNamenode().complete(file, client.clientName, finalBlock, stat.getFileId());
        Assertions.assertEquals((Object)false, (Object)closed);
    }
}

