/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.security.PrivilegedExceptionAction;
import java.util.EnumSet;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FsServerDefaults;
import org.apache.hadoop.fs.InvalidPathException;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.AppendTestUtil;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSOutputStream;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.client.HdfsDataOutputStream;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.DatanodeInfoWithStorage;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManagerTestUtil;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.DataNodeTestUtils;
import org.apache.hadoop.hdfs.server.datanode.SimulatedFSDataset;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.LeaseManager;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.NameNodeAdapter;
import org.apache.hadoop.hdfs.server.namenode.NameNodeAdapterMockitoUtil;
import org.apache.hadoop.hdfs.server.protocol.NamenodeProtocols;
import org.apache.hadoop.io.EnumSetWritable;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.metrics2.MetricsRecordBuilder;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.test.MetricsAsserts;
import org.apache.hadoop.test.ReflectionUtils;
import org.apache.hadoop.util.Time;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.Timeout;
import org.mockito.Mockito;
import org.slf4j.Logger;
import org.slf4j.event.Level;

public class TestFileCreation {
    static final String DIR = "/" + TestFileCreation.class.getSimpleName() + "/";
    private static final String RPC_DETAILED_METRICS = "RpcDetailedActivityForPort";
    static final long seed = 3735928559L;
    static final int blockSize = 8192;
    static final int numBlocks = 2;
    static final int fileSize = 16385;
    boolean simulatedStorage;
    private static final String[] NON_CANONICAL_PATHS = new String[]{"//foo", "///foo2", "//dir//file", "////test2/file", "/dir/./file2", "/dir/../file3"};

    public TestFileCreation() {
        GenericTestUtils.setLogLevel((Logger)LeaseManager.LOG, (Level)Level.TRACE);
        GenericTestUtils.setLogLevel((Logger)FSNamesystem.LOG, (Level)Level.TRACE);
        GenericTestUtils.setLogLevel((Logger)DFSClient.LOG, (Level)Level.TRACE);
        this.simulatedStorage = false;
    }

    public static FSDataOutputStream createFile(FileSystem fileSys, Path name, int repl) throws IOException {
        System.out.println("createFile: Created " + name + " with " + repl + " replica.");
        FSDataOutputStream stm = fileSys.create(name, true, fileSys.getConf().getInt("io.file.buffer.size", 4096), (short)repl, 8192L);
        return stm;
    }

    public static HdfsDataOutputStream create(DistributedFileSystem dfs, Path name, int repl) throws IOException {
        return (HdfsDataOutputStream)TestFileCreation.createFile((FileSystem)dfs, name, repl);
    }

    static void writeFile(FSDataOutputStream stm) throws IOException {
        TestFileCreation.writeFile(stm, 16385);
    }

    public static void writeFile(FSDataOutputStream stm, int size) throws IOException {
        byte[] buffer = AppendTestUtil.randomBytes(3735928559L, size);
        stm.write(buffer, 0, size);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testServerDefaults() throws IOException {
        HdfsConfiguration conf = new HdfsConfiguration();
        conf.setLong("dfs.blocksize", 0x8000000L);
        conf.setInt("dfs.bytes-per-checksum", 512);
        conf.setInt("dfs.client-write-packet-size", 65536);
        conf.setInt("dfs.replication", 4);
        conf.setInt("io.file.buffer.size", 4096);
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).numDataNodes(4).build();
        cluster.waitActive();
        DistributedFileSystem fs = cluster.getFileSystem();
        try {
            FsServerDefaults serverDefaults = fs.getServerDefaults(new Path("/"));
            Assertions.assertEquals((long)0x8000000L, (long)serverDefaults.getBlockSize());
            Assertions.assertEquals((int)512, (int)serverDefaults.getBytesPerChecksum());
            Assertions.assertEquals((int)65536, (int)serverDefaults.getWritePacketSize());
            Assertions.assertEquals((int)4, (int)serverDefaults.getReplication());
            Assertions.assertEquals((int)4096, (int)serverDefaults.getFileBufferSize());
            Assertions.assertEquals((int)7, (int)serverDefaults.getDefaultStoragePolicyId());
        }
        finally {
            fs.close();
            cluster.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testServerDefaultsWithCaching() throws IOException, InterruptedException {
        HdfsConfiguration clusterConf = new HdfsConfiguration();
        long originalBlockSize = 0x10000000L;
        clusterConf.setLong("dfs.blocksize", originalBlockSize);
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)clusterConf).numDataNodes(0).build();
        cluster.waitActive();
        FSNamesystem spyNamesystem = NameNodeAdapterMockitoUtil.spyOnNamesystem(cluster.getNameNode());
        InetSocketAddress nameNodeAddr = cluster.getNameNode().getNameNodeAddress();
        try {
            HdfsConfiguration clientConf = new HdfsConfiguration();
            clientConf.setLong("dfs.client.server-defaults.validity.period.ms", TimeUnit.MINUTES.toMillis(1L));
            DFSClient dfsClient = new DFSClient(nameNodeAddr, (Configuration)clientConf);
            FsServerDefaults defaults = dfsClient.getServerDefaults();
            Assertions.assertEquals((long)originalBlockSize, (long)defaults.getBlockSize());
            long updatedDefaultBlockSize = 0x18000000L;
            FsServerDefaults newDefaults = new FsServerDefaults(updatedDefaultBlockSize, defaults.getBytesPerChecksum(), defaults.getWritePacketSize(), defaults.getReplication(), defaults.getFileBufferSize(), defaults.getEncryptDataTransfer(), defaults.getTrashInterval(), defaults.getChecksumType(), defaults.getKeyProviderUri(), defaults.getDefaultStoragePolicyId());
            ((FSNamesystem)Mockito.doReturn((Object)newDefaults).when((Object)spyNamesystem)).getServerDefaults();
            Thread.sleep(1L);
            defaults = dfsClient.getServerDefaults();
            Assertions.assertEquals((long)originalBlockSize, (long)defaults.getBlockSize());
            DFSClient newDfsClient = new DFSClient(nameNodeAddr, (Configuration)clientConf);
            defaults = newDfsClient.getServerDefaults();
            Assertions.assertEquals((long)updatedDefaultBlockSize, (long)defaults.getBlockSize());
        }
        finally {
            cluster.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testServerDefaultsWithMinimalCaching() throws Exception {
        HdfsConfiguration clusterConf = new HdfsConfiguration();
        long originalBlockSize = 0x10000000L;
        clusterConf.setLong("dfs.blocksize", originalBlockSize);
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)clusterConf).numDataNodes(0).build();
        cluster.waitActive();
        FSNamesystem spyNamesystem = NameNodeAdapterMockitoUtil.spyOnNamesystem(cluster.getNameNode());
        InetSocketAddress nameNodeAddr = cluster.getNameNode().getNameNodeAddress();
        try {
            HdfsConfiguration clientConf = new HdfsConfiguration();
            clientConf.setLong("dfs.client.server-defaults.validity.period.ms", 0L);
            DFSClient dfsClient = new DFSClient(nameNodeAddr, (Configuration)clientConf);
            FsServerDefaults defaults = dfsClient.getServerDefaults();
            Assertions.assertEquals((long)originalBlockSize, (long)defaults.getBlockSize());
            long updatedDefaultBlockSize = 0x18000000L;
            FsServerDefaults newDefaults = new FsServerDefaults(updatedDefaultBlockSize, defaults.getBytesPerChecksum(), defaults.getWritePacketSize(), defaults.getReplication(), defaults.getFileBufferSize(), defaults.getEncryptDataTransfer(), defaults.getTrashInterval(), defaults.getChecksumType(), defaults.getKeyProviderUri(), defaults.getDefaultStoragePolicyId());
            ((FSNamesystem)Mockito.doReturn((Object)newDefaults).when((Object)spyNamesystem)).getServerDefaults();
            GenericTestUtils.waitFor(() -> {
                try {
                    FsServerDefaults currDef = dfsClient.getServerDefaults();
                    return currDef.getBlockSize() == updatedDefaultBlockSize;
                }
                catch (IOException e) {
                    return false;
                }
            }, (long)1L, (long)6000L);
        }
        catch (Throwable throwable) {
            Mockito.reset((Object[])new FSNamesystem[]{spyNamesystem});
            cluster.shutdown();
            throw throwable;
        }
        Mockito.reset((Object[])new FSNamesystem[]{spyNamesystem});
        cluster.shutdown();
    }

    @Test
    public void testFileCreation() throws IOException {
        this.checkFileCreation(null, false);
    }

    @Test
    public void testFileCreationUsingHostname() throws IOException {
        Assumptions.assumeTrue((boolean)System.getProperty("os.name").startsWith("Linux"));
        this.checkFileCreation(null, true);
    }

    @Test
    public void testFileCreationSetLocalInterface() throws IOException {
        Assumptions.assumeTrue((boolean)System.getProperty("os.name").startsWith("Linux"));
        this.checkFileCreation("lo", false);
        try {
            this.checkFileCreation("bogus-interface", false);
            Assertions.fail((String)"Able to specify a bogus interface");
        }
        catch (UnknownHostException e) {
            Assertions.assertEquals((Object)"No such interface bogus-interface", (Object)e.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void checkFileCreation(String netIf, boolean useDnHostname) throws IOException {
        HdfsConfiguration conf = new HdfsConfiguration();
        if (netIf != null) {
            conf.set("dfs.client.local.interfaces", netIf);
        }
        conf.setBoolean("dfs.client.use.datanode.hostname", useDnHostname);
        if (useDnHostname) {
            conf.set("dfs.datanode.hostname", "localhost");
        }
        if (this.simulatedStorage) {
            SimulatedFSDataset.setFactory((Configuration)conf);
        }
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).checkDataNodeHostConfig(true).build();
        DistributedFileSystem fs = cluster.getFileSystem();
        try {
            Path path = new Path("/");
            System.out.println("Path : \"" + path.toString() + "\"");
            System.out.println(fs.getFileStatus(path).isDirectory());
            Assertions.assertTrue((boolean)fs.getFileStatus(path).isDirectory(), (String)"/ should be a directory");
            Path dir1 = new Path("/test_dir");
            fs.mkdirs(dir1);
            System.out.println("createFile: Creating " + dir1.getName() + " for overwrite of existing directory.");
            try {
                fs.create(dir1, true);
                fs.close();
                Assertions.assertTrue((boolean)false, (String)"Did not prevent directory from being overwritten.");
            }
            catch (FileAlreadyExistsException fileAlreadyExistsException) {
                // empty catch block
            }
            Path file1 = new Path("filestatus.dat");
            Path parent = file1.getParent();
            fs.mkdirs(parent);
            DistributedFileSystem dfs = fs;
            dfs.setQuota(file1.getParent(), 100L, 40960L);
            FSDataOutputStream stm = TestFileCreation.createFile((FileSystem)fs, file1, 1);
            Assertions.assertTrue((boolean)fs.getFileStatus(file1).isFile(), (String)(file1 + " should be a file"));
            System.out.println("Path : \"" + file1 + "\"");
            TestFileCreation.writeFile(stm);
            stm.close();
            long len = fs.getFileStatus(file1).getLen();
            Assertions.assertTrue((len == 16385L ? 1 : 0) != 0, (String)(file1 + " should be of size " + 16385 + " but found to be of size " + len));
            long diskSpace = dfs.getContentSummary(file1.getParent()).getLength();
            Assertions.assertEquals((long)16385L, (long)diskSpace, (String)(file1 + " should take " + 16385 + " bytes disk space but found to take " + diskSpace + " bytes"));
            if (this.simulatedStorage) {
                DataNode dn = cluster.getDataNodes().get(0);
                FsDatasetSpi<?> dataset = DataNodeTestUtils.getFSDataset(dn);
                Assertions.assertEquals((long)16385L, (long)dataset.getDfsUsed());
                Assertions.assertEquals((long)0x1FFFFFFBFFFL, (long)dataset.getRemaining());
            }
        }
        finally {
            cluster.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testDeleteOnExit() throws IOException {
        HdfsConfiguration conf = new HdfsConfiguration();
        if (this.simulatedStorage) {
            SimulatedFSDataset.setFactory((Configuration)conf);
        }
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).build();
        DistributedFileSystem fs = cluster.getFileSystem();
        LocalFileSystem localfs = FileSystem.getLocal((Configuration)conf);
        try {
            Path file1 = new Path("filestatus.dat");
            Path file2 = new Path("filestatus2.dat");
            Path file3 = new Path("filestatus3.dat");
            FSDataOutputStream stm1 = TestFileCreation.createFile((FileSystem)fs, file1, 1);
            FSDataOutputStream stm2 = TestFileCreation.createFile((FileSystem)fs, file2, 1);
            FSDataOutputStream stm3 = TestFileCreation.createFile((FileSystem)localfs, file3, 1);
            System.out.println("DeleteOnExit: Created files.");
            TestFileCreation.writeFile(stm1);
            TestFileCreation.writeFile(stm3);
            stm1.close();
            stm2.close();
            stm3.close();
            fs.deleteOnExit(file1);
            fs.deleteOnExit(file2);
            localfs.deleteOnExit(file3);
            fs.close();
            localfs.close();
            fs = null;
            localfs = null;
            fs = cluster.getFileSystem();
            localfs = FileSystem.getLocal((Configuration)conf);
            Assertions.assertTrue((!fs.exists(file1) ? 1 : 0) != 0, (String)(file1 + " still exists inspite of deletOnExit set."));
            Assertions.assertTrue((!fs.exists(file2) ? 1 : 0) != 0, (String)(file2 + " still exists inspite of deletOnExit set."));
            Assertions.assertTrue((!localfs.exists(file3) ? 1 : 0) != 0, (String)(file3 + " still exists inspite of deletOnExit set."));
            System.out.println("DeleteOnExit successful.");
        }
        finally {
            IOUtils.closeStream((Closeable)fs);
            IOUtils.closeStream((Closeable)localfs);
            cluster.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testOverwriteOpenForWrite() throws Exception {
        HdfsConfiguration conf = new HdfsConfiguration();
        SimulatedFSDataset.setFactory((Configuration)conf);
        conf.setBoolean("dfs.permissions.enabled", false);
        final MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).build();
        DistributedFileSystem fs = cluster.getFileSystem();
        UserGroupInformation otherUgi = UserGroupInformation.createUserForTesting((String)"testuser", (String[])new String[]{"testgroup"});
        FileSystem fs2 = (FileSystem)otherUgi.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws Exception {
                return FileSystem.get((Configuration)cluster.getConfiguration(0));
            }
        });
        String metricsName = RPC_DETAILED_METRICS + cluster.getNameNodePort();
        try {
            Path p = new Path("/testfile");
            FSDataOutputStream stm1 = fs.create(p);
            stm1.write(1);
            MetricsAsserts.assertCounter((String)"CreateNumOps", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)metricsName));
            try {
                fs2.create(p, false);
                Assertions.fail((String)"Did not throw!");
            }
            catch (IOException abce) {
                GenericTestUtils.assertExceptionContains((String)"Failed to CREATE_FILE", (Throwable)abce);
            }
            MetricsAsserts.assertCounter((String)"AlreadyBeingCreatedExceptionNumOps", (long)1L, (MetricsRecordBuilder)MetricsAsserts.getMetrics((String)metricsName));
            FSDataOutputStream stm2 = fs2.create(p, true);
            stm2.write(2);
            stm2.close();
            try {
                stm1.close();
                Assertions.fail((String)"Should have exception closing stm1 since it was deleted");
            }
            catch (IOException ioe) {
                GenericTestUtils.assertExceptionContains((String)"File does not exist", (Throwable)ioe);
            }
        }
        finally {
            IOUtils.closeStream((Closeable)fs);
            IOUtils.closeStream((Closeable)fs2);
            cluster.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFileCreationError1() throws IOException {
        HdfsConfiguration conf = new HdfsConfiguration();
        conf.setInt("dfs.namenode.heartbeat.recheck-interval", 1000);
        conf.setInt("dfs.heartbeat.interval", 1);
        if (this.simulatedStorage) {
            SimulatedFSDataset.setFactory((Configuration)conf);
        }
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).build();
        DistributedFileSystem fs = cluster.getFileSystem();
        cluster.waitActive();
        InetSocketAddress addr = new InetSocketAddress("localhost", cluster.getNameNodePort());
        DFSClient client = new DFSClient(addr, (Configuration)conf);
        try {
            DatanodeInfo[] info;
            Path file1 = new Path("/filestatus.dat");
            FSDataOutputStream stm = TestFileCreation.createFile((FileSystem)fs, file1, 1);
            Assertions.assertTrue((boolean)fs.getFileStatus(file1).isFile(), (String)(file1 + " should be a file"));
            System.out.println("Path : \"" + file1 + "\"");
            cluster.shutdownDataNodes();
            while ((info = client.datanodeReport(HdfsConstants.DatanodeReportType.LIVE)).length != 0) {
                System.out.println("testFileCreationError1: waiting for datanode  to die.");
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
            byte[] buffer = AppendTestUtil.randomBytes(3735928559L, 1);
            try {
                stm.write(buffer);
                stm.close();
            }
            catch (Exception e) {
                System.out.println("Encountered expected exception");
            }
            LocatedBlocks locations = client.getNamenode().getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
            System.out.println("locations = " + locations.locatedBlockCount());
            Assertions.assertTrue((locations.locatedBlockCount() == 0 ? 1 : 0) != 0, (String)"Error blocks were not cleaned up");
        }
        finally {
            cluster.shutdown();
            client.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFileCreationError2() throws IOException {
        long leasePeriod = 1000L;
        System.out.println("testFileCreationError2 start");
        HdfsConfiguration conf = new HdfsConfiguration();
        conf.setInt("dfs.namenode.heartbeat.recheck-interval", 1000);
        conf.setInt("dfs.heartbeat.interval", 1);
        if (this.simulatedStorage) {
            SimulatedFSDataset.setFactory((Configuration)conf);
        }
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).build();
        DistributedFileSystem dfs = null;
        try {
            cluster.waitActive();
            dfs = cluster.getFileSystem();
            DFSClient client = dfs.dfs;
            Path file1 = new Path("/filestatus.dat");
            TestFileCreation.createFile((FileSystem)dfs, file1, 1);
            System.out.println("testFileCreationError2: Created file filestatus.dat with one replicas.");
            LocatedBlocks locations = client.getNamenode().getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
            System.out.println("testFileCreationError2: The file has " + locations.locatedBlockCount() + " blocks.");
            LocatedBlock location = client.getNamenode().addBlock(file1.toString(), client.clientName, null, null, 0L, null, null);
            System.out.println("testFileCreationError2: Added block " + location.getBlock());
            locations = client.getNamenode().getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
            int count = locations.locatedBlockCount();
            System.out.println("testFileCreationError2: The file now has " + count + " blocks.");
            cluster.setLeasePeriod(leasePeriod, leasePeriod);
            try {
                Thread.sleep(5L * leasePeriod);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            locations = client.getNamenode().getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
            System.out.println("testFileCreationError2: locations = " + locations.locatedBlockCount());
            Assertions.assertEquals((int)0, (int)locations.locatedBlockCount());
            System.out.println("testFileCreationError2 successful");
        }
        catch (Throwable throwable) {
            IOUtils.closeStream(dfs);
            cluster.shutdown();
            throw throwable;
        }
        IOUtils.closeStream((Closeable)dfs);
        cluster.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFileCreationError3() throws IOException {
        System.out.println("testFileCreationError3 start");
        HdfsConfiguration conf = new HdfsConfiguration();
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).numDataNodes(0).build();
        DistributedFileSystem dfs = null;
        try {
            cluster.waitActive();
            dfs = cluster.getFileSystem();
            DFSClient client = dfs.dfs;
            Path f = new Path("/foo.txt");
            TestFileCreation.createFile((FileSystem)dfs, f, 3);
            try {
                cluster.getNameNodeRpc().addBlock(f.toString(), client.clientName, null, null, 0L, null, null);
                Assertions.fail();
            }
            catch (IOException ioe) {
                FileSystem.LOG.info("GOOD!", (Throwable)ioe);
            }
            System.out.println("testFileCreationError3 successful");
        }
        catch (Throwable throwable) {
            IOUtils.closeStream(dfs);
            cluster.shutdown();
            throw throwable;
        }
        IOUtils.closeStream((Closeable)dfs);
        cluster.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFileCreationNamenodeRestart() throws IOException, ReflectiveOperationException {
        HdfsConfiguration conf = new HdfsConfiguration();
        int MAX_IDLE_TIME = 2000;
        conf.setInt("ipc.client.connection.maxidletime", 2000);
        conf.setInt("dfs.namenode.heartbeat.recheck-interval", 1000);
        conf.setInt("dfs.heartbeat.interval", 1);
        if (this.simulatedStorage) {
            SimulatedFSDataset.setFactory((Configuration)conf);
        }
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).build();
        DistributedFileSystem fs = null;
        try {
            cluster.waitActive();
            fs = cluster.getFileSystem();
            int nnport = cluster.getNameNodePort();
            Path file1 = new Path("/filestatus.dat");
            HdfsDataOutputStream stm = TestFileCreation.create(fs, file1, 1);
            System.out.println("testFileCreationNamenodeRestart: Created file " + file1);
            Assertions.assertEquals((int)1, (int)stm.getCurrentBlockReplication(), (String)(file1 + " should be replicated to 1 datanode."));
            TestFileCreation.writeFile((FSDataOutputStream)stm, 16384);
            stm.hflush();
            Assertions.assertEquals((int)1, (int)stm.getCurrentBlockReplication(), (String)(file1 + " should still be replicated to 1 datanode."));
            Path fileRenamed = new Path("/filestatusRenamed.dat");
            fs.rename(file1, fileRenamed);
            System.out.println("testFileCreationNamenodeRestart: Renamed file " + file1 + " to " + fileRenamed);
            file1 = fileRenamed;
            Path file2 = new Path("/filestatus2.dat");
            FSDataOutputStream stm2 = TestFileCreation.createFile((FileSystem)fs, file2, 1);
            System.out.println("testFileCreationNamenodeRestart: Created file " + file2);
            Path file3 = new Path("/user/home/fullpath.dat");
            FSDataOutputStream stm3 = TestFileCreation.createFile((FileSystem)fs, file3, 1);
            System.out.println("testFileCreationNamenodeRestart: Created file " + file3);
            Path file4 = new Path("/user/home/fullpath4.dat");
            FSDataOutputStream stm4 = TestFileCreation.createFile((FileSystem)fs, file4, 1);
            System.out.println("testFileCreationNamenodeRestart: Created file " + file4);
            fs.mkdirs(new Path("/bin"));
            fs.rename(new Path("/user/home"), new Path("/bin"));
            Path file3new = new Path("/bin/home/fullpath.dat");
            System.out.println("testFileCreationNamenodeRestart: Renamed file " + file3 + " to " + file3new);
            Path file4new = new Path("/bin/home/fullpath4.dat");
            System.out.println("testFileCreationNamenodeRestart: Renamed file " + file4 + " to " + file4new);
            cluster.shutdown(false, false);
            try {
                Thread.sleep(4000L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            cluster = new MiniDFSCluster.Builder((Configuration)conf).nameNodePort(nnport).format(false).build();
            cluster.waitActive();
            cluster.shutdown(false, false);
            try {
                Thread.sleep(5000L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            cluster = new MiniDFSCluster.Builder((Configuration)conf).nameNodePort(nnport).format(false).build();
            cluster.waitActive();
            fs = cluster.getFileSystem();
            DFSOutputStream dfstream = (DFSOutputStream)stm.getWrappedStream();
            ReflectionUtils.setFinalField(DFSOutputStream.class, (Object)dfstream, (String)"src", (Object)file1.toString());
            dfstream = (DFSOutputStream)stm3.getWrappedStream();
            ReflectionUtils.setFinalField(DFSOutputStream.class, (Object)dfstream, (String)"src", (Object)file3new.toString());
            dfstream = (DFSOutputStream)stm4.getWrappedStream();
            ReflectionUtils.setFinalField(DFSOutputStream.class, (Object)dfstream, (String)"src", (Object)file4new.toString());
            byte[] buffer = AppendTestUtil.randomBytes(3735928559L, 1);
            stm.write(buffer);
            stm.close();
            stm2.write(buffer);
            stm2.close();
            stm3.close();
            stm4.close();
            DFSClient client = fs.dfs;
            LocatedBlocks locations = client.getNamenode().getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
            System.out.println("locations = " + locations.locatedBlockCount());
            Assertions.assertTrue((locations.locatedBlockCount() == 3 ? 1 : 0) != 0, (String)("Error blocks were not cleaned up for file " + file1));
            locations = client.getNamenode().getBlockLocations(file2.toString(), 0L, Long.MAX_VALUE);
            System.out.println("locations = " + locations.locatedBlockCount());
            Assertions.assertTrue((locations.locatedBlockCount() == 1 ? 1 : 0) != 0, (String)("Error blocks were not cleaned up for file " + file2));
        }
        catch (Throwable throwable) {
            IOUtils.closeStream(fs);
            cluster.shutdown();
            throw throwable;
        }
        IOUtils.closeStream((Closeable)fs);
        cluster.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testDFSClientDeath() throws IOException, InterruptedException {
        DistributedFileSystem fs;
        HdfsConfiguration conf = new HdfsConfiguration();
        System.out.println("Testing adbornal client death.");
        if (this.simulatedStorage) {
            SimulatedFSDataset.setFactory((Configuration)conf);
        }
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).build();
        DistributedFileSystem dfs = fs = cluster.getFileSystem();
        DFSClient dfsclient = dfs.dfs;
        try {
            Path file1 = new Path("/clienttest.dat");
            FSDataOutputStream stm = TestFileCreation.createFile((FileSystem)fs, file1, 1);
            System.out.println("Created file clienttest.dat");
            TestFileCreation.writeFile(stm);
            dfsclient.close();
            Assertions.assertTrue((boolean)AppendTestUtil.createHdfsWithDifferentUsername((Configuration)conf).exists(file1), (String)(file1 + " does not exist."));
        }
        finally {
            cluster.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFileCreationNonRecursive() throws IOException {
        HdfsConfiguration conf = new HdfsConfiguration();
        if (this.simulatedStorage) {
            SimulatedFSDataset.setFactory((Configuration)conf);
        }
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).build();
        DistributedFileSystem fs = cluster.getFileSystem();
        try {
            TestFileCreation.testFileCreationNonRecursive((FileSystem)fs);
        }
        finally {
            fs.close();
            cluster.shutdown();
        }
    }

    public static void testFileCreationNonRecursive(FileSystem fs) throws IOException {
        Path path = new Path("/" + Time.now() + "-testFileCreationNonRecursive");
        IOException expectedException = null;
        String nonExistDir = "/non-exist-" + Time.now();
        fs.delete(new Path(nonExistDir), true);
        EnumSet<CreateFlag> createFlag = EnumSet.of(CreateFlag.CREATE);
        Assertions.assertNull((Object)TestFileCreation.createNonRecursive(fs, path, 1, createFlag));
        expectedException = TestFileCreation.createNonRecursive(fs, new Path(path, "Create"), 1, createFlag);
        Assertions.assertTrue((expectedException != null && expectedException instanceof ParentNotDirectoryException ? 1 : 0) != 0, (String)"Create a file when parent directory exists as a file should throw ParentNotDirectoryException ");
        fs.delete(path, true);
        Path path2 = new Path(nonExistDir + "/testCreateNonRecursive");
        expectedException = TestFileCreation.createNonRecursive(fs, path2, 1, createFlag);
        Assertions.assertTrue((expectedException != null && expectedException instanceof FileNotFoundException ? 1 : 0) != 0, (String)"Create a file in a non-exist dir using createNonRecursive() should throw FileNotFoundException ");
        EnumSet<CreateFlag> overwriteFlag = EnumSet.of(CreateFlag.CREATE, CreateFlag.OVERWRITE);
        Assertions.assertNull((Object)TestFileCreation.createNonRecursive(fs, path, 1, overwriteFlag));
        expectedException = TestFileCreation.createNonRecursive(fs, new Path(path, "Overwrite"), 1, overwriteFlag);
        Assertions.assertTrue((expectedException != null && expectedException instanceof ParentNotDirectoryException ? 1 : 0) != 0, (String)"Overwrite a file when parent directory exists as a file should throw ParentNotDirectoryException ");
        fs.delete(path, true);
        Path path3 = new Path(nonExistDir + "/testOverwriteNonRecursive");
        expectedException = TestFileCreation.createNonRecursive(fs, path3, 1, overwriteFlag);
        Assertions.assertTrue((expectedException != null && expectedException instanceof FileNotFoundException ? 1 : 0) != 0, (String)"Overwrite a file in a non-exist dir using createNonRecursive() should throw FileNotFoundException ");
    }

    static IOException createNonRecursive(FileSystem fs, Path name, int repl, EnumSet<CreateFlag> flag) throws IOException {
        try {
            System.out.println("createNonRecursive: Attempting to create " + name + " with " + repl + " replica.");
            int bufferSize = fs.getConf().getInt("io.file.buffer.size", 4096);
            FSDataOutputStream stm = fs.createNonRecursive(name, FsPermission.getDefault(), flag, bufferSize, (short)repl, 8192L, null);
            stm.close();
        }
        catch (IOException e) {
            return e;
        }
        return null;
    }

    @Test
    public void testFileCreationSimulated() throws IOException {
        this.simulatedStorage = true;
        this.testFileCreation();
        this.simulatedStorage = false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testConcurrentFileCreation() throws IOException {
        HdfsConfiguration conf = new HdfsConfiguration();
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).build();
        try {
            int i;
            DistributedFileSystem fs = cluster.getFileSystem();
            Path[] p = new Path[]{new Path("/foo"), new Path("/bar")};
            FSDataOutputStream[] out = new FSDataOutputStream[]{fs.create(p[0]), fs.create(p[1])};
            for (i = 0; i < 100; ++i) {
                out[0].write(i);
                out[1].write(i);
            }
            out[0].close();
            while (i < 200) {
                out[1].write(i);
                ++i;
            }
            out[1].close();
            FSDataInputStream[] in = new FSDataInputStream[]{fs.open(p[0]), fs.open(p[1])};
            for (i = 0; i < 100; ++i) {
                Assertions.assertEquals((int)i, (int)in[0].read());
            }
            for (i = 0; i < 200; ++i) {
                Assertions.assertEquals((int)i, (int)in[1].read());
            }
        }
        finally {
            if (cluster != null) {
                cluster.shutdown();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFileCreationSyncOnClose() throws IOException {
        HdfsConfiguration conf = new HdfsConfiguration();
        conf.setBoolean("dfs.datanode.synconclose", true);
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).build();
        try {
            int i;
            DistributedFileSystem fs = cluster.getFileSystem();
            Path[] p = new Path[]{new Path("/foo"), new Path("/bar")};
            FSDataOutputStream[] out = new FSDataOutputStream[]{fs.create(p[0]), fs.create(p[1])};
            for (i = 0; i < 100; ++i) {
                out[0].write(i);
                out[1].write(i);
            }
            out[0].close();
            while (i < 200) {
                out[1].write(i);
                ++i;
            }
            out[1].close();
            FSDataInputStream[] in = new FSDataInputStream[]{fs.open(p[0]), fs.open(p[1])};
            for (i = 0; i < 100; ++i) {
                Assertions.assertEquals((int)i, (int)in[0].read());
            }
            for (i = 0; i < 200; ++i) {
                Assertions.assertEquals((int)i, (int)in[1].read());
            }
        }
        finally {
            if (cluster != null) {
                cluster.shutdown();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testLeaseExpireHardLimit() throws Exception {
        System.out.println("testLeaseExpireHardLimit start");
        long leasePeriod = 1000L;
        int DATANODE_NUM = 3;
        HdfsConfiguration conf = new HdfsConfiguration();
        conf.setInt("dfs.namenode.heartbeat.recheck-interval", 1000);
        conf.setInt("dfs.heartbeat.interval", 1);
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).numDataNodes(3).build();
        DistributedFileSystem dfs = null;
        try {
            cluster.waitActive();
            dfs = cluster.getFileSystem();
            String f = DIR + "foo";
            Path fpath = new Path(f);
            HdfsDataOutputStream out = TestFileCreation.create(dfs, fpath, 3);
            out.write("something".getBytes());
            out.hflush();
            int actualRepl = out.getCurrentBlockReplication();
            Assertions.assertTrue((actualRepl == 3 ? 1 : 0) != 0, (String)(f + " should be replicated to " + 3 + " datanodes."));
            cluster.setLeasePeriod(1000L, 1000L);
            try {
                Thread.sleep(5000L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            LocatedBlocks locations = dfs.dfs.getNamenode().getBlockLocations(f, 0L, Long.MAX_VALUE);
            Assertions.assertEquals((int)1, (int)locations.locatedBlockCount());
            LocatedBlock locatedblock = (LocatedBlock)locations.getLocatedBlocks().get(0);
            int successcount = 0;
            for (DatanodeInfoWithStorage datanodeinfo : locatedblock.getLocations()) {
                DataNode datanode = cluster.getDataNode(datanodeinfo.getIpcPort());
                ExtendedBlock blk = locatedblock.getBlock();
                try (BufferedReader in = new BufferedReader(new InputStreamReader(datanode.getFSDataset().getBlockInputStream(blk, 0L)));){
                    Assertions.assertEquals((Object)"something", (Object)in.readLine());
                    ++successcount;
                }
            }
            System.out.println("successcount=" + successcount);
            Assertions.assertTrue((successcount > 0 ? 1 : 0) != 0);
        }
        catch (Throwable throwable) {
            IOUtils.closeStream(dfs);
            cluster.shutdown();
            throw throwable;
        }
        IOUtils.closeStream((Closeable)dfs);
        cluster.shutdown();
        System.out.println("testLeaseExpireHardLimit successful");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFsClose() throws Exception {
        System.out.println("test file system close start");
        int DATANODE_NUM = 3;
        HdfsConfiguration conf = new HdfsConfiguration();
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).numDataNodes(3).build();
        DistributedFileSystem dfs = null;
        try {
            cluster.waitActive();
            dfs = cluster.getFileSystem();
            String f = DIR + "foofs";
            Path fpath = new Path(f);
            FSDataOutputStream out = TestFileCreation.createFile((FileSystem)dfs, fpath, 3);
            out.write("something".getBytes());
            dfs.close();
        }
        finally {
            System.out.println("testFsClose successful");
            cluster.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFsCloseAfterClusterShutdown() throws IOException {
        System.out.println("test testFsCloseAfterClusterShutdown start");
        int DATANODE_NUM = 3;
        HdfsConfiguration conf = new HdfsConfiguration();
        conf.setInt("dfs.namenode.replication.min", 3);
        conf.setBoolean("ipc.client.ping", false);
        conf.setInt("ipc.ping.interval", 10000);
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).numDataNodes(3).build();
        DistributedFileSystem dfs = null;
        try {
            cluster.waitActive();
            dfs = cluster.getFileSystem();
            String f = DIR + "testFsCloseAfterClusterShutdown";
            Path fpath = new Path(f);
            FSDataOutputStream out = TestFileCreation.createFile((FileSystem)dfs, fpath, 3);
            out.write("something_test".getBytes());
            out.hflush();
            cluster.stopDataNode(2);
            boolean hasException = false;
            try {
                out.close();
                System.out.println("testFsCloseAfterClusterShutdown: Error here");
            }
            catch (IOException e) {
                hasException = true;
            }
            Assertions.assertTrue((boolean)hasException, (String)"Failed to close file after cluster shutdown");
        }
        finally {
            System.out.println("testFsCloseAfterClusterShutdown successful");
            if (cluster != null) {
                cluster.shutdown();
            }
        }
    }

    @Test
    public void testCreateNonCanonicalPathAndRestartRpc() throws Exception {
        this.doCreateTest(CreationMethod.DIRECT_NN_RPC);
    }

    @Test
    public void testCreateNonCanonicalPathAndRestartFromString() throws Exception {
        this.doCreateTest(CreationMethod.PATH_FROM_STRING);
    }

    @Test
    public void testCreateNonCanonicalPathAndRestartFromUri() throws Exception {
        this.doCreateTest(CreationMethod.PATH_FROM_URI);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doCreateTest(CreationMethod method) throws Exception {
        HdfsConfiguration conf = new HdfsConfiguration();
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).numDataNodes(1).build();
        try {
            DistributedFileSystem fs = cluster.getFileSystem();
            NamenodeProtocols nnrpc = cluster.getNameNodeRpc();
            block9: for (String pathStr : NON_CANONICAL_PATHS) {
                System.out.println("Creating " + pathStr + " by " + (Object)((Object)method));
                switch (method) {
                    case DIRECT_NN_RPC: {
                        try {
                            nnrpc.create(pathStr, new FsPermission(493), "client", new EnumSetWritable(EnumSet.of(CreateFlag.CREATE)), true, (short)1, 0x8000000L, null, null, null);
                            Assertions.fail((String)("Should have thrown exception when creating '" + pathStr + "' by " + (Object)((Object)method)));
                        }
                        catch (InvalidPathException invalidPathException) {}
                        continue block9;
                    }
                    case PATH_FROM_URI: 
                    case PATH_FROM_STRING: {
                        Path p = method == CreationMethod.PATH_FROM_URI ? new Path(new URI(fs.getUri() + pathStr)) : new Path(fs.getUri() + pathStr);
                        FSDataOutputStream stm = fs.create(p);
                        IOUtils.closeStream((Closeable)stm);
                        continue block9;
                    }
                    default: {
                        throw new AssertionError((Object)("bad method: " + (Object)((Object)method)));
                    }
                }
            }
            cluster.restartNameNode(new String[0]);
        }
        finally {
            cluster.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFileIdMismatch() throws IOException {
        HdfsConfiguration conf = new HdfsConfiguration();
        MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).numDataNodes(3).build();
        DistributedFileSystem dfs = null;
        try {
            cluster.waitActive();
            dfs = cluster.getFileSystem();
            DFSClient client = dfs.dfs;
            Path f = new Path("/testFileIdMismatch.txt");
            TestFileCreation.createFile((FileSystem)dfs, f, 3);
            long someOtherFileId = -1L;
            try {
                cluster.getNameNodeRpc().complete(f.toString(), client.clientName, null, someOtherFileId);
                Assertions.fail();
            }
            catch (FileNotFoundException e) {
                FileSystem.LOG.info("Caught Expected FileNotFoundException: ", (Throwable)e);
            }
        }
        catch (Throwable throwable) {
            IOUtils.closeStream(dfs);
            cluster.shutdown();
            throw throwable;
        }
        IOUtils.closeStream((Closeable)dfs);
        cluster.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    @Timeout(value=120L)
    public void testFileCreationWithOverwrite() throws Exception {
        Configuration conf = new Configuration();
        conf.setInt("dfs.blocksize", 8192);
        MiniDFSCluster cluster = new MiniDFSCluster.Builder(conf).numDataNodes(3).build();
        DistributedFileSystem dfs = cluster.getFileSystem();
        try {
            dfs.mkdirs(new Path("/foo/dir"));
            String file = "/foo/dir/file";
            Path filePath = new Path(file);
            NameNode nn = cluster.getNameNode();
            FSNamesystem fsn = NameNodeAdapter.getNamesystem(nn);
            BlockManager bm = fsn.getBlockManager();
            byte[] oldData = AppendTestUtil.randomBytes(3735928559L, 16385);
            try (FSDataOutputStream out = dfs.create(filePath);){
                out.write(oldData);
            }
            LocatedBlocks oldBlocks = NameNodeAdapter.getBlockLocations(nn, file, 0L, 16385L);
            this.assertBlocks(bm, oldBlocks, true);
            out = dfs.create(filePath, true);
            BlockManagerTestUtil.waitForMarkedDeleteQueueIsEmpty(cluster.getNamesystem(0).getBlockManager());
            byte[] newData = AppendTestUtil.randomBytes(3735928559L, 16385);
            try {
                out.write(newData);
            }
            finally {
                out.close();
            }
            dfs.deleteOnExit(filePath);
            BlockManagerTestUtil.waitForMarkedDeleteQueueIsEmpty(cluster.getNamesystem(0).getBlockManager());
            LocatedBlocks newBlocks = NameNodeAdapter.getBlockLocations(nn, file, 0L, 16385L);
            this.assertBlocks(bm, newBlocks, true);
            this.assertBlocks(bm, oldBlocks, false);
            byte[] result = null;
            try (FSDataInputStream in = dfs.open(filePath);){
                result = this.readAll(in);
            }
            Assertions.assertArrayEquals((byte[])newData, (byte[])result);
            cluster.restartNameNode(new String[0]);
            nn = cluster.getNameNode();
            in = dfs.open(filePath);
            try {
                result = this.readAll(in);
            }
            finally {
                in.close();
            }
            Assertions.assertArrayEquals((byte[])newData, (byte[])result);
            NameNodeAdapter.enterSafeMode(nn, false);
            NameNodeAdapter.saveNamespace(nn);
            cluster.restartNameNode(new String[0]);
            nn = cluster.getNameNode();
            in = dfs.open(filePath);
            try {
                result = this.readAll(in);
            }
            finally {
                in.close();
            }
            Assertions.assertArrayEquals((byte[])newData, (byte[])result);
        }
        finally {
            if (dfs != null) {
                dfs.close();
            }
            if (cluster != null) {
                cluster.shutdown();
            }
        }
    }

    private void assertBlocks(BlockManager bm, LocatedBlocks lbs, boolean exist) {
        for (LocatedBlock locatedBlock : lbs.getLocatedBlocks()) {
            if (exist) {
                Assertions.assertTrue((bm.getStoredBlock(locatedBlock.getBlock().getLocalBlock()) != null ? 1 : 0) != 0);
                continue;
            }
            Assertions.assertTrue((bm.getStoredBlock(locatedBlock.getBlock().getLocalBlock()) == null ? 1 : 0) != 0);
        }
    }

    private byte[] readAll(FSDataInputStream in) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        byte[] buffer = new byte[1024];
        int n = 0;
        while ((n = in.read(buffer)) > -1) {
            out.write(buffer, 0, n);
        }
        return out.toByteArray();
    }

    private static enum CreationMethod {
        DIRECT_NN_RPC,
        PATH_FROM_URI,
        PATH_FROM_STRING;

    }
}

