/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSInputStream;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.client.HdfsClientConfigKeys;
import org.apache.hadoop.hdfs.protocol.DatanodeInfoWithStorage;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.util.Time;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;

public class TestDFSInputStreamBlockLocations {
    private static final int BLOCK_SIZE = 0x100000;
    private static final String[] RACKS = new String[]{"/d1/r1", "/d1/r1", "/d1/r2", "/d1/r2", "/d1/r2", "/d2/r3", "/d2/r3"};
    private static final int NUM_DATA_NODES = RACKS.length;
    private static final short REPLICATION_FACTOR = 4;
    private final int staleInterval = 8000;
    private final int numOfBlocks = 24;
    private final int fileLength = 0x1800000;
    private final int dfsClientPrefetchSize = 0xC00000;
    private final long dfsInputLocationsTimeout = 3600000L;
    private HdfsConfiguration conf;
    private MiniDFSCluster dfsCluster;
    private DFSClient dfsClient;
    private DistributedFileSystem fs;
    private Path filePath;
    private boolean enableBlkExpiration;

    public static Collection<Object[]> getTestParameters() {
        return Arrays.asList({Boolean.TRUE}, {Boolean.FALSE});
    }

    public void initTestDFSInputStreamBlockLocations(Boolean pEnableExpiration) throws IOException {
        this.enableBlkExpiration = pEnableExpiration;
        this.setup();
    }

    public void setup() throws IOException {
        this.conf = new HdfsConfiguration();
        this.conf.setBoolean("dfs.namenode.avoid.read.stale.datanode", true);
        this.conf.setLong("dfs.namenode.stale.datanode.interval", 8000L);
        this.conf.setInt("dfs.namenode.heartbeat.recheck-interval", 4000);
        this.conf.setBoolean(HdfsClientConfigKeys.Read.ShortCircuit.KEY, false);
        this.conf.setInt("dfs.replication", 4);
        this.conf.setLong("dfs.blocksize", 0x100000L);
        this.conf.setLong("dfs.client.read.prefetch.size", 0xC00000L);
        if (this.enableBlkExpiration) {
            this.conf.setLong("dfs.client.refresh.read-block-locations.ms", 3600000L);
        }
        this.dfsCluster = new MiniDFSCluster.Builder((Configuration)this.conf).numDataNodes(NUM_DATA_NODES).racks(RACKS).build();
        this.dfsCluster.waitActive();
        Assertions.assertEquals((int)NUM_DATA_NODES, (int)this.dfsCluster.getDataNodes().size());
        InetSocketAddress addr = new InetSocketAddress("localhost", this.dfsCluster.getNameNodePort());
        this.dfsClient = new DFSClient(addr, (Configuration)this.conf);
        this.fs = this.dfsCluster.getFileSystem();
    }

    @AfterEach
    public void teardown() throws IOException {
        if (this.dfsClient != null) {
            this.dfsClient.close();
            this.dfsClient = null;
        }
        if (this.fs != null) {
            this.fs.deleteOnExit(this.filePath);
            this.fs.close();
            this.fs = null;
        }
        if (this.dfsCluster != null) {
            this.dfsCluster.shutdown();
            this.dfsCluster = null;
        }
    }

    @MethodSource(value={"getTestParameters"})
    @ParameterizedTest
    public void testRefreshBlockLocations(Boolean pEnableExpiration) throws IOException {
        this.initTestDFSInputStreamBlockLocations(pEnableExpiration);
        String fileName = "/test_cache_locations";
        this.filePath = this.createFile("/test_cache_locations");
        try (DFSInputStream fin = this.dfsClient.open("/test_cache_locations");){
            LocatedBlocks existing = fin.locatedBlocks;
            long lastRefreshedAt = fin.getLastRefreshedBlocksAtForTesting();
            Assertions.assertFalse((boolean)fin.refreshBlockLocations(null), (String)"should not have attempted refresh");
            Assertions.assertEquals((long)lastRefreshedAt, (long)fin.getLastRefreshedBlocksAtForTesting(), (String)"should not have updated lastRefreshedAt");
            Assertions.assertSame((Object)existing, (Object)fin.locatedBlocks, (String)"should not have modified locatedBlocks");
            fin.addToLocalDeadNodes(this.dfsClient.datanodeReport(HdfsConstants.DatanodeReportType.LIVE)[0]);
            Assertions.assertTrue((boolean)fin.refreshBlockLocations(null), (String)"should have attempted refresh");
            this.verifyChanged(fin, existing, lastRefreshedAt);
            lastRefreshedAt = fin.getLastRefreshedBlocksAtForTesting();
            existing = fin.locatedBlocks;
            HashMap<String, InetSocketAddress> mockAddressCache = new HashMap<String, InetSocketAddress>();
            InetSocketAddress unresolved = InetSocketAddress.createUnresolved("www.google.com", 80);
            for (DataNode dataNode : this.dfsCluster.getDataNodes()) {
                mockAddressCache.put(dataNode.getDatanodeUuid(), unresolved);
            }
            Assertions.assertTrue((boolean)fin.refreshBlockLocations(mockAddressCache), (String)"should have attempted refresh");
            this.verifyChanged(fin, existing, lastRefreshedAt);
        }
    }

    private void verifyChanged(DFSInputStream fin, LocatedBlocks existing, long lastRefreshedAt) {
        Assertions.assertTrue((fin.getLastRefreshedBlocksAtForTesting() > lastRefreshedAt ? 1 : 0) != 0, (String)"lastRefreshedAt should have incremented");
        Assertions.assertNotSame((Object)existing, (Object)fin.locatedBlocks, (String)"located blocks should have changed");
        Assertions.assertTrue((boolean)fin.getLocalDeadNodes().isEmpty(), (String)"deadNodes should be empty");
    }

    @MethodSource(value={"getTestParameters"})
    @ParameterizedTest
    public void testDeferredRegistrationStatefulRead(Boolean pEnableExpiration) throws IOException {
        this.initTestDFSInputStreamBlockLocations(pEnableExpiration);
        this.testWithRegistrationMethod(DFSInputStream::read);
    }

    @MethodSource(value={"getTestParameters"})
    @ParameterizedTest
    public void testDeferredRegistrationPositionalRead(Boolean pEnableExpiration) throws IOException {
        this.initTestDFSInputStreamBlockLocations(pEnableExpiration);
        this.testWithRegistrationMethod(fin -> fin.readFully(0L, new byte[1]));
    }

    @MethodSource(value={"getTestParameters"})
    @ParameterizedTest
    public void testDeferredRegistrationGetAllBlocks(Boolean pEnableExpiration) throws IOException {
        this.initTestDFSInputStreamBlockLocations(pEnableExpiration);
        this.testWithRegistrationMethod(DFSInputStream::getAllBlocks);
    }

    @MethodSource(value={"getTestParameters"})
    @ParameterizedTest
    public void testClearIgnoreListChooseDataNode(Boolean pEnableExpiration) throws IOException {
        this.initTestDFSInputStreamBlockLocations(pEnableExpiration);
        String fileName = "/test_cache_locations";
        this.filePath = this.createFile("/test_cache_locations");
        try (DFSInputStream fin = this.dfsClient.open("/test_cache_locations");){
            LocatedBlocks existing = fin.locatedBlocks;
            LocatedBlock block = existing.getLastLocatedBlock();
            ArrayList<DatanodeInfoWithStorage> ignoreList = new ArrayList<DatanodeInfoWithStorage>(Arrays.asList(block.getLocations()));
            Assertions.assertNotNull((Object)fin.chooseDataNode(block, ignoreList, true));
            Assertions.assertEquals((int)0, (int)ignoreList.size());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testWithRegistrationMethod(ThrowingConsumer registrationMethod) throws IOException {
        String fileName = "/test_cache_locations";
        this.filePath = this.createFile("/test_cache_locations");
        DFSInputStream fin = null;
        try {
            fin = this.dfsClient.open("/test_cache_locations");
            Assertions.assertFalse((boolean)this.dfsClient.getLocatedBlockRefresher().isInputStreamTracked(fin), (String)"should not be tracking input stream on open");
            registrationMethod.accept(fin);
            Assertions.assertFalse((boolean)this.dfsClient.getLocatedBlockRefresher().isInputStreamTracked(fin), (String)"should not be tracking input stream after first read");
            fin.setLastRefreshedBlocksAtForTesting(Time.monotonicNow() - 3600001L);
            registrationMethod.accept(fin);
            Assertions.assertEquals((Object)this.enableBlkExpiration, (Object)this.dfsClient.getLocatedBlockRefresher().isInputStreamTracked(fin), (String)"SHOULD be tracking input stream on read after interval, only if enabled");
        }
        finally {
            if (fin != null) {
                fin.close();
                Assertions.assertFalse((boolean)this.dfsClient.getLocatedBlockRefresher().isInputStreamTracked(fin));
            }
            this.fs.delete(this.filePath, true);
        }
    }

    private Path createFile(String fileName) throws IOException {
        Path path = new Path(fileName);
        try (FSDataOutputStream fout = this.fs.create(path, (short)4);){
            fout.write(new byte[0x1800000]);
        }
        return path;
    }

    @FunctionalInterface
    static interface ThrowingConsumer {
        public void accept(DFSInputStream var1) throws IOException;
    }
}

