/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.blockmanagement;

import java.util.Queue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockCollection;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeAdminMonitorInterface;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeStorageInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.HeartbeatManager;
import org.apache.hadoop.hdfs.server.blockmanagement.NumberReplicas;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.Namesystem;
import org.apache.hadoop.net.Node;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.ReflectionUtils;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class DatanodeAdminManager {
    private static final Logger LOG = LoggerFactory.getLogger(DatanodeAdminManager.class);
    private final Namesystem namesystem;
    private final BlockManager blockManager;
    private final HeartbeatManager hbManager;
    private final ScheduledExecutorService executor;
    private DatanodeAdminMonitorInterface monitor = null;

    DatanodeAdminManager(Namesystem namesystem, BlockManager blockManager, HeartbeatManager hbManager) {
        this.namesystem = namesystem;
        this.blockManager = blockManager;
        this.hbManager = hbManager;
        this.executor = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat("DatanodeAdminMonitor-%d").setDaemon(true).build());
    }

    void activate(Configuration conf) {
        int intervalSecs = (int)conf.getTimeDuration("dfs.namenode.decommission.interval", 30L, TimeUnit.SECONDS);
        Preconditions.checkArgument((intervalSecs >= 0 ? 1 : 0) != 0, (Object)"Cannot set a negative value for dfs.namenode.decommission.interval");
        Class cls = null;
        try {
            cls = conf.getClass("dfs.namenode.decommission.monitor.class", Class.forName("org.apache.hadoop.hdfs.server.blockmanagement.DatanodeAdminDefaultMonitor"));
            this.monitor = (DatanodeAdminMonitorInterface)ReflectionUtils.newInstance((Class)cls, (Configuration)conf);
            this.monitor.setBlockManager(this.blockManager);
            this.monitor.setNameSystem(this.namesystem);
            this.monitor.setDatanodeAdminManager(this);
        }
        catch (Exception e) {
            throw new RuntimeException("Unable to create the Decommission monitor from " + cls, e);
        }
        this.executor.scheduleWithFixedDelay(this.monitor, intervalSecs, intervalSecs, TimeUnit.SECONDS);
        LOG.debug("Activating DatanodeAdminManager with interval {} seconds.", (Object)intervalSecs);
    }

    void close() {
        this.executor.shutdownNow();
        try {
            this.executor.awaitTermination(3000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    @VisibleForTesting
    public void startDecommission(DatanodeDescriptor node) {
        if (!node.isDecommissionInProgress() && !node.isDecommissioned()) {
            this.hbManager.startDecommission(node);
            this.blockManager.getDatanodeManager().getNetworkTopology().decommissionNode((Node)node);
            if (node.isDecommissionInProgress()) {
                for (DatanodeStorageInfo storage : node.getStorageInfos()) {
                    LOG.info("Starting decommission of {} {} with {} blocks", new Object[]{node, storage, storage.numBlocks()});
                }
                node.getLeavingServiceStatus().setStartTime(Time.monotonicNow());
                this.monitor.startTrackingNode(node);
            }
        } else {
            LOG.trace("startDecommission: Node {} in {}, nothing to do.", (Object)node, (Object)node.getAdminState());
        }
    }

    @VisibleForTesting
    public void stopDecommission(DatanodeDescriptor node) {
        if (node.isDecommissionInProgress() || node.isDecommissioned()) {
            this.hbManager.stopDecommission(node);
            this.blockManager.getDatanodeManager().getNetworkTopology().recommissionNode((Node)node);
            if (node.isAlive()) {
                this.blockManager.processExtraRedundancyBlocksOnInService(node);
            }
            this.monitor.stopTrackingNode(node);
        } else {
            LOG.trace("stopDecommission: Node {} in {}, nothing to do.", (Object)node, (Object)node.getAdminState());
        }
    }

    @VisibleForTesting
    public void startMaintenance(DatanodeDescriptor node, long maintenanceExpireTimeInMS) {
        node.setMaintenanceExpireTimeInMS(maintenanceExpireTimeInMS);
        if (!node.isMaintenance()) {
            this.hbManager.startMaintenance(node);
            if (node.isEnteringMaintenance()) {
                for (DatanodeStorageInfo storage : node.getStorageInfos()) {
                    LOG.info("Starting maintenance of {} {} with {} blocks", new Object[]{node, storage, storage.numBlocks()});
                }
                node.getLeavingServiceStatus().setStartTime(Time.monotonicNow());
            }
            this.monitor.startTrackingNode(node);
        } else {
            LOG.trace("startMaintenance: Node {} in {}, nothing to do.", (Object)node, (Object)node.getAdminState());
        }
    }

    @VisibleForTesting
    public void stopMaintenance(DatanodeDescriptor node) {
        if (node.isMaintenance()) {
            this.hbManager.stopMaintenance(node);
            if (!node.isAlive()) {
                this.blockManager.removeBlocksAssociatedTo(node);
            } else {
                this.blockManager.processExtraRedundancyBlocksOnInService(node);
            }
            this.monitor.stopTrackingNode(node);
        } else {
            LOG.trace("stopMaintenance: Node {} in {}, nothing to do.", (Object)node, (Object)node.getAdminState());
        }
    }

    protected void setDecommissioned(DatanodeDescriptor dn) {
        dn.setDecommissioned();
        LOG.info("Decommissioning complete for node {}", (Object)dn);
    }

    protected void setInMaintenance(DatanodeDescriptor dn) {
        dn.setInMaintenance();
        LOG.info("Node {} has entered maintenance mode.", (Object)dn);
    }

    protected boolean isSufficient(BlockInfo block, BlockCollection bc, NumberReplicas numberReplicas, boolean isDecommission, boolean isMaintenance) {
        if (this.blockManager.hasEnoughEffectiveReplicas(block, numberReplicas, 0)) {
            LOG.trace("Block {} does not need replication.", (Object)block);
            return true;
        }
        short numExpected = this.blockManager.getExpectedLiveRedundancyNum(block, numberReplicas);
        int numLive = numberReplicas.liveReplicas();
        LOG.trace("Block {} numExpected={}, numLive={}", new Object[]{block, (int)numExpected, numLive});
        if (isDecommission && numExpected > numLive) {
            if (bc.isUnderConstruction() && block.equals((Object)bc.getLastBlock())) {
                if (this.blockManager.hasMinStorage(block, numLive)) {
                    LOG.trace("UC block {} sufficiently-replicated since numLive ({}) >= minR ({})", new Object[]{block, numLive, this.blockManager.getMinStorageNum(block)});
                    return true;
                }
                LOG.trace("UC block {} insufficiently-replicated since numLive ({}) < minR ({})", new Object[]{block, numLive, this.blockManager.getMinStorageNum(block)});
            } else if (numLive >= this.blockManager.getDefaultStorageNum(block)) {
                return true;
            }
        }
        return isMaintenance && numLive >= this.blockManager.getMinMaintenanceStorageNum(block);
    }

    protected void logBlockReplicationInfo(BlockInfo block, BlockCollection bc, DatanodeDescriptor srcNode, NumberReplicas num, Iterable<DatanodeStorageInfo> storages) {
        if (!NameNode.blockStateChangeLog.isInfoEnabled()) {
            return;
        }
        int curReplicas = num.liveReplicas();
        short curExpectedRedundancy = this.blockManager.getExpectedRedundancyNum(block);
        StringBuilder nodeList = new StringBuilder();
        for (DatanodeStorageInfo storage : storages) {
            DatanodeDescriptor node = storage.getDatanodeDescriptor();
            nodeList.append((Object)node).append(' ');
        }
        NameNode.blockStateChangeLog.info("Block: " + block + ", Expected Replicas: " + curExpectedRedundancy + ", live replicas: " + curReplicas + ", corrupt replicas: " + num.corruptReplicas() + ", decommissioned replicas: " + num.decommissioned() + ", decommissioning replicas: " + num.decommissioning() + ", maintenance replicas: " + num.maintenanceReplicas() + ", live entering maintenance replicas: " + num.liveEnteringMaintenanceReplicas() + ", replicas on stale nodes: " + num.replicasOnStaleNodes() + ", readonly replicas: " + num.readOnlyReplicas() + ", excess replicas: " + num.excessReplicas() + ", Is Open File: " + bc.isUnderConstruction() + ", Datanodes having this block: " + nodeList + ", Current Datanode: " + srcNode + ", Is current datanode decommissioning: " + srcNode.isDecommissionInProgress() + ", Is current datanode entering maintenance: " + srcNode.isEnteringMaintenance());
    }

    @VisibleForTesting
    public int getNumPendingNodes() {
        return this.monitor.getPendingNodeCount();
    }

    @VisibleForTesting
    public int getNumTrackedNodes() {
        return this.monitor.getTrackedNodeCount();
    }

    @VisibleForTesting
    public int getNumNodesChecked() {
        return this.monitor.getNumNodesChecked();
    }

    @VisibleForTesting
    public Queue<DatanodeDescriptor> getPendingNodes() {
        return this.monitor.getPendingNodes();
    }

    @VisibleForTesting
    void runMonitorForTest() throws ExecutionException, InterruptedException {
        this.executor.submit(this.monitor).get();
    }

    public void refreshPendingRepLimit(int pendingRepLimit, String key) {
        this.ensurePositiveInt(pendingRepLimit, key);
        this.monitor.setPendingRepLimit(pendingRepLimit);
    }

    @VisibleForTesting
    public int getPendingRepLimit() {
        return this.monitor.getPendingRepLimit();
    }

    public void refreshBlocksPerLock(int blocksPerLock, String key) {
        this.ensurePositiveInt(blocksPerLock, key);
        this.monitor.setBlocksPerLock(blocksPerLock);
    }

    @VisibleForTesting
    public int getBlocksPerLock() {
        return this.monitor.getBlocksPerLock();
    }

    private void ensurePositiveInt(int val, String key) {
        Preconditions.checkArgument((val > 0 ? 1 : 0) != 0, (Object)(key + " = '" + val + "' is invalid. It should be a positive, non-zero integer value."));
    }
}

